/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		ReflectorSession.cpp

	Contains:	Implementation of object defined in ReflectorSession.h. 
					
	$Log: ReflectorSession.cpp,v $
	Revision 1.1  1999/02/19 23:10:05  ds
	Created
	

*/

#include "ReflectorSession.h"
#include "OS.h"
#include "RTCPPacket.h"
#include "SocketUtils.h"

#pragma mark _REFLECTOR_SESSION_

ReflectorSession::ReflectorSession(SDPParser* inParser, StrPtrLen* inUniqueID)
: 	fMutex('rarm'),
	fNumStreams(inParser->GetNumStreams()),
 	fSessionArray(NULL),
	fNumBuckets(kMinNumBuckets),
	fFirstEmptyBucket(1),
	fNumElements(0),
	fNumPlaying(0)
{
	//Store a separate copy of the URL in order to identify this session.
	fURL = new ('rssp') char[inUniqueID->Len];
	::memcpy(fURL, inUniqueID->Ptr, inUniqueID->Len);
	StrPtrLen tempPtr(fURL, inUniqueID->Len);
	fRef.Set(tempPtr, this);
			
	//Get the settings for bucket size and delay time
	fClientSpacing = RTSPServerInterface::GetRTSPPrefs()->GetReflectorDelayTimeInMilSecs();
	fBucketSize = 	RTSPServerInterface::GetRTSPPrefs()->GetReflectorBucketSize();
	
	//allocate the session array
	this->AllocateSessionArray(fNumBuckets);
		
	fStreamArray = new ('rfsa') ReflectorStream*[fNumStreams];
	for (UInt32 x = 0; x < fNumStreams; x++)
	{
		SDPParser::StreamInfo* theStreamInfoP = inParser->GetStreamInfo(x);
		fStreamArray[x] = new ('rfsm') ReflectorStream(this, theStreamInfoP->fIPAddr, 
														theStreamInfoP->fPort,
														theStreamInfoP->fTimeToLive, x+1,
														theStreamInfoP->fCodecType,
														&theStreamInfoP->fCodecName);
	}	
}

ReflectorSession::~ReflectorSession()
{
	Assert(fNumElements == 0);

	for (UInt32 x = 0; x < fNumStreams; x++)
		delete fStreamArray[x];
	delete [] fStreamArray;
	delete [] fURL;

	//delete every client Bucket
	for (UInt32 y = 0; y < fNumBuckets; y++)
		delete [] fSessionArray[y];
	delete [] fSessionArray;
}

void ReflectorSession::AddSession(RTPSession* inSession)
{
	OSMutexLocker locker(&fMutex);
	if (fNumElements == (fBucketSize * fNumBuckets))
		this->AllocateSessionArray(fNumBuckets * 2);

	Assert(fNumElements < fBucketSize * fNumBuckets);

	//find the first open spot in the array
	for (UInt32 x = 0; x < fNumBuckets; x++)
	{
		for(UInt32 y = 0; y < fBucketSize; y++)
		{
			if (fSessionArray[x][y] == NULL)
			{
				fSessionArray[x][y] = inSession;
				//if this session is filling a bucket that was
				//previously empty, make sure to track that!
				if (x >= fFirstEmptyBucket)
					fFirstEmptyBucket = x+1;
				fNumElements++;
				return;
			}
		}
	}
	Assert(0);
}

void  ReflectorSession::RemoveSession(RTPSession* inSession)
{
	OSMutexLocker locker(&fMutex);
	Assert(fNumElements > 0);
	
	//look at all the indexes in the array
	for (UInt32 x = 0; x < fNumBuckets; x++)
	{
		for (UInt32 y = 0; y < fBucketSize; y++)
		{
			//The array may have blank spaces!
			if (fSessionArray[x][y] == inSession)
			{
				//if this session is playing, make sure to decrement the playing count.
				//we need to do this atomically because other manipulations of this variable
				//aren't protected by a mutex
				if (inSession->GetState() == RTPSession::kPlayingState)
					atomic_add(&fNumPlaying, -1);

				fSessionArray[x][y] = NULL;//just clear out the pointer
				
				//check to see if eliminating this session has created a new empty
				//bucket. This is important to track because it eases the
				//buffering / smoothing burden
				if (fFirstEmptyBucket == (x+1))
				{
					bool isEmpty = true;
					for (UInt32 z = 0; z < fBucketSize; z++)
						if (fSessionArray[x][z] != NULL)
							isEmpty = false;
					if (isEmpty)
						fFirstEmptyBucket--;
				}
				
				fNumElements--;
				return;				
			}
		}
	}
	Assert(0);
}

void ReflectorSession::AllocateSessionArray(UInt32 inNumBuckets)
{
	Bucket* oldArray = fSessionArray;
	//allocate the 2-dimensional array
	fSessionArray = new ('buck') Bucket[inNumBuckets];
	for (UInt32 x = 0; x < inNumBuckets; x++)
	{
		fSessionArray[x] = new ('rsma') RTPSession*[fBucketSize];
		::memset(fSessionArray[x], 0, sizeof(RTPSession*) * fBucketSize);
	}
	
	//copy over the old information if there was an old array
	if (oldArray != NULL)
	{
		Assert(inNumBuckets > fNumBuckets);
		for (UInt32 y = 0; y < fNumBuckets; y++)
		{
			::memcpy(fSessionArray[y],oldArray[y], fBucketSize * sizeof(RTPSession*));
			delete [] oldArray[y];
		}
		delete [] oldArray;
	}
	fNumBuckets = inNumBuckets;
}

RTSPProtocol::RTSPStatusCode ReflectorSession::BindReflectorSockets(RTSPRequestInterface* inRTSPRequest)
{
	for (UInt32 x = 0; x < fNumStreams; x++)
	{
		RTSPProtocol::RTSPStatusCode error = fStreamArray[x]->BindSockets(inRTSPRequest);
		if (error != RTSPProtocol::kSuccessOK)
			return error;
	}
	return RTSPProtocol::kSuccessOK;
}

void	ReflectorSession::CloseSourceSockets()
{
	for (UInt32 x = 0; x < fNumStreams; x++)
		fStreamArray[x]->CloseSockets();
}

#pragma mark __REFLECTOR_STREAM__

ReflectorSocketPool	ReflectorStream::sSocketPool;

ReflectorStream::ReflectorStream(ReflectorSession* inSession, UInt32 inSrcAddr,
									UInt16 inSrcPort, UInt16 inTimeToLive, 
									UInt32 inStreamID, UInt32 inCodecType,
									StrPtrLen* inCodecName)
: 	fSockets(NULL),
	fSession(inSession),
	fRTPSender(this, inSession, false, inStreamID),
	fRTCPSender(this, inSession, true, inStreamID),
	fCodecType(inCodecType),
	fAddr(inSrcAddr),
	fPort(inSrcPort),
	fTtl(inTimeToLive)
{
	//copy the codec name to a new, local buffer
	if ((inCodecName != NULL) && (inCodecName->Len > 0))
	{
		fCodecName.Ptr = new ('CODE') char[inCodecName->Len + 2];
		::memcpy(fCodecName.Ptr, inCodecName->Ptr, inCodecName->Len);
		fCodecName.Ptr[inCodecName->Len] = '\0';
		fCodecName.Len = inCodecName->Len;
	}
	
	//write as much of the RTCP RR as is possible right now (most of it never changes)
	UInt32 theSsrc = (UInt32)::rand();
	char theTempCName[RTPServerInterface::kMaxCNameLen];
	UInt32 cNameLen = RTPServerInterface::GetACName(theTempCName);
	
	//write the RR (just header + ssrc)
	UInt32* theRRWriter = (UInt32*)&fReceiverReportBuffer[0];
	*theRRWriter = htonl(0x80c90001);
	theRRWriter++;
	*theRRWriter = htonl(theSsrc);
	theRRWriter++;

	//SDES length is the length of the CName, plus 2 32bit words, minus 1
	*theRRWriter = htonl(0x81ca0000 + (cNameLen >> 2) + 1);
	theRRWriter++;
	*theRRWriter = htonl(theSsrc);
	theRRWriter++;
	::memcpy(theRRWriter, theTempCName, cNameLen);
	theRRWriter += cNameLen >> 2;
	
	//APP packet format, QTSS specific stuff
	*theRRWriter = htonl(0x80cc0008);
	theRRWriter++;
	*theRRWriter = htonl(theSsrc);
	theRRWriter++;
	*theRRWriter = 'QTSS';
	theRRWriter++;
	*theRRWriter = htonl(0);
	theRRWriter++;
	*theRRWriter = htonl(0x00000004);
	theRRWriter++;
	*theRRWriter = htonl(0x6579000c);
	theRRWriter++;
	
	fEyeLocation = theRRWriter;
	fReceiverReportSize = kReceiverReportSize + kAppSize + cNameLen;
}

RTSPProtocol::RTSPStatusCode ReflectorStream::BindSockets(RTSPRequestInterface* inRTSPRequest)
{
	//get a pair of sockets
	fSockets = sSocketPool.GetUDPSocketPair(INADDR_ANY, fPort, fAddr, fPort);
	if (fSockets == NULL)
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kServerInternal,
												RTSPMessages::kCantBindReflectorSocket);
#if !OS_SUPPORTS_UDP_DEST_ADDR
	//In order to have multiple multicasts on the same port pair, we
	//need to be able to demux based on destination addr, which is not
	//always supported by sockets implementations
 	if (((ReflectorSocket*)fSockets->GetSocketA())->HasSender())
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kServerInternal,
												RTSPMessages::kCantBindReflectorSocket);
#endif 		
	
	//If the broadcaster is sending RTP directly to us, we don't
	//need to join a multicast group because we're not using multicast
	if (SocketUtils::IsMulticastIPAddr(fAddr))
	{
		QTSS_ErrorCode err = fSockets->GetSocketA()->JoinMulticast(fAddr, fTtl);
		if (err == QTSS_NoErr)
			err = fSockets->GetSocketB()->JoinMulticast(fAddr, fTtl);
		if (err != QTSS_NoErr)
			return inRTSPRequest->SendErrorResponse(RTSPProtocol::kServerInternal,
												RTSPMessages::kCantJoinMulticastGroup);
	}

	//also put this stream onto the socket's queue of streams
	((ReflectorSocket*)fSockets->GetSocketA())->AddSender(&fRTPSender);
	((ReflectorSocket*)fSockets->GetSocketB())->AddSender(&fRTCPSender);
	return RTSPProtocol::kSuccessOK;
}

void ReflectorStream::CloseSockets()
{
	if (fSockets != NULL)
	{
		//first things first, let's take this stream off the socket's queue
		//of streams. This will basically ensure that no reflecting activity
		//can happen on this stream.
		((ReflectorSocket*)fSockets->GetSocketA())->RemoveSender(&fRTPSender);
		((ReflectorSocket*)fSockets->GetSocketB())->RemoveSender(&fRTCPSender);
		
		//leave the multicast group. Because this socket is shared amongst several
		//potential multicasts, we don't want to remain a member of a stale multicast
		if (SocketUtils::IsMulticastIPAddr(fAddr))
		{
			fSockets->GetSocketA()->LeaveMulticast(fAddr);
			fSockets->GetSocketB()->LeaveMulticast(fAddr);
		}
		//now release the socket pair
		sSocketPool.ReleaseUDPSocketPair(fSockets);
	}
	fSockets = NULL;
}

void ReflectorStream::SendReceiverReport()
{
	//write the current eye count information based on the array
	//this is not really pre-emptive safe, but who cares... it's just statistics
	UInt32 theNumPlaying = fSession->fNumPlaying;
	UInt32 theNumElements = fSession->fNumElements;
	
	if (theNumPlaying > theNumElements)
		theNumPlaying = theNumElements;
		
	UInt32* theEyeWriter = fEyeLocation;
	*theEyeWriter = htonl(theNumElements) & 0x7fffffff;//no idea why we do this!
	theEyeWriter++;
	*theEyeWriter = htonl(theNumPlaying) & 0x7fffffff;
	theEyeWriter++;
	*theEyeWriter = htonl(theNumElements - theNumPlaying) & 0x7fffffff;
	
	//send the packet to the multicast RTCP addr & port for this stream
	(void)fSockets->GetSocketB()->SendTo(fAddr, fPort + 1, fReceiverReportBuffer, fReceiverReportSize);
}


ReflectorSender::ReflectorSender(ReflectorStream* inStream, ReflectorSession* inSession, 
									bool isRTCP, UInt32 inStreamID)
: 	fStream(inStream),
	fSession(inSession),
	fIsRTCP(isRTCP),
	fStreamID(inStreamID),
	fSocketQueueElem(this),
	fHasNewPackets(false),
	fNextTimeToRun(0),
	fLastRRTime(0)
{}

ReflectorSender::~ReflectorSender()
{
	//dequeue and delete every buffer
	while (fPacketQueue.GetLength() > 0)
	{
		ReflectorPacket* packet = (ReflectorPacket*)fPacketQueue.DeQueue()->GetEnclosingObject();
		delete packet;
	}
}

bool ReflectorSender::ShouldReflectNow(const SInt64& inCurrentTime, SInt64* ioWakeupTime)
{
	Assert(ioWakeupTime != NULL);
	//check to make sure there actually is work to do for this stream.
	if ((!fHasNewPackets) && ((fNextTimeToRun == 0) || (inCurrentTime < fNextTimeToRun)))
	{
		//We don't need to do work right now, but
		//this stream must still communicate when it needs to be woken up next
		SInt64 theWakeupTime = fNextTimeToRun - inCurrentTime;
		if ((fNextTimeToRun > 0) && (theWakeupTime < *ioWakeupTime))
			*ioWakeupTime = theWakeupTime;
#if !OS_SUPPORTS_UDP_DEST_ADDR
		//if this is the case, only one sender per socket is allowed, and the 1
		//sender should ALWAYS have work to do!
		if (!fIsRTCP)
			Assert(0);
#endif
		return false;
	}
	return true;	
}

void ReflectorSender::ReflectPackets(SInt64* ioWakeupTime, OSQueue* inFreeQueue)
{
#if DEBUG
	Assert(ioWakeupTime != NULL);
	bool didSomething = false;
#endif

	//make sure to reset these state variables
	fHasNewPackets = false;
	fNextTimeToRun = 0;
	
	SInt64 currentTime = OS::Milliseconds();
	
	//determine if we need to send a receiver report to the multicast source
	if ((fIsRTCP) && (currentTime > (fLastRRTime + kRRInterval)))
	{
		fLastRRTime = currentTime;
		fStream->SendReceiverReport();
	}
	
	//the rest of this function must be atomic wrt the ReflectorSession, because
	//it involves iterating through the RTPSession array, which isn't thread safe
	OSMutexLocker locker(&fSession->fMutex);

	//iterate through the array of packets, sending any data that needs to be sent at this time
	for (OSQueueIter qIter(&fPacketQueue); !qIter.IsDone(); )
	{
		ReflectorPacket* thePacket = (ReflectorPacket*)qIter.GetCurrent()->GetEnclosingObject();
			
		//based on currentTime - arrivalTime, we can figure out how many clients to reflect this
		//packet to
		SInt64 timeElapsed = currentTime - thePacket->fTimeArrived;
		Assert(timeElapsed >= 0);
		
		SInt64 numToSendTo = 1 + (timeElapsed / fSession->fClientSpacing);
		UInt32 bucketsToSendTo = (UInt32)numToSendTo;
		//make sure to adjust this value down to the number of actual partially
		//filled buckets!
		if (fSession->fFirstEmptyBucket < bucketsToSendTo)
			bucketsToSendTo = fSession->fFirstEmptyBucket;

		//loop through all the streams that haven't sent this packet yet, sending it.
		for (UInt32 x = thePacket->fBucketsSeenThisPacket; x < bucketsToSendTo; x++)
		{
#if DEBUG
			didSomething = true;
#endif
				
			for (UInt32 y = 0; y < fSession->fBucketSize; y++)
			{			
				RTPSession* theSession = fSession->fSessionArray[x][y];
				if ((theSession != NULL) && (theSession->GetState() == RTPSession::kPlayingState))
				{
					//make sure all RTP streams with this ID see this packet
					for (OSQueueIter streamIter(theSession->GetStreamQueue());
							!streamIter.IsDone(); streamIter.Next())
					{
						RTPStream* theStream = (RTPStream*)streamIter.GetCurrent()->GetEnclosingObject();
						if (theStream->GetStreamID() == fStreamID)
						{
							//We may have thinned this client in the past. If that's the case,
							//the sequence number of this packet needs to be adjusted, but only
							//for this client, so we need to make sure we can restore the old sequence
							//number after we mess around with it.
							UInt16 theSeqNumber = this->GetPacketSeqNumber(thePacket->fPacketPtr);
							
							if (fIsRTCP)
								theStream->SendRTCP(thePacket->fPacketPtr.Ptr, thePacket->fPacketPtr.Len);
							else if (!this->PacketShouldBeThinned(theStream, thePacket->fPacketPtr))
								theStream->SendRTP(thePacket->fPacketPtr.Ptr, thePacket->fPacketPtr.Len);
							
							//Reset the old sequence number
							this->SetPacketSeqNumber(thePacket->fPacketPtr, theSeqNumber);
						}
					}
				}
			}
			thePacket->fBucketsSeenThisPacket++;
		}
		
		//at this point, move onto the next queue element, because we may be altering
		//the queue itself in the code below
		qIter.Next();
		
		//alter time to send based on the timestamp of this packet, but only if there
		//are clients of this packet that haven't received it yet
		if (thePacket->fBucketsSeenThisPacket < fSession->fFirstEmptyBucket)
		{
			SInt64 nextTimeForThisPacket = fSession->fClientSpacing -
											(timeElapsed % fSession->fClientSpacing);
			if (fNextTimeToRun == 0)
				fNextTimeToRun = nextTimeForThisPacket;
			else if (fNextTimeToRun > nextTimeForThisPacket)
				fNextTimeToRun = nextTimeForThisPacket;
		}
		else
		{
			//every client has seen this packet, so pull it off the in use queue and
			//put it back on the free queue.
			OSQueueElem* elem = fPacketQueue.DeQueue();
			Assert(elem == &thePacket->fQueueElem);
			inFreeQueue->EnQueue(&thePacket->fQueueElem);
		}
	}
#if DEBUG
	//Warn(didSomething);
#endif

	//Don't forget that the caller also wants to know when we next want to run
	if (*ioWakeupTime == 0)
		*ioWakeupTime = fNextTimeToRun;
	else if ((fNextTimeToRun > 0) && (*ioWakeupTime > fNextTimeToRun))
		*ioWakeupTime = fNextTimeToRun;
	//fNextTimeToRun is in real time, not relative time.
	fNextTimeToRun += currentTime;
}

UInt16 ReflectorSender::GetPacketSeqNumber(const StrPtrLen& inPacket)
{
	if (inPacket.Len < 4)
		return 0;
	
	//The RTP seq number is the second short of the packet
	UInt16* seqNumPtr = (UInt16*)inPacket.Ptr;
	return ntohs(seqNumPtr[1]);
}

void ReflectorSender::SetPacketSeqNumber(const StrPtrLen& inPacket, UInt16 inSeqNumber)
{
	if (inPacket.Len < 4)
		return;

	//The RTP seq number is the second short of the packet
	UInt16* seqNumPtr = (UInt16*)inPacket.Ptr;
	seqNumPtr[1] = htons(inSeqNumber);
}

bool ReflectorSender::PacketShouldBeThinned(RTPStream* inStream, const StrPtrLen& inPacket)
{
	//This function determines whether the packet should be dropped.
	//It also adjusts the sequence number if necessary

	//Only thin video
	if ((inStream->GetCodecType() != RTPStream::kVideoCodecType) || (inPacket.Len < 4))
		return false;
	
	UInt16 curSeqNum = this->GetPacketSeqNumber(inPacket);
	
	//Check to see if we need to drop to audio only
	if ((inStream->GetQualityLevel() == ReflectorSession::kAudioOnlyQuality) &&
		(inStream->GetNextSeqNumber() == 0))
	{
#if REFLECTOR_THINNING_DEBUGGING
		printf("Reflector Dropping to audio only\n");
#endif
		//All we need to do in this case is mark the sequence number of the first dropped packet
		//fNextSeqNumber = curSeqNum;
		inStream->SetNextSeqNumber(curSeqNum);
	}
	
	//Check to see if we can reinstate video
	if ((inStream->GetQualityLevel() == ReflectorSession::kNormalQuality) &&
		(inStream->GetNextSeqNumber() != 0))
	{
		//Compute the offset amount for each subsequent sequence number. This offset will
		//alter the sequence numbers so that they increment normally (providing the illusion to the
		//client that there are no missing packets)
		//fSeqNumberOffset += curSeqNum - fNextSeqNumber;
		inStream->SetSeqNumberOffset(inStream->GetSeqNumberOffset() + (curSeqNum - inStream->GetNextSeqNumber()));
		inStream->SetNextSeqNumber(0);
		//fNextSeqNumber = 0;
#if REFLECTOR_THINNING_DEBUGGING
		printf("Reflector Reinstating video to probe the client. Offset = %d\n", inStream->GetSeqNumberOffset());
#endif
	}
	
	//tell the caller whether to drop this packet or not.
	if (inStream->GetQualityLevel() == ReflectorSession::kAudioOnlyQuality)
		return true;
	else
	{
		//Adjust the sequence number of the current packet based on the offset, if any
		curSeqNum -= inStream->GetSeqNumberOffset();
		this->SetPacketSeqNumber(inPacket, curSeqNum);
		return false;
	}
}



#pragma mark __REFLECTOR_SOCKET__

UDPSocketPair* ReflectorSocketPool::ConstructUDPSocketPair()
{
	return new ('usce') UDPSocketPair(new ('rsck') ReflectorSocket(),
										new ('rsck') ReflectorSocket());
}

void ReflectorSocketPool::DestructUDPSocketPair(UDPSocketPair *inPair)
{
	//The socket's run function may be executing RIGHT NOW! So we can't
	//just delete the thing, we need to send the sockets kill events.
	((ReflectorSocket*)inPair->GetSocketA())->Signal(Task::kKillEvent);
	((ReflectorSocket*)inPair->GetSocketB())->Signal(Task::kKillEvent);
	delete inPair;
}

ReflectorSocket::ReflectorSocket()
: UDPSocket(Socket::kNonBlocking | Socket::kWantsEvents, this), IdleTask(), fMutex('rsss'), fSleepTime(0)
{
	//construct all the preallocated packets
	for (UInt32 numPackets = 0; numPackets < kNumPreallocatedPackets; numPackets++)
	{
		//If the local port # of this socket is odd, then all the packets
		//used for this socket are rtcp packets.
		ReflectorPacket* packet = new ('rpka') ReflectorPacket();
		fFreeQueue.EnQueue(&packet->fQueueElem);//put this packet onto the free queue
	}
}

ReflectorSocket::~ReflectorSocket()
{
	while (fFreeQueue.GetLength() > 0)
	{
		ReflectorPacket* packet = (ReflectorPacket*)fFreeQueue.DeQueue()->GetEnclosingObject();
		delete packet;
	}
}

void	ReflectorSocket::AddSender(ReflectorSender* inSender)
{
	OSMutexLocker locker(&fMutex);
	fSenderQueue.EnQueue(&inSender->fSocketQueueElem);
}

void	ReflectorSocket::RemoveSender(ReflectorSender* inSender)
{
	OSMutexLocker locker(&fMutex);
	fSenderQueue.Remove(&inSender->fSocketQueueElem);
}

SInt64 ReflectorSocket::Run()
{
	//We want to make sure we can't get idle events WHILE we are inside
	//this function. That will cause us to run the queues unnecessarily
	//and just get all confused.
	this->CancelTimeout();
	
	Task::EventFlags theEvents = this->GetEvents();
	//if we have been told to delete ourselves, do so.
	if (theEvents & Task::kKillEvent)
		return -1;

	OSMutexLocker locker(&fMutex);
	SInt64 theMilliseconds = OS::Milliseconds();
	
	//Only check for data on the socket if we've actually been notified to that effect
	if (theEvents & Task::kReadEvent)
		this->GetIncomingData(theMilliseconds);

#if DEBUG
	//make sure that we haven't gotten here prematurely! This wouldn't mess
	//anything up, but it would waste CPU.
	if (theEvents & Task::kIdleEvent)
	{
		SInt32 temp = (SInt32)(fSleepTime - theMilliseconds);
		char tempBuf[20];
		::sprintf(tempBuf,"%ld",temp);
		WarnV(fSleepTime <= theMilliseconds, tempBuf);
	}
#endif
	
	fSleepTime = 0;
	//Now that we've gotten all available packets, have the streams reflect
	for (OSQueueIter iter2(&fSenderQueue); !iter2.IsDone(); iter2.Next())
	{
		ReflectorSender* theSender2 = (ReflectorSender*)iter2.GetCurrent()->GetEnclosingObject();
		if (theSender2->ShouldReflectNow(theMilliseconds, &fSleepTime))
			theSender2->ReflectPackets(&fSleepTime, &fFreeQueue);
	}
	
#if DEBUG
	theMilliseconds = OS::Milliseconds();
#endif
	
	//For smoothing purposes, the streams can mark when they want to wakeup.
	if (fSleepTime > 0)
		this->SetIdleTimer(fSleepTime);

#if DEBUG
	//The debugging check above expects real time.
	fSleepTime += theMilliseconds;
#endif
		
	return 0;
}

void ReflectorSocket::GetIncomingData(const SInt64& inMilliseconds)
{
	UInt32 theMulticastAddr = 0;
	//get all the outstanding packets for this socket
	while (true)
	{
		//get a packet off the free queue.
		ReflectorPacket* thePacket = this->GetPacket();

		this->RecvPacket(&theMulticastAddr, thePacket->fPacketPtr.Ptr,
							ReflectorPacket::kMaxReflectorPacketSize, &thePacket->fPacketPtr.Len);
		if (thePacket->fPacketPtr.Len == 0)
		{
			//put the packet back on the free queue, because we didn't actually
			//get any data here.
			fFreeQueue.EnQueue(&thePacket->fQueueElem);
			break;//no more packets on this socket!
		}
		if (GetLocalPort() & 1)
		{
			//if this is a new RTCP packet, check to see if it is a sender report.
			//We should only reflect sender reports. Because RTCP packets can't have both
			//an SR & an RR, and because the SR & the RR must be the first packet in a
			//compound RTCP packet, all we have to do to determine this is look at the
			//packet type of the first packet in the compound packet.
			RTCPPacket theRTCPPacket;
			if ((!theRTCPPacket.ParsePacket((UInt8*)thePacket->fPacketPtr.Ptr, thePacket->fPacketPtr.Len)) ||
				(theRTCPPacket.GetPacketType() != RTCPPacket::kSRPacketType))
			{
				//pretend as if we never got this packet
				fFreeQueue.EnQueue(&thePacket->fQueueElem);
				continue;
			}
		}
	
		//We're going to have to reflect this packet, so put it on the proper queue
		bool hasStream = false;
		
		//find the right stream for this packet
		for (OSQueueIter iter(&fSenderQueue); (!iter.IsDone() && !hasStream); iter.Next())
		{
			ReflectorSender* theSender = (ReflectorSender*)iter.GetCurrent()->GetEnclosingObject();
			if ((theSender->fStream->fAddr == theMulticastAddr) || (theMulticastAddr == 0))
			{
				//Here we're trying to route packets to the right reflector stream
				//based on destination (multicast) addr. If the OS doesn't support
				//this feature, the above function will return 0 for the addr.
#if OS_SUPPORTS_UDP_DEST_ADDR
				Assert(theMulticastAddr != 0);
#endif
				thePacket->fBucketsSeenThisPacket = 0;
				thePacket->fTimeArrived = inMilliseconds;
				theSender->fPacketQueue.EnQueue(&thePacket->fQueueElem);
				theSender->fHasNewPackets = true;
				hasStream = true;
			}
		}
		//if none of the Reflector streams wanted this packet, make sure not to leak it!
		if (!hasStream)
			fFreeQueue.EnQueue(&thePacket->fQueueElem);
	}
}


ReflectorPacket* ReflectorSocket::GetPacket()
{
	if (fFreeQueue.GetLength() == 0)
		//if the port number of this socket is odd, this packet is an RTCP packet.
		return new ('rpka') ReflectorPacket();
	else
		return (ReflectorPacket*)fFreeQueue.DeQueue()->GetEnclosingObject();
}
