
/*
 * LIB/EXPIRE.C
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

Prototype int GetExpire(const char *msgid, const char *nglist, int lncount, int size);
Prototype void FindExpireCtl(const char *group, ExpireCtl *save);
Prototype int GetOverExpire(const char *groupName);
Prototype void LoadExpireCtl(int force);

#ifdef NOTDEF

typedef struct ExpireCtl {
    struct ExpireCtl	*ex_Next;
    char		*ex_Wild;
    int			ex_Slot;
    int			ex_MaxSize;
    int			ex_ExpireLines;
    int			ex_MaxArts;
    double		ex_ExpireDays;
    double		ex_CrossFactor;
} ExpireCtl;

#endif

ExpireCtl *ExBase;
MemPool *EXMemPool;

int findExpireCtl(const char *group, int ngcount, int lncount, int size, int *goe);

/*
 * GetExpire() return the expiration, in slot #n (0-100), for an
 * article given the msgid, history record, and group list
 */

int
GetExpire(const char *msgid, const char *nglist, int lncount, int size)
{
    char group[MAXGNAME];
    const char *p;
    int slot = 100;
    int ngcount = 0;

    /*
     * Scan the grouplist and expire based on the SHORTEST expire
     * time.
     */

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;
    for (p = nglist; p; p = strchr(p + 1, ','))
	++ngcount;
    for (p = nglist; p; p = strchr(p, ',')) {
	int l;
	int lslot;

	if (*p == ',')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != '*' && p[l] != '?' && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(group, p, l);
	group[l] = 0;
	lslot = findExpireCtl(group, ngcount, lncount, size, NULL);
	if (slot > lslot)
	    slot = lslot;
    }
    if (DebugOpt > 1)
	ddprintf("ngcount=%d size=%d expire=%d (%s)\n", ngcount, size, slot, nglist);
    return(slot);
}

int
GetOverExpire(const char *groupName)
{
    int goe = -1;
    findExpireCtl(groupName, -1, -1, 0, &goe);
    return(goe);
}

void
FindExpireCtl(const char *group, ExpireCtl *save)
{
    ExpireCtl *ex;

    bzero(save, sizeof(ExpireCtl));

    save->ex_Slot = 65535;
    save->ex_MaxSize = -1;
    save->ex_ExpireLines = -1;
    save->ex_ExpireDays = -1.0;
    save->ex_CrossFactor = -1.0;

    for (ex = ExBase; ex; ex = ex->ex_Next) {
	if (WildCmp(ex->ex_Wild, group) == 0) {
	    if (ex->ex_Slot < save->ex_Slot)
		save->ex_Slot = ex->ex_Slot;
	    if (ex->ex_MaxSize >= 0 && (save->ex_MaxSize < 0 || ex->ex_MaxSize < save->ex_MaxSize))
		save->ex_MaxSize = ex->ex_MaxSize;
	    if (ex->ex_CrossFactor >= 0.0 && (save->ex_CrossFactor < 0.0 || ex->ex_CrossFactor < save->ex_CrossFactor))
		save->ex_CrossFactor = ex->ex_CrossFactor;
	    if (ex->ex_ExpireLines >= 0 && (save->ex_ExpireLines < 0 || ex->ex_ExpireLines < save->ex_ExpireLines))
		save->ex_ExpireLines = ex->ex_ExpireLines;
	    if (ex->ex_MaxArts > 0)
		save->ex_MaxArts = ex->ex_MaxArts;
	    if (ex->ex_ExpireDays >= 0.0 && (save->ex_ExpireDays < 0.0 || ex->ex_ExpireDays < save->ex_ExpireDays))
		save->ex_ExpireDays = ex->ex_ExpireDays;
	}
    }
}

int
findExpireCtl(const char *group, int ngcount, int lncount, int size, int *goe)
{
    ExpireCtl save;
    double slot;

    FindExpireCtl(group, &save);

    if (goe)
	*goe = (int)(save.ex_ExpireDays * 60.0 * 60.0 * 24.0);

    slot = (double)save.ex_Slot;

    if (save.ex_CrossFactor >= 0.0 && ngcount > 0)
	slot = slot * pow(save.ex_CrossFactor, (double)ngcount);

    if (save.ex_ExpireLines > 0 && lncount > 0) {
	double dexl = (double)save.ex_ExpireLines;
	double dlnc = (double)lncount;
	double f = dexl * dexl / ((dlnc + dexl) * (dlnc + dexl));
	slot *= f;
    }

    if (save.ex_MaxSize >= 0 && size >= save.ex_MaxSize)
	slot = 0.0;
    if (slot < 0.0)
	slot = 0.0;
    if (slot > 100.0)
	slot = 100.0;
    return((int)slot);
}

uint32 ExGmtMin = (uint32)-1;
time_t ExMTime = 0;

void
LoadExpireCtl(int force)
{
    time_t gmt = time(NULL) / 60;

    /*
     * check for dexpire.ctl file modified once a minute
     */

    if (force || gmt != ExGmtMin) {
	struct stat st = { 0 };

	ExGmtMin = gmt;

	/*
	 * dexpire.ctl
	 */

	{
	    FILE *fi;

	    fi = fopen(PatLibExpand(DExpireCtlPat), "r");

	    if (fi == NULL)  
		syslog(LOG_EMERG, "%s file not found", PatLibExpand(DExpireCtlPat));

	    if (force || fi == NULL || 
		(fstat(fileno(fi), &st) == 0 && st.st_mtime != ExMTime)
	    ) {
		char buf[MAXGNAME+256];
		ExpireCtl **pex = &ExBase;

		ExMTime = st.st_mtime;	/* may be 0 if file failed to open */

		freePool(&EXMemPool);
		ExBase = NULL;
#ifdef NOTDEF
		    while ((ex = ExBase) != NULL) {
			ExBase = ex->ex_Next;
			free(ex);
		    }
#endif
		while (fi && fgets(buf, sizeof(buf), fi) != NULL) {
		    char *p;

		    if (buf[0] == '/')
			continue;
		    if (buf[0] == '#')
			continue;
		    if (buf[0] == '\n')
			continue;
		    if ((p = strchr(buf, ':')) != NULL) {
			ExpireCtl *ex = zalloc(&EXMemPool, sizeof(ExpireCtl) + (p - buf) + 1);

			ex->ex_Wild = (char *)(ex + 1);
			ex->ex_Slot = strtol(p + 1, NULL, 0);
			ex->ex_MaxSize = -1;
			ex->ex_CrossFactor = -1.0;
			ex->ex_ExpireLines = -1;
			ex->ex_ExpireDays = -1.0;

			memmove(ex->ex_Wild, buf, p - buf);
			ex->ex_Wild[p-buf] = 0;
			*pex = ex;
			pex = &ex->ex_Next;

			while ((p = strchr(p + 1, ':')) != NULL) {
			    switch(p[1]) {
			    case 'a':
				ex->ex_MaxArts = strtol(p + 2, NULL, 0);
				break;
			    case 'c':	/* per cross post    */
				ex->ex_CrossFactor = strtod(p + 2, NULL);
				break;
			    case 'b':
				ex->ex_ExpireLines = strtol(p + 2, NULL, 0);
				break;
			    case 's':	/* per million bytes */
				/* DEPRECIATED */
				break;
			    case 'm':
				ex->ex_MaxSize = strtol(p + 2, &p, 0);

				switch(*p) {
				case 'g':
				    ex->ex_MaxSize *= 1024;
				    /* fall through */
				case 'm':
				    ex->ex_MaxSize *= 1024;
				    /* fall through */
				case 'k':
				    ex->ex_MaxSize *= 1024;
				    break;
				}
				break;
			    case 'x':
				ex->ex_ExpireDays = strtod(p + 2, NULL);
				break;
			    }
			}
		    }
		}
		*pex = NULL;
	    }
	    if (fi)
		fclose(fi);
	}
    }
}

