
/*
 * DIDUMP.C	Dump or trace a dhistory file
 *
 * DIDUMP [-t] [-x] [-l] [-v] [-f[#]] dhistory
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

/*
 * OldHistory - versions <= 1.07
 *
 */

typedef struct OldHistory {
    uint32	next;   /* next link            */
    uint32	gmt;    /* gmt time in minutes  */
    hash_t      hv;     /* hash value           */
    uint16	iter;   /* file id              */
    uint16	exp;    /* hours relative to gmt minutes */
} OldHistory;

int TraceMode;
int DNPOpt;
int FOpt;
int FCount;
int OldOpt;
int StartOff;
int LineModeOpt;
int VerboseOpt;

void DumpTrace(int fd, int hsize, int rsize);
void DumpQuick(int fd, int hsize, int rsize);

int
main(int ac, char **av)
{
    int fd;
    int i;
    int hsize = 1024 * 1024;
    int rsize = sizeof(History);
    struct stat st;
    char *fileName = NULL;

    LoadDiabloConfig(ac, av);

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];

	if (*ptr != '-') {
	    if (*ptr == '@') {
		StartOff = strtol(ptr + 1, NULL, 0);
		continue;
	    }
	    if (fileName) {
		fprintf(stderr, "unexpected argument\n");
		exit(1);
	    }
	    fileName = ptr;
	    continue;
	}
	ptr += 2;
	switch(ptr[-1]) {
	case 'C':
	    if (*ptr == 0)
		++i;
	    break;
	case 'v':
	    if (*ptr)
		VerboseOpt = strtol(ptr, NULL, 0);
	    else
		++VerboseOpt;
	    break;
	case 'f':
	    FOpt = 1;
	    if (*ptr)
		FCount = strtol(ptr, NULL, 0);
	    break;
	case 'l':
	    LineModeOpt = 1;
	    break;
	case 'x':
	    DNPOpt = 1;
	    break;
	case 't':
	    TraceMode = 1;
	    break;
	case 'o':
	    OldOpt = 1;
	    break;
	default:
	    if (isdigit(ptr[-1])) {
		FCount = strtol(ptr - 1, NULL, 0);
	    } else {
		fprintf(stderr, "illegal option: %s\n", ptr - 2);
		exit(1);
	    }
	}
    }

    if (fileName == NULL) {
	printf("didump [-t] [-x] [-o] [-f] [@offset] dhistory-file\n");
	printf("    default	- quick dump\n");
	printf("    -t		- hash table trace (slow)\n");
	printf("    -x		- dump / DNP entries over 16 days old\n");
	printf("    -o		- dump old-style (diablo V <= 1.07) history file\n");
	printf("    -f		- 'tail -f' the history file\n");
	printf("    -v		- Include additional (synthesized) info\n");
	exit(0);
    }

    if (OldOpt) {
	char *paramName = malloc(strlen(fileName) + 32);
	FILE *fi;

	sprintf(paramName, "%s.param", fileName);

	rsize = sizeof(OldHistory);

	if ((fi = fopen(paramName, "r")) != NULL) {
	    char buf[256];

	    hsize = 0;

	    while (fgets(buf, sizeof(buf), fi) != NULL) {
		if (buf[0] == 'H')
		    hsize = strtol(buf + 1, NULL, 0);
	    }
	    fclose(fi);

	    if (hsize == 0) {
		fprintf(stderr, "dhistory parameter file error\n");
		exit(1);
	    }
	}
    }

    if (DNPOpt && TraceMode) {
	fprintf(stderr, "-x only works for quick dumps\n");
	exit(1);
    }

    if ((fd = open(fileName, O_RDONLY)) >= 0 && fstat(fd, &st) == 0) {
	/*
	 * new style history file has a header
	 */

	if (OldOpt == 0) {
	    HistHead hh;

	    if (read(fd, &hh, sizeof(hh)) != sizeof(hh)) {
		fprintf(stderr, "corrupted history file\n");
		exit(1);
	    }
	    if (hh.hmagic != HMAGIC) {
		fprintf(stderr, "corrupted history file\n");
		exit(1);
	    }
	    if (hh.version != HVERSION) {
		fprintf(stderr, "WARNING! Version mismatch file V%d, expecting V%d\n", hh.version, HVERSION);
		fprintf(stderr, "dump may be invalid\n");
	    }
	    rsize = hh.henSize;
	    hsize = hh.hashSize;

	    lseek(fd, hh.headSize, 0);
	}

	fprintf(stderr, "Dumping, hash table %d entries, record size %d\n",
	    hsize,
	    rsize
	);

	if (TraceMode)
	    DumpTrace(fd, hsize, rsize);
	else
	    DumpQuick(fd, hsize, rsize);
	close(fd);
    } else {
	fprintf(stderr, "open failed\n");
	exit(1);
    }
    return(0);
}

void
DumpTrace(int fd, int hsize, int rsize)
{
    int i;
    int32 *Ary = calloc(hsize, sizeof(int32));

    if (read(fd, Ary, hsize * sizeof(int32)) != hsize * sizeof(int32)) {
	printf("Unable to read hash table array\n");
	exit(1);
    }

    for (i = 0; i < hsize; ++i) {
	int32 off;

	if ((off = Ary[i]) != 0) {
	    int32 loff = hsize * sizeof(int32) - 4;
	    int maxChainLen = 1000;

	    printf("Index %d: ", i);

	    while (off) {
		History h = { 0 };

		lseek(fd, (off_t)(uint32)off, 0);
		if (read(fd, &h, rsize) != rsize) {
		    printf("read error @ %d", off);
		    break;
		} else {
		    printf(" [%u,%u %08x.%08x.%04x gm=%d ex=%d boff=%d bsize=%d F=%s]",
			(unsigned int)off, (unsigned int)h.next,
			h.hv.h1,
			h.hv.h2,
			(int)h.iter,
			(int)h.gmt,
			(int)H_EXP(h.exp),
			(int)h.boffset,
			(int)h.bsize,
			((h.exp & EXPF_HEADONLY) ? "H" : "")
		    );
		}
		loff = off;
		off = h.next;
		if (--maxChainLen == 0) {
		    printf(" MAXIMEM CHAIN LENGTH EXCEEDED!");
		    break;
		}
	    }
	    if (off)
		printf(" offset error: %d", off);
	    printf("\n");
	}
    }
}

void
DumpQuick(int fd, int hsize, int rsize)
{
    char hbuf[32768];
    int n;
    int hlen = (sizeof(hbuf) / rsize) * rsize;
    int rememberMins = (DiabloRememberDays + 1) * 24 * 60;
    uint32 gmt = time(NULL) / 60;

    if (StartOff)
	lseek(fd, StartOff, 0);
    else if (FOpt || FCount)
	lseek(fd, -FCount * rsize, 2);
    else
	lseek(fd, hsize * sizeof(int32), 1);

    fprintf(stderr, "@%d\n", (int)lseek(fd, 0L, 1));

top:
    while ((n = read(fd, &hbuf, hlen)) > 0) {
	int i;

	n /= rsize;

	for (i = 0; i < n; ++i) {
	    History *h = (History *)(hbuf + i * rsize);

	    if ((i & 255) == 0)
		gmt = time(NULL) / 60;

	    if (DNPOpt && H_EXP(h->exp) == H_EXP((unsigned short)-1)) {
		int32 dgmt = gmt - h->gmt;	/* DELTA MINUTES */
		if (dgmt < -rememberMins || dgmt > rememberMins)
		    continue;
	    }

	    printf("DUMP %08x.%08x.%04x gm=%d ex=%-2d",
		h->hv.h1,
		h->hv.h2,
		(int)h->iter,
		(int)h->gmt,
		(int)H_EXP(h->exp)
	    );
	    if (!OldOpt) {
		printf(" boff=%-7d bsize=%-6d", (int)h->boffset, (int)h->bsize);
		if (VerboseOpt) {
		    printf(" file=\"D.%08x/B.%04x\"",
			(int)(h->gmt - h->gmt % 10),
			(int)h->iter 
		    );
		}
		printf(" flags=%s",
		    ((h->exp & EXPF_HEADONLY) ? "H" : "")
		);
	    }
	    printf("\n");
	    if (LineModeOpt)
		fflush(stdout);
	}
    }
    if (FOpt) {
	sleep(1);
	goto top;
    }
    printf(".\n");
}

