
/*
 * DILOADFROMSPOOL.C	
 *
 *	Scan the specified spool directory or spool file, extract the
 *	Message-ID, offset, and article size, and create the appropriate
 *	entry in the dhistory file.  This command is typically used if 
 *	you have lost your history file entirely and need to regenerate it
 *	from the existing spool or to do a partial recovery from backup
 *	and regeneration the lost portion from the existing spool.
 *	
 *	diloadfromspool ... [-F dhistory] D.directory ... D.directory/B.file...
 *
 *	NOTE:  File specifications must be in the form of D.directory for
 *	a spool directory, or D.directory/B.file for a spool file in order
 *	for diloadfromspool to figure out the history file fields.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

int VerboseOpt;
int QuietOpt;
int LoadDupCount;
int LoadCount;
int ForReal = 1;
int RequeueOpt = 0;

void ScanEntireSpool(void);
void ScanSpoolDirectory(char *dpath, int gmt);
void ScanSpoolFile(char *fpath, int gmt, int iter);
void ScanSpoolFileMap(const char *base, int bytes, int gmt, int iter, char *dpath);
int strSort(const void *s1, const void *s2);

int
main(int ac, char **av)
{
    int fileMax = 128;
    int fileIdx = 0;
    int flags = 0;
    int uflag = 0;
    int aflag = 0;
    char **fileAry = calloc(sizeof(char *), fileMax);
    char *historyFileName = NULL;

    LoadDiabloConfig(ac, av);

    {
	int i;

	for (i = 1; i < ac; ++i) {
	    char *ptr = av[i];

	    if (*ptr != '-') {
		if (fileIdx == fileMax) {
		    fileMax = fileMax * 2;
		    fileAry = realloc(fileAry, sizeof(char *) * fileMax);
		}
		fileAry[fileIdx++] = ptr;
		continue;
	    }
	    ptr += 2;
	    switch(ptr[-1]) {
	    case 'C':
		if (*ptr == 0)
		    ++i;
		break;
	    case 'n':
		ForReal = 0;
		break;
	    case 'F':
		historyFileName = (*ptr) ? ptr : av[++i];
		break;
	    case 'h':
		NewHSize = strtol(((*ptr) ? ptr : av[++i]), &ptr, 0);
		if (*ptr == 'k' || *ptr == 'K')
		    NewHSize *= 1024;
		if (*ptr == 'm' || *ptr == 'M')
		    NewHSize *= 1024 * 1024;
		if ((NewHSize ^ (NewHSize - 1)) != (NewHSize << 1) - 1) {
		    fprintf(stderr, "specified history size is not a power of 2\n");
		    exit(1);
		}
		break;
	    case 'v':
		VerboseOpt = (*ptr) ? strtol(ptr, NULL, 0) : 1;
		break;
	    case 'q':
		QuietOpt = 1;
		break;
	    case 'Q':
		RequeueOpt = 1;
		break;
	    case 'f':
		flags |= HGF_FAST | HGF_NOSEARCH;
		break;
	    case 'u':
		uflag = 1;
		break;
	    case 'a':
		aflag = 1;
		break;
	    }
	}
    }

    if (flags & HGF_FAST) {
	struct stat st;

	if (historyFileName == NULL) {
	    fprintf(stderr, "You cannot run fastmode without specifying a filename!\n");
	    exit(1);
	}
	if (stat(historyFileName, &st) == 0 && uflag == 0) {
	    fprintf(stderr, "-f history files may not previously exist unless you also specify -u\n");
	    fprintf(stderr, "WARNING! -f -u is NOT suggested!\n");
	    exit(1);
	}
	if (uflag)
	    flags &= ~HGF_NOSEARCH;
    }

    if (RequeueOpt) {
	ForReal = 0;
	aflag = 1;
	QuietOpt = 1;
    } else {
    /*
     * historyFileName can be NULL and causes the default dhistory path
     * to be used.
     */

	HistoryOpen(historyFileName, flags);
    }

    {
	int i;

	for (i = 0; i < fileIdx; ++i) {
	    struct stat st;
	    if (stat(fileAry[i], &st) == 0) {
		if (S_ISDIR(st.st_mode)) {
		    int gmt;

		    if (sscanf(fileAry[i], "D.%x", &gmt) == 1) {
			ScanSpoolDirectory(fileAry[i], gmt);
		    } else {
			fprintf(stderr, "Illegal path format for directory: %s\n", fileAry[i]);
		    }
		} else {
		    int gmt;
		    int iter;

		    if (sscanf(fileAry[i], "D.%x/B.%x", &gmt, &iter) == 2) {
			ScanSpoolFile(fileAry[i], gmt, iter);
		    } else {
			fprintf(stderr, "Illegal path format for file: %s\n", fileAry[i]);
		    }
		}
	    }
	}
    }
    if (aflag) {
	ScanEntireSpool();
    }
    printf("diload: %d/%d entries loaded\n", LoadCount, LoadCount + LoadDupCount);

    if (!RequeueOpt) 
    {
	int r = HistoryClose();

	if (r == RCOK)
	    return(0);
	else
	    return(1);
    }
    return(0);
    /* not reached */
}

void
ScanEntireSpool(void)
{
    DIR *dir;
    int fileIdx = 0;
    int fileMax = 2048;
    char **fileAry = calloc(sizeof(char *), fileMax);

    if (chdir(PatExpand(SpoolHomePat)) < 0) {
	fprintf(stderr, "Unable to cd to %s\n", PatExpand(SpoolHomePat));
	exit(1);
    }

    /*
     * Scan directory
     */

    if ((dir = opendir(".")) != NULL) {
	den_t *den;

	while ((den = readdir(dir)) != NULL) {
	    if (fileIdx == fileMax) {
		fileMax = fileMax * 2;
		fileAry = realloc(fileAry, fileMax * sizeof(char *));
	    }
	    fileAry[fileIdx++] = strdup(den->d_name);
	}
	closedir(dir);
    }

    /*
     * Sort directory
     */

    if (fileIdx > 1)
	qsort(fileAry, fileIdx, sizeof(char *), strSort);

    /*
     * Process directory
     */

    {
	int i;

	for (i = 0; i < fileIdx; ++i) {
	    int gmt;

	    if (sscanf(fileAry[i], "D.%x", &gmt) == 1) {
		ScanSpoolDirectory(fileAry[i], gmt);
	    }
	}
    }
}

void
ScanSpoolDirectory(char *dpath, int gmt)
{
    DIR *dir;

    if ((dir = opendir(dpath)) != NULL) {
	den_t *den;
	char path[256];

	if (VerboseOpt)
	    printf("%s:\n", dpath);

	while ((den = readdir(dir)) != NULL) {
	    int iter;

	    if (sscanf(den->d_name, "B.%x", &iter) == 1) {
		snprintf(path, sizeof(path), "%s/%s", dpath, den->d_name);
		ScanSpoolFile(path, gmt, iter);
	    }
	}
	closedir(dir);
    }
}

void
ScanSpoolFile(char *fpath, int gmt, int iter)
{
    int fd;
    char *base;
    int bytes;
    struct stat st;

    errno = 0;
    if ((fd = open(fpath, O_RDONLY)) < 0) {
	printf("    %s\t%s\n", fpath, strerror(errno));
	return;
    }
    if (fstat(fd, &st) < 0) {
	close(fd);
	printf("    %s\t%s\n", fpath, strerror(errno));
	return;
    }
    bytes = st.st_size;

    base = xmap(NULL, bytes, PROT_READ, MAP_SHARED, fd, 0);
    if (base == NULL) {
	close(fd);
	printf("    %s\t%s\n", fpath, strerror(errno));
	return;
    }

    if (!QuietOpt)
	printf("    %s:\n", fpath);

    ScanSpoolFileMap(base, bytes, gmt, iter, fpath);

    xunmap(base, bytes);
    close(fd);
}

void
ScanSpoolFileMap(const char *base, int bytes, int gmt, int iter, char *dpath)
{
    int b = 0;

    /*
     * scan file
     */

    while (b < bytes) {
	int i = b;
	int inHeader = 1;
	int linesLeft = -1;
	int numLines = 0;
	char msgId[MAXMSGIDLEN];
	char newsgroups[2048];

	msgId[0] = 0;

	/*
	 * scan article
	 */

	while (i < bytes && linesLeft) {
	    int l;

	    /*
	     * Scan line
	     */

	    for (l = i; l < bytes && base[l] != '\n'; ++l)
		;
	    if (l < bytes)
		++l;
	    if (inHeader) {
		if (l - i == 1) {
		    inHeader = 0;
		} else if (strncasecmp(base + i, "Lines:", 6) == 0) {
		    linesLeft = strtol(base + i + 6, NULL, 0);
		} else if (strncasecmp(base + i, "Message-ID:", 11) == 0) {
		    diablo_strlcpynl(msgId, base + i + 11, l - i - 11, sizeof(msgId));
		} else if (strncasecmp(base + i, "Newsgroups:", 11) == 0) {
		    diablo_strlcpynl(newsgroups, base + i + 11, l - i - 11, sizeof(newsgroups));
		}
	    } else {
		--linesLeft;
		++numLines;
	    }
	    i = l;
	}
	if (i < bytes && base[i] == 0) {
	    const char *id = MsgId(msgId);
	    int r = 0;

	    if (!RequeueOpt) {
		History h = { 0 };
		History htmp;

		h.hv = hhash(id);
		h.iter = iter;
		h.gmt = gmt;
		h.exp = H_EXP(49);
		h.boffset = b;
		h.bsize = i - b;
		if (numLines == 0)
		    h.exp |= EXPF_HEADONLY;

		if ((r = HistoryLookupByHash(h.hv, &htmp)) == 0) {
		    ++LoadDupCount;
		} else {
		    if (ForReal)
			HistoryAdd(id, &h);
		}
	    }
	    ++LoadCount;
	    if (VerboseOpt > 1 || (VerboseOpt && r != 0))
		printf("\tMessage %d,%d %s %s\n", 
		    b, 
		    i - b, 
		    ((r == 0) ? "dup" : "add"),
		    id
		);
	    if (RequeueOpt) {
		printf("SOUT %s %s %d,%d %s\n", 
		    dpath, id,
		    b, 
		    i - b, 
		    newsgroups
		);
	    }
	    ++i;
	} else {
	    if (!QuietOpt) {
		printf("\tFailed %d,%d %s\n", b, i - b, MsgId(msgId));
		fflush(stdout);
	    }
	    /*
	    write(1, base + b, i - b);
	    write(1, "*", 1);
	    printf("(%d)\n", base[i]);
	    */

	    while (i < bytes && base[i] != 0)
		++i;
	    if (i < bytes)
		++i;
	}
	b = i;
    }
}

int     
strSort(const void *s1, const void *s2)
{ 
    char *str1 = *(char **)s1;
    char *str2 = *(char **)s2;

    return(strcmp(str1, str2));
}  

