#include "cs.h"                                          /*  HETRO.C   */
#include "soundio.h"
#include <math.h>

#ifdef macintosh
#include "MacTransport.h"
#endif

/*  #define PIE	3.14159265358979323846 */
/*  #define TWOPI   6.28318530717958647692 */
#define	SQRTOF3	1.73205080756887729352
#define SQUELCH	0.5	/* % of max ampl below which delta_f is frozen */
#define HMAX    50

/* Authors:   Tom Sullivan, Nov'86, Mar'87;  bv revised Jun'92, Aug'92  */
/* Function:  Fixed frequency heterodyne filter analysis.		     */
/* Simplifications and partial recoding by John Fitch Dec 1994 */

/* lowest heterodyne freq = sr/bufsiz */

static float  	x1,x2,yA,y2,y3,		/* lpf coefficients*/
	cur_est,			/* current freq. est.*/
	freq_est = 0.0f, max_frq, max_amp,/* harm freq. est. & max vals found */
	fund_est = 100.0f,		/* fundamental est.*/
	t,				/* fundamental period est.*/
	delta_t, outdelta_t,		/* sampling period, outpnt period */
	sr = 0.0f, 			/* sampling rate */
       	freq_c = 0.0f,			/* filter cutoff freq.*/
	beg_time = 0.0f, input_dur = 0.0f,/* begin time & sample input duration*/
	**MAGS, **FREQS;		/* magnitude and freq. output buffers*/

static double *cos_mul, *sin_mul,       /* quad. term buffers*/
       	*a_term, *b_term,               /*real & imag. terms*/
        *r_ampl,		        /* pt. by pt. amplitude buffer*/
	*r_phase,		        /* pt. by pt. phase buffer*/
	*a_avg,			        /* output dev. freq. buffer*/
	new_ph,				/* new phase value*/
       	old_ph = 0.0,			/* previous phase value*/
       	jmp_ph = 0.0,			/* for phase unwrap*/
        *ph_av1, *ph_av2, *ph_av3,      /*tempor. buffers*/
        *amp_av1, *amp_av2, *amp_av3,   /* same for ampl.*/
	m_ampsum = 32767.0;		/* maximum amplitude at output*/
       
static long    windsiz; 		/* # of pts. in one per. of sample*/

static short    hmax = 10;		/* max harmonics requested */
static int 	num_pts = 256,          /* breakpoints per harmonic */
        	amp_min = 64;   	/* amplitude cutout threshold */

static int     skip,			/* flag to stop analysis if zeros*/
	bufsiz = 1;			/* circular buffer size */

static long	smpsin;			/* num sampsin */

static long    midbuf,			/* set to bufsiz / 2   */
	bufmask;		        /* set to bufsiz - 1   */

static char    *infilnam,		/* input file name */
	*outfilnam;			/* output file name */
static float	*auxp;			/* pointer to input file */

static float 	*adp; 		/*pointer to front of sample file*/
static double	*c_p,*s_p;	/* pointers to space for sine and cos terms */

static	double	GETVAL(double *, long);
static  double 	sq(double);
static	void	PUTVAL(double *, long, double);
static  void    hetdyn(int), lpinit(void);
static	void	lowpass(double *, double *, long);
static	void	average(long,double *,double *,long);
static	void	output(long, int, int);
static  void    output_ph(long), filedump(void), quit(char *);
extern	int	close(int);

#define	sgn(x)	(x<0.0 ? -1 : 1)
#define u(x)	(x>0.0 ? 1 : 0)

#define FIND(MSG)   if (*s == '\0')  \
		        if (!(--argc) || (s = *++argv) && *s == '-')  \
			    quit(MSG);

int hetro(int argc, char **argv)  /* called from main.c or anal/adsyn/main.c */
{
	int      i, hno, infd, channel = 1;
	long     nsamps, smpspc, bufspc, mgfrspc;
	char     *dsp, *dspace, *mspace;
	double   *begbufs, *endbufs;

	SOUNDIN  *p;      /* space allocated by SAsndgetset() */
extern	int      SAsndgetset(char *, SOUNDIN**, float*, float*, float*, int);
extern	long     getsndin(int, float*, long, SOUNDIN*);
extern  char     *retfilnam;

        if (!(--argc))
#ifdef CWIN
        {
extern int __cdecl hetro_dialog(float *, int *, float *, float *,
                                float *, short *, double *, int *,
                                int *, float *, char **, char **);
            if (!hetro_dialog(&sr, &channel, &beg_time, &input_dur,
                              &fund_est, &hmax, &m_ampsum, &amp_min,
                              &num_pts, &freq_c, &infilnam, &outfilnam))
              return 0;
        }
        else {
#else
        {
               quit("no arguments");
        }
#endif
	do {
	    char *s = *++argv;
	    if (*s++ == '-')
	        switch (*s++) {
		    case 's':   FIND("no sampling rate")
				sscanf(s,"%f",&sr);
				break;
		    case 'c':   FIND("no channel")
				sscanf(s,"%d",&channel);
				break;
		    case 'b':   FIND("no begin time")
				sscanf(s,"%f",&beg_time);
				break;
		    case 'd':   FIND("no duration time")
				sscanf(s,"%f",&input_dur);
				break;
		    case 'f':   FIND("no fundamental estimate")
				sscanf(s,"%f",&fund_est);
				break;
		    case 'h':   FIND("no harmonic count")
				sscanf(s,"%hd",&hmax);
		                if (hmax > HMAX)
				 err_printf("over %d harmonics but continuing",
					     HMAX);
				if (hmax < 1) {
				  err_printf("h of %d too low, reset to 1\n",
					     hmax);
				  hmax = 1;
				}
				break;
		    case 'M':   FIND("no amplitude maximum")
				sscanf(s,"%f",&m_ampsum);
				break;
		    case 'm':   FIND("no amplitude minimum")
				sscanf(s,"%d",&amp_min);
				break;
		    case 'n':   FIND("no number of output points")
				sscanf(s,"%d",&num_pts);
				break;
		    case 'l':   FIND("no filter cutoff")
				sscanf(s,"%f",&freq_c);
				break;
		    case '-':   FIND("no log file");
#ifndef macintosh
			        dribble = fopen(s, "w");
#endif
                                while (*s++); s--;
			        break;
		    default:   quit("Invalid switch option");
		}
	    else break;
	} while (--argc);

	if (argc != 2)	quit("incorrect number of filenames");
	infilnam = *argv++;
	outfilnam = *argv;

	if (freq_c > 1)
            fprintf (stderr,"Filter cutoff freq. = %f\n",freq_c);

	if ((input_dur < 0) || (beg_time < 0))
	    quit("input and begin times cannot be less than zero");
#ifdef CWIN
        }
#endif
					   /* open sndfil, do skiptime */
	if (
	 (infd = SAsndgetset(infilnam,&p,&beg_time,&input_dur,&sr,channel))<0) {
	    sprintf(errmsg,"Cannot open %s", retfilnam);
	    quit (errmsg);
	}
	nsamps = p->getframes;
	auxp = (float*)mmalloc(nsamps * sizeof(float));   /* alloc for floats */
	if ((smpsin = getsndin(infd,auxp,nsamps,p)) <= 0) { /* & read them in */
	    sprintf(errmsg,"Read error on %s\n", retfilnam);
	    quit(errmsg);
	}
	close(infd);				

	sr = (float)p->sr;                              /* sr now from open  */
	windsiz = (long)(sr / fund_est + 0.5f);         /* samps in fund prd */
	if (num_pts > 32767 || num_pts >= nsamps - windsiz) 
	    quit("number of output points is too great");
	delta_t = 1.0f/sr;
	t = 1.0f/fund_est;
	outdelta_t = (float)num_pts / (smpsin - windsiz);

	while (bufsiz < (sr/fund_est + 0.5)) 
	    bufsiz *= 2;
	midbuf = bufsiz/2;
	bufmask = bufsiz - 1;
				
	smpspc = smpsin * sizeof(double);
	bufspc = bufsiz * sizeof(double);
	
	dsp = dspace = mmalloc(smpspc * 2 + bufspc * 13);
	c_p = (double *) dsp;		dsp += smpspc;	/* space for the    */
	s_p = (double *) dsp;		dsp += smpspc;	/* quadrature terms */
	begbufs = (double *) dsp;
	cos_mul = (double *) dsp;	dsp += bufspc;  /* bufs that will be */
	sin_mul = (double *) dsp;	dsp += bufspc;  /* refilled each hno */
	a_term = (double *) dsp;	dsp += bufspc;
	b_term = (double *) dsp;	dsp += bufspc;
	r_ampl = (double *) dsp;	dsp += bufspc;
	ph_av1 = (double *) dsp;	dsp += bufspc;
	ph_av2 = (double *) dsp;	dsp += bufspc;
	ph_av3 = (double *) dsp;	dsp += bufspc;
	r_phase = (double *) dsp;	dsp += bufspc;
	amp_av1 = (double *) dsp;	dsp += bufspc;
	amp_av2 = (double *) dsp;	dsp += bufspc;
	amp_av3 = (double *) dsp;	dsp += bufspc;
	a_avg = (double *) dsp;		dsp += bufspc;
	endbufs = (double *) dsp;

	mgfrspc = num_pts * sizeof(float);
	dsp = mspace = mmalloc(mgfrspc * hmax * 2);
	MAGS = (float **) mmalloc(hmax * sizeof(float*));
	FREQS = (float **) mmalloc(hmax * sizeof(float*));
	for (i = 0; i < hmax; i++) {
	    MAGS[i] = (float *) dsp;	;dsp += mgfrspc;
	    FREQS[i] = (float *) dsp;	;dsp += mgfrspc;
	}
	lpinit();				/* calculate LPF coeffs.  */
        if (!POLL_EVENTS()) exit(1);
	adp = auxp;			/* point to beg sample data block */
	for (hno = 0; hno < hmax; hno++) {	/* for requested harmonics*/
	    double *dblp;
	    freq_est += fund_est;		/*   do analysis */
	    cur_est = freq_est;
	    dblp = begbufs;
	    do  *dblp++ = 0.0f;		/* clear all refilling buffers*/
	    while (dblp < endbufs);
	    max_frq = 0.0f;
	    max_amp = 0.0f;

	    err_printf("analyzing harmonic #%d\n",hno);
	    err_printf("freq est %6.1f,", cur_est);
	    hetdyn(hno);		/* perform actual computation */
#ifdef macintosh
		if (transport.state & kGenKilled) exit(1);
#endif	    
	    err_printf(" max found %6.1f, rel amp %6.1f\n", max_frq, max_amp);
	}
	mfree(dspace);
#ifdef macintosh
if (!(transport.state & kGenKilled))
#endif
	filedump();			/* write output to adsyn file */
	mfree(mspace);
#if !defined(macintosh) && !defined(CWIN)
	exit(0);
#endif
	return (-1);		/* To keep compiler quiet */
}

static double 
GETVAL(double *inb, long smpl)    /* get value at position smpl in array inb */
{
	return(*(inb + ((smpl + midbuf) & bufmask)));
}

static void PUTVAL(double *outb, long smpl, double value)  /* put value in array outb at postn smpl */
{
	*(outb + ((smpl + midbuf) & bufmask)) = value;
}

static void hetdyn(int hno)                           /* HETERODYNE FILTER */
{
        long 	smplno;
	double 	temp_a, temp_b, tpidelest;
	double *cos_p, *sin_p, *cos_wp, *sin_wp;
	long	n;
	int   	outpnt, lastout = -1;
	float *ptr;

	jmp_ph = 0;   			/* set initial phase to 0 */
	temp_a = temp_b = 0;
	cos_p = c_p;
	sin_p = s_p;
	tpidelest = TWOPI * cur_est * delta_t;
	for (smplno = 0; smplno < smpsin; smplno++) {
	    double phase = smplno * tpidelest;     /* do all quadrature calcs */
	    ptr = adp + smplno;  		   /* at once and point to it */
	    *cos_p++ = (double)(*ptr) * cos(phase);
	    *sin_p++ = (double)(*ptr) * sin(phase);
	}
        if (!POLL_EVENTS()) exit(1);

	cos_p = cos_wp = c_p;
	sin_p = sin_wp = s_p;
	for (smplno = 0; smplno < smpsin - windsiz; smplno++) {
	    if (smplno == 0 && smpsin >= windsiz) {   /* for first smplno */
	        n = windsiz;
	        do {
		    temp_a += *cos_wp++;     /* sum over windsiz = nsmps in */
		    temp_b += *sin_wp++;     /*    1 period of fund. freq.  */
		} while (--n);
	    }
	    else {	/* if more than 1 fund. per. away from file end */
			/* remove front value and add on new rear value */
			/* to obtain summation term for new sample! */
	        if (smplno <= smpsin - windsiz) {		
		    temp_a += (*cos_wp++ - *cos_p++);  /* _wp = _p + windsiz */
		    temp_b += (*sin_wp++ - *sin_p++);
		}
		else {      
		    skip = 1;
		    temp_a = temp_b = 0;
		}
	    }
            if (!POLL_EVENTS()) exit(1);
	    PUTVAL(cos_mul,smplno,temp_a);     /* store values into buffers*/
	    PUTVAL(sin_mul,smplno,temp_b);
	    if ((freq_c <= 1) || (smplno < 3)) {
	        average(windsiz,cos_mul,a_term,smplno); /* average over previous */
		average(windsiz,sin_mul,b_term,smplno); /* values 1 fund prd ago */
	    }
	    else {
	        lowpass(a_term,cos_mul,smplno);
		lowpass(b_term,sin_mul,smplno);
	    }
	    output_ph(smplno);              /* calculate mag. & phase for sample */
	    if ((outpnt = (int)(smplno * outdelta_t)) > lastout) {  /* if next out-time */
	        output(smplno, hno, outpnt);                 /*     place in     */
		lastout = outpnt;	                     /*     output array */
	    }
	    if (skip) {
	        skip = 0;       /* quit if no more samples in file */
		break;
	    }
            if (!POLL_EVENTS()) exit(1);
	}
}

static void lpinit(void) /* lowpass coefficient ititializer */
{	        /* 3rd order butterworth LPF coefficients calculated using */
		/* impulse invariance */
  float costerm,sinterm;
  double omega_c;

        omega_c = freq_c*TWOPI;
	costerm = (float)cos(SQRTOF3*omega_c*delta_t/2.0);
	sinterm = (float)sin(SQRTOF3*omega_c*delta_t/2.0);
	x1 = (float)(omega_c*delta_t*(exp(-omega_c*delta_t) +
				      exp(-omega_c*delta_t/2.0)
	    * (-costerm + sinterm/SQRTOF3)));
	x2 = (float)(omega_c*delta_t*(exp(-omega_c*delta_t) -
				      exp(-3*omega_c*delta_t/2)
	    * (costerm + sinterm/SQRTOF3)));
	yA = (-((float)exp(-omega_c*delta_t) +
		2.0f*(float)exp(-omega_c*delta_t/2)*costerm));
	y2 = 2.0f * (float)exp(-3.0*omega_c*delta_t/2.0)*costerm +
	  (float)exp(-omega_c*delta_t);
	y3 = (-(float)exp(-2.0*omega_c*delta_t));
}

static void lowpass(double *out, double *in, long smpl)    /* call with x1,x2,yA,y2,y3 initialized  */
		/* calls LPF function */
{
	PUTVAL(out, smpl, (x1 *
	    GETVAL(in,smpl-1) + x2 * GETVAL(in,smpl-2) -
	    yA * GETVAL(out,smpl-1) - y2 *
	    GETVAL(out,smpl-2) - y3 * GETVAL(out,smpl-3)));
}

static void average(long window,double *in,double *out, long smpl)  /* AVERAGES OVER 'WINDOW' SAMPLES */
			     /* this is actually a comb filter with 'Z' */
			/* transform of (1/w *[1 - Z**-w]/[1 - Z**-1]) */
			/* ie. zeros at all harmonic frequencies except*/
			     /* the current one where the pole cancels it */
{			     
	PUTVAL(out, smpl, (double)(GETVAL(out,smpl-1) +
	    (1/(double)window) * (GETVAL(in,smpl) - GETVAL(in,smpl-window))));
}

                                 /* update phase counter */
static void output_ph(long smpl)	/* calculates magnitude and phase components */
				/* for each samples quadrature components, & */
				/* and unwraps the phase.  A phase difference*/
{				/* is taken to represent the freq. change.   */
double 	delt_temp;	        /* the pairs are then comb filtered.	     */
double 	temp_a;

	if ((temp_a=GETVAL(a_term,smpl)) == 0)
	    new_ph=(-PI/2.0f)*sgn(GETVAL(b_term,smpl));
	else new_ph= -atan(GETVAL(b_term,smpl)/temp_a) - PI*u(-temp_a);

	if (fabs((double)new_ph - old_ph)>PI) 
	    jmp_ph -= TWOPI*sgn(temp_a);

	old_ph = new_ph;
	PUTVAL(r_phase,smpl,old_ph+jmp_ph);
	delt_temp = ((GETVAL(r_phase,smpl) - GETVAL(r_phase,smpl-1))/
		      (TWOPI*delta_t));
	if ((freq_c <= 1) || (smpl < 3)) {
	    PUTVAL(amp_av1,smpl,(float)sqrt(sq(GETVAL(a_term,smpl))
	    	  + sq(GETVAL(b_term,smpl))));
	    average(windsiz,amp_av1,amp_av2,smpl);
	    average(windsiz,amp_av2,amp_av3,smpl);
	    average(windsiz,amp_av3,r_ampl,smpl);
	    PUTVAL(ph_av1,smpl,delt_temp);
	    average(windsiz,ph_av1,ph_av2,smpl);
	    average(windsiz,ph_av2,ph_av3,smpl);
	    average(windsiz,ph_av3,a_avg,smpl);
	}
	else {
	    PUTVAL(r_ampl,smpl,(float)sqrt(sq(GETVAL(a_term,smpl))
	    	  + sq(GETVAL(b_term,smpl))));
	    PUTVAL(a_avg,smpl,delt_temp);
	}
}

static void output(long smpl, int hno, int pnt)    /* output one freq_mag pair */
			/* when called, gets frequency change */
			/* and adds it to current freq. stores*/
{				/* current amp and new freq in arrays */
  double delt_freq;
  float  new_amp, new_freq;

        if (pnt < num_pts) {
	    delt_freq = GETVAL(a_avg,smpl);        /* 0.5 for rounding ? */
	    FREQS[hno][pnt] = new_freq = (float)(delt_freq + cur_est);
	    MAGS[hno][pnt] = new_amp = (float)GETVAL(r_ampl,smpl);
	    if (new_freq > max_frq)
	        max_frq = new_freq;
	    if (new_amp > max_amp)
	        max_amp = new_amp;
  /*printf("A=%f\tF=%f\t%f\n",MAGS[hno][pnt],FREQS[hno][pnt],delt_freq);*/
	}
}

/* If this function worthwhile?  Need to coinsider recalculation */
static double sq(double num)     /* RETURNS SQUARE OF ARGUMENT */
{
	return(num*num);
}	

static void quit (char *msg)
{
#ifdef macintosh
	char temp[256];
	sprintf(temp,"hetro:  %s\n\tanalysis aborted\n",msg);
	die(temp);
#else
	err_printf("hetro:  %s\n\tanalysis aborted\n",msg);
#endif
	exit(1);
}

#define END  32767

static void filedump(void)     /* WRITE OUTPUT FILE in DATA-REDUCED format */
{
        int 	h, pnt, ofd, nbytes;
	double 	scale,x,y;
	short   **mags, **freqs, *magout, *frqout;
	double  ampsum, maxampsum = 0.0;
	long    lenfil = 0;
	short   *TIME;
	float   timesiz;
extern  int     openout(char *, int);

        mags = (short **) mmalloc(hmax * sizeof(short*));
        freqs = (short **) mmalloc(hmax * sizeof(short*));
	for (h = 0; h < hmax; h++) {
	    mags[h] = (short *)mmalloc((long)num_pts * sizeof(short));
	    freqs[h] = (short *)mmalloc((long)num_pts * sizeof(short));
	}

	TIME = (short *)mmalloc((long)num_pts * sizeof(short));
	timesiz = 1000.0f * input_dur / num_pts;
	for (pnt = 0; pnt < num_pts; pnt++)
	    TIME[pnt] = (short)(pnt * timesiz);
	
	if ((ofd = openout(outfilnam, 1)) < 0)     /* fullpath else cur dir */
	    quit("cannot create output file\n");

        write(ofd, (char*)&hmax, sizeof(hmax)); /* Write header */

	for (pnt=0; pnt < num_pts; pnt++) {
	    ampsum = 0.0;
	    for (h = 0; h < hmax; h++)
	        ampsum += MAGS[h][pnt];
	    if (ampsum > maxampsum)
		maxampsum = ampsum;
	}
	scale = m_ampsum / maxampsum;
	err_printf("scale = %f\n", scale);

	for (h = 0; h < hmax; h++) {
	    for (pnt = 0; pnt < num_pts; pnt++) {
	        x = MAGS[h][pnt] * scale;
		mags[h][pnt] = (short)(x*u(x));
		y = FREQS[h][pnt];
		freqs[h][pnt] = (short)(y*u(y)); 
	    }
	}

	magout = (short *)mmalloc((long)(num_pts + 1) * 2 * sizeof(short));
	frqout = (short *)mmalloc((long)(num_pts + 1) * 2 * sizeof(short));
	for (h = 0; h < hmax; h++) {
	    short *mp = magout, *fp = frqout;
	    short *lastmag, *lastfrq, pkamp = 0;
	    int mpoints, fpoints, contig = 0;
	    *mp++ = -1;                      /* set brkpoint type codes  */
	    *fp++ = -2;
	    lastmag = mp;
	    lastfrq = fp;
	    for (pnt = 0; pnt < num_pts; pnt++) {
	        short tim, mag, frq;
		tim = TIME[pnt];
		frq = freqs[h][pnt];
		if ((mag = mags[h][pnt]) > pkamp)
		    pkamp = mag;
		if (mag > amp_min) {
		    if (contig > 1) {        /* if third time this value  */
		        if (mag == *(mp-1) && mag == *(mp-3)
			                     /*    or 2nd time this slope */
			  || (float)(mag - *(mp-1)) / (tim - *(mp-2)) ==
			     (float)(*(mp-1) - *(mp-3)) / (*(mp-2) - *(mp-4)))
			    mp -= 2;              /* overwrite the previous */
		        if (frq == *(fp-1) && frq == *(fp-3)
			  || (float)(frq - *(fp-1)) / (tim - *(fp-2)) ==
			     (float)(*(fp-1) - *(fp-3)) / (*(fp-2) - *(fp-4)))
			    fp -= 2;
		    }
		    *mp++ = tim;
		    *mp++ = mag;
		    *fp++ = tim;
		    *fp++ = frq;
		    lastmag = mp;         /* record last significant seg  */
		    lastfrq = fp;
		    contig++;
		}
		else {
		    if (mp > lastmag) {   /* for non-significant segments */
		        mp = lastmag + 2; /*   only two points necessary  */
			fp = lastfrq + 2; /*   to peg the ends            */
		    }
		    *mp++ = tim;
		    *mp++ = 0;
		    *fp++ = tim;
		    *fp++ = frq;
		    contig = 0;
		}
	    }
	    if (lastmag < mp) {          /* if last signif not last point */
		mp = lastmag + 2;        /*   make it the penultimate mag */
		fp = lastfrq + 2;
	    }
	    *(mp-1) = 0;                 /* force the last mag to zero    */
	    if (fp - frqout > 3)
	        *(fp-1) = *(fp-3);       /*   & zero the freq change      */
	    *mp++ = END;                 /* add the sequence delimiters   */
	    *fp++ = END;
	    mpoints = ((mp - magout) / 2) - 1;
	    nbytes = (mp - magout) * sizeof(short);
	    write(ofd, (char *)magout, nbytes);
#ifdef DEBUG
	    {
		int i;
		for (i=0; i<(mp-magout); i++)
		    err_printf( "%hd,", magout[i]);
		err_printf( "\n");
	    }
#endif
	    lenfil += nbytes;
	    fpoints = ((fp - frqout) / 2) - 1;
	    nbytes = (fp - frqout) * sizeof(short);
	    write(ofd, (char *)frqout, nbytes);
#ifdef DEBUG
	    {
		int i;
		for (i=0; i<(fp-frqout); i++)
		    err_printf( "%hd,", frqout[i]);
		err_printf( "\n");
	    }
#endif
	    lenfil += nbytes;
	    printf("harmonic #%d:\tamp points %d, \tfrq points %d,\tpeakamp %d\n",
		   h,mpoints,fpoints,pkamp);
    	}
	err_printf("wrote %ld bytes to %s\n", lenfil, outfilnam);
	close(ofd);
#if !defined (macintosh) && !defined(SYMANTEC)
        if (access(outfilnam, 0x6))
          chmod(outfilnam, S_IREAD | S_IWRITE); /* Make read-write */
#endif
	mfree(magout);
	mfree(frqout);
	mfree(TIME);
	for (h = 0; h < hmax; h++) {
	    mfree(mags[h]);
	    mfree(freqs[h]);
	}
	mfree(mags);
	mfree(freqs);
}
