#!/bin/bash
#
# This script wraps GCC (or similar) and applies a filter to source code
# that converts all references to "long" to "int".  It also changes
# "%ld" and its ilk to plain "%d" (or whatever's appropriate).  The
# objective is to allow code where sizeof(long) is assumed to be 32 bits
# to be compiled on systems where sizeof(long) is 64 bits but sizeof(int)
# is 32 bits.  As a special case, "LONG" is passed through as "long" for
# the few cases where the name long is really required.
#
# Asumes GNU SED (or some other SED that supports "\<\>" to delimit words).
#
# Warning: assumes that "gcc -E" passes through line and file markers of
# the form "# 999 file", and furthermore that system headers (or other
# files that shouldn't be filtered) are absolute pathnames (i.e. start
# with "/" in the marker's file name) and that local includes (which we
# want filtered) are relative pathnames (i.e. the file name part of the
# marker doesn't start with "/").
#
# Script should work with KSH after just a change to the magic header line.
#
# Ed Hall (edhall@ayched.com), November 1, 1998
#

# temp file prefix, and cleanup
tmp=/tmp/cook$$
trap "rm -f $tmp.*" 0 1 2 15

# set a different compiler here, if you dare.
cc=gcc

# init before arg parsing loop.
oflag=false     # boolean to indicate "-o" flag
cflag=false     # boolean to indicate "-c" flag
srcs=""         # index of source (.c) files.
ofile=""        # argument supplied for "-o" flag (i.e. GCC output file)
need_oarg=false # internal flag set if "-o" arg is next

# scan args
for arg in "$@"; do
    if $need_oarg; then
	ofile="$arg"
	need_oarg=false
	continue
    fi

    case "$arg" in
	*.c) srcs="$srcs $arg" ;;
	-o) oflag=true; need_oarg=true ;;
	-c) cflag=true ;;
	*) args="$args "'"'"$arg"'"' ;;
    esac
done

# these two tests set command-line subcomponents (see "eval" statements below)
oargs=""
if $oflag; then
    oargs="-o $ofile"
fi

cargs=""
if $cflag; then
    cargs="-c"
fi

# loop through source files, if any
for file in $srcs; do
    # if no output file name has been set, define one.
    if [ $oflag = false ]; then
	oargs="-o ${file%.c}.o"
    fi
    # first pass--just preprocess through GCC
    eval "$cc -E $args $file > $tmp.I"
    # filter the preprocessed code
    sed -n -e ': proc' -e '/^#  *[0-9][0-9]*  *"\//b noproc' \
        -e '/".*%l[duxg]/s/%l\([duxg]\)/%\1/g' \
	-e 's/\<unsigned\>  *\<long\>/unsigned int/g' -e 's/\<long\>/int/g' \
	-e 's/\<LONG\>/long/g' -e 'p' -e 'n' -e 'b proc' \
	-e ': noproc' -e '/^#  *[0-9][0-9]*  *"[^\/]/b proc' \
	-e 'p' -e 'n' -e 'b noproc' $tmp.I >$tmp.i
    # now finish the compilation process
    eval "$cc $cargs $oargs $args $tmp.i"
    stat=$?
    if [ $stat -gt 0 ]; then
	exit $stat
    fi
done

# If no source files, just pass straight to GCC for linking or whatever
if [ -z "$srcs" ]; then
    eval "$cc $oargs $args"
    # return results immediately
    exit $?
fi
# success
exit 0
