#include <stdio.h>
#include <ctype.h>

#include <X11/Xlib.h>
#include "generic.h"
#include "list.h"


typedef struct {
	ltgenericd g;
	char *name;
	} ltfontnamed, *ltfontname, ltfontstyled, *ltfontstyle;

typedef struct {
	ltgenericd g;
	int size;
	} ltfontsized, *ltfontsize;

typedef struct XFONTFAMILY {
/* used for a list of the available fonts */
    char *fam_name;             /* family name, for menu */
    list styles; 		/* list of the various styles of this font */
    } xfontfamilyd, *xfontfamily;

typedef struct XFONTSTYLE {
/* used for a list of the available styles for a font family */
    xfontfamily family;         /* backpointer to family */
    char *style_name;           /* style name, for menu */
	int macnum;		/* for quickdraw */
	int macstyle;
    list fonts;    		/* list of the various sizes of this font */
    } xfontstyled, *xfontstyle;

typedef struct XFONTSIZE {
/* used for a list of the available font sizes, and data on them */
    xfontstyle style;           /* backpointer to style */
    char *x_name;               /* name for loading into X */
    char *laser_name;           /* name for printing on laserwriter */
	XFontStruct *font_data;	/* X Font data, NULL if not loaded yet */
    int sz;                     /* font size */
    } xfontsized, *xfontsize;

typedef struct {
	ltgenericd g;
	tdisplay dpy;
	XFontStruct *data;
	int h,a,d;		/* height,ascent,descent */
	int reqsize;		/* requested size */
	int macnum;		/* for quickdraw */
	int macstyle;
	ltfontname name;
	ltfontstyle style;
	ltfontsize size;	/* actual size */
	xfontfamily xfamily;
	xfontstyle xstyle;
	xfontsize xsize;
	} ltfontd, *ltfont;

/* list of fonts, and sizes and laserwriter names for each */
list font_list=NULL;
/* list of names, styles, and sizes */
list font_names;
list font_styles;
list font_sizes;
/* list of ltfont structures created */
list fonts=NULL;

int ch;

int tfont_gogetachar(f)
/* get a character, skipping comments if found at start of line, and newlines */
FILE *f;
{
    int t;

    t=fgetc(f);
    while(t=='#' || t=='\n')
    {
        if(t=='#')
            /* skip comment (rest of line) */
            while((t=fgetc(f))!='\n');
        t=fgetc(f);
    }
    return(t);
}

int tfont_skipspace(f)
/* skips space and tabs in the given input file */
FILE *f;
{
    int t;

    t=tfont_gogetachar(f);
    while(t==' ' || t=='\t')
        t=tfont_gogetachar(f);
    return(t);
}

char *tfont_read_fname(f, type)
/* read string till next , or ;, skipping newlines */
FILE *f;
int type;   /* =1 if we are testing for more entries, so don't print error */
{
    static char buf[100];
    int i=0;

    ch=tfont_skipspace(f);
    while(ch!=',' && ch!=';')
    {
        if(ch==EOF)
        {
            if(type!=1)
                fprintf(stderr, "Unexpected EOF in font_map file.\n");
            return(NULL);
        }
        else
            buf[i++]=ch;
        ch=tfont_gogetachar(f);
		if(i==99)
			break;
    }
    buf[i]='\0';
    return((char *)strdup(buf));
}

int tfont_read_num(f)
/* read number till next , or ;, skipping newlines */
FILE *f;
{
    int total=0;

    ch=tfont_skipspace(f);
    while(ch!=',' && ch!=';')
    {
        if(ch==EOF)
        {
            fprintf(stderr, "Unexpected EOF in font_map file.\n");
            return(-1);
        }
        else if(!isdigit(ch) && ch!=' ')
        {
            fprintf(stderr, "String found when expecting number");
            fprintf(stderr, " in font_map file.\n");
            while(ch!=';')
            {
                ch=tfont_gogetachar(ch);
                if(ch==EOF)
                {
                    fprintf(stderr, "Unexpected EOF in font_map file.\n");
                    return(-1);
                }
            }
            return(-1);
        }
        else if(ch==' ')
        {
            ch=tfont_skipspace(f);
            if(ch!=',' && ch!=';')
            {
                fprintf(stderr, "Unexpected spaces in number");
                fprintf(stderr, " in font_map file\n");
                while(ch!=';')
                {
                    ch=tfont_gogetachar(ch);
                    if(ch==EOF)
                    {
                        fprintf(stderr, "Unexpected EOF in font_map file.\n");
                        return(-1);
                    }
                }
                return(-1);
            }
        }
        else
        {
            total=total*10+ch-'0';
            ch=tfont_gogetachar(f);
        }
    } return(total);
}

xfontfamily tfont_get_family(n)
/* returns the xfontfamily if the given name exists in font_list */
char *n;
{
	list t;
	xfontfamily family;

	lfor(font_list, t)
	{
	    family=nodeobj(t);
	    if(strcmp(n, family->fam_name)==0)
		return(family);
	}
	return(NULL);
}

xfontstyle tfont_get_style(n, family)
/* returns the xfontstyle if the given name exists in the
	family's style list */
char *n;
xfontfamily family;
{
	list t;
	xfontstyle style;

	lfor(family->styles, t)
	{
	    style=nodeobj(t);
	    if(strcmp(n, style->style_name)==0)
		return(style);
	}
	return(NULL);
}

xfontsize tfont_get_size(n, style)
/* returns the xfontsize if the given name exists in the style's size list */
int n;
xfontstyle style;
{
	list t;
	xfontsize tsize;

	lfor(style->fonts, t)
	{
	    tsize=nodeobj(t);
	    if(n<=tsize->sz)
			return(tsize);
	}
	return(NULL);
}

ltfontname tfontname_new(name)
char *name;
{
	ltfontname tmp;

	tmp=(ltfontname)titem_new(NULL, lt_fontname, sizeof(ltfontnamed));
	if(tmp==NULL)
		return(NULL);

	tmp->name=(char *)strdup(name);
	return(tmp);
}

ltfontstyle tfontstyle_new(style)
char *style;
{
	ltfontstyle tmp;

	tmp=(ltfontstyle)titem_new(NULL, lt_fontstyle, sizeof(ltfontstyled));
	if(tmp==NULL)
		return(NULL);

	tmp->name=(char *)strdup(style);
	return(tmp);
}

ltfontsize tfontsize_new(size)
int size;
{
	ltfontsize tmp;

	tmp=(ltfontsize)titem_new(NULL, lt_fontsize, sizeof(ltfontsized));
	if(tmp==NULL)
		return(NULL);

	tmp->size=size;
	return(tmp);
}

int tfontname_free(n)
ltfontname n;
{
	/* check type */
	if(n==NULL)
		return(0);
	if(titem_type(n)!=lt_fontname)
		return(0);

	free(n->name);
	return(titem_free(n));
}

int tfontstyle_free(n)
ltfontstyle n;
{
	/* check type */
	if(n==NULL)
		return(0);
	if(titem_type(n)!=lt_fontstyle)
		return(0);

	free(n->name);
	return(titem_free(n));
}

int tfontsize_free(n)
ltfontsize n;
{
	/* check type */
	if(n==NULL)
		return(0);
	if(titem_type(n)!=lt_fontsize)
		return(0);

	return(titem_free(n));
}

int tfont_list(where, names, styles, sizes)
/* returns a lists of font names, styles, and sizes that can be used with the
	Laserwriter */
titem where;
list *names, *styles, *sizes;
{
    FILE *font_map;         /* for opening font map file */
    xfontsize t_font;       /* temporary structure holders, for placing into */
    xfontfamily t_family;   /*   fontlist */
    xfontstyle t_style;
    char *f_name;               /* temporary name & size holders */
    char *x_name;
    char *style;
    char *lasername;
	int macnum, macstyle;
    char t_name[100];
    int sz;
    char **fonts;           /* for XListFonts */
    int nfonts;
    int trying=0;           /* 0 if font_map found,
                                1 if times loaded,
                                2 if fixed loaded */
	Display *dpy;
	ltfontname tmpname;
	ltfontstyle tmpstyle;
	ltfontsize tmpsize;
	int alreadythere;

	if(font_list!=NULL)
	{
		*names=font_names;
		*styles=font_styles;
		*sizes=font_sizes;
		return(1);
	}

	/* check type of 'where', and get display */
	if(where==NULL)
		return(0);
	dpy=(Display *)tdisplay_X(titem_display(where));

    font_list=newlist();
    font_names=newlist();
    font_styles=newlist();
    font_sizes=newlist();

	{
		char *home;
		char buf[1024];

		home=(char *)getenv("HOME");
		font_map=NULL;

		if(home!=NULL)
		{
			strcpy(buf, home);
			strcat(buf, "/.font_map");
			font_map=fopen(buf, "r");
			if(font_map==NULL)
			{
				strcpy(buf, home);
				strcat(buf, "/font_map");
				font_map=fopen(buf, "r");
			}
		}
		if(font_map==NULL)
			font_map=fopen("/usr/lib/font_map", "r");
	}
    if(font_map==NULL)
    {
        /* try Helvetica or fixed */
        fprintf(stderr, "Could not open font file, only the Helvetica-12");
        fprintf(stderr, " font will be available.\n");
        trying=1;
        fonts=XListFonts(dpy, "-adobe-helvetica-medium-r-normal--12*",
            1, &nfonts);
        if(fonts==NULL)
        {
            fprintf(stderr, "Couldn't load Helvetica-12,");
            fprintf(stderr, " still a chance, trying fixed...\n");
            trying=2;
            fonts=XListFonts(dpy, "fixed", 1, &nfonts);
			macnum=21;
			macstyle=0;
            if(fonts==NULL)
            {
				/* couldn't load any fonts, not doable */
                fprintf(stderr, "Couldn't load a font!\n");
                fprintf(stderr, "Must Quit!\n");
				freelist(font_list);
				freelist(font_names);
				freelist(font_styles);
				freelist(font_sizes);
                return(0);
            }
        }
		else
		{
			macnum=21;
			macstyle=0;
		}
        /* now have Times or fixed */
        t_family=(xfontfamily)malloc(sizeof(xfontfamilyd));
        t_style=(xfontstyle)malloc(sizeof(xfontstyled));
        t_font=(xfontsize)malloc(sizeof(xfontsized));
        t_font->style=t_style; /* setup back pointers */
        t_style->family=t_family;
		t_style->macnum=macnum;
		t_style->macstyle=macstyle;
		t_family->styles=newlist();         /* setup list of styles */
		addnode(t_family->styles, t_style);
        t_style->fonts=newlist();           /* setup list of fonts */
        addnode(t_style->fonts, t_font);
        addnode(font_list, t_family);
        if(trying==1)
        {
            f_name="Helvetica";
            x_name="-adobe-helvetica-medium-r-normal--12";
            style="Normal";
            lasername="Helvetica";
            sz=12;
        }
        else
        {
            f_name="Fixed";
            x_name="fixed";
            style="Normal";
            lasername="Courier";
            sz=12;
        }
        t_font->laser_name=(char *)strdup(lasername);
        t_font->x_name=(char *)strdup(x_name);
        t_font->sz=sz;
		t_font->font_data=NULL;
		addnode(font_sizes, tfontsize_new(sz));
        t_style->style_name=(char *)strdup(style);
		addnode(font_styles, tfontstyle_new(style));

        t_family->fam_name=(char *)strdup(f_name);
		addnode(font_names, tfontname_new(f_name));
    }
	else
	{
		ch='\n';
		/* for each font in the file */
		while(ch!=EOF)
		{
			x_name=tfont_read_fname(font_map,1);
			if(x_name==NULL)
				break;
			lasername=tfont_read_fname(font_map,0);
			if(lasername==NULL)
				break;
			f_name=tfont_read_fname(font_map,0);
			if(f_name==NULL)
				break;
			style=tfont_read_fname(font_map,0);
			if(style==NULL)
				break;
			macnum=tfont_read_num(font_map);
			macstyle=tfont_read_num(font_map);
			t_family=tfont_get_family(f_name);
			if(t_family==NULL)
			{
				t_family=(xfontfamily)malloc(sizeof(xfontfamilyd));
				addnode(font_list, t_family);
				t_family->styles=newlist();
				t_family->fam_name=(char *)strdup(f_name);

				/* add to fontname list */
				addnode(font_names, tfontname_new(f_name));
			}
			t_style=(xfontstyle)malloc(sizeof(xfontstyled));
			addnode(t_family->styles, t_style);
			t_style->family=t_family;
			t_style->fonts=newlist();
			t_style->style_name=(char *)strdup(style);
			t_style->macnum=macnum;
			t_style->macstyle=macstyle;

			/* add to style list, if not already there */
			startlist(font_styles);
			for(alreadythere=0;(tmpstyle=listnext(font_styles))!=NULL;)
				if(strcmp(tmpstyle->name, style)==0)
				{
					alreadythere=1;
					break;
				}
			endlist(font_styles);
			if(!alreadythere)
				addnode(font_styles, tfontstyle_new(style));

			while(ch!=';')
			{
				sz=tfont_read_num(font_map);
				if(sz==-1)
					break;
				t_font=(xfontsize)malloc(sizeof(xfontsized));
				addnode(t_style->fonts, t_font);
				t_font->style=t_style;
				t_font->laser_name=(char *)strdup(lasername);
				sprintf(t_name, "%s--%d", x_name, sz);
				t_font->x_name=(char *)strdup(t_name);
				t_font->sz=sz;
				t_font->font_data=NULL;

				/* add to size list, if not already there */
				startlist(font_sizes);
				for(alreadythere=0;(tmpsize=listnext(font_sizes))!=NULL;)
					if(tmpsize->size==sz)
					{
						alreadythere=1;
						break;
					}
				endlist(font_sizes);
				if(!alreadythere)
					addnode(font_sizes, tfontsize_new(sz));
			}
		}
	}
	/* now font_list contains list of font families, styles, sizes */
	/* font_names, font_styles, font_sizes are setup correctly */
	*names=font_names;
	*styles=font_styles;
	*sizes=font_sizes;
	return(1);
}

tfontname tfont_get_name_from_item(i)
tpitem i;
{
	ltfontname tmp;

	/* check types */
	if(i==NULL)
		return(NULL);
	if(titem_type(i)!=lt_pitem)
		return(NULL);

	tmp=(tpitem)titem_get_intdata(i);

	if(tmp==NULL)
		return(NULL);
	if(titem_type(tmp)!=lt_fontname)
		return(NULL);
	return(tmp);
}

tfontstyle tfont_get_style_from_item(i)
tpitem i;
{
	ltfontstyle tmp;

	/* check types */
	if(i==NULL)
		return(NULL);
	if(titem_type(i)!=lt_pitem)
		return(NULL);

	tmp=(tpitem)titem_get_intdata(i);

	if(tmp==NULL)
		return(NULL);
	if(titem_type(tmp)!=lt_fontstyle)
		return(NULL);
	return(tmp);
}

tfontsize tfont_get_size_from_item(i)
tpitem i;
{
	ltfontsize tmp;

	/* check types */
	if(i==NULL)
		return(NULL);
	if(titem_type(i)!=lt_pitem)
		return(NULL);

	tmp=(tpitem)titem_get_intdata(i);

	if(tmp==NULL)
		return(NULL);
	if(titem_type(tmp)!=lt_fontsize)
		return(NULL);
	return(tmp);
}

tpchoice tfont_make_name_choicemenu(parent, x,y,w,h,right,below, l, function)
/* make a choice menu of font names from the given font name list, with the
	given callback routines */
tpanel parent;
int x,y,w,h;
titem right, below;
list l;
int (*function)();
{
    list tmp=listnode(l);
    ltfontname current;
    tpchoice t_choice;
	tpitem t_item;
	tpitem firstitem=NULL;

    t_choice=(tpchoice)tpchoice_new(parent, x, y, w, h, right, below,
		"Fonts", 1);
	if(t_choice==NULL)
		return(NULL);

    startlist(l);
    while((current=listnext(l))!=NULL)
    {
		/* create an item */
		t_item=(tpitem)tpitem_new(current->name, function);
		titem_set_intdata(t_item, current);

        /* add to the choice */
		tpchoice_add(t_choice, t_item, NULL);
    }
    tpchoice_set_default_item(t_choice, 0);
    /*tpchoice_fix_width(t_choice);*/
    setnode(l, tmp);
    return(t_choice);
}

tpchoice tfont_make_style_choicemenu(parent, x,y,w,h,right,below, l, function)
/* make a choice menu of font styles from the given font style list, with the
	given callback routines */
titem parent;
int x,y,w,h;
titem right, below;
list l;
int (*function)();
{
    list tmp=listnode(l);
    ltfontstyle current;
    tpchoice t_choice;
	tpitem t_item;
	tpitem firstitem=NULL;

    t_choice=(tpchoice)tpchoice_new(parent, x, y, w, h, right, below,
		"Styles", 1);
	if(t_choice==NULL)
		return(NULL);

    startlist(l);
    while((current=listnext(l))!=NULL)
    {
		/* create an item */
		t_item=(tpitem)tpitem_new(current->name, function);
		titem_set_intdata(t_item, current);

        /* add to the choice */
		tpchoice_add(t_choice, t_item, NULL);
    }
    tpchoice_set_default_item(t_choice, 0);
    /*tpchoice_fix_width(t_choice);*/
    setnode(l, tmp);
    return(t_choice);
}

tpchoice tfont_make_size_choicemenu(parent, x,y,w,h,right,below, l, function)
/* make a choice menu of font sizes from the given font size list, with the
	given callback routines */
titem parent;
int x,y,w,h;
titem right, below;
list l;
int (*function)();
{
    list tmp=listnode(l);
    ltfontsize current;
    tpchoice t_choice;
	tpitem t_item;
	char sizetext[500];
	tpitem firstitem=NULL;

    t_choice=(tpchoice)tpchoice_new(parent, x, y, w, h, right, below,
		"Sizes", 1);
	if(t_choice==NULL)
		return(NULL);

    startlist(l);
    while((current=listnext(l))!=NULL)
    {
		/* create an item */
		sprintf(sizetext, "%d", current->size);
		t_item=(tpitem)tpitem_new(sizetext, function);
		titem_set_intdata(t_item, current);

        /* add to the choice */
		tpchoice_add(t_choice, t_item, NULL);
    }
    tpchoice_set_default_item(t_choice, 0);
    /*tpchoice_fix_width(t_choice);*/
    setnode(l, tmp);
    return(t_choice);
}

int tfont_grey_menus(names, styles, sizes)
/* greys out the menu items according to the selections in each */
tpchoice names, styles, sizes;
{
    char *t_name, *f_name;
    tpitem item;
	xfontfamily tfam;
	xfontstyle tsty;
	int needtoselect;
	int tsize;
	ltfontname fn;
	ltfontstyle fs;
		
	/* check types */
	if(names==NULL || styles==NULL || sizes==NULL)
		return(0);
	if(titem_type(names)!=lt_pchoice ||
			titem_type(styles)!=lt_pchoice ||
			titem_type(sizes)!=lt_pchoice)
		return(0);

	/* get xfontfamily */
	fn=tfont_get_name_from_item(tpchoice_selected(names));
	tfam=tfont_get_family(fn->name);

	/* grey out style menu */
	item=tpchoice_first_item(styles);
	while(item!=NULL)
	{
		t_name=tpitem_get_name(item);
		if(tfont_get_style(t_name, tfam)!=NULL)
			tpitem_enable(item);
		else
		{
			if(tpitem_enabled(item))
				needtoselect=1;
			tpchoice_disable_item(styles, item);
		}
		item=tpchoice_next_item(styles);
	}

	/* get xfontstyle */
	fs=tfont_get_style_from_item(tpchoice_selected(styles));
	tsty=tfont_get_style(fs->name, tfam);

	/* grey out size menu */
	item=tpchoice_first_item(sizes);
	while(item!=NULL)
	{
		t_name=tpitem_get_name(item);
		sscanf(t_name, "%d", &tsize);
		if(tfont_get_size(tsize, tsty)!=NULL)
			tpitem_enable(item);
		else
		{
			if(tpitem_enabled(item))
				needtoselect=1;
			tpchoice_disable_item(sizes, item);
		}
		item=tpchoice_next_item(sizes);
	}
}

tfont tfont_new(where, name, style, size)
titem where;
ltfontname name;
ltfontstyle style;
ltfontsize size;
{
	ltfont tmp, q;
	tdisplay dpy;
	Display *xdpy;
	xfontfamily tfam;
	xfontstyle tsty;
	xfontsize tsz;
	static char search[1023];

	/* check types */
	if(where==NULL || name==NULL || style==NULL || size==NULL)
		return(NULL);
	if(titem_type(name)!=lt_fontname ||
			titem_type(style)!=lt_fontstyle ||
			titem_type(size)!=lt_fontsize)
		return(NULL);
	
	dpy=(tdisplay)titem_display(where);

	xdpy=(Display *)tdisplay_X(dpy);

	tfam=tfont_get_family(name->name);
	if(tfam==NULL)
		return(NULL);
	tsty=tfont_get_style(style->name, tfam);
	if(tsty==NULL)
		return(NULL);
	tsz=tfont_get_size(size->size, tsty);
	if(tsz==NULL)
		return(NULL);
	
	tmp=(ltfont)titem_new(where, lt_font, sizeof(ltfontd));
	if(tmp==NULL)
		return(NULL);

	tmp->reqsize=size->size;
	tmp->xsize=tsz;

	if(fonts==NULL)
		fonts=newlist();
	startlist(fonts);
	while((q=listnext(fonts))!=NULL)
		if(q->xsize==tmp->xsize && q->reqsize==tmp->reqsize)
			break;
	if(q==NULL)
	{
		startlist(fonts);
		addnode(fonts, tmp);
		tmp->dpy=dpy;
		tmp->name=name;
		tmp->style=style;
		tmp->size=size;
		tmp->xfamily=tfam;
		tmp->xstyle=tsty;
		tmp->macnum=tsty->macnum;
		tmp->macstyle=tsty->macstyle;
	}
	else
	{
		titem_free(tmp);
		tmp=q;
	}

	if(tsz->font_data==NULL)
	{
		sprintf(search, "%s*", tsz->x_name);
		tsz->font_data=XLoadQueryFont(xdpy, search);
		if(tsz->font_data==NULL)
		{
			titem_free(tmp);
			return(NULL);
		}
	}
	tmp->data=tsz->font_data;
	tmp->h=tmp->data->ascent+tmp->data->descent;
	tmp->d=tmp->data->descent;
	tmp->a=tmp->data->ascent;


	return(tmp);
}

tfont tfont_create_from_fontname(where, s)
titem where;
char *s;
{
	tdisplay dpy;
	Display *xdpy;
	ltfont tmp;

	/* check types */
	if(where==NULL || s==NULL)
		return(NULL);
	
	dpy=(tdisplay)titem_display(where);

	xdpy=(Display *)tdisplay_X(dpy);

	tmp=(ltfont)titem_new(where, lt_font, sizeof(ltfontd));
	if(tmp==NULL)
		return(NULL);
	
	tmp->dpy=dpy;
	tmp->name=NULL;
	tmp->style=NULL;
	tmp->size=NULL;
	tmp->xfamily=NULL;
	tmp->xstyle=NULL;
	tmp->xsize=NULL;
	tmp->reqsize=0;
	tmp->data=XLoadQueryFont(xdpy, s);
	tmp->macnum=3;
	tmp->macstyle=0;
	if(tmp->data==NULL)
	{
		titem_free(tmp);
		return(NULL);
	}
	tmp->h=tmp->data->ascent+tmp->data->descent;
	tmp->d=tmp->data->descent;
	tmp->a=tmp->data->ascent;

	return(tmp);
}

int tfont_free(f)
/* frees data for this font */
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
    return(titem_free(f));
}

int tfont_get_characteristics(f, name, style, size)
ltfont f;
tfontname *name;
tfontstyle *style;
tfontsize *size;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	*name=f->name;
	*style=f->style;
	*size=f->size;
	return(1);
}

int tfont_height(f)
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	return(f->h);
}

int tfont_ascent(f)
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	return(f->a);
}

int tfont_descent(f)
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	return(f->d);
}

int tfont_text_width(f, str, l)
ltfont f;
char *str;
int l;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	if(l<=0)
		return(0);
	
	return(XTextWidth(f->data, str, l));
}

Font tfont_Xid(f)
/* returns the xid of the font */
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(NULL);
	if(titem_type(f)!=lt_font)
		return(NULL);

	return(f->data->fid);
}

int tfont_get_extents(f, s, l, x, y, w, h)
/* given a string, and an x,y location, will return a rectangle that will
	enclose it */
ltfont f;
char *s;
int l, *x, *y, *w, *h;
{
	static XCharStruct overall;
	int dir, a, d;

	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	XTextExtents(f->data, s, l, &dir, &a, &d, &overall);
	*h=a+d+1;
	*w=overall.rbearing-overall.lbearing+2;
	*y=*y-a-1;
	*x=*x+overall.lbearing-1;
	return(1);
}

int tfont_exist(name, style, size)
ltfontname name;
ltfontstyle style;
ltfontsize size;
{
	xfontfamily tfam;
	xfontstyle tsty;
	xfontsize tsz;

	if(name==NULL || style==NULL || size==NULL)
		return(0);
	if(titem_type(name)!=lt_fontname ||
			titem_type(style)!=lt_fontstyle ||
			titem_type(size)!=lt_fontsize)
		return(0);
	
	tfam=tfont_get_family(name->name);
	if(tfam==NULL)
		return(0);
	tsty=tfont_get_style(style->name, tfam);
	if(tsty==NULL)
		return(0);
	tsz=tfont_get_size(size->size, tsty);
	if(tsz==NULL)
		return(0);
	return(1);
}

int tfont_set_selection(f,nm,sm,zm)
ltfont f;
tpchoice nm,sm,zm;
{
	tpitem i;

	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	i=tpchoice_first_item(nm);
	while(i!=NULL)
	{
	    if((ltfontname)titem_get_intdata(i)==f->name)
	    {
		tpchoice_select_item(nm, i);
		break;
	    }
	    i=tpchoice_next_item(nm);
	}
	i=tpchoice_first_item(sm);
	while(i!=NULL)
	{
	    if((ltfontstyle)titem_get_intdata(i)==f->style)
	    {
		tpchoice_select_item(sm, i);
		break;
	    }
	    i=tpchoice_next_item(sm);
	}
	i=tpchoice_first_item(zm);
	while(i!=NULL)
	{
	    if((ltfontsize)titem_get_intdata(i)==f->size)
	    {
		tpchoice_select_item(zm, i);
		break;
	    }
	    i=tpchoice_next_item(zm);
	}
	return(1);
}

tfont tfont_get_selected_font(where, namem, stylem, sizem)
/* returns the font selected in the menus */
titem where;
tpchoice namem, stylem, sizem;
{
	ltfontname n;
	ltfontstyle s;
	ltfontsize z;

	if(where==NULL || namem==NULL || stylem==NULL || sizem==NULL)
		return(NULL);
	if(titem_type(namem)!=lt_pchoice ||
			titem_type(stylem)!=lt_pchoice ||
			titem_type(sizem)!=lt_pchoice)
		return(NULL);
	
	n=tfont_get_name_from_item(tpchoice_selected(namem));
	s=tfont_get_style_from_item(tpchoice_selected(stylem));
	z=tfont_get_size_from_item(tpchoice_selected(sizem));
	return(tfont_new(where, n, s, z));
}

char *tfont_get_lasername(f)
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(NULL);
	if(titem_type(f)!=lt_font)
		return(NULL);

	return(f->xsize->laser_name);
}

int tfont_get_lasersize(f)
ltfont f;
{
	/* check type */
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);

	return(f->reqsize);
}

tfont tfont_get_font_by_names(where, n, s, z)
titem where;
char *n, *s;
int z;
{
	ltfontname fn;
	ltfontstyle fs;
	ltfontsize fz;

	if(where==NULL || n==NULL || s==NULL || z==0)
		return(NULL);
	
	startlist(font_names);
	while((fn=listnext(font_names))!=NULL)
		if(strcmp(n, fn->name)==0)
			break;
	if(fn==NULL)
		return(NULL);
	startlist(font_styles);
	while((fs=listnext(font_styles))!=NULL)
		if(strcmp(s, fs->name)==0)
			break;
	if(fs==NULL)
		return(NULL);
	startlist(font_sizes);
	while((fz=listnext(font_sizes))!=NULL)
		if(z==fz->size)
			break;
	if(fz==NULL)
		return(NULL);

	return(tfont_new(where, fn, fs, fz));
}

tfont tfont_get_first_font(where)
titem where;
{
	ltfontname fn;
	ltfontstyle fs;
	ltfontsize fz;

	if(where==NULL)
		return(NULL);
	
	startlist(font_names);
	fn=listnext(font_names);
	if(fn==NULL)
		return(NULL);
	startlist(font_styles);
	fs=listnext(font_styles);
	if(fs==NULL)
		return(NULL);
	startlist(font_sizes);
	fz=listnext(font_sizes);
	if(fz==NULL)
		return(NULL);

	return(tfont_new(where, fn, fs, fz));
}

int tfont_get_mac_number(f)
ltfont f;
{
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
	return(f->macnum);
}

int tfont_get_mac_style(f)
ltfont f;
{
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
	return(f->macstyle);
}

char *tfont_name(f)
ltfont f;
{
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
	return(f->name->name);
}

char *tfont_style(f)
ltfont f;
{
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
	return(f->style->name);
}

int tfont_size(f)
ltfont f;
{
	if(f==NULL)
		return(0);
	if(titem_type(f)!=lt_font)
		return(0);
	
	return(f->reqsize);
}
