#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

/* load.c:
 *
 * this module handles all the concerns for actually loading a tree.  It makes
 * sure modified trees are saved, deals with errors reading a tree, and deals
 * with loading a species list instead of a tree
 */

#include "interface.h"

#include "treedata.h"

int do_load(s, format)
char *s;
file_type format;
{
	FILE *f;
	list load_list;
	int old_trees;
	static char errormesg[1024];


	f=fopen(s, "r");
	if(f==NULL)
	{
		/* can't open file */
		sprintf(errormesg, "Couldn't open file '%s' for reading", s);
		tterr(errormesg, Notice);
		return(0);
	}

	if(format==Treetool)
	{
		tree t;
		int oneread=0;

		old_trees=number_of_trees();
		/* will load new trees into a new list, just in case something
		 * goes wrong */
		load_list=newlist();

		while((t=(tree)readrootedtree(f, 0))!=NULL)
		{
			addnode(load_list, t);
			oneread=1;
		}
		/* error if didn't load any trees */
		if(!oneread)
		{
			sprintf(errormesg, "Error Reading Tree File '%s'", s);
			tterr(errormesg, Notice);
			return(0);
		}
		/* remove selection */
		add_to_selection(NULL, NoSel, 0, 0.0);
		if(old_trees>0)
			/* in this case new trees were added to same page as old trees, so
			 * we've modified the treetool data file */
			modify();
		startlist(load_list);
		/* add all trees in load_list to canvas, and select all of them */
		while((t=listnext(load_list))!=NULL)
		{
			tree_add(t);
			add_to_selection(treeroot(t), SubtreeSel, 1, 0.0);
		}
		freelist(load_list);
		/* if we've added trees, then we leave the current file name alone.
		 * if we've loaded a file without having a file loaded, then we
		 * change the current filename to the one just loaded */
		if(old_trees==0)
			change_filename(s);
	}
	else if(format==List)
	{
		char buf[80];
		char *s;
		list c;

		/* a species list is just a list of node names.  When it is loaded, the
		 * nodes that are specified in the file are selected on screen */

		load_list=newlist();
		/* add all the names in the file to the load_list */
		while(fgets(buf, 80, f)!=NULL)
			if(buf[0]!='#')
			{
					s=(char *)index(buf, '\n');
					if(s!=NULL)
						*s='\0';
					addnode(load_list, strdup(buf));
			}
		/* select the nodes with the names in the load_list */
		select_names_in_list(load_list);
		/* free the strings in the load_list, and then free the load_list */
		lfor(load_list, c)
			free(nodeobj(c));
		freelist(load_list);
	}
	else
		return(0);

	return(1);
}
