/*
 * Rockwell.c
 *
 * This file contains the DYNALINK V1414VQE specific hardware stuff.
 *
 * This file is originally written for the Elsa 28.8 modem by:
 * Stefan Froehlich <e9025800@student.tuwien.ac.at>.
 *
 * Since commands are the same I've copied the code, and left out some
 * Elsa features....
 * The rockwell/dynalink V1414VQE is now maintained by:
 * Ard van Breemen <ard@cstmel.hobby.nl>.
 *
 */

#include "../include/voice.h"

char *Rockwell_c = "$Id: Rockwell.c,v 1.6 1996/08/06 19:50:24 marc Exp $";

/*
 * Here we save the current mode of operation of the voice modem when
 * switching to voice mode, so that we can restore it afterwards.
 */

static char rockwell_mode_save[VOICE_BUF_LEN] = "";

/*
 * Internal status variables for aborting some voice modem actions.
 */

static int stop_dialing;
static int stop_playing;
static int stop_recording;
static int stop_waiting;

/*
 * The ROCKWELL samples with 7200 samples per second with a maximum of 4 bit
 * per sample. We want to buffer voice data for 0.1 second, so we need a
 * buffer less or equal to 7200 * 0.5 * 0.1 = 360 bytes.
 */

#define ROCKWELL_BUFFER_SIZE 360
static int rockwell_buffer_size = ROCKWELL_BUFFER_SIZE;

/*
 * This function handles the <DLE> shielded codes.
 */

#define ST_NO_INPUT (0x00)
#define ST_GOT_DLE  (0x01)

static int rockwell_answer_state = ST_NO_INPUT;

static void handle_rockwell_answer _P1((new_byte),char new_byte) {
     switch (rockwell_answer_state) {
          case ST_NO_INPUT: {
               switch (new_byte) {
                    case DLE: {
                         rockwell_answer_state = ST_GOT_DLE;
                         break;
                    }
                    case XON: {
                         lprintf( L_WARN, "%s: Received XON", program_name, new_byte);
                         break;
                    }
                    case XOFF: {
                         lprintf(L_WARN, "%s: Received XOFF", program_name, new_byte);
                         break;
                    }
                    case NL:
                    case CR: {
                         break;
                    }
                    default: {
                         lprintf(L_ERROR, "%s: Illegal modem answer 0x%2x", program_name, new_byte);
                         break;
                    }
               }
               return;
          }
          case ST_GOT_DLE: {
               switch (new_byte) {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                    case '*':
                    case '#':
                    case 'A':
                    case 'B':
                    case 'C':
                    case 'D': {
                         voice_handle_event(RECEIVED_DTMF,
                          (event_data) new_byte);
                         break;
                    }
                    case 'c': {
                         voice_handle_event(FAX_CALLING_TONE,
                          (event_data) 0);
                         break;
                    }
                    case 'e': {
                         voice_handle_event(DATA_CALLING_TONE,
                          (event_data) 0);
                         break;
                    }
                    case 's': {
                         voice_handle_event(NO_VOICE_ENERGY,
                          (event_data) 0);
                         break;
                    }
                    case 'q': {
                         voice_handle_event(SILENCE_DETECTED,
                          (event_data) 0);
                         break;
                    }
                    case 'b': {
                         voice_handle_event(BUSY_TONE,
                          (event_data) 0);
                         break;
                    }
                    case 'd': {
                         voice_handle_event(DIAL_TONE,
                          (event_data) 0);
                         break;
                    }
                    case 't': {
                         voice_handle_event(HANDSET_OFF_HOOK,
                          (event_data) 0);
                         break;
                    }
                    case 'h': {
                         voice_handle_event(HANDSET_ON_HOOK,
                          (event_data) 0);
                         break;
                    }
                    case 'a': /* ANSWER TONE CCITT V.25/T.30 2100Hz (Data/Fax) */
                    case 'f': /* BELL ANSWER TONE 2225 Hz */
                    case 'o': /* OVERRUN */
                    case 'u': /* UNDERRUN */
                    case 'T': /* TIMER MARK */
                    default: {
                         lprintf(L_ERROR, "%s: Unknown <DLE> shielded code 0x%x", program_name, new_byte);
                    }
               }
               rockwell_answer_state = ST_NO_INPUT;
               return;
          }
     }
}

/*
 * This is the main handle event routine for the ROCKWELL.
 */

int Rockwell_handle_event _P2((event,data),int event, event_data data) {
     char buffer[VOICE_BUF_LEN];
     switch (event) {
          case VOICE_ANSWER_PHONE: {
               return(voice_command("ATA", "VCON"));
          }
          case VOICE_BEEP: {
               sprintf(
                    buffer,
                    "AT#VTS=[%d,0,%d]",
                    data.beep.frequency,data.beep.length
               );
               if (voice_command(buffer, "OK") != VMA_USER_1) return(ERROR);
               return(OK);
          }
          case VOICE_DIAL: {
               int result = ERROR;
               voice_modem_state = DIALING;
               stop_dialing = FALSE;
               sprintf(buffer, "ATD %s", (char*) data.p);
               if (voice_command(buffer, "") != OK) return(ERROR);
               while (!stop_dialing) {
                    voice_read(buffer);
                    result = voice_analyze(buffer, NULL);
                    switch (result) {
                         case VMA_BUSY:
                         case VMA_FAIL:
                         case VMA_ERROR:
                         case VMA_NO_ANSWER:
                         case VMA_NO_CARRIER:
                         case VMA_NO_DIAL_TONE: {
                              stop_dialing = TRUE;
                              result = ERROR;
                              break;
                         }
                         case VMA_VCON: {
                              stop_dialing = TRUE;
                              result = OK;
                              break;
                         }
                    }
               }
               voice_modem_state = IDLE;
               return(result);
          }
          case VOICE_INIT: {
               lprintf(L_MESG, "initializing ROCKWELL voice modem");
               voice_modem_state = INITIALIZING;
               Rockwell_handle_event(VOICE_MODE_ON, (event_data) 0);
               sprintf(buffer, "AT#VSP=%1u", cvd.rec_silence_len.d.i);
               if (voice_command(buffer, "OK") != VMA_USER_1) {
                    lprintf(L_WARN, "can't set silence period");
               }
               if (voice_command("AT#VSD=1", "OK") != VMA_USER_1) {
                    lprintf(L_WARN, "can't set silence detection");
               }
               if (voice_command("AT#VTD=3F,3F,3F","OK") != VMA_USER_1) {
                    lprintf(L_WARN, "can't set DLE responses");
               }
               if (
                    (cvd.rec_silence_threshold.d.i > 100) ||
                    (cvd.rec_silence_threshold.d.i < 0)
               ) { lprintf(L_ERROR, "Invalid threshold value.");
                    return(ERROR);
               }
               sprintf(buffer, "AT#VSS=%1u", cvd.rec_silence_threshold.d.i * 3 / 100);
               if (voice_command(buffer, "OK") != VMA_USER_1) {
                    lprintf(L_WARN, "can't set silence threshold");
               }
               Rockwell_handle_event(VOICE_MODE_OFF, (event_data) 0);
               if (voice_command("AT&K3", "OK") == VMA_USER_1) {
                    TIO tio;
                    tio_get(voice_fd, &tio);
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD);
                    tio_set(voice_fd, &tio);
               } else {
                    lprintf(L_WARN, "can't turn on hardware flow control");
               }
               voice_modem_state = IDLE;
               return(OK);
          }
          case VOICE_MESSAGE_LIGHT_OFF: {
               voice_command("ATS0=0", "OK");
               return(OK);
          }
          case VOICE_MESSAGE_LIGHT_ON: {
               voice_command("ATS0=0", "OK");
               return(OK);
          }
          case VOICE_MODE_OFF: {
               sprintf(buffer, "AT#CLS=%s", rockwell_mode_save);
               voice_command(buffer, "OK");
               return(OK);
          }
          case VOICE_MODE_ON: {
               voice_command("AT#CLS?", "");
               voice_read(rockwell_mode_save);
               voice_flush(1);
               voice_command("AT#CLS=8", "OK");
               return(OK);
          }
          case VOICE_PLAY_FILE: {
               TIO tio_save;
               TIO tio;
               char input_buffer[ROCKWELL_BUFFER_SIZE];
               char output_buffer[2 * ROCKWELL_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int bytes_written;
               stop_playing = FALSE;
               voice_modem_state = PLAYING;
               voice_command("AT#VTX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;
               tio_set_flow_control(voice_fd, &tio, FLOW_HARD);
               tio_set(voice_fd, &tio);
               while (!stop_playing) {
                    if (
                         (bytes_in = read(data.i, input_buffer, rockwell_buffer_size)) <= 0
                    ) {
                         break;
                    }
                    bytes_out = 0;
                    for(i = 0; i < bytes_in; i++) {
                         output_buffer[bytes_out] = input_buffer[i];
                         if (output_buffer[bytes_out++] == DLE) {
                              output_buffer[bytes_out++] = DLE;
                         }
                    }
                    lprintf(
                         L_JUNK,
                         "%s: <DATA %d bytes>",
                         program_name, bytes_out
                    );
                    errno = 0;
                    bytes_written = 0;
                    while ( ( (bytes_written += write(voice_fd, &output_buffer[bytes_written], bytes_out - bytes_written)) != bytes_out) && (errno == 0)) ;
                    if (bytes_written != bytes_out) {
                         lprintf(
                              L_ERROR,
                              "%s: could only write %d bytes of %d bytes (errno 0x%x)",
                              program_name, bytes_written, bytes_out, errno
                         );
                    }
                    while (check_for_input(voice_fd)) {
                         char modem_byte;
                         if (read(voice_fd, &modem_byte, 1) != 1) {
                              lprintf(L_ERROR, "%s: could not read byte from voice modem", program_name);
                         } else {
                              handle_rockwell_answer(modem_byte);
                         }
                    }
               }
               if (stop_playing) {
                    sprintf(output_buffer, "%c%c", DLE, CAN);
                    lprintf(L_JUNK, "%s: <DLE><CAN>", program_name);
               } else {
                    sprintf(output_buffer, "%c%c", DLE, ETX);
                    lprintf(L_JUNK, "%s: <DLE><ETX>", program_name);
               }
               write(voice_fd, output_buffer, strlen(output_buffer));
               tio_set(voice_fd, &tio_save);
               voice_command("", "VCON");
               voice_modem_state = IDLE;
               if (stop_playing) {
                    return(INTERRUPTED);
               }
               return(OK);
          }
          case VOICE_RECORD_FILE: {
               TIO tio_save;
               TIO tio;
               char input_buffer[ROCKWELL_BUFFER_SIZE];
               char output_buffer[ROCKWELL_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int got_DLE_ETX = FALSE;
               int was_DLE = FALSE;
               stop_recording = FALSE;
               voice_modem_state = RECORDING;
               voice_command("AT#VRX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;
               tio_set_flow_control(voice_fd, &tio, FLOW_HARD);
               tio.c_cc[VMIN] = (rockwell_buffer_size > 0xff) ? 0xff : rockwell_buffer_size;
               tio.c_cc[VTIME] = 1;
               tio_set(voice_fd, &tio);
               while (!got_DLE_ETX) {
                    if ((bytes_in = read(voice_fd, input_buffer, rockwell_buffer_size)) <= 0) {
                         lprintf(L_ERROR, "%s: could not read byte from voice modem", program_name);
                         return(FAIL);
                    };
                    bytes_out = 0;
                    for (i = 0; (i < bytes_in) && !got_DLE_ETX; i++) {
                         if (was_DLE) {
                              was_DLE = FALSE;
                              lprintf(L_JUNK, "DLE followed by char(%u): %c", (unsigned int)input_buffer[i], input_buffer[i]);
                              switch (input_buffer[i]) {
                                   case DLE: {
                                        output_buffer[bytes_out++] = DLE;
                                        break;
                                   }
                                   case ETX: {
                                        got_DLE_ETX = TRUE;
                                        lprintf(L_JUNK, "%s: <DATA %d bytes>",
                                        voice_modem_name, bytes_out);
                                        lprintf(L_JUNK, "%s: <DLE><ETX>", voice_modem_name);
                                        break;
                                   }
                                   default: {
                                        handle_rockwell_answer(DLE);
                                        handle_rockwell_answer(input_buffer[i]);
                                        break;
                                   }
                              }
                         } else {
                              if (input_buffer[i] == DLE) {
                                   lprintf(L_JUNK, "Read a DLE");
                                   was_DLE = TRUE;
                              } else {
                                   output_buffer[bytes_out++] = input_buffer[i];
                              }
                         }
                    }
                    write(data.i, output_buffer, bytes_out);
               }
               tio_set(voice_fd, &tio_save);
               if (voice_analyze(&input_buffer[i], "\r\nVCON") == VMA_USER_1) {
                    lprintf(L_JUNK, "%s: VCON", voice_modem_name);
               } else {
                    int j;
                    lprintf(L_JUNK, "%s: data left in buffer: ", voice_modem_name);
                    for (j = i; j < bytes_in ; j++) {
                         lputc(L_JUNK, input_buffer[j]);
                    }
                    voice_command("AT", "OK");
               }
               voice_modem_state = IDLE;
               return(OK);
          }
          case VOICE_SET_COMPRESSION: {
               switch (data.i) {
                    case 0:
                    case 2: {
                         rockwell_buffer_size = ROCKWELL_BUFFER_SIZE * 2 / 4;
                         voice_command("AT#VBS=2", "OK");
                         return(OK);
                    }
                    case 4: {
                         rockwell_buffer_size = ROCKWELL_BUFFER_SIZE * 4 / 4;
                         voice_command("AT#VBS=4", "OK");
                         return(OK);
                    }
                    /* NOT TESTED, I HAVE NO PCM MODEM! */
                    case 8: {
                         rockwell_buffer_size = ROCKWELL_BUFFER_SIZE * 4 / 4;
                         voice_command("AT#VBS=8", "OK");
                         return(OK);
                    }
                    case 16: {
                         rockwell_buffer_size = ROCKWELL_BUFFER_SIZE * 4 / 4;
                         voice_command("AT#VBS=16", "OK");
                         return(OK);
                    }
               }
               lprintf(L_WARN, "ROCKWELL handle event: Illegal voice compression method (%d)", data.i);
               return(FAIL);
          }
          case VOICE_SET_DEVICE: {
               switch (data.i) {
                    case NO_DEVICE: {
                         voice_command("AT#VLS=0", "OK");
                         return(OK);
                    }
                    case DIALUP_LINE: {
                         voice_command("AT#VLS=4", "OK");
                         return(OK);
                    }
                    case EXTERNAL_MICROPHONE: {
                         voice_command("AT#VLS=3", "VCON");
                         return(OK);
                    }
                    case INTERNAL_SPEAKER: {
                         voice_command("AT#VLS=2", "VCON");
                         return(OK);
                    }
                    case LOCAL_HANDSET: {
                         voice_command("AT#VLS=1","VCON");
                         return(OK);
                    }
               }
               lprintf(L_WARN, "ROCKWELL handle event: Unknown output device (%d)", data.i);
               return(FAIL);
          }
          case VOICE_STOP_DIALING: {
               stop_dialing = TRUE;
               return(OK);
          }
          case VOICE_STOP_PLAYING: {
               stop_playing = TRUE;
               return(OK);
          }
          case VOICE_STOP_RECORDING: {
               stop_recording = TRUE;
               voice_write("!");
               return(OK);
          }
          case VOICE_STOP_WAITING: {
               stop_waiting = TRUE;
               return(OK);
          }
          case VOICE_SWITCH_TO_DATA_FAX:
               sprintf(buffer, "AT#CLS=%s", (char *) data.p);
               return(voice_command(buffer, "OK"));
          case VOICE_WAIT: {
               stop_waiting = FALSE;
               voice_modem_state = WAITING;
               alarm(data.i);
               while (!stop_waiting) {
                    while (check_for_input(voice_fd)) {
                         char modem_byte;
                         if (read(voice_fd, &modem_byte, 1) != 1) {
                              lprintf(L_ERROR, "%s: could not read byte from voice modem", program_name);
                         } else {
                              handle_rockwell_answer(modem_byte);
                         }
                    }
                    delay(100);
               }
               voice_modem_state = IDLE;
               alarm(0);
               return(OK);
          }
     }
     lprintf(L_WARN, "ROCKWELL handle event: Unknown event %04x", event);
     return(FAIL);
}
