/*
 * US_Robotics.c
 *
 * This file contains specific hardware stuff for US Robotics based
 * voice modems.
 *
 * Steve Wormley <wormley@step.mother.com> mailed me this first version
 * of the hardware driver. He suspects, that there are many bugs in it.
 * Well, let's see...
 *
 */

#include "../include/voice.h"

char *libvoice_US_Robotics_c = "$Id: US_Robotics.c,v 1.6 1996/08/06 19:50:25 marc Exp $";

/*
 * Here we save the current mode of operation of the voice modem when
 * switching to voice mode, so that we can restore it afterwards.
 */

static char mode_save[VOICE_BUF_LEN] = "";

/*
 * Internal status variables for aborting some voice modem actions.
 */

static int stop_dialing;
static int stop_playing;
static int stop_recording;
static int stop_waiting;

/*
 * The US Robotics modems sample with 7200 samples per second
 * with a maximum of 4 bit per sample. We want to buffer voice
 * data for 0.1 second, so we need a buffer less or equal to
 * 7200 * 0.5 * 0.1 = 360 bytes.
 */

#define USR_BUFFER_SIZE 360
static int usr_buffer_size = USR_BUFFER_SIZE;

/*
 * This function handles the <DLE> shielded codes.
 */

#define ST_NO_INPUT (0x00)
#define ST_GOT_DLE  (0x01)

static int modem_answer_state = ST_NO_INPUT;

static void handle_modem_answer(char new_byte)
     {

     switch (modem_answer_state)
          {
          case ST_NO_INPUT:

               switch (new_byte)
                    {
                    case DLE:
                         modem_answer_state = ST_GOT_DLE;
                         break;
                    case XON:
                         lprintf(L_WARN, "%s: Received XON",
                          program_name, new_byte);
                         break;
                    case XOFF:
                         lprintf(L_WARN, "%s: Received XOFF",
                          program_name, new_byte);
                         break;
                    case NL:
                    case CR:
                         break;
                    default:
                         lprintf(L_ERROR, "%s: Illegal modem answer 0x%2x",
                          program_name, new_byte);
                    };

               return;
          case ST_GOT_DLE:

               switch (new_byte)
                    {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                    case '*':
                    case '#':
                         voice_handle_event(RECEIVED_DTMF,
                          (event_data) new_byte);
                         break;
                    case 'c':
                         voice_handle_event(FAX_CALLING_TONE,
                          (event_data) 0);
                         break;
                    case 'e':
                         voice_handle_event(DATA_CALLING_TONE,
                          (event_data) 0);
                         break;
                    case 's':
                         voice_handle_event(NO_VOICE_ENERGY,
                          (event_data) 0);
                         break;
                    case 'q':
                         voice_handle_event(SILENCE_DETECTED,
                          (event_data) 0);
                         break;
                    case 'b':
                         voice_handle_event(BUSY_TONE,
                          (event_data) 0);
                         break;
                    case 'd':
                         voice_handle_event(DIAL_TONE,
                          (event_data) 0);
                         break;
                    default:
                         lprintf(L_ERROR, "%s: Unknown <DLE> shielded"
                          " code 0x%x", program_name, new_byte);
                    };

               modem_answer_state = ST_NO_INPUT;
               return;
          };

     };

/*
 * This is the main handle event routine for the US Robotics modems.
 */

int US_Robotics_handle_event(int event, event_data data)
     {
     char buffer[VOICE_BUF_LEN];

     switch (event)
          {
          case VOICE_ANSWER_PHONE:
               return(voice_command("AT#VLS=4#CLS=8A", "VCON"));
          case VOICE_BEEP:
               sleep(1);
               sprintf(buffer, "AT#VTS=[%d,0,%d]", data.beep.frequency,
                data.beep.length);

               if (voice_command("AT","OK") != VMA_USER_1)
                    return(ERROR);

               sleep(1);

               if (voice_command(buffer, "OK") != VMA_USER_1)
                    return(ERROR);

               if (voice_command("AT","OK") != VMA_USER_1)
                    return(ERROR);

               return(OK);
          case VOICE_DIAL:
               {
               int result = ERROR;

               voice_modem_state = DIALING;
               stop_dialing = FALSE;
               sprintf(buffer, "ATD%s", (char*) data.p);

               if (voice_command(buffer, "") != OK)
                    return(ERROR);

               while (!stop_dialing)
                    {
                    voice_read(buffer);
                    result = voice_analyze(buffer, NULL);

                    switch (result)
                         {
                         case VMA_BUSY:
                         case VMA_FAIL:
                         case VMA_ERROR:
                         case VMA_NO_ANSWER:
                         case VMA_NO_CARRIER:
                         case VMA_NO_DIAL_TONE:
                              stop_dialing = TRUE;
                              result = ERROR;
                              break;
                         case VMA_VCON:
                              stop_dialing = TRUE;
                              result = OK;
                              break;
                         };

                    };

               voice_modem_state = IDLE;
               return(result);
               };
          case VOICE_INIT:
               lprintf(L_MESG, "initializing USR voice modem");
               voice_modem_state = INITIALIZING;

               if ( voice_command("ATS30=60#VTD=3f,1,3f", "OK") !=
                VMA_USER_1)
                    lprintf(L_WARN, "voice init failed, continuing");

               sprintf(buffer, "AT#VSD=1#VSS=%d#VSP=%d",
                cvd.rec_silence_threshold.d.i * 3 / 100,
                 cvd.rec_silence_len.d.i);

               if (voice_command(buffer, "OK") != VMA_USER_1)
                    lprintf(L_WARN,
                     "setting recording preferences didn't work");

               if (voice_command("AT&R2&I2", "OK") == VMA_USER_1)
                    {
                    TIO tio;

                    tio_get(voice_fd, &tio);
                    tio_set_flow_control(voice_fd, &tio, FLOW_BOTH);
                    tio_set(voice_fd, &tio);
                    }
               else
                    lprintf(L_WARN,
                     "can't turn on hardware flow control");

               voice_modem_state = IDLE;
               return(OK);
          case VOICE_MESSAGE_LIGHT_OFF:
               return(voice_command("ATS0=0", "OK"));
          case VOICE_MESSAGE_LIGHT_ON:
               return(voice_command("ATS0=255", "OK"));
          case VOICE_MODE_OFF:
               sprintf(buffer, "AT#CLS=%s", mode_save);
               voice_command(buffer, "OK");
               return(OK);
          case VOICE_MODE_ON:
               voice_command("AT#CLS?", "");
               voice_read(mode_save);
               voice_flush(1);
               voice_command("AT#CLS=8", "OK");
               return(OK);
          case VOICE_PLAY_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[USR_BUFFER_SIZE];
               char output_buffer[2 * USR_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int bytes_written;

               stop_playing = FALSE;
               voice_modem_state = PLAYING;
               voice_command("AT#VTX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;

               if (cvd.do_hard_flow.d.i)
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD);
               else
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD |
                     FLOW_XON_OUT);

               tio_set(voice_fd, &tio);

               while (!stop_playing)
                    {

                    if ((bytes_in = read(data.i, input_buffer,
                     usr_buffer_size)) <= 0)
                         break;

                    bytes_out = 0;

                    for(i = 0; i < bytes_in; i++)
                         {
                         output_buffer[bytes_out] = input_buffer[i];

                         if (output_buffer[bytes_out++] == DLE)
                              output_buffer[bytes_out++] = DLE;

                         };

                    lprintf(L_JUNK, "%s: <DATA %d bytes>", program_name,
                     bytes_out);

                    errno = 0;
                    bytes_written = 0;

                    while (((bytes_written += write(voice_fd,
                     &output_buffer[bytes_written], bytes_out -
                     bytes_written)) != bytes_out) && (errno == 0))
                         ;

                    if (bytes_written != bytes_out)
                         lprintf(L_ERROR,
                          "%s: could only write %d bytes of %d bytes (errno 0x%x)",
                          program_name, bytes_written, bytes_out, errno);

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    };

               if (stop_playing)
                    {
                    sprintf(output_buffer, "%c%c", DLE, CAN);
                    lprintf(L_JUNK, "%s: <DLE><CAN>", program_name);
                    }
               else
                    {
                    sprintf(output_buffer, "%c%c", DLE, ETX);
                    lprintf(L_JUNK, "%s: <DLE><ETX>", program_name);
                    };

               write(voice_fd, output_buffer, strlen(output_buffer));
               tio_set(voice_fd, &tio_save);
               voice_command("", "VCON");
               voice_modem_state = IDLE;

               if (stop_playing)
                    return(INTERRUPTED);

               return(OK);
               };
          case VOICE_RECORD_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[USR_BUFFER_SIZE];
               char output_buffer[USR_BUFFER_SIZE];
               int i = 0;
               int bytes_in = 0;
               int bytes_out;
               int got_DLE_ETX = FALSE;
               int was_DLE = FALSE;

               stop_recording = FALSE;
               voice_modem_state = RECORDING;
               voice_command("AT#VRX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;

               if (cvd.do_hard_flow.d.i)
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD);
               else
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD |
                    FLOW_XON_IN);

               tio.c_cc[VMIN] = (usr_buffer_size > 0xff) ? 0xff :
                usr_buffer_size;
               tio.c_cc[VTIME] = 1;
               tio_set(voice_fd, &tio);

               while (!got_DLE_ETX)
                    {

                    if ((bytes_in = read(voice_fd, input_buffer,
                     usr_buffer_size)) <= 0)
                         {
                         lprintf(L_ERROR,
                          "%s: could not read byte from voice modem",
                          program_name);
                         return(FAIL);
                         };

                    bytes_out = 0;

                    for (i = 0; (i < bytes_in) && !got_DLE_ETX; i++)
                         {

                         if (was_DLE)
                              {
                              was_DLE = FALSE;

                              switch (input_buffer[i])
                                   {
                                   case DLE:
                                        output_buffer[bytes_out++] = DLE;
                                        break;
                                   case ETX:
                                        got_DLE_ETX = TRUE;
                                        lprintf(L_JUNK, "%s: <DATA %d bytes>",
                                         voice_modem_name, bytes_out);
                                        lprintf(L_JUNK, "%s: <DLE><ETX>",
                                         voice_modem_name);
                                        break;
                                   default:
                                        handle_modem_answer(DLE);
                                        handle_modem_answer(input_buffer[i]);
                                   };

                              }
                         else
                              {

                              if (input_buffer[i] == DLE)
                                   was_DLE = TRUE;
                              else
                                   output_buffer[bytes_out++] =
                                    input_buffer[i];

                              };

                         };

                    write(data.i, output_buffer, bytes_out);

                    if (!got_DLE_ETX)
                         lprintf(L_JUNK, "%s: <DATA %d bytes>",
                          voice_modem_name, bytes_out);

                    };

               tio_set(voice_fd, &tio_save);

               if (voice_analyze(&input_buffer[i], "\r\nVCON") == VMA_USER_1)
                    lprintf(L_JUNK, "%s: VCON", voice_modem_name);
               else
                    {
                    int j;

                    lprintf(L_JUNK, "%s: data left in buffer: ",
                     voice_modem_name);

                    for (j = i; j < bytes_in ; j++)
                         lputc(L_JUNK, input_buffer[j]);

                    voice_command("AT", "OK");
                    };

               voice_modem_state = IDLE;
               return(OK);
               };
          case VOICE_SET_COMPRESSION:

               switch (data.i)
                    {
                    case 0:
                    case 2:
                         usr_buffer_size = USR_BUFFER_SIZE * 2 / 4;
                       if ( voice_command("AT#VBS=2", "OK") != VMA_FAIL)
                       { return(OK) ; } else {return(VMA_FAIL);}
                    case 3:
                         usr_buffer_size = USR_BUFFER_SIZE * 3 / 4;
                       if ( voice_command("AT#VBS=3", "OK") != VMA_FAIL)
                       { return(OK) ; } else {return(VMA_FAIL);}
                    case 4:
                         usr_buffer_size = USR_BUFFER_SIZE * 4 / 4;
                       if ( voice_command("AT#VBS=4", "OK") != VMA_FAIL)
                       { return(OK) ; } else {return(VMA_FAIL);}
                    };

               lprintf(L_WARN,
                "USR handle event: Illeagal voice compression method (%d)",
                data.i);
               return(FAIL);
          case VOICE_SET_DEVICE:

               switch (data.i)
                    {
                    case NO_DEVICE:
                         voice_write("AT#VLS=0");

                         if (voice_command("", "AT#VLS=0|OK|VCON") == VMA_USER_1)
                              voice_command("", "OK|VCON");

                         return(OK);
                    case DIALUP_LINE:
                         voice_command("AT#VLS=0", "OK|VCON");
                         return(OK);
                    case EXTERNAL_MICROPHONE:
                         voice_command("AT#VLS=1", "OK|VCON");
                         return(OK);
                    case INTERNAL_SPEAKER:
                         voice_command("AT#VLS=2", "OK|VCON");
                         return(OK);
                    case INTERNAL_MICROPHONE:
                         voice_command("AT#VLS=3", "OK|VCON");
                         return(OK);
                    };

               lprintf(L_WARN,
                "USR handle event: Unknown output device (%d)", data.i);
               return(FAIL);
          case VOICE_STOP_DIALING:
               stop_dialing = TRUE;
               return(OK);
          case VOICE_STOP_PLAYING:
               stop_playing = TRUE;
               return(OK);
          case VOICE_STOP_RECORDING:
               stop_recording = TRUE;
               voice_write("AT");
               return(OK);
          case VOICE_STOP_WAITING:
               stop_waiting = TRUE;
               return(OK);
          case VOICE_SWITCH_TO_DATA_FAX:
               sprintf(buffer, "AT+FCLASS=%s", (char *) data.p);
               return(voice_command(buffer, "OK"));
          case VOICE_WAIT:
               {
               stop_waiting = FALSE;
               voice_modem_state = WAITING;
               alarm(data.i);

               while (!stop_waiting)
                    {

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    delay(100);
                    };

               voice_modem_state = IDLE;
               alarm(0);
               return(OK);
               };

          };

     lprintf(L_WARN, "USR handle event: Unknown event %04x", event);
     return(FAIL);
     };
