
/****************************************************************************
 * This module is all new
 * by Rob Nation 
 ****************************************************************************/
/***********************************************************************
 *
 * mwm pager handling code
 *
 ***********************************************************************/
#include "mwm.h"

extern XEvent   Event;

XGCValues       Globalgcv;
unsigned long   Globalgcm;

void            DrawPartitionLines(void);
MwmWindow      *FindCounterpart(Window target);
Bool            pagerOn = True;
Bool            EnablePagerRedraw = True;
Bool            DoHandlePageing = True;

/****************************************************************************
 * Cause the pager window to be re-drawn
 *
 * Trying to get clever - re-draw the pager by causing an expose event,
 * the really redraw only when an expose arrives. Advantage is that the
 * number of re-draws will be minimized
 ***************************************************************************/
void
RedrawPager()
{
    if ((Scr.MwmPager) && (EnablePagerRedraw))
	XClearArea(dpy, Scr.Pager_w, 0, 0, Scr.MwmPager->frame_width,
		   Scr.MwmPager->frame_height, True);
}

void
ReallyRedrawPager()
{
    MwmWindow      *t;
    Pixel           TextColor, BackColor;

    if (!Scr.MwmPager)
	return;

    flush_expose(Scr.Pager_w);
    flush_expose(Scr.CPagerWin);

    if (Scr.PagerFont.height > 0) {
	if (Scr.Hilite != NULL) {
	    if (!(Scr.Hilite->flags & STICKY) &&
		(!(Scr.Hilite->flags & ICONIFIED) ||
		 (!(Scr.Hilite->flags & SUPPRESSICON))) &&
		(!(Scr.Hilite->flags & ICONIFIED) ||
		 (!(Scr.flags & StickyIcons))) &&
		(Scr.Hilite->icon_image != NULL)) {
		TextColor = Mwm.components[MWM_PAGER].active_foreground;
		BackColor = Mwm.components[MWM_PAGER].active_background;
		NewFontAndColor(Scr.PagerFont.font->fid, TextColor, BackColor);
		flush_expose(Scr.Hilite->pager_view);
		XDrawImageString(dpy, Scr.Hilite->pager_view, Scr.FontGC,
				 2, Scr.PagerFont.y + 2,
		      Scr.Hilite->icon_image, strlen(Scr.Hilite->icon_image));
	    }
	}

	TextColor = Mwm.components[MWM_PAGER].foreground;
	BackColor = Mwm.components[MWM_PAGER].background;
	NewFontAndColor(Scr.PagerFont.font->fid, TextColor, BackColor);


	for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	    if (t != Scr.Hilite) {
		if (!(t->flags & STICKY) &&
		(!(t->flags & ICONIFIED) || (!(t->flags & SUPPRESSICON))) &&
		(!(t->flags & ICONIFIED) || (!(Scr.flags & StickyIcons))) &&
		    (t->icon_image != NULL)) {
		    flush_expose(t->pager_view);
		    XDrawImageString(dpy, t->pager_view, Scr.FontGC,
				     2, Scr.PagerFont.y + 2,
				     t->icon_image, strlen(t->icon_image));
		}
	    }
	}
    }
    DrawPartitionLines();
}

void
DrawPartitionLines(void)
{
    int             y, y1, y2, x, x1, x2;
    int             MaxW, MaxH, width, height;

    MaxW = Scr.VxMax + Scr.d_width;
    MaxH = Scr.VyMax + Scr.d_height;

    width = Scr.MwmPager->frame_width - 2 * Scr.MwmPager->boundary_width;
    height = Scr.MwmPager->frame_height - Scr.MwmPager->title_height
	- 2 * Scr.MwmPager->boundary_width;

    x = Scr.d_width;
    y1 = 0;
    y2 = height;
    while (x < MaxW) {
	x1 = x * width / MaxW;
	XDrawLine(dpy, Scr.Pager_w, Scr.pager_GC, x1, y1, x1, y2);
	x += Scr.d_width;
    }

    y = Scr.d_height;
    x1 = 0;
    x2 = width;
    while (y < MaxH) {
	y1 = y * height / MaxH;
	XDrawLine(dpy, Scr.Pager_w, Scr.pager_GC, x1, y1, x2, y1);
	y += Scr.d_height;
    }
}

void
SwitchPages(Bool align, Bool ChangeFocus)
{
    int             x, y;
    unsigned int    width, height;
    MwmWindow      *tmp_win;
    Window          dumwin;

    if (!Scr.MwmPager)
	return;

    XTranslateCoordinates(dpy, Event.xbutton.window, Scr.Pager_w,
			  Event.xbutton.x, Event.xbutton.y, &x, &y, &dumwin);

    width = Scr.MwmPager->frame_width - 2 * Scr.MwmPager->boundary_width;
    height = Scr.MwmPager->frame_height - Scr.MwmPager->title_height
	- 2 * Scr.MwmPager->boundary_width;

    if (x < 0)
	x = 0;
    if (y < 0)
	y = 0;
    x = x * (Scr.VxMax + Scr.d_width) / width;
    y = y * (Scr.VyMax + Scr.d_height) / height;
    if (align) {
	x = (x / Scr.d_width) * Scr.d_width;
	y = (y / Scr.d_height) * Scr.d_height;
    }
    if (x < 0)
	x = 0;
    if (y < 0)
	y = 0;
    MoveViewport(x, y, True);
    if ((ChangeFocus) && (Mwm.keyboard_focus_policy == XmEXPLICIT)) {
	tmp_win = FindCounterpart(Event.xbutton.subwindow);
	if (tmp_win) {
	    RaiseWindow(tmp_win);
	    SetFocus(tmp_win->w, tmp_win);
	    ClearCirculatedFlag();
	}
    }
}


void
RaiseWindow(MwmWindow * t)
{
    MwmWindow      *t2;
    int             count, i;
    Window         *wins;

    SetTimer(0);

    /* raise the target, at least */
    count = 1;

    for (t2 = Scr.MwmRoot.next; t2 != NULL; t2 = t2->next) {
	if (t2->flags & ONTOP)
	    count++;
	if ((t2->flags & TRANSIENT) && (t2->transientfor == t->w) &&
	    (t2 != t)) {
	    if ((Scr.Pager_w) && !(t2->flags & STICKY))
		XRaiseWindow(dpy, t2->pager_view);
	    count++;
	    if ((t2->flags & ICONIFIED) && (!(t2->flags & SUPPRESSICON))) {
		count += 2;
	    }
	}
    }
    if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
	count += 2;
    }

    if ((Scr.Pager_w) && !(t->flags & STICKY)) {
	XRaiseWindow(dpy, t->pager_view);
    }

    wins = (Window *) XtMalloc(count * sizeof(Window));

    i = 0;

    /* ONTOP windows on top */
    for (t2 = Scr.MwmRoot.next; t2 != NULL; t2 = t2->next) {
	if (t2->flags & ONTOP)
	    wins[i++] = t2->frame;
    }

    /* now raise transients */
#ifndef DONT_RAISE_TRANSIENTS
    for (t2 = Scr.MwmRoot.next; t2 != NULL; t2 = t2->next) {
	if ((t2->flags & TRANSIENT) && (t2->transientfor == t->w) &&
	    (t2 != t) && (!(t2->flags & ONTOP))) {
	    wins[i++] = t2->frame;
	    if ((t2->flags & ICONIFIED) && (!(t2->flags & SUPPRESSICON))) {
		if (!(t2->flags & NOICON_TITLE))
		    wins[i++] = t2->icon_w;
		wins[i++] = t2->icon_pixmap_w;
	    }
	}
    }
#endif


    if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
	if (!(t->flags & NOICON_TITLE))
	    wins[i++] = t->icon_w;
	wins[i++] = t->icon_pixmap_w;
    }
    if (!(t->flags & ONTOP))
	wins[i++] = t->frame;
    if (!(t->flags & ONTOP))
	Scr.LastWindowRaised = t;

    if (i > 0)
	XRaiseWindow(dpy, wins[0]);

    XRestackWindows(dpy, wins, i);
    free(wins);

    RedrawPager();
    raisePanFrames();
}


void
LowerWindow(MwmWindow * t)
{
    XLowerWindow(dpy, t->frame);

    SetTimer(0);

    if ((Scr.Pager_w) && !(t->flags & STICKY))
	XLowerWindow(dpy, t->pager_view);

    if ((t->flags & ICONIFIED) && (!(t->flags & SUPPRESSICON))) {
	XLowerWindow(dpy, t->icon_w);
	XLowerWindow(dpy, t->icon_pixmap_w);
    }
    Scr.LastWindowRaised = (MwmWindow *) 0;
    if (Scr.CPagerWin)
	XLowerWindow(dpy, Scr.CPagerWin);
    RedrawPager();
}


void
PagerMoveWindow()
{
    MwmWindow      *tmp_win;
    unsigned int    width, height;
    int             Xoff, Yoff, x, y, MaxW, MaxH, xl, yt, xl1, yt1;
    Window          target;
    Bool            done, finished = False;

    /* I tried to implement a feature so that when windows which could be
     * opaque moved in a normal move would get the full size windows moved in
     * conjunction with the pager version of the window, but it seems to
     * be buggy on some machines */
#ifdef BROKEN_STUFF
    Bool            OpaqueMove = False;
    int             dwidth, dheight;

#endif
    if (!Scr.MwmPager)
	return;


    EnablePagerRedraw = False;
    target = Event.xbutton.subwindow;
    tmp_win = FindCounterpart(target);

    if (tmp_win == NULL)
	return;


    MaxW = Scr.VxMax + Scr.d_width;
    MaxH = Scr.VyMax + Scr.d_height;

    width = Scr.MwmPager->frame_width - 2 * Scr.MwmPager->boundary_width;
    height = Scr.MwmPager->frame_height - Scr.MwmPager->title_height
	- 2 * Scr.MwmPager->boundary_width;

    XQueryPointer(dpy, target, &JunkRoot, &JunkChild,
		  &JunkX, &JunkY, &Xoff, &Yoff, &JunkMask);

    XQueryPointer(dpy, Scr.Pager_w, &JunkRoot, &JunkChild,
		  &JunkX, &JunkY, &xl, &yt, &JunkMask);
    if (xl < 0)
	xl = 0;
    if (yt < 0)
	yt = 0;
    if (xl > width)
	xl = width;
    if (yt > height)
	yt = height;
    xl -= Xoff;
    yt -= Yoff;
    xl1 = xl;
    yt1 = yt;

    while (!finished) {
	/* block until there is an interesting event */
	XMaskEvent(dpy, ButtonPressMask | ButtonReleaseMask | KeyPressMask |
		   PointerMotionMask | ButtonMotionMask | ExposureMask |
		   VisibilityChangeMask, &Event);
	StashEventTime(&Event);

	if (Event.type == MotionNotify)
	    /* discard any extra motion events before a logical release */
	    while (XCheckMaskEvent(dpy, PointerMotionMask | ButtonMotionMask |
				   ButtonRelease, &Event)) {
		StashEventTime(&Event);
		if (Event.type == ButtonRelease)
		    break;
	    }

	done = FALSE;

	/* Handle a limited number of key press events to allow mouseless
	 * operation */
	if (Event.type == KeyPress)
	    Keyboard_shortcuts(&Event, ButtonRelease);
	switch (Event.type) {
	case ButtonPress:
	case KeyPress:
	    /* throw away enter and leave events until release */
	    done = TRUE;
	    break;
	case ButtonRelease:
	    XQueryPointer(dpy, Scr.Pager_w, &JunkRoot, &JunkChild,
			  &JunkX, &JunkY, &xl, &yt, &JunkMask);
	    if (xl < 0)
		xl = 0;
	    if (yt < 0)
		yt = 0;
	    if (xl > width)
		xl = width;
	    if (yt > height)
		yt = height;
	    xl -= Xoff;
	    yt -= Yoff;
	    done = TRUE;
	    finished = TRUE;
	    break;

	case MotionNotify:
	    XQueryPointer(dpy, Scr.Pager_w, &JunkRoot, &JunkChild,
			  &JunkX, &JunkY, &xl, &yt, &JunkMask);
	    if (xl < 0)
		xl = 0;
	    if (yt < 0)
		yt = 0;
	    if (xl > width)
		xl = width;
	    if (yt > height)
		yt = height;

	    /* redraw the rubberband */
	    xl -= Xoff;
	    yt -= Yoff;

	    done = TRUE;
	    break;

	default:
	    break;
	}
	if (!done) {
	    DispatchEvent();
	}
	XMoveWindow(dpy, target, xl, yt);
	DrawPartitionLines();

    }

    x = xl * MaxW / (int) width - Scr.Vx;
    y = yt * MaxH / (int) height - Scr.Vy;

    MoveResizePagerView(tmp_win);
    if ((xl1 != xl) || (yt1 != yt)) {
	if ((tmp_win->flags & ICONIFIED) && (!(tmp_win->flags & SUPPRESSICON))) {
	    tmp_win->icon_x_loc = x;
	    tmp_win->icon_xl_loc = x -
		(tmp_win->icon_w_width - tmp_win->icon_p_width) / 2;
	    tmp_win->icon_y_loc = y;
	    XMoveWindow(dpy, tmp_win->icon_w, tmp_win->icon_xl_loc, y + tmp_win->icon_p_height);

	    if (tmp_win->icon_pixmap_w != None)
		XMoveWindow(dpy, tmp_win->icon_pixmap_w, x, y);

	    tmp_win->flags |= ICON_MOVED;

	}
	else {
	    /* show the actual window */
	    SetupFrame(tmp_win, x, y, tmp_win->frame_width, tmp_win->frame_height, FALSE);
	}
    }
    EnablePagerRedraw = True;
    RedrawPager();
}



MwmWindow      *
FindCounterpart(Window target)
{
    MwmWindow      *t, *tmp_win = 0;

    tmp_win = NULL;
    for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	if (t->pager_view == target) {
	    tmp_win = t;
	}
    }
    return tmp_win;
}

/***************************************************************************
 * 
 * Check to see if the pointer is on the edge of the screen, and scroll/page
 * if needed 
 ***************************************************************************/
void
HandlePaging(int HorWarpSize, int VertWarpSize, int *xl, int *yt,
	     int *delta_x, int *delta_y, Bool Grab)
{
    int             x, y, total;

    *delta_x = 0;
    *delta_y = 0;

    if (DoHandlePageing) {
	if ((Scr.ScrollResistance >= 10000) ||
	    ((HorWarpSize == 0) && (VertWarpSize == 0)))
	    return;

	/* need to move the viewport */
	if ((*xl >= SCROLL_REGION) && (*xl < Scr.d_width - SCROLL_REGION) &&
	    (*yt >= SCROLL_REGION) && (*yt < Scr.d_height - SCROLL_REGION))
	    return;

	total = 0;
	while (total < Scr.ScrollResistance) {
	    sleep_a_little(10000);
	    total += 10;
	    if (XCheckWindowEvent(dpy, Scr.PanFrameTop.win,
				  LeaveWindowMask, &Event)) {
		StashEventTime(&Event);
		return;
	    }
	    if (XCheckWindowEvent(dpy, Scr.PanFrameBottom.win,
				  LeaveWindowMask, &Event)) {
		StashEventTime(&Event);
		return;
	    }
	    if (XCheckWindowEvent(dpy, Scr.PanFrameLeft.win,
				  LeaveWindowMask, &Event)) {
		StashEventTime(&Event);
		return;
	    }
	    if (XCheckWindowEvent(dpy, Scr.PanFrameRight.win,
				  LeaveWindowMask, &Event)) {
		StashEventTime(&Event);
		return;
	    }
	}

	XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
		      &x, &y, &JunkX, &JunkY, &JunkMask);

	/* Turn off the rubberband if its on */
	MoveOutline(Scr.root_win, 0, 0, 0, 0);

	/* Move the viewport */
	/* and/or move the cursor back to the approximate correct location */
	/* that is, the same place on the virtual desktop that it */
	/* started at */
	if (x < SCROLL_REGION)
	    *delta_x = -HorWarpSize;
	else if (x >= Scr.d_width - SCROLL_REGION)
	    *delta_x = HorWarpSize;
	else
	    *delta_x = 0;
	if (y < SCROLL_REGION)
	    *delta_y = -VertWarpSize;
	else if (y >= Scr.d_height - SCROLL_REGION)
	    *delta_y = VertWarpSize;
	else
	    *delta_y = 0;

	/* Ouch! lots of bounds checking */
	if (Scr.Vx + *delta_x < 0) {
	    if (!(Scr.flags & EdgeWrapX)) {
		*delta_x = -Scr.Vx;
		*xl = x - *delta_x;
	    }
	    else {
		*delta_x += Scr.VxMax + Scr.d_width;
		*xl = x + *delta_x % Scr.d_width + HorWarpSize;
	    }
	}
	else if (Scr.Vx + *delta_x > Scr.VxMax) {
	    if (!(Scr.flags & EdgeWrapX)) {
		*delta_x = Scr.VxMax - Scr.Vx;
		*xl = x - *delta_x;
	    }
	    else {
		*delta_x -= Scr.VxMax + Scr.d_width;
		*xl = x + *delta_x % Scr.d_width - HorWarpSize;
	    }
	}
	else
	    *xl = x - *delta_x;

	if (Scr.Vy + *delta_y < 0) {
	    if (!(Scr.flags & EdgeWrapY)) {
		*delta_y = -Scr.Vy;
		*yt = y - *delta_y;
	    }
	    else {
		*delta_y += Scr.VyMax + Scr.d_height;
		*yt = y + *delta_y % Scr.d_height + VertWarpSize;
	    }
	}
	else if (Scr.Vy + *delta_y > Scr.VyMax) {
	    if (!(Scr.flags & EdgeWrapY)) {
		*delta_y = Scr.VyMax - Scr.Vy;
		*yt = y - *delta_y;
	    }
	    else {
		*delta_y -= Scr.VyMax + Scr.d_height;
		*yt = y + *delta_y % Scr.d_height - VertWarpSize;
	    }
	}
	else
	    *yt = y - *delta_y;

	if (*xl <= SCROLL_REGION)
	    *xl = SCROLL_REGION + 1;
	if (*yt <= SCROLL_REGION)
	    *yt = SCROLL_REGION + 1;
	if (*xl >= Scr.d_width - SCROLL_REGION)
	    *xl = Scr.d_width - SCROLL_REGION - 1;
	if (*yt >= Scr.d_height - SCROLL_REGION)
	    *yt = Scr.d_height - SCROLL_REGION - 1;

	if ((*delta_x != 0) || (*delta_y != 0)) {
	    if (Grab)
		XGrabServer(dpy);
	    XWarpPointer(dpy, None, Scr.root_win, 0, 0, 0, 0, *xl, *yt);
	    MoveViewport(Scr.Vx + *delta_x, Scr.Vy + *delta_y, False);
	    XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
			  xl, yt, &JunkX, &JunkY, &JunkMask);
	    if (Grab)
		XUngrabServer(dpy);
	}
    }
}



void
MoveResizeViewPortIndicator(void)
{
    int             width, height, x1, x2, y1, y2;

    if ((Scr.CPagerWin) && (Scr.MwmPager)) {
	width = Scr.MwmPager->frame_width - 2 * Scr.MwmPager->boundary_width;
	height = Scr.MwmPager->frame_height - Scr.MwmPager->title_height
	    - 2 * Scr.MwmPager->boundary_width;
	x1 = Scr.Vx * width / (Scr.VxMax + Scr.d_width) + 1;
	y1 = Scr.Vy * height / (Scr.VyMax + Scr.d_height) + 1;
	x2 = (Scr.d_width) * width / (Scr.VxMax + Scr.d_width) - 1;
	y2 = (Scr.d_height) * height / (Scr.VyMax + Scr.d_height) - 1;
	if (x1 == 1) {
	    x1--;
	    x2++;
	}
	if (y1 == 1) {
	    y1--;
	    y2++;
	}
	XMoveResizeWindow(dpy, Scr.CPagerWin, x1, y1, x2, y2);
    }
}


void
MoveResizePagerView(MwmWindow * t)
{
    unsigned int    width, height;
    int             ww, wh;
    int             wx, wy;
    int             MaxH, MaxW;

    if ((!Scr.MwmPager) || (!pagerOn))
	return;

    width = Scr.MwmPager->frame_width - 2 * Scr.MwmPager->boundary_width;
    height = Scr.MwmPager->frame_height - Scr.MwmPager->title_height
	- 2 * Scr.MwmPager->boundary_width;

    MaxW = Scr.VxMax + Scr.d_width;
    MaxH = Scr.VyMax + Scr.d_height;

    if ((!(t->flags & STICKY) &&
	 (!(t->flags & ICONIFIED) || (!(t->flags & SUPPRESSICON))) &&
	 (!(t->flags & ICONIFIED) || (!(t->flags & SUPPRESSICON)))) &&
	(!((t->flags & ICONIFIED) && (t->flags & ICON_UNMAPPED))) &&
	(t->Desk == Scr.CurrentDesk)) {
	if (t->flags & ICONIFIED) {
	    /* show the icon loc */
	    wx = (t->icon_x_loc + Scr.Vx) * (int) width / MaxW;;
	    wy = (t->icon_y_loc + Scr.Vy) * (int) height / MaxH;
	    ww = t->icon_w_width * (int) width / MaxW;
	    wh = (t->icon_w_height + t->icon_p_height) * (int) height / MaxH;
	}
	else {
	    /* show the actual window */
	    wx = (t->frame_x + Scr.Vx) * (int) width / MaxW;
	    wy = (t->frame_y + Scr.Vy) * (int) height / MaxH;
	    ww = t->frame_width * (int) width / MaxW;
	    wh = t->frame_height * (int) height / MaxH;
	}
	if (ww < 2)
	    ww = 2;
	if (wh < 2)
	    wh = 2;
	XMoveResizeWindow(dpy, t->pager_view, wx, wy, ww, wh);
    }
    else {
	/* window is sticky - make sure that the pager_view window is not 
	 * visible */
	XMoveResizeWindow(dpy, t->pager_view, -10, -10, 5, 5);
    }
    RedrawPager();
}


/***********************************************************************
 *
 *  Procedure:
 *	Initialize_pager - creates the pager window, if needed
 *
 *  Inputs:
 *	x,y location of the window
 *
 ***********************************************************************/
char           *pager_name = "Mwm Pager";
XSizeHints      sizehints =
{
    (PMinSize | PResizeInc | PBaseSize | PWinGravity),
    0, 0, 100, 100,		/* x, y, width and height */
    1, 1,			/* Min width and height */
    0, 0,			/* Max width and height */
    1, 1,			/* Width and height increments */
    {0, 0},
    {0, 0},			/* Aspect ratio - not used */
    1, 1,			/* base size */
    (NorthWestGravity)		/* gravity */
};

void
initialize_pager(int x, int y)
{
    XTextProperty   name;
    int             width, height, window_x, window_y;
    unsigned long   valuemask;
    XSetWindowAttributes attributes;

    width = (Scr.VxMax + Scr.d_width) / Scr.VScale;
    height = (Scr.VyMax + Scr.d_height) / Scr.VScale;

    if (x >= 0)
	window_x = x;
    else {
	sizehints.win_gravity = NorthEastGravity;
	window_x = Scr.d_width - width + x - 2;
    }

    if (y >= 0)
	window_y = y;
    else {
	window_y = Scr.d_height - height + y - 2;
	if (x < 0)
	    sizehints.win_gravity = SouthEastGravity;
	else
	    sizehints.win_gravity = SouthWestGravity;
    }
    valuemask = (CWBackPixel | CWBorderPixel | CWEventMask | CWCursor);
    attributes.background_pixel = Mwm.components[MWM_PAGER].background;
    attributes.border_pixel = Mwm.components[MWM_PAGER].background;
    attributes.cursor = Scr.MwmCursors[DEFAULT_CURS];
    attributes.event_mask = (ExposureMask | EnterWindowMask | ButtonReleaseMask |
			     ButtonMotionMask);
    sizehints.width = width;
    sizehints.height = height;
    sizehints.x = window_x;
    sizehints.y = window_y;

    Scr.Pager_w = XCreateWindow(dpy, Scr.root_win, window_x, window_y, width,
				height, (unsigned int) 1,
				CopyFromParent, InputOutput,
				(Visual *) CopyFromParent,
				valuemask, &attributes);
    XSetWMNormalHints(dpy, Scr.Pager_w, &sizehints);
    XStringListToTextProperty(&pager_name, 1, &name);
    XSetWMName(dpy, Scr.Pager_w, &name);
    XSetWMIconName(dpy, Scr.Pager_w, &name);
    XFree((char *) name.value);

    attributes.event_mask = KeyPressMask | ExposureMask;
    attributes.background_pixel = Mwm.components[MWM_PAGER].background;
    attributes.cursor = Scr.MwmCursors[DEFAULT_CURS];
    Scr.CPagerWin = XCreateWindow(dpy, Scr.Pager_w, -10, -10, 10, 10, 0,
				  CopyFromParent,
				  InputOutput, CopyFromParent,
				  CWEventMask | CWBackPixel | CWCursor,
				  &attributes);
    XMapRaised(dpy, Scr.CPagerWin);
}
