/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <strings.h>

#include <X11/Xlib.h>
#include "McApp.h"
#include "McGadget.h"
#include "McSlider.h"
#include "McText.h"
#include "McBitmap.h"
#include "McString.h"
#include "McSelector.h"
#include "McResource.h"
#include "McUtils.h"
#include "McAlloc.h"

#include "editor.h"
#include "struct.h"
#include "cd.h"
#include "discid.h"

/*****************************************************************************
 *
 * Private definitons
 *
 */
enum {
  BTN_APPLY,
  BTN_CLOSE,
  BTN_CANCEL,
  BTN_UNDO,

  TXT_TITLE,
  STR_TITLE,

  SEL_SONGS,
  PRP_SONGS,

  STR_SONG,

  NUM_BUTTONS
};

#define WIDTH  320
#define HEIGHT 320

#undef POPUP

typedef struct name_t {
  unsigned char name[MAX_NAMELEN+8];
} name_t;

/*****************************************************************************
 *
 * Private variables and procedures
 *
 */

static McWindow *editwin = NULL;
static McGadget *EGad[NUM_BUTTONS];
static int close_edit = 0;
static name_t *titlelist = NULL;
static unsigned char **titleptr = NULL;
static int ntracks;
static unsigned int discid;

static void configure_proc(McWindow *mcw, int willredraw);
static int event_proc(McWindow *mcw, XEvent *event);
static void select_up(McGadget *gadget);
static void select_down(McGadget *gadget);
static void close_proc(McGadget *gadget);
static void undo_proc(McGadget *gadget);
static void cancel_proc(McGadget *gadget);
static void apply_proc(McGadget *gadget);
static void song_proc(McGadget *gadget);
static void set_selection(int sel);
static void create_titlelist(void);
static void apply_list(void);

/*****************************************************************************
 *
 * Pop up the editwin window
 *
 */
extern void create_edit_window(void) {

  /* Only one edit window at a time */
  if (editwin) return;
  close_edit = 0;

  if ((cur_cdmode==CDEJECT) || (cur_cdmode==CDMOUNT) || (thiscd.ntracks==0)) {
    XBell(app->display,0);
    return;
  }

  ntracks = thiscd.ntracks;
  discid = get_discid();

  /*
   * Create the window
   */

  editwin=McCreateSimpleWindow(app, "XPlaycd - Database editor",
			       WIDTH, HEIGHT, WIDTH, 150, 0, 0,
			       configure_proc, event_proc);

  EGad[0] = MakeButton(editwin, 248, 294, 60, 16, BTN_CANCEL, "Cancel",
		       cancel_proc);
  EGad[0]->topLeftGravity = SouthEastGravity;
  EGad[1] = MakeButton(editwin,  88, 294, 60, 16, BTN_APPLY, "Apply",
		       apply_proc);
  EGad[1]->topLeftGravity = SouthWestGravity;
  EGad[2] = MakeButton(editwin,  12, 294, 60, 16, BTN_CLOSE, "Close",
		       close_proc);
  EGad[2]->topLeftGravity = SouthWestGravity;
  EGad[3] = MakeButton(editwin, 164, 294, 60, 16, BTN_UNDO, "Undo",
		       undo_proc);
  EGad[3]->topLeftGravity = SouthWestGravity;

  EGad[4] =  MakeRText(editwin, 40, 10, TXT_TITLE, "Title:");

  EGad[5] = MakeString(editwin, 46, 10, 262, 17, STR_TITLE, thiscd.cdname,
		      NULL, NULL);
  EGad[5]->topLeftGravity = NorthWestGravity;
  EGad[5]->bottomRightGravity = NorthEastGravity;

  EGad[6] = MakeSelector(editwin, 12, 37, 277, 220, SEL_SONGS,
			 app->defaultFont, select_up, select_down);
  EGad[6]->bottomRightGravity = SouthEastGravity;
  EGad[7] = MakeProp(editwin, 296, 37, 12, 220, PRP_SONGS, NULL);
  EGad[7]->topLeftGravity = NorthEastGravity;
  EGad[7]->bottomRightGravity = SouthWestGravity;
  McSelectorBindSlider(EGad[6], EGad[7]);

  EGad[8] = MakeString(editwin, 12, 267, 296, 17, STR_SONG, "",
	       song_proc, song_proc);
  EGad[8]->topLeftGravity = SouthWestGravity;
  EGad[8]->bottomRightGravity = NorthEastGravity;

  EGad[5]->specialInfo->prev=EGad[5];
  EGad[5]->specialInfo->next=EGad[6];
  EGad[6]->specialInfo->prev=EGad[5];
  EGad[6]->specialInfo->next=EGad[8];
  EGad[8]->specialInfo->prev=EGad[6];
  EGad[8]->specialInfo->next=EGad[8];

  create_titlelist();

  /*
   * Display the window
   */
  McInitGadgets(editwin);
  XMapWindow(app->display, editwin->window);

  McSetFocus(editwin, EGad[STR_TITLE]);
  
  /* Setup complete. Let the event-loop do the rest. */
}

static void configure_proc(McWindow *mcw, int willredraw) {
}

void check_edit_window(void) {

  if ((cur_cdmode==CDEJECT) || (cur_cdmode==CDMOUNT) || (thiscd.ntracks==0)) {
    close_edit=1;
  }

  if (close_edit) {
    if (editwin) {
      McFreeWindow(app, editwin);
      if (titlelist) McFree(titlelist);
      editwin = NULL;
      titlelist = NULL;
    }
    close_edit = 0;
  }
}

/**************************************************************************
 * static int event_proc(McWindow *mcw, XEvent *event)
 *
 */
static int event_proc(McWindow *mcw, XEvent *event) {
  switch(event->type) {
  case ClientMessage:
    if ((event->xclient.format == 32) &&
	(event->xclient.data.l[0] == app->wmDelWin)) {
      if (editwin && mcw==editwin) {
	close_edit = 1;
      }
      return 1;
    }
  }
  return 0;
}

static void select_up(McGadget *gadget) {
  McSetFocus(editwin, EGad[STR_SONG]);
  McGadgetUpdate(editwin, EGad[STR_SONG]);
}

static void select_down(McGadget *gadget) {
  McSelector *selector = (McSelector *)gadget->specialInfo;
  set_selection(selector->selection);
  if (selector->flags & SEL_DOUBLE_CLICK) {
    McSetFocus(editwin, EGad[STR_SONG]);
    McGadgetUpdate(editwin, EGad[STR_SONG]);
  }
}

static void set_selection(int sel) {
  unsigned char *c;
  if (!(c=strchr(titlelist[sel].name, '-'))) c=titlelist[sel].name; else c+=2;
  McStringSetText(editwin, EGad[STR_SONG], c);
}

static void close_proc(McGadget *gadget) {
  apply_list();
  if (editwin) {
    close_edit = 1;
  }
}

static void undo_proc(McGadget *gadget) {
  create_titlelist();
}

static void cancel_proc(McGadget *gadget) {
  if (editwin) {
    close_edit = 1;
  }
}

static void apply_proc(McGadget *gadget) {
  apply_list();
}

static void song_proc(McGadget *gadget) {
  unsigned char *c;
  int sel =((McSelector *)EGad[SEL_SONGS]->specialInfo)->selection;
  if (!(c=strchr(titlelist[sel].name, '-'))) c=titlelist[sel].name; else c+=2;
  
  if (strncmp(c, ((McString *)gadget->specialInfo)->buf, MAX_NAMELEN)) {
    gadget->specialInfo->next=gadget;
    strncpy(c, ((McString *)gadget->specialInfo)->buf, MAX_NAMELEN);
    McGadgetUpdate(editwin, EGad[SEL_SONGS]);
  }
  if (++sel>=ntracks) sel=0;
  McSelectorSelect(editwin, EGad[SEL_SONGS], sel);
  set_selection(((McSelector *)EGad[SEL_SONGS]->specialInfo)->selection);
}

static void create_titlelist(void) {
  int i;
  if (titlelist) McFree(titlelist);

  if ((thiscd.ntracks != ntracks) || (discid != get_discid())) {
    XBell(app->display, 0);
    return;
  }

  titlelist = (name_t *)McAlloc((sizeof(name_t)*ntracks) +
				(sizeof(unsigned char *)*ntracks));
  titleptr  = (unsigned char **)(&titlelist[ntracks]);

  for (i = 0; i < ntracks; i++) {
    char *c = titlelist[i].name;
    int p;
    *c=' ';
    if (i<9) {
      c[1]=' ';
      c[2]=' ';
      p=3;
    } else {
      c[1]=((i+1) / 10)+0x30;
      p=2;
    }
    c[p++]=((i+1) % 10)+0x30;
    c[p++]=' '; c[p++]='-'; c[p++]=' ';
    strncpy(titlelist[i].name+p, thiscd.trk[i].songname, MAX_NAMELEN);
    titleptr[i]=titlelist[i].name;
  }
  McSelectorSetList(editwin, EGad[SEL_SONGS], titleptr, ntracks, 0);
  set_selection(0);
}

static void apply_list(void) {
  int i;
  unsigned char *c;

  if ((thiscd.ntracks != ntracks) || (discid != get_discid())) {
    XBell(app->display, 0);
    return;
  }

  strncpy(thiscd.cdname, ((McString *)EGad[STR_TITLE]->specialInfo)->buf,
	  MAX_NAMELEN-1);

  for (i = 0; i < ntracks; i++) {
    if (!(c=strchr(titlelist[i].name, '-'))) c=titlelist[i].name; else c+=2;
    strncpy(thiscd.trk[i].songname, c, MAX_NAMELEN-1);
  }

  update_db(0);
  write_db(currentFile);
}

