#include <string.h>
#include <stdio.h>

#include "packet.h"


// Endianness utility routines: Unlike Real Internet Protocols, this
// heap of dung uses little-endian byte sex.

// return short (16-bit) stored in little-endian format, possibly unaligned
unsigned short get_le_short(char *p)
{
   unsigned char *q = (unsigned char *)p;
   return q[0] + (q[1] << 8);
}

// return int (32-bit) stored in little-endian format, possibly unaligned
unsigned int get_le_int(char *p)
{
   unsigned char *q = (unsigned char *)p;
   return q[0] + (q[1] << 8) + (q[2] << 16) + (q[3] << 24);
}

// return long (32-bit) stored in little-endian format, possibly unaligned
unsigned long get_le_long(char *p)
{
   unsigned char *q = (unsigned char *)p;
   return q[0] + (q[1] << 8) + (q[2] << 16) + (q[3] << 24);
}

// store 16-bit short in little-endian format, possibly unaligned
void put_le_short(unsigned short x, char *p)
{
   unsigned char *q = (unsigned char*)p;
   q[0] = x & 0xff;
   q[1] = (x >> 8) & 0xff;
}

// store 32-bit int in little-endian format, possibly unaligned
void put_le_int(unsigned int x, char *p)
{
   unsigned char *q = (unsigned char*)p;
   q[0] = x & 0xff;
   q[1] = (x >> 8) & 0xff;
   q[2] = (x >> 16) & 0xff;
   q[3] = (x >> 24) & 0xff;
}

// store 32-bit int in little-endian format, possibly unaligned
void put_le_long(unsigned long x, char *p)
{
   unsigned char *q = (unsigned char*)p;
   q[0] = x & 0xff;
   q[1] = (x >> 8) & 0xff;
   q[2] = (x >> 16) & 0xff;
   q[3] = (x >> 24) & 0xff;
}


Packet::Packet(void)
{
   maxSize = MAX_PACKET_SIZE;
   buff = new char[maxSize];
   nextData = buff;
   sizeVal = 0;
}


Packet::Packet(Packet *packet)
{
   maxSize = packet->maxSize;
   buff = new char[maxSize];
   nextData = packet->nextData;
   sizeVal = packet->sizeVal;   
   for (unsigned long i = 0; i < sizeVal; i++) buff[i] = packet->buff[i];
}


Packet::Packet(char *newBuff, unsigned long buffSize)
{
    maxSize = MAX_PACKET_SIZE;
    if (buffSize > maxSize) buffSize = maxSize;
    buff = new char[maxSize];
    memcpy(buff, newBuff, buffSize);
    nextData = buff;
    sizeVal = buffSize;
}


Packet::~Packet()
{
    delete(buff);
}

char *Packet::print(void)
{
    static char p[2048];
    char *pPos = p;

    sprintf(pPos, "  ");
    pPos += 2;
    for(unsigned long i = 0; i < sizeVal; i++)
    {
       sprintf(pPos, "%02X ", (unsigned char)buff[i]);
       pPos += 3;
       if((i + 1) % 24 == 0) { sprintf(pPos, "\n  "); pPos += 3; }
    }
    sprintf(pPos, "\n%c", '\0');
    return(p);
}

void Packet::clearPacket(void)
{
   nextData = buff;
   sizeVal = 0;
}


Packet& Packet::operator<<(unsigned int data)
{
   put_le_int(data, nextData);
   sizeVal += sizeof(unsigned int);
   nextData += sizeof(unsigned int);
   return(*this);    
}


Packet& Packet::operator<<(unsigned short data)
{
   put_le_short(data, nextData);
   sizeVal += sizeof(unsigned short);
   nextData += sizeof(unsigned short);
   return(*this);    
}


Packet& Packet::operator<<(unsigned long data)
{
   put_le_long(data, nextData);
   sizeVal += sizeof(unsigned long);
   nextData += sizeof(unsigned long);
   return(*this);    
}


Packet& Packet::operator<<(char data)
{
   int s = sizeof(unsigned char);
   memcpy(nextData, &data, s);
   sizeVal += s;
   nextData += s;
   return(*this);    
}


Packet& Packet::operator<<(char *data)
{
   int s = strlen((char *)data) + 1;
   memcpy(nextData, data, s);
   sizeVal += s;
   nextData += s;
   return(*this);
}

/*
template <class T>
Packet& Packet::operator>>(T &in)
{
   if(nextData + sizeof(T) > (buff + sizeVal)) 
      in = 0;
   else
   {
      in = *((T *)nextData);
      nextData += sizeof(T);
   }
   return(*this);
}
*/

Packet& Packet::operator>>(char &in)
{
   if(nextData + sizeof(char) > (buff + sizeVal)) 
      in = 0;
   else
   {
      in = *((char *)nextData);
      nextData += sizeof(char);
   }
   return(*this);
}


Packet& Packet::operator>>(unsigned short &in)
{
   if(nextData + sizeof(unsigned short) > (buff + sizeVal)) 
      in = 0;
   else
   {
      in = get_le_short(nextData);
      nextData += sizeof(unsigned short);
   }
   return(*this);
}


Packet& Packet::operator>>(unsigned long &in)
{
   if(nextData + sizeof(unsigned long) > (buff + sizeVal)) 
      in = 0;
   else
   {
      in = get_le_long(nextData);
      nextData += sizeof(unsigned long);
   }
   return(*this);
}
