/*
** This program translates a C program which contains the special functions
** "ET()", "ET_INT()", "ET_STR()" and/or "ET_DBL()" into a valid C program
** which can be compiled with a C compiler.
**
** The afore mentioned functions are special in that their sole argument is
** not a C expression, but a Tcl/Tk script.
**
**    ET()        Returns the usual Tcl status code. (Ex: TCL_OK, TCL_ERROR,
**                and so on.)
**    ET_INT()    Converts the string returned by the Tcl interpreter into
**                and integer (using atoi()) and returns that integer.  Zero
**                is returned if there is an error.
**    ET_STR()    Returns the string returned by the Tcl interpreter, or ""
**                if there is an error.
**    ET_DBL()    Converts the return of the interpreter to a double. 0.0 is
**                returned if there is an error.
**
** All of the functions use the interpreter which is pointed to by the
** global variable "extern struct Tcl_Interp *Et_Interp;".
**
** Strings, integers and floating point values of C can be included in
** the Tk script within a ET() function call by use of the following
** constructs within the Tcl/Tk script:
**
**    %d(e)       "e" must be a valid C expression of type "int".  The
**                expression is evaluated before the Tcl/Tk interpreter
**                is invoked and the result of the expression is converted
**                into an ascii string and stored at the in place of the
**                macro.
**
**    %s(e)       This work just like %d(), except that the enclosed
**    %f(e)       expressions evaluate to types "char*" and "double",
**                respectively.
**
**    %q(e)       This works like %s() except that the string is quoted.
**                Extra backslashes are added in front of $, ", \ and [
**                characters to prevent them from being interpreted by
**                Tcl/Tk.
**
** The afore mentioned ET_XXX() functions all work like C functions.  That
** is to say, they all return a value.  The next macro implemented by
** this translator works more like a procedure -- it does not return
** a value and cannot be included in an expression.  ET_INCLUDE(f) reads
** and evaluates the Tcl source code file "f".  The text of the file "f"
** actually becomes part of the program -- it is read it at compile time,
** not at run time.  Also, the %d(e) type substitutions do not occur
** on "f".  The ET_INCLUDE() macro is useful for making the various
** Tcl and Tk initialization scripts part of the executable.
**
** An important note:  All Tcl/Tk code is stripped of comments and
** leading spaces before being embedded in C.  It is possible that this
** can break a Tcl/Tk script, though unlikely.
**
** Under the assumption that this program (the program generated by compiling
** this file) is named "et2c" and that the source code module containing
** the special ET_XXX() functions is called "module.et", then the C program
** is generated as follows:
**
**       et2c module.et >module.c
**
** The translated program must be linked with et.c in order to run.
**
** AN ADDITION:
**
** A couple of new macros have been added which make it easier to add
** new Tcl/Tk commands to the interpreter.  The first is the ET_PROC
** macro which is used to define a new command.  For example:
**
**         ET_PROC( newCmd ){
**            // C code goes here
**         }
**
** The ET_PROC macro takes the place of the normal C function declaration.
** (In fact, ET_PROC is expanded to an appropriate C function declaration
** by the et2c translator.)  The declaration defines 3 useful parameters:
**
**   argc        The number of arguments passed to the Tcl/Tk command
**   argv        A null-terminated array of pointers to strings of
**               arguments of the command
**   interp      A pointer to the Tcl/Tk interpreter.
**
** Thus, the ET_PROC macro saves one the trouble of writing out the
** details of a Tcl/Tk command procedure declaration.  But, ET_PROC does
** more than that.  ET_PROC helps in registering the command with the
** interpreter.  It works like this...
**
** Suppose that several new Tcl/Tk commands are coded using the ET_PROC
** macro in a file named "myfile.c".  These commands can be registered
** with the Tcl/Tk interpreter by executing the following macro:
** 
**         ET_INSTALL_COMMANDS( myfile.c )
**
** This macro generates appropriate code to register every command built
** using the ET_PROC macro in the file "myfile.c".  If the parentheses
** and file name are omitted, then all ET_PROC macros from the present
** source file are registered.  The argument to ET_INSTALL_COMMANDS can
** also be the name of an individual command, in order to cause that
** command to be installed.
**
** D. Richard Hipp. 1994.
**
** CHANGES:
**
** December 4, 1995:
** Normally, et2c attempts to reduce the size of Tcl/Tk code be removing
** comments and white-space that occurs at the beginning of lines.  However,
** with the new -nocompress switch, no such attempts are made, and every
** character of the Tcl/Tk is included in the C file verbatim.
*/
#include <stdio.h>
#include <assert.h>
#include <ctype.h>

#if 0
/* We need to know the values for TCL_OK and TCL_ERROR, but nothing else
** from <tcl.h>.  Therefore, in order to make this program easier to
** compiler, will hard-code in appropriate values.  This prevents the
** user from having to put a "-I" command-line option on the compiler
** line.
*/
#include <tcl.h>
#else
#define TCL_OK 0
#define TCL_ERROR 1
#endif

#if defined(__WIN32__) || defined(__WIN32)
#define WIN32 1
#define UNIX  0
#else
#define WIN32 0
#define UNIX  1
#endif

#define cant_happen fprintf(stderr,"This can't happen.  %s Line %d.\n",__FILE__,__LINE__);exit(1);

extern void free();
extern int strncmp();

/* The following variable is TRUE if Tcl/Tk comments should compressed by
** removing comments and initial white space.
*/
static int compress_flag = 1;

/*
** The following definitions give the precise text used for all
** keywords
*/
#define TX_BEGIN_TCL       "ET"
#define TX_BEGIN_TCL_INT   "ET_INT"
#define TX_BEGIN_TCL_STR   "ET_STR"
#define TX_BEGIN_TCL_DBL   "ET_DBL"
#define TX_INCLUDE_TCL     "ET_INCLUDE"
#define TX_DEFINE_PROC     "ET_PROC"
#define TX_INSTALL_CMD     "ET_INSTALL_COMMAND"
#define TX_INSTALL_CMDS    "ET_INSTALL_COMMANDS"

/*
** The input file is broken into tokens.  Each token is of one of
** the following types:
*/
enum egTokenType {
  SPACE,                  /* Spaces and tabs, but not new-lines */
  LEFT_PAREN,             /* Open parenthesis  ( */
  RIGHT_PAREN,            /* Close parenthesis ) */
  LEFT_BRACE,             /* Open curly brace  { */
  RIGHT_BRACE,            /* Close curly brace } */
  BEGIN_TCL,              /* Keyword:  "ET" */
  BEGIN_TCL_INT,          /* Keyword:  "ET_INT" */
  BEGIN_TCL_STR,          /* Keyword:  "ET_STR" */
  BEGIN_TCL_DBL,          /* Keyword:  "ET_DBL" */
  INCLUDE_TCL,            /* Keyword:  "ET_INCLUDE" */
  DEFINE_PROC,            /* Keyword:  "ET_PROC" */
  INSTALL_CMDS,           /* Keyword:  "ET_INSTALL_COMMANDS" */
  NEWLINE,                /* A newline or form-feed */
  END_OF_FILE,            /* End of file */

  MISCELLANEOUS,          /* None of the above */
};

/*
** The following variables hold the name and FILE* pointer for
** the input file.  "fileLine" is the line number of the file
** currently being read.
*/
static char *fileName;       /* Name of the input file (for error messages) */
static FILE *fileIn;         /* Stream pointer for the input file */
static int fileLine;         /* Line number of the input file */


/*
** The next set of variables hold information about the token currently
** being read from the input file.  The text of the token is
** stored in "token".  The memory pointed to by "token" is enlarged
** as necessary.  "tokenNext" is the next free slot in the "token" buffer.
** "tokenSize" is the total number of slots in the token buffer.
** tokenType is the type of the most recently seen token.
*/
static char *token;                  /* Text of the next token */
static int tokenNext = 0;            /* Next free slot in token[] */
static int tokenSize = 0;            /* Number of slots in token[] */
static enum egTokenType tokenType;   /* Type of the current token */

/*
** The following variables are used to manage the diversion buffer.
** The diversion buffer holds text whose output is to be delayed.
** The diversion buffer is used during ET macro processing.
*/
static char *diversion;              /* Text of the diversion */
static int diverNext = 0;            /* Next free slot in diversion */
static int diverSize = 0;            /* Number of slotes in diversion[] */

/*
** The -I command line switch can be used to specify paths of where
** to look for the files to read for the "ET_INCLUDE" macro.  There
** can be any number of these switches.  The following variables hold
** the number of paths, the name of each path, and the maximum length
** of each pathname.  The variables are written by main() and read
** by ProcessIncludeMacro().  Note that the directory "." is implicit,
** as if there were always a "-I." at the beginning of every command
** line.
*/
static char **includePath;          /* The names of include paths */
static int nIncludePath;            /* The number of include paths */
static int maxIncludePathLength;    /* Maximum length of any path */

/*
** The number of errors seen while processing the input file
*/
static int errorCount = 0;          /* Number of errors seen */

/*
** Prototype memory allocation functions, to avoid compiler complaints
*/
extern char *malloc();
extern char *realloc();

/*
** Call malloc() but exit with an error if it returns NULL.  Also
** a version of realloc() which is safe.
*/
static char *
SafeMalloc(nBytes)
int nBytes;      /* Amount of memory to allocate */
{
  char *buf;

  buf = malloc( nBytes );
  if( buf==0 ){
    fprintf(stderr,"Out of memory.\n");
    exit(1);
  }
  return buf;
}
static char *
SafeRealloc(oldbuf,nBytes)
char *oldbuf;    /* Previous allocation */
int nBytes;      /* Amount of memory to allocate */
{
  char *buf;

  buf = realloc( oldbuf, nBytes );
  if( buf==0 ){
    fprintf(stderr,"Out of memory.\n");
    exit(1);
  }
  return buf;
}

/*
** Increase the size of the static buffer used to hold the text of
** a single token.
**
** This routine should be call very infrequently.  Most executions
** will call it only once.
*/
static void
EnlargeTokenBuffer()
{
  if( tokenSize==0 ){
    tokenSize = 1000;
    token = SafeMalloc( tokenSize );
  }else{
    tokenSize *= 2;
    token = SafeRealloc( token, tokenSize );
  }
}

/* Add a single character to the token buffer.  Increase the
** size of the token buffer, if necessary.
*/
static void
AddCharToToken(c)
int c;    /* Character to be added to the token buffer */
{
  if( c!=EOF ){
    if( tokenNext>=tokenSize ) EnlargeTokenBuffer();
    token[tokenNext++] = c;
  }
}

/*
** Increase the size of the static buffer used to hold the text of
** the diversion.
**
** This routine should be call very infrequently.  Most executions
** will call it only once.
*/
static void
EnlargeDiversionBuffer(amt)
int amt;    /* Amount of new space needed */
{
  if( diverSize==0 ){
    diverSize = 1000 + amt;
    diversion = SafeMalloc( diverSize );
  }else{
    diverSize = diverSize*2 + amt;
    diversion = SafeRealloc( diversion, diverSize );
  }
}

/* Add a string to the end of the diversion buffer.
*/
static void
WriteToDiversion(s)
char *s;      /* String to be appended to the diversion */
{
  int length = strlen(s);
  if( diverNext + length + 1 >= diverSize ){
    EnlargeDiversionBuffer(length);
  }
  while( *s ) diversion[diverNext++] = *s++;
  diversion[diverNext] = 0;
}

/* Get a single character of input.  Update the global variable
** fileLine which tracks the line number of the input file, as
** necessary */
static int
GetInput()
{
  int c = getc(fileIn);
  if( c=='\n' ) fileLine++;
  return c;
}

/* Return the single character 'c' back into the input stream so
** that it is retrieved by the next call to GetInput(). 
*/
static void
PushBackIntoInput(c)
int c;
{
  ungetc(c,fileIn);
  if( c=='\n' ) fileLine--;
}

/*
** Read the next token of C source code.
** Store it's text in the "token" variable.
** Record the type in "tokenType", and also return the type.
*/
static enum egTokenType
GetNextCToken()
{
  int c;           /* Next character read */

  tokenNext = 0;
  c = GetInput();
  if( c==EOF ){
    tokenType = END_OF_FILE;
  }else if( c=='\n' || c=='\f' ){
    AddCharToToken(c);
    tokenType = NEWLINE;
  }else if( c=='(' ){
    AddCharToToken(c);
    tokenType = LEFT_PAREN;
  }else if( c==')' ){
    AddCharToToken(c);
    tokenType = RIGHT_PAREN;
  }else if( c==' ' || c=='\t' ){
    do{
      AddCharToToken(c);
      c = GetInput();
    }while( c==' ' || c=='\t' );
    PushBackIntoInput(c);
    tokenType = SPACE;
  }else if( c=='#' && tokenType==NEWLINE ){   /* Preprocessor statements */
    do{
      int cPrev;   /* Holds the previous character read */
      AddCharToToken(c);
      cPrev = c;
      c = GetInput();
      if( c=='\n' && cPrev=='\\' ){
        AddCharToToken(c);
        c = GetInput();
      }
    }while( c!=EOF && c!='\n' );
    AddCharToToken(c);
    tokenType = MISCELLANEOUS;
  }else if( isalnum(c) || c=='_' ){        /* Identifiers and numbers */
    do{
      AddCharToToken(c);
      c = GetInput();
    }while( isalnum(c) || c=='_' );
    PushBackIntoInput(c);
    AddCharToToken(0);
    if( strcmp(token,TX_BEGIN_TCL)==0 )           tokenType = BEGIN_TCL;
    else if( strcmp(token,TX_BEGIN_TCL_INT)==0 )  tokenType = BEGIN_TCL_INT;
    else if( strcmp(token,TX_BEGIN_TCL_STR)==0 )  tokenType = BEGIN_TCL_STR;
    else if( strcmp(token,TX_BEGIN_TCL_DBL)==0 )  tokenType = BEGIN_TCL_DBL;
    else if( strcmp(token,TX_INCLUDE_TCL)==0 )    tokenType = INCLUDE_TCL;
    else if( strcmp(token,TX_DEFINE_PROC)==0 )    tokenType = DEFINE_PROC;
    else if( strcmp(token,TX_INSTALL_CMD)==0 )    tokenType = INSTALL_CMDS;
    else if( strcmp(token,TX_INSTALL_CMDS)==0 )   tokenType = INSTALL_CMDS;
    else                                          tokenType = MISCELLANEOUS;
  }else if( c=='\"' || c=='\'' ){         /* String and character literals */
    int cStart = c;
    int startLine = fileLine;
    do{
      int cPrev;
      AddCharToToken(c);
      cPrev = c;
      c = GetInput();
      if( cPrev=='\\' ){
        AddCharToToken(c);
        c = GetInput();
      }
    }while( c!=EOF && c!=cStart );
    if( c==EOF ){
      fprintf(stderr,"ERROR: Unterminated string beginning on line %d of %s.\n",
        startLine,fileName);
      errorCount++;
    }
    AddCharToToken(c);
    tokenType = MISCELLANEOUS;
  }else if( c=='/' ){
    AddCharToToken(c);
    c = GetInput();
    if( c=='/' ){                          /* C++ comments */
      do{
        AddCharToToken(c);
        c = GetInput();
      }while( c!=EOF && c!='\n' );
      AddCharToToken(c);
    }else if( c=='*' ){                    /* C comments */
      int cPrev;
      int startLine = fileLine;
      do {
        AddCharToToken(c);
        cPrev = c;
        c = GetInput();
      }while( c!=EOF && (c!='/' || cPrev!='*') );
      if( c==EOF ){
        fprintf(stderr,"ERROR: Unterminated comment starting at line %d of %s.\n",
          startLine,fileName);
        errorCount++;
      }
      AddCharToToken(c);
    }else{
      PushBackIntoInput(c);
    }
    tokenType = MISCELLANEOUS;
  }else{
    AddCharToToken(c);
    tokenType = MISCELLANEOUS;
  }
  AddCharToToken(0);
  return tokenType;
}

/* Print the contents of the token buffer on standard output */
static void
PrintToken()
{
  AddCharToToken(0);
  printf("%s",token);
}

/* Copy tokens from input to output.
** Stop when the first unmatched close parenthesis ")" is seen.
** The close parenthesis is retained.
**
** Unlike PrintTokenAsString(), the input text is not translated into the
** C string format.
*/
static void
MoveExpressionToDiversion()
{
  int depth = 1;

  WriteToDiversion("(");
  while( depth ){
    GetNextCToken();
    if( tokenType==LEFT_PAREN ) depth++;
    if( tokenType==RIGHT_PAREN ) depth--;
    AddCharToToken(0);
    WriteToDiversion(token);
  }
}

/* This function is called when a "ET", "ET_INT", "ET_STR" or
** "ET_DBL" token is seen in the C source code.  It processes
** all input up to the next matching close parenthesis.
*/
static void
ProcessTcl(zSuffix)
char *zSuffix;   /* Suffix for the evaluator function name */
{
  int inLiteral = 0;        /* True if the next character of output will
                            ** be part of a string literal in the resulting
                            ** C program */
  int c;                    /* Next character of input */
  int atLineStart = 1;      /* True if at the beginning of a line */
  char buf[20];             /* A bit of text to be appended to the
                            ** diversion.  Never more than 2 or 3 characters */
  int n;                    /* Number of characters in buf[] */
  int doOneNewline = 0;     /* True to cause a newline to be output */
  int inComment = 0;        /* True if currently within a Tcl/Tk comment */
  int startLine;            /* Line number containing the "ET_XXX" */
  int inQuote = 0;          /* True if inside a double-quoted string */
  int nBrace = 0;           /* Number of unmatched "{" seen */
  int nParen = 1;           /* Number of unmatched "(" seen.  Parenthesis
                            ** are only counted if they occur outside of
                            ** all curly braces, i.e. when nBrace==0 */

  /* Get the first "(" which must immediately follow the opening "ET_XXX" */
  c = GetInput();
  if( c!='(' ){
    PrintToken();
    PushBackIntoInput(c);
    return;
  }

  /* Now, read and process the Tcl/Tk source */
  startLine = fileLine;
  printf("Et_Eval%s(__FILE__,__LINE__,\"",zSuffix);
  do{
    c = GetInput();
    n = 0;
    buf[n++] = c;
    switch( c ){
      case EOF:
        fprintf(stderr,"Unclosed ET() macro beginning on line %d\n",startLine);
        exit(1);
      case '%':
        c = GetInput();
        if( c=='d' || c=='f' || c=='s' || c=='q' ){
          int paren = GetInput();
          if( paren!='(' ){
            buf[n++] = c;
            PushBackIntoInput(paren);
          }else{
            n--;
            if( inLiteral ) WriteToDiversion("\",");
            else            WriteToDiversion(",");
            printf("%c",c);
            MoveExpressionToDiversion();
            inLiteral = 0;
          }
        }else{
          PushBackIntoInput(c);
        }
        atLineStart = 0;
        break;

      case '#':
        if( atLineStart ) inComment = 1;
        break;

      case '(':
        if( !inQuote && !inComment && nBrace==0 ) nParen++;
        atLineStart = 0;
        break;

      case ')':
        if( !inQuote && !inComment && nBrace==0 ) nParen--;
        if( nParen==0 ) n--;
        atLineStart = 0;
        break;

      case '{':
        if( !inQuote && !inComment ) nBrace++;
        atLineStart = 0;
        break;

      case '}':
        if( !inQuote && !inComment ) nBrace--;
        atLineStart = 0;
        break;

      case '"':
        buf[0] = '\\';
        buf[1] = '"';
        n = 2;
        if( !inComment ) inQuote = !inQuote;
        atLineStart = 0;
        break;

      case '\\':
        buf[n++] = '\\';
        c = GetInput();
        if( c=='\n' ){
          PushBackIntoInput(c);
        }else{
          if( c=='\\' || c=='"' ){
            buf[n++] = '\\';
          }
          buf[n++] = c;
        }
        atLineStart = 0;
        break;

      case '\n':
        buf[0] = '\\';
        buf[1] = 'n';
        n = 2;
        atLineStart = compress_flag;
        doOneNewline = 1;
        inComment = 0;
        break;

      case ' ':
      case '\t':
        if( atLineStart ){ n--; break; }
        break;

      default:
        atLineStart = 0;
        break;
    }
    if( n>0 ){
      if( !inLiteral ){ 
        printf("s"); 
        WriteToDiversion(",\"");
        inLiteral = 1; 
      }
      buf[n] = 0;
      if( !inComment ) WriteToDiversion(buf);
      if( doOneNewline ){
        WriteToDiversion("\"");
        inLiteral = 0;
        WriteToDiversion("\n");
        doOneNewline = 0;
      }
    }
  }while( nParen );
  printf("\"%s",diversion);
  diverNext = 0;
  if( inLiteral ) printf("\"");
  printf(")");
}

/*
** On WIN32 systems, change the \ marks in a pathname into /.
*/
static void
FixDirectorySeparators(char *zBuf){
#if WIN32
  int i;
  for(i=0; zBuf[i]; i++){
    if( zBuf[i]=='\\' ) zBuf[i] = '/';
  }
#endif
}

/* Return TRUE if the given pathname is absolute and FALSE if it
** is relative.
*/
static int
IsAbsolutePath(char *zPath){
#if UNIX
  return zPath[0]=='/';
#else
  return zPath[0]=='/' || zPath[0]=='\\'
      || (isalpha(zPath[0]) && zPath[1]==':' &&
         (zPath[2]=='/' || zPath[2]=='\\'));
#endif
}

/*
** Process one of the ET_INCLUDE(f) macros.  
**
** Code is generated which make the text of the file "f" into a
** string constant.  Then, code is generated which make a call to
** evaluate that constant as a Tcl command string.
**
** If the krFlag is TRUE (indicating that code is being generated
** for an older non-ANSI compiler) then the string constant is 
** constructed one character at a time.  This is because many older
** compilers have restrictive limits on the length of strings.  If
** the krFlag is FALSE, then string constants are used since they
** will often compile faster on a modern ANSI C compiler.
**
** If the dynamicFlag is true, then we don't really read in the
** include file, but instead generate an "ET(source FILENAME)" command
** which will case the file to be read at run-time.
*/
static void
ProcessIncludeMacro(dynamicFlag,krFlag)
int dynamicFlag;    /* True if the file should be read at runtime */
int krFlag;         /* True if the target compiler is not ANSI */
{
  int c;            /* The next character read */
  FILE *pIn;        /* For reading the include file */
  int atLineStart;  /* True if the read point is at the beginning of
                    ** a line in the include file.  Use to suppress
                    ** leading space and comments */
  int nChar;        /* Number of character constants output.  Used to
                    ** known when to break a line */
  int i;            /* For looping thru the includePath[] names */
  char *zPath;      /* Full pathname of the input file */

  /* Get the first "(" which must immediately follow the opening
  ** ET_INCLUDE keyword */
  c = GetInput();
  if( c!='(' ){
    PrintToken();
    PushBackIntoInput(c);
    return;
  }

  /* Find the file name.  The file name is the first space-delimited token
  ** seen. */
  do{
    c = GetInput();
  }while( isspace(c) );
  tokenNext = 0;
  do{
    AddCharToToken(c);
    c = GetInput();
  }while( c!=EOF && !isspace(c) && c!=')' );
  AddCharToToken(0);
  while( c!=EOF && c!=')' ) c = GetInput();

  /* Open the file */
  zPath = SafeMalloc( strlen(token) + maxIncludePathLength + 2 );
  pIn = 0;
  if( token[0]=='/' ){
    strcpy(zPath,token);
    FixDirectorySeparators(zPath);
    pIn = fopen(zPath,"r");
  }else{
    for(i=0; pIn==0 && i<nIncludePath; i++){
      sprintf(zPath,"%s/%s",includePath[i],token);
      FixDirectorySeparators(zPath);
      pIn = fopen(zPath,"r");
    }
  }
  if( pIn==0 ){
    char zErrMsg[400];

    sprintf(zErrMsg,"ERROR: %.*s line %d: can't open include file \"%.*s\"",
       (int)sizeof(zErrMsg)/2-50,fileName,fileLine,
       (int)sizeof(zErrMsg)/2-50,token);
    perror(zErrMsg);
    errorCount++;
  }

  /* Generate code to read the include file at run-time if the 
  ** dynamicFlag is true */
  if( dynamicFlag ){
    if( IsAbsolutePath(zPath) ){
      printf("Et_Eval(__FILE__,__LINE__,\"s\",\"source %s\")",zPath);
    }else{
      int i;
      char cwd[5000];
      extern char *getcwd();
      if( getcwd(cwd,sizeof(cwd)-1)==0 ) strcpy(cwd,".");
      FixDirectorySeparators(cwd);
      printf("Et_Eval(__FILE__,__LINE__,\"s\",\"source %s/%s\")",cwd,zPath);
    }
    if( pIn ) fclose(pIn);
    pIn = 0;
  }
  free( zPath );
  if( pIn==0 ) return;

  /* Control only reaches this point if the dynamicFlag is false.  That
  ** means we need to read the include file now, at compile time. */
  if( krFlag ){
    /* Generate code for older compilers that can't handle long strings
    ** or concatenated string */
    printf("{ static char _ET_script_[] = {\n");
    atLineStart = 1;
    nChar = 0;
    while( (c=getc(pIn))!=EOF ){
      switch( c ){
        case ' ':
        case '\t':
          if( !atLineStart ){
             printf(" ' ',");
             nChar++;
          }
          break;

        case '\n':
          printf(" '\\n',");
          atLineStart = compress_flag;
          nChar++;
          break;

        case '\'':
        case '\"':
        case '\\':
          printf(" '\\%c',",c);
          nChar++;
          atLineStart = 0;
          break;

        case '#':
          if( atLineStart ){
            while( c!=EOF ){
              if( c=='\\' ){
                c = getc(pIn);
                if( c=='\n' ){
                  printf(" '\\n',");
                  nChar++;
                }else if( c==EOF ){
                  break;
                }
              }else if( c=='\n' ){
                printf(" '\\n',");
                nChar++;
                break;
              }
              c = getc(pIn);
            }
          }else{
            printf(" '#',");
            nChar++;
          }
          break;
      
        default:
          if( isprint(c) ){
            printf(" '%c',",c);
          }else{
            printf(" '\\%03o',",c);
          }
          atLineStart = 0;
          nChar++;
          break;
      }
      if( nChar>12 ){ printf("\n"); nChar = 0; }
    }
    printf("0};\n");
  }else /* if( !krFlag ) */{
    /* This is the case for a modern ANSI-C compiler */
    printf("{ static char _ET_script_[] = \n\"");
    atLineStart = 1;
    nChar = 0;
    while( (c=getc(pIn))!=EOF ){
      switch( c ){
        case ' ':
        case '\t':
          if( !atLineStart ){
             printf(" ");
          }
          break;

        case '\n':
          printf("\\n\"\n\"");
          atLineStart = compress_flag;
          break;

        case '\"':
        case '\\':
          printf("\\%c",c);
          atLineStart = 0;
          break;

        case '#':
          if( atLineStart ){
            while( c!=EOF ){
              if( c=='\\' ){
                c = getc(pIn);
                if( c=='\n' ){
                  printf("\\n\"\n\"");
                }else if( c==EOF ){
                  break;
                }
              }else if( c=='\n' ){
                printf("\\n\"\n\"");
                break;
              }
              c = getc(pIn);
            }
          }else{
            printf("#");
          }
          /* atLineStart = 1; */
          break;
      
        default:
          if( isprint(c) ){
            printf("%c",c);
          }else{
            printf("\\%03o",c);
          }
          atLineStart = 0;
          nChar++;
          break;
      }
    }
    printf("\";\n");
  }
  fclose(pIn);

  /* Finish out the command */
  printf("#line %d \"%s\"\n",fileLine,fileName);
  printf("Et_EvalInclude(\"%s\",_ET_script_); }",token);
}

/*
** The characters that follow should be an open-parenthesis, optional
** white space, an identifier or filename, more optional white-space
** and finally a close-parenthesis.  
**
**                (   ID   )
**
** If the characters of the input stream do NOT match this, then
** print the current token, followed by the input characters and
** return TRUE.
**
** If the characters do match this pattern, load the text of ID into
** the buffer "buf" and return FALSE.
*/
static int
GetArgument(pbuf)
char **pbuf;   /* Space in which to write the indentifier or filename */
{
  int i = 0;
  int c;
  int nBuf = 0;   /* Amount of space malloc'ed for *pbuf */

  c = GetInput();
  if( c!='(' ){
    PrintToken();
    printf("(");
    return 1;
  }
  do{
    c = GetInput();
  }while( isspace(c) );
  while( !isspace(c) && c!=EOF && c!=')' ){
    if( i>=nBuf-1 ){ 
      if( nBuf==0 ){
        nBuf = 100;
        *pbuf = SafeMalloc( nBuf );
      }else{
        nBuf *= 2;
        *pbuf = SafeRealloc( *pbuf, nBuf );
      }
    }
    (*pbuf)[i++] = c;
    c = GetInput();
  }
  (*pbuf)[i] = 0;
  while( isspace(c) ) c = GetInput();
  if( c!=')' ){
    PrintToken();
    printf(" %s ",*pbuf);
    free( *pbuf );
    return 1;
  }
  return 0;
}


/*
** Convert a name into a valid C identifier by converting non-alphanum
** characters into underscores.  Return a pointer to the converted name.
** The space to hold the converted name is obtained from malloc() and
** must be freed by the calling function.
*/
static char *
SanitizeName(name,baseOnly)
char *name;
int baseOnly;
{
  int i;
  char *newName;
  extern char *strrchr();

  if( baseOnly ){
    char *cp = strrchr(name,'/');
    if( cp ) name = &cp[1];
    cp = strrchr(name,'\\');
    if( cp ) name = &cp[1];
  }
  newName = SafeMalloc( strlen(name) + 1 );
  strcpy(newName,name);
  for(i=0; newName[i]; i++){
    if( !isalnum(newName[i]) ) newName[i] = '_';
    if( isupper(newName[i]) ) newName[i] = tolower(newName[i]);
  }
  return newName;
}

/*
** Process the "ET_PROC" macro.
**
** The parameter pnProcs is a pointer to an integer which holds the
** number of previously defined procedures.  pazProcs is a pointer
** to an array of procedure names.  Both of these variables will
** be updated.
*/
static void
ProcessDefineProc(pnProcs,pazProcs,krFlag)
int *pnProcs;        /* Number of procedures prior to this one */
char ***pazProcs;    /* Array of previously defined procedures */
int krFlag;          /* True to generate K&R.  False for ANSI-C */
{
  char *zCmdname = 0;       /* Name of the Tcl/Tk command */
  char *zSanitized = 0;     /* The name with special characters removed */

  if( GetArgument(&zCmdname) ) return;
  zSanitized = SanitizeName(zCmdname,0);
  if( krFlag ){
    printf("\
static int Et_Proc_%s(clientData,interp,argc,argv) \
void *clientData; \
struct Tcl_Interp *interp; \
int argc; \
char **argv;",
      zSanitized);
  }else{
    printf("\
static int Et_Proc_%s(void *clientData,struct Tcl_Interp *interp,\
int argc,char **argv)",
      zSanitized);
  }
  if( *pnProcs==0 ){
    *pazProcs = (char**)SafeMalloc( sizeof(char*) );
  }else{
    *pazProcs = (char**)SafeRealloc( *pazProcs, sizeof(char*)*(1 + *pnProcs) );
  }
  (*pazProcs)[*pnProcs] = SafeMalloc( strlen(zCmdname) + 1 );
  strcpy((*pazProcs)[*pnProcs],zCmdname);
  ++*pnProcs;
  free( zCmdname );
  free( zSanitized );
}   

/*
** Process the "ET_INSTALL_COMMANDS" macro.
**
*/
static void
ProcessInstallCmds(krFlag)
int krFlag;    /* True to generate K&R.  False for ANSI-C */
{
  int c;
  char *arg = 0;
  char *zFilename = 0;

  c = GetInput();
  PushBackIntoInput(c);
  if( c=='(' ){
    if( GetArgument(&arg) ) return;
    zFilename = SanitizeName(arg,1);
    free(arg);
  }else{
    zFilename = SanitizeName(fileName,1);
  }
  if( krFlag ){
    printf("{void Et_Init_%s(); Et_Init_%s();}",zFilename,zFilename);
  }else{
    printf("{void Et_Init_%s(void); Et_Init_%s();}",zFilename,zFilename);
  }
  free( zFilename );
}

/*
** Print code which defines EXTERN appropriately for either C or C++
*/
static void
DefineExtern(){
  printf("#ifdef __cplusplus\n");
  printf("#  define EXTERN extern \"C\"\n");
  printf("#else\n");
  printf("#  define EXTERN extern\n");
  printf("#endif\n");
}

/*
** Add code to the end of a program to install a set of new Tcl/Tk
** commands.  nProcs is the number of new commands, and azProcs is
** an array of pointers to strings, each of which contains the name
** of a command.
*/
static void
AppendProcInstaller(nProcs,azProcs,krFlag)
int nProcs;           /* Number of new commands to install */
char **azProcs;       /* An array of names of new commands */
int krFlag;           /* True to generate K&R.  False for ANSI-C */
{
  int i;
  char *zFilename;

  printf("\n/******* Automatically generated code follows ********/\n");
  zFilename = SanitizeName(fileName,1);
  if( krFlag ){
    printf("extern void Et_InstallCommand();\n");
    printf("void Et_Init_%s(){\n",zFilename);
  }else{
    DefineExtern();
    printf("EXTERN void Et_InstallCommand(char *,int (*)(void*,struct Tcl_Interp*,int,char**));\n");
    printf("void Et_Init_%s(void){\n",zFilename);
  }
  for(i=0; i<nProcs; i++){
    char *z = SanitizeName(azProcs[i]);
    printf("  Et_InstallCommand(\"%s\",Et_Proc_%s);\n",
      azProcs[i],z);
    free( z );
  }
  printf("}\n");
  free(zFilename);

  /* ET used to provide a feature where ET_INSTALL_COMMANDS(...) would
  ** accept a command name instead of an argument name.  The following
  ** code used to implement that feature.  As the feature was never used,
  ** it has been removed. */
#if 0
  for(i=0; i<nProcs; i++){
    char *z = SanitizeName(azProcs[i]);
    if( krFlag ){
      printf("void Et_Init_%s(){ Et_InstallCommand(\"%s\",Et_Proc_%s); }\n",
        z,azProcs[i],z);
    }else{
      printf("void Et_Init_%s(void){ Et_InstallCommand(\"%s\",Et_Proc_%s); }\n",
        z,azProcs[i],z);
    }
    free( z );
  }
#endif
}

/* Read all input, make the necessary conversion, and send the
** output to standard-out.
*/
static void
ProcessFile(dynamicFlag,krFlag)
int dynamicFlag;        /* True if ET_INCLUDE should be read at run-time */
int krFlag;             /* True to generate K&R C code only */
{
  char **azProcs = 0;   /* Names contained in every ET_PROC() macro */
  int nProcs = 0;       /* Number of ET_PROC() macros seen */
  int i;                /* Loop counter */

  printf("/* DO NOT EDIT! \n");
  printf("** This file has been automatically generated from \"%s\" */\n",
     fileName);
  if( krFlag ){
    printf("#define EXTERN extern\n");
    printf("extern int Et_Eval();\n");
    printf("extern int Et_EvalInt();\n");
    printf("extern char *Et_EvalString();\n");
    printf("extern double Et_EvalDouble();\n");
    printf("extern void Et_EvalInclude();\n");
    printf("extern void Et_Init();\n");
    printf("extern void Et_MainLoop();\n");
    printf("extern void Et_ReadStdin();\n");
  }else{
    DefineExtern();
    printf("EXTERN int Et_Eval(const char *,int,const char*,...);\n");
    printf("EXTERN int Et_EvalInt(const char *,int,const char*,...);\n");
    printf("EXTERN char *Et_EvalString(const char *,int,const char*,...);\n");
    printf("EXTERN double Et_EvalDouble(const char *,int,const char*,...);\n");
    printf("EXTERN void Et_EvalInclude(const char *,char *);\n");
    printf("EXTERN void Et_Init(int *, char **);\n");
    printf("EXTERN void Et_MainLoop(void);\n");
    printf("EXTERN void Et_ReadStdin(void);\n");
  }
  printf("EXTERN struct Tcl_Interp *Et_Interp;\n");
  printf("EXTERN struct Tk_Window_ *Et_MainWindow;\n");
  printf("EXTERN struct _XDisplay *Et_Display;\n");
  printf("#define ET_OK %d\n",TCL_OK);
  printf("#define ET_ERROR %d\n",TCL_ERROR);
  printf("#undef EXTERN\n");
  if( dynamicFlag ){
    printf("#define ET_DYNAMIC 1\n");
  }
  printf("#define UNIX %d\n#define WIN32 %d\n",UNIX,WIN32);
  printf("#line 1 \"%s\"\n",fileName);
  do{
    GetNextCToken();
    switch( tokenType ){
      case BEGIN_TCL:
        ProcessTcl("");
        break;
      case BEGIN_TCL_INT:
        ProcessTcl("Int");
        break;
      case BEGIN_TCL_STR:
        ProcessTcl("String");
        break;
      case BEGIN_TCL_DBL:
        ProcessTcl("Double");
        break;
      case INCLUDE_TCL:
        ProcessIncludeMacro(dynamicFlag,krFlag);
        break;
      case DEFINE_PROC:
        ProcessDefineProc(&nProcs,&azProcs,krFlag);
        break;
      case INSTALL_CMDS:
        ProcessInstallCmds(krFlag);
        break;
      case END_OF_FILE:
        break;
      default:
        PrintToken();
    }
  }while( tokenType!=END_OF_FILE );
  AppendProcInstaller(nProcs,azProcs,krFlag);
  for(i=0; i<nProcs; i++) free(azProcs[i]);
  if( azProcs ) free(azProcs);
}

/* Figure out what the name of the input file is, open the input file,
** then call ProcessFile() to do the work.
*/
void
main(argc, argv)
int argc;
char **argv;
{
  int i,j;               /* Loop counters */
  int noMoreSwitches;    /* True if subsequent arguments are filenames 
                         ** even if they begin with '-' */
  int dynamicFlag = 0;   /* True to load ET_INCLUDE() macros at run-time */
  int krFlag = 0;        /* True to generate K&R code instead of ANSI */
  char *inputFile = 0;   /* Name of the input file */

  /* Find and process every -I command line switch */
  nIncludePath = 1;      /* Always have at least one path, namely "." */
  for(i=1; i<argc; i++){
    if( strncmp(argv[i],"-I",2)==0 ) nIncludePath++;
    if( strcmp(argv[i],"--")==0 ) break;
  }
  includePath = (char **)SafeMalloc( sizeof(char*)*nIncludePath );
  includePath[0] = ".";
  maxIncludePathLength = 1;
  j = 1;
  for(i=1; i<argc; i++){
    if( strncmp(argv[i],"-I",2)==0 ){
      includePath[j] = &argv[i][2];
      if( maxIncludePathLength < strlen(includePath[j]) ){
        maxIncludePathLength = strlen(includePath[j]);
      }
      j++;
    }else if( strcmp(argv[i],"-dynamic")==0 ){
      dynamicFlag = 1;
    }else if( strcmp(argv[i],"-K+R")==0 ){
      krFlag = 1;
    }else if( strcmp(argv[i],"-nocompress")==0 ){
      compress_flag = 0;
    }
    if( strcmp(argv[i],"--")==0 ) break;
  }

  /* Get the command line argument which is the name of the input
  ** file */
  noMoreSwitches = 0;
  for(i=1; i<argc; i++){
    if( strncmp(argv[i],"-I",2)==0 && !noMoreSwitches ) continue;
    if( strcmp(argv[i],"-dynamic")==0 && !noMoreSwitches ) continue;
    if( strcmp(argv[i],"-nocompress")==0 && !noMoreSwitches ) continue;
    if( strcmp(argv[i],"-K+R")==0 && !noMoreSwitches ) continue;
    if( strcmp(argv[i],"-noshroud")==0 && !noMoreSwitches ) continue;
    if( strcmp(argv[i],"--")==0 && !noMoreSwitches ){
      noMoreSwitches = 1;
      continue;
    }
    if( inputFile ){
      fprintf(stderr,"%s: only one input file allowed\n",*argv);
      exit(1);
    }
    inputFile = argv[i];
  }

  /* Process the input file */
  if( inputFile==0 ){
    fileName = "(stdin)";
    fileIn = stdin;
  }else{
    fileName = inputFile;
    fileIn = fopen(fileName,"r");
    fileLine = 1;
    if( fileIn==0 ){
      char zErrMsg[1000];
      sprintf(zErrMsg,"Can't open \"%.*s\"",(int)sizeof(zErrMsg)-100,fileName);
      perror(zErrMsg);
      exit(1);
    }
  }
  ProcessFile(dynamicFlag,krFlag);
  exit(errorCount);
}
