/* 
   EODatabaseContext.h

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Mircea Oancea <mircea@jupiter.elcom.pub.ro>
   Date: 1996

   This file is part of the GNUstep Database Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef __EODatabaseContext_h__
#define __EODatabaseContext_h__

#include <Foundation/NSObject.h>

@class NSArray;
@class NSMutableArray;
@class NSDictionary;
@class NSMutableDictionary;
@class NSString;
@class NSMutableString;

@class EOAdaptorContext;
@class EOEntity;

@class EOObjectUniquer;
@class EODatabase;
@class EODatabaseContext;
@class EODatabaseChannel;

typedef enum {
    EOUpdateWithOptimisticLocking,
    EOUpdateWithPessimisticLocking,
    EOUpdateWithNoLocking,
    EONoUpdate,
} EOUpdateStrategy;

struct _EOTransactionScope;

@interface EODatabaseContext : NSObject
{
    EOAdaptorContext*		adaptorContext;
    EODatabase*			database;
    NSMutableArray*		channels;
    EOUpdateStrategy		updateStrategy;
    struct _EOTransactionScope*transactionStackTop;
    int 			transactionNestingLevel;
	
    // These fields should be in a bitfield but are ivars for debug purposes
    
    BOOL			isKeepingSnapshots;
    BOOL			isUniquingObjects;
}

// Initializing instances
- initWithDatabase:(EODatabase*)aDatabase;

// Getting the database object
- (EODatabase*)database;

// Getting the adaptor context
- (EOAdaptorContext*)adaptorContext;

// Finding channels
- (BOOL)hasBusyChannels;
- (BOOL)hasOpenChannels;
- (NSArray*)channels;
- createChannel;

// Controlling transactions
- (BOOL)beginTransaction;
- (BOOL)commitTransaction;
- (BOOL)rollbackTransaction;

// Notifying of other transactions
- (void)transactionDidBegin;
- (void)transactionDidCommit;
- (void)transactionDidRollback;

// Nesting transactions
- (BOOL)canNestTransactions;
- (unsigned)transactionNestingLevel; 

// Setting the update strategy
- (void)setUpdateStrategy:(EOUpdateStrategy)aStrategy;
- (EOUpdateStrategy)updateStrategy;
- (BOOL)keepsSnapshots;

// Handle Objects
- (void)forgetObject:anObj;
- objectForPrimaryKey:(NSDictionary*)aKey
  entity:(EOEntity*)anEntity;
- (NSDictionary*)snapshotForObject:anObj;
- (NSDictionary*)primaryKeyForObject:anObj;
- (void)primaryKey:(NSDictionary**)aKey
  andSnapshot:(NSDictionary**)aSnapshot
  forObject:anObj;
- (void)recordObject:anObj
  primaryKey:(NSDictionary*)aKey 
  snapshot:(NSDictionary*)aSnapshot;
- (void)recordObject:anObj
  primaryKey:(NSDictionary*)aKey
  entity:(EOEntity*)anEntity
  snapshot:(NSDictionary*)aSnapshot;
- (void)recordLockedObject:anObj;
- (BOOL)isObjectLocked:anObj;
- (void)recordUpdatedObject:anObj;
- (BOOL)isObjectUpdated:anObj;

@end /* EODatabaseContext */

/*
 * Methods used by database classess internally
 */

@interface EODatabaseContext(Private)
- (void)channelDidInit:aChannel;
- (void)channelWillDealloc:aChannel;
- (void)privateBeginTransaction;
- (void)privateCommitTransaction;
- (void)privateRollbackTransaction;
@end

#endif /* __EODatabaseContext_h__ */
