/* 
   EOFaultResolver.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Mircea Oancea <mircea@jupiter.elcom.pub.ro>
   Date: 1996

   This file is part of the GNUstep Database Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <Foundation/NSObject.h>
#include <Foundation/NSUtilities.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSString.h>
#include <Foundation/NSObjCRuntime.h>

#include <extensions/NSException.h>
#include <extensions/exceptions/GeneralExceptions.h>

#include <eoaccess/EOEntity.h>
#include <eoaccess/EOQualifier.h>

#include "EOFault.h"
#include "EOFaultResolver.h"
#include "EODatabaseContext.h"
#include "EODatabaseChannel.h"

@implementation EOFaultResolver

- initWithDatabaseChannel:(EODatabaseChannel*)aChannel
  zone:(NSZone*)aZone  
  targetClass:(Class)aTargetClass 
  fault:aFault
{
    [super init];
    channel = aChannel;
    targetClass = aTargetClass;
    fault = aFault;
    zone = aZone;
    faultReferences = 0;
    return self;
}

- (void)dealloc
{
    [super dealloc];
}

- (BOOL)beforeFault
{
    return NO;
}

- (BOOL)fault
{
    return NO;
}

- (BOOL)afterFault
{
    return NO;
}

- (EODatabaseChannel*)databaseChannel
{
    return channel;
}

- (Class)targetClass;
{
    return targetClass;
}

- (NSDictionary*)primaryKey
{
    return nil;
}

- (EOEntity*)entity
{
    return nil;
}

- (EOQualifier*)qualifier
{
    return nil;
}

- (NSArray*)fetchOrder
{
    return nil;
}

@end /* EOFaultResolver */

@implementation EOArrayFault

- initWithQualifier:(EOQualifier*)aQualifier
  fetchOrder:(NSArray*)aFetchOrder 
  databaseChannel:(EODatabaseChannel*)aChannel 
  zone:(NSZone*)aZone  
  targetClass:(Class)aTargetClass 
  fault:aFault;
{
    [super initWithDatabaseChannel:aChannel zone:aZone 
    	targetClass:aTargetClass fault:aFault];
    qualifier = [aQualifier retain];
    fetchOrder = [aFetchOrder retain];
    return self;
}

- (void) dealloc
{
    [qualifier autorelease];
    [fetchOrder autorelease];
    [super dealloc];
}

- (EOEntity*)entity
{
    return [qualifier entity];
}

- (EOQualifier*)qualifier
{
    return qualifier;
}

- (NSArray*)fetchOrder
{
    return fetchOrder;
}

- (BOOL)fault
{
    id anObj;
    BOOL tran;
    
    [EOFault clearFault:fault];
    [(id)fault init];
    
    if ([channel isFetchInProgress])
	THROW([[InvalidArgumentException alloc]
		initWithFormat:@"attempt to fault with busy channel: %@",
		[self description]]);
    
    tran = [[channel databaseContext] transactionNestingLevel] != 0;
    if (!tran)
	if (![[channel databaseContext] beginTransaction])
	    return NO;
    
    if (![channel selectObjectsDescribedByQualifier:qualifier
	    fetchOrder:fetchOrder]) {
	if (!tran)
	    [[channel databaseContext] rollbackTransaction];
	return NO;
    }
    
    while ((anObj = [channel fetchWithZone:zone])) {
	[(id)fault addObject:anObj];
    }
    
    [channel cancelFetch];

    if (!tran)
	if (![[channel databaseContext] commitTransaction])
	    return NO;

    return YES;
}

- (NSString*)description
{
    return [NSString stringWithFormat:
	@"<Array fault 0x%x (qualifier=%@, order=%@, channel=%@)>",
	fault,
	[qualifier description], 
	[fetchOrder description], 
	[channel description]];
}

@end /* EOArrayFault */

@implementation EOObjectFault

- initWithPrimaryKey:(NSDictionary*)aKey
  entity:(EOEntity*)anEntity 
  databaseChannel:(EODatabaseChannel*)aChannel 
  zone:(NSZone*)aZone  
  targetClass:(Class)aTargetClass 
  fault:aFault
{
    [super initWithDatabaseChannel:aChannel zone:aZone 
    	targetClass:aTargetClass fault:aFault];
    entity = [anEntity retain];
    primaryKey = [aKey retain];
    return self;
}

- (void) dealloc
{
    [entity autorelease];
    [primaryKey autorelease];
    [super dealloc];
}

- (NSDictionary*)primaryKey
{
    return primaryKey;
}

- (EOEntity*)entity
{
    return entity;
}

- (BOOL)fault
{
    EOQualifier* qualifier;
    BOOL tran;
    id anObj = nil;
    
    if ([channel isFetchInProgress])
	THROW([[InvalidArgumentException alloc]
		initWithFormat:@"attempt to fault with busy channel: %@",
		[self description]]);
    
    qualifier = [EOQualifier qualifierForPrimaryKey:primaryKey entity:entity];
    if (!qualifier)
	THROW([[InvalidArgumentException alloc]
		initWithFormat:@"could not build qualifier for fault: %@",
		[self description]]);
    
    tran = [[channel databaseContext] transactionNestingLevel] != 0;
    if (!tran)
	if (![[channel databaseContext] beginTransaction])
	    goto done;
    
    if (![channel selectObjectsDescribedByQualifier:qualifier
	    fetchOrder:nil]) {
	if (!tran)
	    [[channel databaseContext] rollbackTransaction];
	goto done;
    }

    // Fetch the object
    anObj = [channel fetchWithZone:zone];

    // The fetch failed!
    if (!anObj)
      {
	[channel cancelFetch];
	if (!tran)
	  [[channel databaseContext] rollbackTransaction];
	goto done;
      }

    // Make sure we only fetched one object
    if ([channel fetchWithZone:zone]) {
	anObj = nil;
    }
    
    [channel cancelFetch];
    
    if (!tran)
	if (![[channel databaseContext] commitTransaction])
	    anObj = nil;

done:
    if (anObj != fault) {
	if ([EOFault isFault:fault])
	    [EOFault clearFault:fault];
	[(id)fault unableToFaultWithPrimaryKey:primaryKey 
	    entity:entity databaseChannel:channel];
	return NO;
    }
    
    return YES;
}

- (NSString*)description
{
    return [NSString stringWithFormat:
	@"<Object fault 0x%x (class=%@, entity=%@, key=%@, channel=%@)>",
	fault,
	NSStringFromClass(targetClass), 
	[entity name], 
	[primaryKey description], 
	[channel description]];
}

@end /* EOObjectFault */
