/*
   PXKScrollView.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@net-community.com>
   Date: July 1997
   
   This file is part of the GNUstep GUI X/DPS Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"
#include <AppKit/NSClipView.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSGraphics.h>

#include <gnustep/xdps/PXKScrollView.h>
#include <gnustep/xdps/PXKDPSContext.h>

@implementation PXKScrollView

/* Rewrite some methods in order to deal with the border type */

+ (NSSize)contentSizeForFrameSize:(NSSize)frameSize
  hasHorizontalScroller:(BOOL)hFlag
  hasVerticalScroller:(BOOL)vFlag
  borderType:(NSBorderType)borderType
{
  NSSize size = [super contentSizeForFrameSize:frameSize
		       hasHorizontalScroller:hFlag
		       hasVerticalScroller:vFlag
		       borderType:borderType];

  /* Substract 1 from the width and height for the line that separates the
     horizontal and vertical scroller from the clip view. */
  if (hFlag)
    size.height -= 1;
  if (vFlag)
    size.width -= 1;

  switch (borderType) {
    case NSNoBorder:
      break;

    case NSLineBorder:
      size.width -= 2;
      size.height -= 2;
      break;

    case NSBezelBorder:
    case NSGrooveBorder:
      size.width -= 4;
      size.height -= 4;
      break;
  }

  return size;
}

+ (NSSize)frameSizeForContentSize:(NSSize)contentSize
  hasHorizontalScroller:(BOOL)hFlag
  hasVerticalScroller:(BOOL)vFlag
  borderType:(NSBorderType)borderType
{
  NSSize size = [super frameSizeForContentSize:contentSize
		       hasHorizontalScroller:hFlag
		       hasVerticalScroller:vFlag
		       borderType:borderType];

  /* Add 1 to the width and height for the line that separates the
     horizontal and vertical scroller from the clip view. */
  if (hFlag)
    size.height += 1;
  if (vFlag)
    size.width += 1;

  switch (borderType) {
    case NSNoBorder:
      break;

    case NSLineBorder:
      size.width += 2;
      size.height += 2;
      break;

    case NSBezelBorder:
    case NSGrooveBorder:
      size.width += 4;
      size.height += 4;
      break;
  }

  return size;
}

- (void)tile
{
  NSRect boundsRect = [self bounds];
  NSSize contentSize = [isa contentSizeForFrameSize:boundsRect.size
			    hasHorizontalScroller:_hasHorizScroller
			    hasVerticalScroller:_hasVertScroller
			    borderType:_borderType];
  float scrollerWidth = [NSScroller scrollerWidth];
  NSRect contentRect = { NSZeroPoint, contentSize };
  NSRect vertScrollerRect = NSZeroRect;
  NSRect horizScrollerRect = NSZeroRect;
  float borderThickness = 0;

  switch ([self borderType]) {
    case NSNoBorder:
      break;

    case NSLineBorder:
      borderThickness = 1;
      break;

    case NSBezelBorder:
    case NSGrooveBorder:
      borderThickness = 2;
      break;
  }

  contentRect.origin.x = borderThickness;
  contentRect.origin.y = borderThickness;

  if (_hasVertScroller) {
    vertScrollerRect.origin.x = boundsRect.origin.x + borderThickness;
    vertScrollerRect.origin.y = boundsRect.origin.y + borderThickness;
    vertScrollerRect.size.width = scrollerWidth;
    vertScrollerRect.size.height = bounds.size.height - 2 * borderThickness;

    contentRect.origin.x += scrollerWidth + 1;
  }

  if (_hasHorizScroller) {
    horizScrollerRect.origin.x = boundsRect.origin.x
				  + vertScrollerRect.origin.x
				  + vertScrollerRect.size.width
				  + 1;
    horizScrollerRect.origin.y = boundsRect.origin.y + borderThickness;
    horizScrollerRect.size.width = boundsRect.size.width
				  - horizScrollerRect.origin.x
				  - borderThickness;
    horizScrollerRect.size.height = scrollerWidth;

    contentRect.origin.y += scrollerWidth + 1;
  }

  [_contentView setFrame:contentRect];
  [_horizScroller setFrame:horizScrollerRect];
  [_vertScroller setFrame:vertScrollerRect];

  /* If the document view is not flipped then reverse the meaning of the
     the vertical scroller float value */
  if (![_contentView isFlipped])
    [_vertScroller setFloatValue:1];

  [self display];
}

- (void)drawRect:(NSRect)rect
{
  float scrollerWidth = [NSScroller scrollerWidth];
  float horizLinePosition, horizLineLength = [self bounds].size.width;
  float borderThickness = 0;

  PSgsave ();
  switch ([self borderType]) {
    case NSNoBorder:
      break;

    case NSLineBorder:
      borderThickness = 1;
      NSFrameRect (rect);
      break;

    case NSBezelBorder:
      borderThickness = 2;
      NSDrawWhiteBezel (rect, rect);
      break;

    case NSGrooveBorder:
      borderThickness = 2;
      NSDrawGroove (rect, rect);
      break;
  }

  horizLinePosition = borderThickness;

  PSsetlinewidth (1);
  PSsetgray (0);
  if (_hasVertScroller) {
    horizLinePosition = scrollerWidth + borderThickness;
    horizLineLength -= scrollerWidth + 2 * borderThickness;
    PSmoveto (horizLinePosition, borderThickness);
    PSrlineto (0, [self bounds].size.height - 2 * borderThickness);
    PSstroke ();
  }

  if (_hasHorizScroller) {
    PSmoveto (horizLinePosition, scrollerWidth + borderThickness + 1);
    PSrlineto (horizLineLength, 0);
    PSstroke ();
  }

  PSgrestore ();
}

@end
