/* 
   PXKScroller.m

   NSScroller for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: July 1997
   A completely rewritten version of the original source by Scott Christley.

   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "config.h"
#include <gnustep/xdps/PXKScroller.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <AppKit/NSWindow.h>

@implementation PXKScroller

static const float scrollerWidth = 17;
static const float buttonsWidth = 16;
static const float buttonsDistance = 1;

+ (float)scrollerWidth
{
  return scrollerWidth;
}

- (NSRect)rectForPart:(NSScrollerPart)partCode
{
  NSRect scrollerFrame = [self frame];
  float x = 1, y = 1, width = 0, height = 0, floatValue;
  NSScrollArrowPosition arrowsPosition;
  NSUsableScrollerParts usableParts;

  /* If the scroller is disabled then the scroller buttons and the knob are
     not displayed at all. */
  if (!_isEnabled)
    usableParts = NSNoScrollerParts;
  else
    usableParts = _usableParts;

  /* Since we don't have yet flipped views we have to swap the meaning
     of the arrows position if the scroller's orientation is vertical. */
  if (!_isHorizontal) {
    if (_arrowsPosition == NSScrollerArrowsMaxEnd)
      arrowsPosition = NSScrollerArrowsMinEnd;
    else if (_arrowsPosition == NSScrollerArrowsMinEnd)
      arrowsPosition = NSScrollerArrowsMaxEnd;
    else
      arrowsPosition = NSScrollerArrowsNone;
  }
  else
    arrowsPosition = _arrowsPosition;

  /* Assign to `width' and `height' values so that they represent the width
     and height of the scroller no matter what is its orientation.
     Also compute the `floatValue' which is essentially the same with
     _floatValue but it keeps track of the scroller orientation. */
   if (_isHorizontal) {
      width = scrollerFrame.size.height;
      height = scrollerFrame.size.width;
      floatValue = _floatValue;
    }
    else {
      width = scrollerFrame.size.width;
      height = scrollerFrame.size.height;
      floatValue = 1 - _floatValue;
    }

  /* The x, y, width and height values are computed below for the vertical
     scroller. The height of the scroll buttons is assumed to be equal with
     the width. */
  switch (partCode) {
    case NSScrollerKnob: {
      float knobHeight, knobPosition, slotHeight;

      /* If scroller has no parts or no knob then return the zero rect. */
      if (usableParts == NSNoScrollerParts
	  || usableParts == NSOnlyScrollerArrows)
	return NSZeroRect;

      /* Compute first `knobHeight', which shows how big the knob is */
      slotHeight = height - (arrowsPosition == NSScrollerArrowsNone
				? 0
				: 2 * (buttonsWidth + buttonsDistance) + 1);
      knobHeight = _knobProportion * slotHeight;
      if (knobHeight < buttonsWidth)
        knobHeight = buttonsWidth;

      /* Now determine the position of knob, knobPosition. */
      knobPosition = floatValue * (slotHeight - knobHeight);

      /* Now compute the actual position of the knob. */
      y = knobPosition + (arrowsPosition == NSScrollerArrowsMaxEnd
			  || arrowsPosition == NSScrollerArrowsNone
				? 0
				: 2 * (buttonsWidth + buttonsDistance));
      width = buttonsWidth;
      height = knobHeight;
      break;
    }

    case NSScrollerKnobSlot:
      x = 0;
      width = scrollerWidth + 1; /* for the lines surrounding the knob */

      /* If the scroller has no buttons then the knob slot completely fills
         the scroller. */
      if (usableParts == NSNoScrollerParts) {
	y = 1;
	/* `height' is unchainged */
	break;
      }

      if (arrowsPosition == NSScrollerArrowsMaxEnd) {
	y = 1;
	height -= 2 * (buttonsWidth + buttonsDistance) + 1;
      }
      else if (arrowsPosition == NSScrollerArrowsMinEnd) {
	y = 2 * (buttonsWidth + buttonsDistance) + 1;
	height -= y;
      }
      else {
	y = 1;
	/* `height' is unchanged */
      }
      break;

    case NSScrollerDecrementLine:
    case NSScrollerDecrementPage:
      /* If scroller has no parts then return the zero rect. */
      if (usableParts == NSNoScrollerParts)
	return NSZeroRect;

      width = buttonsWidth;
      if (arrowsPosition == NSScrollerArrowsMaxEnd)
	y = height - 2 * (buttonsWidth + buttonsDistance);
      else if (arrowsPosition == NSScrollerArrowsMinEnd)
	y = 1;
      else
	return NSZeroRect;
      height = buttonsWidth;
      break;

    case NSScrollerIncrementLine:
    case NSScrollerIncrementPage:
      /* If scroller has no parts then return the zero rect. */
      if (usableParts == NSNoScrollerParts)
	return NSZeroRect;

      width = buttonsWidth;
      if (arrowsPosition == NSScrollerArrowsMaxEnd)
	y = height - (buttonsWidth + buttonsDistance);
      else if (arrowsPosition == NSScrollerArrowsMinEnd)
	y = buttonsWidth + buttonsDistance + 1;
      else
	return NSZeroRect;
      height = buttonsWidth;
      break;

    case NSScrollerNoPart:
      return NSZeroRect;
  }

  if (_isHorizontal)
    return NSMakeRect (y, x, height, width);
  else
    return NSMakeRect (x, y, width, height);
}

- (void)drawKnobSlot
{
  NSRect rect = [self rectForPart:NSScrollerKnobSlot];

  PSgsave ();
  PSsetgray (0.5);
  PSrectfill (rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);
  PSgrestore ();
  [window _view: self needsFlushInRect: rect];
}

@end
