/*
   XRMenu.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   A completely rewritten version of the original source by Scott Christley.
   Author: Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: July 1998
   
   This file is part of the GNUstep GUI X/RAW Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <config.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "wraster/wraster.h"

#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSNotification.h>

#include <AppKit/NSMatrix.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSMenu.h>

#include <gnustep/xraw/XRContext.h>
#include <gnustep/xraw/XRMenu.h>
#include <gnustep/xraw/XR.h>

NSString* NSMenuLocationsKey = @"NSMenuLocations";

//
//  The XRMenuWindow class is required to override some window behaviours
//  so that menus don't mess up the behaviour of the application.
//  For instance, a menu window should never become the key or main
//  window and should never be seen in the windows menu.  Also, it should
//  NOT be released when closed!
//
@interface XRMenuWindow : NSWindow
@end

@implementation XRMenuWindow
- (BOOL) canBecomeMainWindow
{
  return NO;
}
- (BOOL) canBecomeKeyWindow
{
  return NO;
}
- (id) initWithContentRect: (NSRect)contentRect
                 styleMask: (unsigned int)aStyle
                   backing: (NSBackingStoreType)bufferingType
                     defer: (BOOL)flag
                    screen: (NSScreen*)aScreen
{
  self = [super initWithContentRect: contentRect
                          styleMask: aStyle
                            backing: bufferingType
                              defer: flag
                             screen: aScreen];
  [self setExcludedFromWindowsMenu: YES];
  [self setReleasedWhenClosed: NO];
  return self;
}
@end

//*****************************************************************************
//
// 		XRMenuMatrix 
//
//*****************************************************************************

@implementation XRMenuMatrix

- (BOOL)_processMouseEvent:(NSEvent*)theEvent level:(int)level
{
unsigned eventMask = NSLeftMouseUpMask | NSLeftMouseDownMask | NSMouseMovedMask 
					| NSRightMouseUpMask | NSRightMouseDraggedMask | 
					NSLeftMouseDraggedMask | NSPeriodicMask;
NSPoint lastLocation = [theEvent locationInWindow];
NSApplication* theApp = [NSApplication sharedApplication];
NSDate *theDistantFuture = [NSDate distantFuture];
float height = [self frame].size.height;
NSEvent* lastEvent = nil;
id aCell;
BOOL done = NO;
BOOL newlySelected = (selectedCell == nil);
int index;
NSWindow* lastWindow;
float mousex = 0;
static NSTimeInterval timeLast, timeNow;
BOOL should;
float menuItemHeight = (cellSize.height + INTERCELL_SPACE);

	ASSIGN(lastEvent, theEvent);
	[self lockFocus];

	while (!done) 
		{									// Convert point from the event's
		BOOL shouldGetAnotherEvent = YES;	// window to our window.  For main  
											// menu's window they are the same.
		if((lastWindow = [lastEvent window]) != window) 
			{											 
			lastLocation = [lastWindow convertBaseToScreen:lastLocation];
			lastLocation = [window convertScreenToBase:lastLocation];
			}
    	NSDebugLog (@"location = (%f, %f)", lastLocation.x, lastLocation.y);
   		lastLocation = [self convertPoint:lastLocation fromView:nil];

		if (lastLocation.x >= 0) 		// The mouse may be inside our window
			{							// or one of our attached menus 
			if (lastLocation.x < cellSize.width) 
				{						// The mouse may be inside our window, 
										// check the y coordinate. 
				if (lastLocation.y >= 0 && lastLocation.y < height) 
					{			// mouse is inside our window.  Determine the
								// selected cell.  If the mouse is between
								// cells simply assume it is on the upper cell. 
					index = (height - lastLocation.y) / menuItemHeight;
					aCell = [cells objectAtIndex:index];
													// unselect previous cell
					timeNow = [theEvent timestamp];	// and select the new one 
													// unless cells are the 
													// same.  If the previous 
					if (selectedCell != aCell) 		// cell has a submenu,
						{							// close submenu's window.
						should = YES;						
						if(lastLocation.x > (mousex + 4))	// but first make
							{								// sure the user
							if(timeNow < (timeLast + 300))	// is not doing a
								should = NO;				// diagonal drag
							}								// into a submenu
						if(should == YES)		
							{					
							if (selectedCell) 	
								{						
								if ([selectedCell hasSubmenu]) 	 
									{							
									[[selectedCell target] close];	
									[selectedCell setState:0];
									}						 
								[selectedCell highlight:NO 	
											  withFrame:selectedCellRect
											  inView:self];
								[window flushWindow];
								}
							if ([aCell isEnabled])
								{
								selectedCell = aCell;
								selectedCellRect = [self cellFrameAtRow:index];
								[selectedCell highlight:YES 
											  withFrame:selectedCellRect
											  inView:self];	// If selected cell
								[window flushWindow];		// has a submenu
								if ([selectedCell hasSubmenu]) 	// open it 
									[[selectedCell target] display];
								newlySelected = YES;
								}
							}
						}	
					timeLast = timeNow;
					mousex = lastLocation.x;
					}	
				}
			else 								// The mouse could be in one of
				{								// the right attached menus. 
				NSMenu* attachedMenu = [menu attachedMenu];
				id menuCells = [attachedMenu menuCells];

				if (attachedMenu && [menuCells _processMouseEvent:lastEvent 
															level:level + 1]) 
					{
					done = YES;
					shouldGetAnotherEvent = NO;
					}
				if (selectedCell) 
					{
					if (attachedMenu) 
						{
						[attachedMenu close];
						[selectedCell setState:0];
						}					
					[selectedCell highlight:NO 
								  withFrame:selectedCellRect
								  inView:self];
					selectedCell = nil;
					[window flushWindow];
			}	}	}
		else 							// The mouse is to the left of menu. 
			{							// Close the current menu window. 
			if (level) 
				{
				if (selectedCell) 
					{
					[selectedCell setState:0];
					[selectedCell highlight:NO 
								  withFrame:selectedCellRect
			      				  inView:self];
					[menu close];
  					[window flushWindow];
					selectedCell = nil;
					}
  				[self unlockFocus];
				[lastEvent release];

				return NO;
				}
    		}

		while (shouldGetAnotherEvent) 					
			{											// Get the next event
			theEvent = [theApp nextEventMatchingMask:eventMask
								untilDate:theDistantFuture
								inMode:NSEventTrackingRunLoopMode
								dequeue:YES];
			switch ([theEvent type]) 
				{
				case NSRightMouseUp:					// right mouse up or
				case NSLeftMouseUp:						// left mouse up means
					done = YES;							// we're done
					ASSIGN(lastEvent, theEvent);
				case NSPeriodic:
					shouldGetAnotherEvent = NO;
					break;
				default:
					ASSIGN(lastEvent, theEvent);
					continue;
				}
			}
    	lastLocation = [lastEvent locationInWindow];
		}
														// Unselect cell unless
	if (![selectedCell hasSubmenu]) 					// it has a submenu
		{	
		[selectedCell highlight:NO withFrame:selectedCellRect inView:self];
  		[window flushWindow];							// Send the menu item's  
		[menu performActionForItem:selectedCell];		// action to its target
		selectedCell = nil;
 		}							// Set the state of the selected cell, if 1
	else 							// this indicates the submenu is opened.
		{							
		if ([[selectedCell target] hasTornOffMenu] || newlySelected == NO)
			{
      		[selectedCell setState:0];
			[selectedCell highlight:NO withFrame:selectedCellRect inView:self];
			[window flushWindow];
			[[selectedCell target] close];
			selectedCell = nil;
    		}
    	else 
			{
      		[selectedCell highlight:YES 
						  withFrame:selectedCellRect 
						  inView:self];
			[window flushWindow];
      		[[selectedCell target] display];
    		}
  		}

	[self unlockFocus];
	[lastEvent release];

	return YES;
}

- (void)mouseDown:(NSEvent*)theEvent
{
BOOL rightMouseEvent = ([theEvent type] == NSRightMouseDown) ? YES : NO;

	[NSEvent startPeriodicEventsAfterDelay:0.05 withPeriod:0.05];
	[self _processMouseEvent:theEvent level:0];
	if(rightMouseEvent)								// if trigger event was a
		[menu close];								// right mouse, close menu
	[NSEvent stopPeriodicEvents];
}

@end /* XRMenuMatrix */

//*****************************************************************************
//
// 		XRMenu 
//
//*****************************************************************************

typedef struct 
{
	XRWindow* window;
	XRMenuWindowTitleView* titleView;
	BOOL isTornOff;
	BOOL hasTornOffMenu;
	XRMenu* isCopyOfMenu;
	id superSelCell;

} XRMenuPrivate;

#define WINDOW 			(((XRMenuPrivate*)be_menu_reserved)->window)
#define TITLEVIEW 		(((XRMenuPrivate*)be_menu_reserved)->titleView)
#define ISTORNOFF 		(((XRMenuPrivate*)be_menu_reserved)->isTornOff)
#define HASTORNOFFMENU 	(((XRMenuPrivate*)be_menu_reserved)->hasTornOffMenu)
#define ISCOPYOFMENU 	(((XRMenuPrivate*)be_menu_reserved)->isCopyOfMenu)
#define SUPERSELCELL 	(((XRMenuPrivate*)be_menu_reserved)->superSelCell)

static int titleHeight = 0;

@implementation XRMenu

+ (void)initialize
{
	if (self == [XRMenu class])
		[self setVersion:1];								// Initial version
}

- (void)_createWindowWithFrame:(NSRect)winRect
{
XSetWindowAttributes winattrs;
unsigned long valuemask;
NSRect menuWin = winRect;

	be_menu_reserved = calloc (sizeof (XRMenuPrivate), 1);
	titleHeight = [menuCells cellSize].height + 4;
	menuWin.size.height += titleHeight;
	WINDOW = [[XRMenuWindow alloc] initWithContentRect:menuWin
								styleMask:NSBorderlessWindowMask
								backing:NSBackingStoreRetained
								defer:NO];
	valuemask = (CWSaveUnder|CWOverrideRedirect);
	winattrs.save_under = True;
	winattrs.override_redirect = True;
	XChangeWindowAttributes ([XRContext currentXDisplay], [WINDOW xWindow], 
							valuemask, &winattrs);
	[[NSApplication sharedApplication] removeWindowsItem:WINDOW];
	
	NSDebugLog (@"create menu with title '%@', frame = (%f, %f, %f, %f)",
				[self title], winRect.origin.x, winRect.origin.y,
				winRect.size.width, winRect.size.height);

	TITLEVIEW = [XRMenuWindowTitleView new];
	[TITLEVIEW setFrameOrigin:NSMakePoint(0, winRect.size.height)];
	[TITLEVIEW setFrameSize:NSMakeSize (winRect.size.width, titleHeight)];
	[[WINDOW contentView] addSubview:menuCells];
	[[WINDOW contentView] addSubview:TITLEVIEW];
	[TITLEVIEW setMenu:self];
}

- (id)initWithTitle:(NSString*)aTitle
{
NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
NSApplication* theApp = [NSApplication sharedApplication];
NSRect winRect = {{0, 0}, {20, 17}};
	
	[super initWithTitle:aTitle];
	[self _createWindowWithFrame:winRect];
	
	[defaultCenter addObserver:self
					selector:@selector(_showTornOffMenuIfAny:)
					name:NSApplicationWillFinishLaunchingNotification
					object:theApp];
	[defaultCenter addObserver:self
					selector:@selector(_showTornOffMenuIfAny:)
					name:NSApplicationWillBecomeActiveNotification
					object:theApp];
	
	return self;
}

- (void)dealloc
{
	NSDebugLog (@"XRMenu '%@' dealloc", [self title]);
	
	[WINDOW release];
	[TITLEVIEW release];
	free (be_menu_reserved);

	[super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
XRMenu* copy = [super copyWithZone:zone];
NSRect frame = [XRMenuWindow frameRectForContentRect:[menuCells frame]
						 			   styleMask:[WINDOW styleMask]];

	[copy _createWindowWithFrame:frame];
	
	return copy;
}
											// This method is invoked when the 
- (void)_performMenuClose:sender			// user presses on the close button
{											// of a torn off menu window.
NSUserDefaults* defaults;
NSMutableDictionary* menuLocations;
NSString* key;
NSPoint location;

	NSDebugLog (@"_performMenuClose");
	
	[self close];											// close the window

	[SUPERSELCELL setTarget:self];							// return self to
	supermenu = [ISCOPYOFMENU supermenu];					// original status
	ISTORNOFF = NO;											// (state before we 
	location = [supermenu locationForSubmenu:self];			// were tornoff)
	[WINDOW setFrameOrigin:location];
	[TITLEVIEW _releaseCloseButton];

	defaults = [NSUserDefaults standardUserDefaults];
	menuLocations = [[[defaults objectForKey:NSMenuLocationsKey] mutableCopy] 
																 autorelease];

								// Inform our copy we are no longer available 
	(((XRMenuPrivate*)((ISCOPYOFMENU)->be_menu_reserved))->hasTornOffMenu) =NO;
	
	key = [self title];								// Remove window's position  
	if (key) 										// info from defaults db
		{
		[menuLocations removeObjectForKey:key];
		[defaults setObject:menuLocations forKey:NSMenuLocationsKey];
		[defaults synchronize];
		}
		
	[sender unlockFocus];
	[ISCOPYOFMENU autorelease];								// destroy the copy
	ISCOPYOFMENU = nil;
}
													// Show the torn off menus 
- (void)_showTornOffMenuIfAny:(NSNotification*)notification
{
NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
NSDictionary* menuLocations = [defaults objectForKey:NSMenuLocationsKey];
NSString* key;
NSArray* array;

	if ([[NSApplication sharedApplication] mainMenu] == self)					
		key = nil; 									// Ignore the main menu
	else
		key = [self title];
	
	if (key) 
		{
		array = [menuLocations objectForKey:key];
		if (array && [array isKindOfClass:[NSArray class]]) 
			{
			[self windowBecomeTornOff];
			[self display];
			}
		}
}

- (void)sizeToFit
{
  Display *xDisplay = [XRContext currentXDisplay];
  int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
  NSRect frame;

  [super sizeToFit];
  frame = [XRMenuWindow frameRectForContentRect: [menuCells frame]
                                      styleMask: [WINDOW styleMask]];
  frame.size.height += titleHeight;

  [WINDOW setFrame: frame display: NO];
  /*
   * Now we adjust the frame origin to be correctly positioned
   * relative to the parent menu (or as a torn-off menu.
   */
  if (supermenu)
    {
      frame.origin = [supermenu locationForSubmenu: self];
    }
  else 
    {
      NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
      NSDictionary* menuLocations = [defaults 
										objectForKey:NSMenuLocationsKey];
      NSString* key;
      NSArray* array;
	
      if ([[NSApplication sharedApplication] mainMenu] == self)
        key = @"Main menu";
      else
        key = [self title];
      
      if (key) 
        {
          array = [menuLocations objectForKey:key];
          if (array && [array isKindOfClass:[NSArray class]]) 
            {
              frame.origin.x = [[array objectAtIndex:0] floatValue];
              frame.origin.y = [[array objectAtIndex:1] floatValue];
            }
        }
    }
  /*
   * If the top of the menu would be off-screen - adjust it to fit.
   */
  if (frame.origin.y > DisplayHeight(xDisplay, xScreen) - titleHeight
						- [menuCells frame].size.height)
    frame.origin.y = DisplayHeight(xDisplay, xScreen) - titleHeight
						- [menuCells frame].size.height;

  [WINDOW setFrame: frame display: [WINDOW isVisible]];

  [TITLEVIEW setFrameOrigin: NSMakePoint (0,frame.size.height - titleHeight)];
  [TITLEVIEW setFrameSize: NSMakeSize (frame.size.width, titleHeight)];
  [TITLEVIEW setNeedsDisplay: YES];
  menuHasChanged = NO;
}

- (void)windowBecomeTornOff					// user has torn off a menu.  The
{											// new menu is released when the
XRMenu* copy = [self copy]; 				// user presses its close button
id supermenuMenuCells = [supermenu menuCells];
id supermenuSelectedCell = [supermenuMenuCells selectedCell];
NSRect rect;
									// There is no selected cell in super menu,
	if (!supermenuSelectedCell) 	// this happens when the method was invoked
		{							// to create torn off menus. Identify the
									// corresponding cell by comparing the cell 
									// targets from the upper menu cells array. 
		NSArray* cells = [supermenuMenuCells itemArray];
		int i, count = [cells count];
	
		for (i = 0; i < count; i++) 
			{
			id cell = [cells objectAtIndex:i];
	
			if ([cell target] == self) 
				{
				supermenuSelectedCell = cell;
				break;
		}	}	}
	else											// deselect the selected 
		{											// cell in the supermenu
		[supermenuMenuCells lockFocus];			
												
		rect = [supermenuMenuCells selectedCellRect];
		[supermenuSelectedCell setState:0];
		[supermenuSelectedCell highlight:NO
							   withFrame:rect
							   inView:supermenuMenuCells];
	
		[[supermenuMenuCells window] flushWindow];
		[supermenuMenuCells setSelectedCell:nil];
		[supermenuMenuCells unlockFocus];
		}
	/*
     * Must retain self 'cos changing our parent menu items target will
     * Cause us to be released.
     */
    [self retain];
	[supermenuSelectedCell setTarget:copy];
	
	((XRMenu*)supermenu)->attachedMenu = nil;
	supermenu = nil;
	((XRMenuPrivate*)copy->be_menu_reserved)->hasTornOffMenu = YES;

	[TITLEVIEW windowBecomeTornOff];
	ISTORNOFF = YES;
	ISCOPYOFMENU = copy;
	SUPERSELCELL = supermenuSelectedCell;
}

- (BOOL)isTornOff
{
	return ISTORNOFF;
}

- (NSPoint)locationForSubmenu:(NSMenu*)aSubmenu
{
NSRect frame = [WINDOW frame];
NSRect submenuFrame;

	if (aSubmenu)
    	submenuFrame = [((XRMenuPrivate*) 
					   (((XRMenu*)aSubmenu)->be_menu_reserved))->window frame];
	else
		submenuFrame = NSZeroRect;

  	return NSMakePoint (frame.origin.x + frame.size.width + 1,
		      			frame.origin.y + frame.size.height
						- submenuFrame.size.height);
}

- (void)update
{
  if ([WINDOW isVisible])
    {
      [super update];
    }
}

- (void)_rightMouseDisplay							// triggered by right mouse
{													// down. displays a copy of 
	if (supermenu) 									// the main menu under the
		{											// cursor
		NSPoint location = [supermenu locationForSubmenu:self];
													// we are not main menu so 
		[WINDOW setFrameOrigin:location];			// query super menu for our
		((XRMenu*)supermenu)->attachedMenu = self;	// position
		}
	else 
		{
		NSSize menuSize = [WINDOW frame].size;
		NSPoint mousePoint = [WINDOW mouseLocationOutsideOfEventStream];

		mousePoint = [WINDOW convertBaseToScreen:mousePoint];
		mousePoint.y -= (menuSize.height - 9);
		mousePoint.x = mousePoint.x - (menuSize.width/2);		
		[WINDOW setFrameOrigin:mousePoint];
		}
										// Invoke the -submenuAction: method 
	[self submenuAction:nil];			// so subclasses can use it. 
	
	[WINDOW orderFront:nil];
}

- (void)display
{													// we are not main menu so   
    if (menuHasChanged)
        [self sizeToFit];
	if (supermenu) 									// query super menu for our
		{											// position
		NSPoint location = [supermenu locationForSubmenu:self];

		[WINDOW setFrameOrigin:location];
		((XRMenu*)supermenu)->attachedMenu = self;
		}
	else 
		{
		NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
		NSDictionary* menuLocations = [defaults 
										objectForKey:NSMenuLocationsKey];
		NSString* key;
		NSArray* array;
		NSPoint origin;
	
		if ([[NSApplication sharedApplication] mainMenu] == self)
			key = @"Main menu";
		else
			key = [self title];
		
		if (key) 
			{
			array = [menuLocations objectForKey:key];
			if (array && [array isKindOfClass:[NSArray class]]) 
				{
				origin.x = [[array objectAtIndex:0] floatValue];
				origin.y = [[array objectAtIndex:1] floatValue];
				[WINDOW setFrameOrigin:origin];
				}
			}
		}
										// Invoke the -submenuAction: method 
	[self submenuAction:nil];			// so subclasses can use it. 
	
	[WINDOW orderFront:nil];
}

- (void)close
{
id selectedCell = [menuCells selectedCell];
										// Also close the attached menu. This 
	if (attachedMenu)					// is recursive, if the attached menu
		[attachedMenu close];			// has an attached menu this will also
										// get closed.
	if (selectedCell && [selectedCell isHighlighted]) 
		{
		NSRect selectedCellRect = [menuCells selectedCellRect];

		[menuCells lockFocus];				
		[selectedCell setState:0];
		[selectedCell highlight:NO 
					  withFrame:selectedCellRect 
					  inView:menuCells];
		[WINDOW flushWindow];
		[menuCells unlockFocus];
		[menuCells setSelectedCell:nil];
		}
										// Invoke the -submenuAction: method so 
	[self submenuAction:nil];			// subclasses can use it. 

	[WINDOW orderOut:nil];
	if (supermenu) 
		((XRMenu*)supermenu)->attachedMenu = nil;
}

- (XRWindow*)menuWindow					{ return WINDOW; }
- (XRMenuWindowTitleView*)titleView		{ return TITLEVIEW; }

- (BOOL)hasTornOffMenu
{
	return (((XRMenuPrivate*)be_menu_reserved)->hasTornOffMenu);
}

@end /* XRMenu */

//*****************************************************************************
//
// 		XRMenuWindowTitleView 
//
//*****************************************************************************

@implementation XRMenuWindowTitleView

static XRCell* titleCell = nil;

+ (void)initialize
{
	titleCell = [[XRCell alloc] initTextCell:@""];
	[titleCell setBordered:NO];
	[titleCell setEnabled:NO];
	[titleCell setFont:[NSFont boldSystemFontOfSize: 12]];
	[titleCell setAlignment:NSLeftTextAlignment];
}

- (void)setMenu:(NSMenu*)aMenu			{ menu = aMenu; }
- (NSMenu*)menu							{ return menu; }

- (void)drawRect:(NSRect)rect
{												
RImage *menuGradient;
RXImage *mySplatch;
RContext *rc;
RColor from, to;

	rc = RCreateContext([XRContext currentXDisplay], 
						[(XRScreen *)[NSScreen mainScreen] xScreen], NULL);

    from.red = 0x1e;  									// a blue gradient?
    from.green = 0x1e;
    from.blue = 0x8a;
    from.alpha = 0xff;
    to.red = 0x1e;
    to.green = 0x1e;
    to.blue = 0x63;
    to.alpha = 0xff; 

	if ((int)rect.size.width != 0)						// draw menu title view
		{
		menuGradient = RRenderGradient(rect.size.width, rect.size.height,
										&from, &to, RGRD_DIAGONAL);
		RBevelImage(menuGradient, RBEV_RAISED2);
		mySplatch = RConvertImage2(rc, menuGradient);
	
		XRDrawXImage(mySplatch->image, rect);
		}
								// Ugly, we use a private method to draw; but
								// this way we avoid creating another cell
	[titleCell _displayTitle:[menu title]
				inFrame:NSMakeRect(rect.origin.x + 3, rect.origin.y + 3,
				rect.size.width - 4, rect.size.height - 4)
	      		titleGray:NSWhite];
}

- (void)mouseDown:(NSEvent*)theEvent
{
XRContext* context = (XRContext *)[XRContext currentContext];
Display *xDisplay = [context xDisplay];
int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
unsigned int eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
							| NSLeftMouseDraggedMask | NSMouseMovedMask
							| NSPeriodicMask;
NSApplication *app = [NSApplication sharedApplication];
NSPoint point, lastPoint;
NSDate *theDistantFuture = [NSDate distantFuture];
NSEventType eventType;
Window xWindow = [window xWindow];
float wx, wy, originalX, originalY, menuWidth;
int init = 0;
XRMenu* attachedMenu;
NSUserDefaults* defaults;
NSMutableDictionary* menuLocations;
XRWindow* menuWindow;
XRMenu* appMainMenu;
NSPoint origin;
NSArray* array;
NSString* key;
NSRect menuWindowXFrame;
													
	menuWindowXFrame = [window xFrame];				// get menu window's global	
	wx = menuWindowXFrame.origin.x;					// X coordinates
	wy = menuWindowXFrame.origin.y;					
	menuWidth = menuWindowXFrame.size.width;					
	originalX = wx;
	originalY = wy;									// set periodic events rate
													// to achieve max of ~30fps
	[NSEvent startPeriodicEventsAfterDelay:0.02 withPeriod:0.03];
	[[NSRunLoop currentRunLoop] limitDateForMode:NSEventTrackingRunLoopMode];
													// convert cursor location
								 					// to X screen coordinates
	lastPoint = [[window contentView] convertPoint:[theEvent locationInWindow] 
									  fromView:nil];
	lastPoint = [window convertBaseToScreen:lastPoint];
	lastPoint.y = DisplayHeight(xDisplay, xScreen) - lastPoint.y;

	while ((eventType = [theEvent type]) != NSLeftMouseUp)				 
		{											// user is moving menu's
		if (eventType != NSPeriodic)				// loop until left mouse up
			{
			point = [theEvent locationInWindow];
			if(init < 1)							// first point has already
				{									// been converted to global
				init++;								// X screen coordinates	
				point = lastPoint;						 
				}
			else									// convert point to global
				{									// X screen coordinates
				point.y = menuWindowXFrame.size.height - point.y;
				point.y += wy;
				point.x += wx;
				}
			}
		else
			{									
			if (point.x != lastPoint.x || point.y != lastPoint.y) 
				{
				XRMenu* currentMenu = menu;
				float x, y;
				
				wx += (point.x - lastPoint.x);				// Move the menu
				wy += (point.y - lastPoint.y);
				XMoveWindow (xDisplay, xWindow, (int)wx, (int)wy);	
				lastPoint = point;						 
														 
				x = wx + menuWidth + 1;						// Move attached
				y = wy;										// menus if any
				while ((attachedMenu = (XRMenu*)[currentMenu attachedMenu])) 
					{
					menuWindow = [attachedMenu menuWindow];
					XMoveWindow (xDisplay, [menuWindow xWindow],(int)x,(int)y);
					currentMenu = attachedMenu;
					x += [menuWindow frame].size.width + 1;
					}
				}
			}
		theEvent = [app nextEventMatchingMask:eventMask
						untilDate:theDistantFuture 
						inMode:NSEventTrackingRunLoopMode
						dequeue:YES];
  		}												// left mouse button
														// has gone up so clean
	[NSEvent stopPeriodicEvents];						// up

	appMainMenu = (XRMenu*)[app mainMenu];
	defaults = [NSUserDefaults standardUserDefaults];
	menuLocations = [[[defaults objectForKey:NSMenuLocationsKey] mutableCopy] 
																 autorelease];
	[window _updateWindowGeometry];
	attachedMenu = (XRMenu*)[menu attachedMenu];
	while (attachedMenu) 
		{
		menuWindow = [attachedMenu menuWindow];
		[menuWindow _updateWindowGeometry];
		attachedMenu = (XRMenu*)[attachedMenu attachedMenu];
		}
	[context flush];
													// do not save menu pos or
	if (wx == originalX && wy == originalY)			// display as torn off if	
		return;										// position is unchanged
											
	if (!menuLocations)									
		menuLocations = [NSMutableDictionary dictionaryWithCapacity:2];
	origin = [window frame].origin;
	array = [NSArray arrayWithObjects:
						[[NSNumber numberWithInt:origin.x] stringValue],
						[[NSNumber numberWithInt:origin.y] stringValue], nil];
	if (menu == appMainMenu)
		key = @"Main menu";
	else
		key = [menu title];								// Save menu window pos
	[menuLocations setObject:array forKey:key];			// in defaults database
	[defaults setObject:menuLocations forKey:NSMenuLocationsKey];
	[defaults synchronize];

	NSDebugLog (@"move ready, save under name '%@'!", key);
														// display window close 
	if (menu != appMainMenu && ![menu isTornOff])		// button if the menu
		[(XRMenu*)menu windowBecomeTornOff];			// is not the main menu
}														// and not torn off

- (void)windowBecomeTornOff
{
	if ([menu isTornOff])								// do nothing if menu
		return;											// is already torn off
	else 
		{												// show close button
		NSImage* closeImage = [NSImage imageNamed:@"common_Close"];
		NSImage* closeHImage = [NSImage imageNamed:@"common_CloseH"];
    	NSSize imageSize = [closeImage size];
    	NSRect rect = { { frame.size.width - imageSize.width - 4,
	  					(frame.size.height - imageSize.height) / 2},
						{ imageSize.height, imageSize.width } };
    	int mask = NSViewMinXMargin | NSViewMinYMargin | NSViewMaxYMargin;
 
		button = [[[NSButton alloc] initWithFrame:rect] autorelease];
		[button setButtonType:NSMomentaryLight];		// configure the menu's
		[button setImagePosition:NSImageOnly];			// close button
		[button setImage:closeImage];
		[button setAlternateImage:closeHImage];
		[button setBordered:NO];
		[button setTarget:menu];
		[button setAction:@selector(_performMenuClose:)];
                [button setAutoresizingMask: NSViewMinXMargin];

		[self addSubview:button];
		[self setAutoresizingMask:mask];

		[button display];
  		}
}

- (void)_releaseCloseButton
{
	[button removeFromSuperview];
}

@end /* XRMenuWindowTitleView */
