"""Maintains graphical map window display."""

#    Copyright (C) 1998 Kevin O'Connor
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from Tkinter import *
import string
import operator

import empDb

###########################################################################
#############################  Map window     #############################
class mapSubWin:
    """Maintain map display"""

    def __init__(self, master):
	self.gridsize = (18.0, 24.0)
	self.maxCoord = empDb.megaDB['version']['worldsize']
	self.origin = (self.maxCoord[0]/2, self.maxCoord[1]/2)
	self.cursor = self.start = ()
	self.dimen = []

	scrollX = Scrollbar(master, name='scrollX', orient=HORIZONTAL)
	scrollX.pack(side=BOTTOM, fill=X)
	scrollY = Scrollbar(master, name='scrollY', orient=VERTICAL)
	scrollY.pack(side=RIGHT, fill=Y)
	self.Map = Canvas(master, name='sectors',
			  scrollregion= (-self.gridsize[0], -self.gridsize[1],
					 self.maxCoord[0]*self.gridsize[0],
					 self.maxCoord[1]*self.gridsize[1]),
			  xscrollcommand=scrollX.set,
			  yscrollcommand=scrollY.set)
	self.Map.pack(side=LEFT, expand=1, fill=BOTH)
	scrollX['command'] = self.Map.xview
	scrollY['command'] = self.Map.yview
	self.Map.create_rectangle(-self.gridsize[0], -self.gridsize[1],
				  self.maxCoord[0]*self.gridsize[0]-1,
				  self.maxCoord[1]*self.gridsize[1]-1,
				  tags='outline')

	self.Map.bind('<ButtonRelease>', self.DoCoordEndBox, "+")
	self.Map.bind('<Enter>', self.DoCoord)
	self.Map.bind('<Motion>', self.DoCoord)
	self.Map.bind('<Leave>', self.DoCoordClear)

	self.Map.bind('<Button-2>', self.DoCoordBox)
	self.Map.bind('<Button2-ButtonRelease>', self.DoSelect)
	self.Map.bind('<Shift-Button-1>', self.DoCoordBox)
	self.Map.bind('<Shift-Button1-ButtonRelease>', self.DoSelect)

	self.Map.bind('<Control-Button-1>',
		      (lambda e, self=self:
		       self.adjustSector(0.90)))
	self.Map.bind('<Control-Button-3>',
		      (lambda e, self=self:
		       self.adjustSector(1.10)))
	self.Map.bind('<Control-Shift-Button-1>',
		      (lambda e, self=self:
		       self.adjustSector(0.50)))
	self.Map.bind('<Control-Shift-Button-3>',
		      (lambda e, self=self:
		       self.adjustSector(2.00)))
	self.Map.bind('<Control-Button-2>',
		      (lambda e, self=self:
		       self.redraw(1)))

	self.Map.bind('<Button-1>', self.DoCensor)
	self.Map.bind('<Button-3>',
		      (lambda e, self=self:
		       self.setOrigin((self.cursor[0], self.cursor[2]))))
	self.Map.bind('<Configure>', self.DoResize)

	# Register automatic updating
	viewer.updateList.append(self)

    def adjustSector(self, ratio):
	"""Scale the size of the map by RATIO"""
	self.gridsize = map(operator.mul, [ratio]*len(self.gridsize),
			    self.gridsize)
	# Redisplay sectors without forcing a redraw.
	xview = self.Map.xview()
	yview = self.Map.yview()
	winWidth = (xview[1] - xview[0])/2.0
	winHeight = (yview[1] - yview[0])/2.0
	ws = empDb.megaDB['version']['worldsize']
	self.Map['scrollregion'] = (-self.gridsize[0], -self.gridsize[1],
				    ws[0]*self.gridsize[0],
				    ws[1]*self.gridsize[1])
	self.Map.scale(ALL, 0, 0, ratio, ratio)
	self.Map.delete('outline')
	self.Map.create_rectangle(-self.gridsize[0], -self.gridsize[1],
				  ws[0]*self.gridsize[0]-1,
				  ws[1]*self.gridsize[1]-1,
				  tags='outline')
	self.Map.xview('moveto', xview[0]+winWidth*(1-1.0/ratio))
	self.Map.yview('moveto', yview[0]+winHeight*(1-1.0/ratio))

    def DoResize(self, event):
	new = (event.width, event.height)
	if not self.dimen:
	    # The window is being drawn for the first time
	    self.redraw(1)
	    self.center()
	    self.dimen = new
	    return
	xview = self.Map.xview()
	yview = self.Map.yview()
	win = map(string.atof, string.split(self.Map['scrollregion']))
	scrWidth, scrHeight = win[2]-win[0], win[3]-win[1]
	self.Map.xview('moveto', xview[0]
		       +(self.dimen[0]-new[0])/2.0/scrWidth)
	self.Map.yview('moveto', yview[0]
		       +(self.dimen[1]-new[1])/2.0/scrHeight)
	self.dimen = new

    def getCoord(self, coord):
	"""Convert empire coords to screen coordinates"""
	return ((coord[0]+self.origin[0])%self.maxCoord[0]*self.gridsize[0],
		(coord[1]+self.origin[1])%self.maxCoord[1]*self.gridsize[1])

    def see(self, coord):
	"""If COORD isn't currently viewable, scroll window so that it is"""
	win = map(string.atof, string.split(self.Map['scrollregion']))
	scrWidth, scrHeight = win[2]-win[0], win[3]-win[1]
	xview = self.Map.xview()
	yview = self.Map.yview()
	x, y = self.getCoord(coord)
	xpos, ypos = (x-win[0])/scrWidth, (y-win[1])/scrHeight
	if (xpos < xview[0] or xpos > xview[1]
	    or ypos < yview[0] or ypos > yview[1]):
	    winWidth = (xview[1] - xview[0])/2.0
	    winHeight = (yview[1] - yview[0])/2.0
	    self.Map.xview('moveto', xpos - winWidth)
	    self.Map.yview('moveto', ypos - winHeight)

    def center(self):
	"""Center window"""
	xview = self.Map.xview()
	yview = self.Map.yview()
	winWidth = (xview[1] - xview[0])/2
	winHeight = (yview[1] - yview[0])/2
	self.Map.xview('moveto', 0.5 - winWidth)
	self.Map.yview('moveto', 0.5 - winHeight)

    def setOrigin(self, loc):
	"""Change origin to LOC - the origin is the center of the window"""
	# Center window
	self.center()

	# Set origin for future redraws
	loc = (loc[0]-self.maxCoord[0]/2,
	       loc[1]-self.maxCoord[1]/2)
	if self.origin == (-loc[0], -loc[1]):
	    # Nothing has changed
	    return
	x, y = self.getCoord(loc)
	self.origin = (-loc[0], -loc[1])

	# Redisplay sectors without forcing a redraw.
	win = map(string.atof, string.split(self.Map['scrollregion']))
	scrWidth, scrHeight = win[2], win[3]
	self.Map.addtag_enclosed('move_a',
				 -99999, -99999,
				 x+1, 99999)
	self.Map.addtag_enclosed('move_b',
				 x, -99999,
				 99999, 99999)
	# It is not possible to capture all the x coordinates by drawing a
	# vertical line.  (The x coordinate of every other row is
	# staggered.)  This loop catches these odd starting coordinates.
	for i in (range(y, scrHeight+1, self.gridsize[1])+
		  range(y, -1, -self.gridsize[1])):
	    self.Map.addtag_enclosed('move_b',
				     x-self.gridsize[0],
				     i-self.gridsize[1]*2/3,
				     x+self.gridsize[0]+1,
				     i+self.gridsize[1]*2/3+1)
	self.Map.dtag('move_b', 'move_a')
	self.Map.move('move_a', scrWidth-x, 0)
	self.Map.move('move_b', -x, 0)
	self.Map.dtag('move_a')
	self.Map.dtag('move_b')
	self.Map.addtag_enclosed('move_a',
				 -99999, -99999,
				 99999, y-self.gridsize[1]/3+1)
	self.Map.addtag_enclosed('move_b',
				 -99999, y-self.gridsize[1]*2/3,
				 99999, 99999)
	self.Map.move('move_a', 0, scrHeight-y)
	self.Map.move('move_b', 0, -y)
	self.Map.dtag('move_a')
	self.Map.dtag('move_b')

    def redraw(self, total=0):
	try:
	    ws = empDb.megaDB['updated']['version']['worldsize']
	except KeyError:
	    pass
	else:
	    self.Map['scrollregion'] = (-self.gridsize[0], -self.gridsize[1],
					ws[0]*self.gridsize[0],
					ws[1]*self.gridsize[1])
	    self.Map.delete('outline')
	    self.Map.create_rectangle(-self.gridsize[0], -self.gridsize[1],
				      ws[0]*self.gridsize[0]-1,
				      ws[1]*self.gridsize[1]-1,
				      tags='outline')
	    self.origin = (ws[0]/2, ws[1]/2)
	    self.maxCoord = ws
	    self.center()
	    total = 1

	for k in (('LAND UNITS', 'LAND', 1, -1, 'brown'),
		  ('SHIPS', 'SHIP', -1, -1, 'blue'),
		  ('PLANES', 'PLANE', -1, 1, 'pink'),
		  ('NUKES', 'NUKE', 1, 1, 'thistle'),
		  ):
	    if not total and not empDb.megaDB['updated'][k[0]]:
		continue
	    db = empDb.megaDB.get(k[0], {})
	    self.Map.delete(k[1])
	    for i in db.getSec(('x', 'y')).values():
		for j in i.values():
## 		    if j.get('owner') == empDb.CN_OWNED:
		    if j.get('owner') != empDb.CN_UNOWNED:
			break
		else:
		    continue
		x, y = self.getCoord((string.atoi(j['x']),
				      string.atoi(j['y'])))
		x1, y1, x2, y2 = (
		    k[2]*(self.gridsize[0]/3+2), k[3]*(self.gridsize[1]*2/6+2),
		    k[2]*(self.gridsize[0]/3-2), k[3]*(self.gridsize[1]*2/6-2))
		self.Map.create_rectangle(x+x1, y+y1,
					  x+x2, y+y2,
					  tags=k[1],
					  fill=k[4], outline='')

	if total:
	    db = empDb.megaDB['SECTOR']
	    self.Map.delete('SECTOR')
	else:
	    db = empDb.megaDB['updated']['SECTOR']
	for i, j in db.items():
	    t = j.get('des')
	    if not t:
		continue
	    own = j.get('owner')
	    if own == empDb.CN_OWNED:
		hcolor = 'green'
	    elif own == empDb.CN_ENEMY or own > 0:
		hcolor = 'red'
	    elif t == '.' or t == '\\':
		hcolor = None
	    elif own == empDb.CN_UNOWNED:
		hcolor = 'grey'
	    else:
		hcolor = 'black'

	    x, y = self.getCoord((string.atoi(j['x']), string.atoi(j['y'])))
	    if not total:
		# Delete old sector
		group = self.Map.find_enclosed(x-self.gridsize[0],
					       y-self.gridsize[1],
					       x+self.gridsize[0],
					       y+self.gridsize[1])
		for i in group:
		    if 'SECTOR' in self.Map.gettags(i):
			self.Map.delete(i)
	    # draw hex around sector
	    if hcolor:
		full = self.gridsize[0]
		half = self.gridsize[1]*2/3
		quar = self.gridsize[1]/3
## 		self.Map.create_polygon((x-full)+1, (y-quar)+1,
## 					x, (y-half)+1,
## 					x+1, (y-half)+1,
## 					x+full, (y-quar)+1,
## 					x+full, y+quar,
## 					x+1, y+half,
## 					x, y+half,
## 					(x-full)+1, y+quar,
## 					tags='SECTOR',
## 					fill='', outline=hcolor)
		self.Map.create_line((x-full)+1, (y-quar)+1,
				     x, (y-half)+1,
				     x+1, (y-half)+1,
				     x+full, (y-quar)+1,
				     x+full, y+quar,
				     x+1, y+half,
				     x, y+half,
				     (x-full)+1, y+quar,
				     (x-full)+1, (y-quar)+1,
				     tags='SECTOR',
				     fill=hcolor)
	    # Draw text description
	    self.Map.create_text(x, y, tags='SECTOR', text=t)
	# 0,0 axis decoration
	x, y = self.getCoord((0,0))
	self.Map.delete('origin')
	color = (self.Map.option_get('origin', "Mark") or "black")
	self.Map.create_line(
	    x-self.gridsize[0]/2, y-self.gridsize[1]*2/3+1,
	    x-self.gridsize[0]/2, y+self.gridsize[1]/2,
	    x+self.gridsize[0], y+self.gridsize[1]/2,
	    tags='origin', fill=color)

    def markSectors(self, coords, name='mark'):
	self.Map.delete(name)
	color = (self.Map.option_get(name, "Mark") or "black")
	for coord in coords:
	    coord = self.getCoord(tuple(coord))
	    self.Map.create_oval(coord[0]-self.gridsize[0]*3/4+1,
				 coord[1]-self.gridsize[1]/2+1,
				 coord[0]+self.gridsize[0]*3/4,
				 coord[1]+self.gridsize[1]/2,
				 tags=name,
				 fill='', outline=color)

    def DoCoord(self, event):
	# convert screen coords to empire coordinates
	x, y = self.ploc = [
	    int((self.Map.canvasx(event.x)+self.gridsize[0]/2)
		/self.gridsize[0]),
	    int((self.Map.canvasy(event.y)+self.gridsize[1]/2)
		/self.gridsize[1])]

	if self.start:
	    # Set box to contain the coordinates in proper order
	    if x > self.start[0]:
		box = [self.start[0], x]
	    else:
		box = [x, self.start[0]]
	    if y > self.start[1]:
		box = box + [self.start[1], y]
	    else:
		box = box + [y, self.start[1]]
	else:
	    box = [x, x, y, y]

	# Check for bogus x ranges
	if box[0] == box[1] and box[2] == box[3]:
	    x = box[0] = box[1] = (
		int(self.Map.canvasx(event.x)/self.gridsize[0])
		+(~y&1))/2*2+(y&1)

	rbox = list(box)

	# Compensate for the origin
	for i, j in [[0, 0], [1, 0], [2, 1], [3, 1]]:
	    box[i] = ((box[i]-self.origin[j]+self.maxCoord[j]/2)
		      %self.maxCoord[j]-self.maxCoord[j]/2)

	# Set st to contain a string representation of box
	if box[2] == box[3]:
	    st = ",%d" % (box[2],)
	else:
	    st = ",%d:%d" % tuple(box[2:])
	if box[0] == box[1]:
	    st = "%d" % (box[0],) + st
	else:
	    st = "%d:%d" % tuple(box[:2]) + st

	if box == self.cursor:
	    # Nothing has changed
	    return
	self.cursor = box

	# Display new entry in coord box
	viewer.coord.set(st)

	# redraw box if necessary
	self.Map.delete('rubberbox')
	if self.start:
	    self.CoordBox = self.Map.create_rectangle(
		rbox[0]*self.gridsize[0] - self.gridsize[0]/2,
		rbox[2]*self.gridsize[1] - self.gridsize[1]/2,
		rbox[1]*self.gridsize[0] + self.gridsize[0]/2,
		rbox[3]*self.gridsize[1] + self.gridsize[1]/2,
		tags='rubberbox')

    def DoCoordClear(self, event):
	if self.start:
	    # When a box is "stretched" beyond the window boundaries a
	    # dummy leave event is generated - just ignore it.
	    return
	self.cursor = self.start = ()
	viewer.coord.set("")
	self.Map.delete('rubberbox')

    def DoCoordBox(self, event):
	if self.start:
	    # If a button press occurs while "stretching" a box around
	    # sectors, cancel all buttons.  (This allows the user to
	    # cancel a bad range.)
	    self.start = self.cursor = ()
	else:
	    self.cursor = ()
	    self.start = self.ploc
	self.DoCoord(event)

    def DoCoordEndBox(self, event):
	self.start = self.cursor = ()
	self.DoCoord(event)

    def DoCensor(self, event):
##	if not self.start:
##	    return
	viewer.cen.SetSect(self.cursor)
	self.DoCoordEndBox(event)

    def DoSelect(self, event):
	if not self.start:
	    return
	rng = viewer.coord.get()
	inpbox = self.Map.focus_displayof()
	cmd = inpbox.get()
	idx = inpbox.index(INSERT)
	end = inpbox.index(END)
	if idx > 1 and cmd[idx-1] != ' ':
	    rng = " " + rng
	if idx == end or cmd[idx] != ' ':
	    rng = rng + " "
	inpbox.insert(INSERT, rng)
	self.DoCoordEndBox(event)
