/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     difpic.c                                                       *
 *            Funktionen zum Bilden von Differenzen                          *
 * Autor:     Andreas Tille                                                  *
 * Datum:     27.02.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

#define  MINCOL  0xFF
#define  MAXCOL  0x00

static int strxcmp(const char *s1, const char *s2)
/* obtain the matching first characters of two strings
 * --- Parameter: ---
 * const char *s1, *s2: strings to compare
 * --- R"uckgabe: ---
 * int   strxcmp()    : number of matching characters in the beginning of the
 *                      two strings
 */
{
   register char *p1 = (char *)s1 - 1, *p2 = (char *)s2 - 1;

   while ( *(++p1) && *(++p2) ) 
      if ( *p1 != *p2 ) break;
   return p1 - s1;
}

static int strbxcmp(const char *s1, const char *s2)
/* obtain the matching last characters of two strings
 * --- Parameter: ---
 * const char *s1, *s2: strings to compare
 * --- R"uckgabe: ---
 * int   strbxcmp()   : number of matching characters in the end of the
 *                      two strings
 */
{
   register char *p1 = (char *)s1 + strlen(s1), 
                 *p2 = (char *)s2 + strlen(s2),
		 *ss;
   
   ss = p1 - 1;
   while ( *(--p1) && *(--p2) ) 
      if ( *p1 != *p2 ) break;
   return ss - p1;
}

static char *DiffBildName(char *bild1, char *bild2)
/* Ermittelt aus zwei Bilddateinamen den Namen fr das Differenzbild
 * Gemeinsame Dateinamensbestandteile am Anfang und am Ende des Dateinamens
 * (ohne Erweiterung) werden bernommen, der Teil, in dem sich die Namen
 * unterscheiden wird in der Form _[bild1]-[bild2]_ dargestellt
 * --- Parameter: ---
 * char *bild1:          Dateiname des ersten Bildes
 * char *bild2:          Dateiname des zweiten Bildes
 * --- R"uckgabe: ---
 * char *DiffBildName(): Name des Differenzbildes (Speicher wird hier allociert 
 *                       ... und Platz f"ur Erweiterung gelassen)
 */
{
   char         buf[200], *pb;
   register int n, m, i;
	  
   strncpy(buf, bild1, n = strxcmp(bild1,bild2));
   *(pb = buf + n) = '_';
   if        ( n == strlen(bild1) ) {
      *(++pb)                   = '-';
      ++pb;
      if ( n != strlen(bild2) ) strcpy(pb, bild2 + n);
      *(pb += strlen(bild2+n))  = '_';
      ++pb;
   } else if ( n == strlen(bild2) ) {
      ++pb;
      if ( n != strlen(bild1) ) strcpy(pb, bild1 + n);
      *(pb += strlen(bild1+n))  = '-';
      *(++pb)                   = '_';
      ++pb;
   } else {
      m       = strbxcmp(bild1,bild2);
      if ( (i = strlen(bild1) - n - m) > 0 )  
         strncpy(pb, bild1 + n, i);
      else
         i = 1; /* This makes not much sense but avoids crashs for some unusual cases */
      *(pb += i) = '-';
      if ( (i = strlen(bild2) - n - m) > 0 ) 
         strncpy(pb, bild2 + n, i);
      else
         i = 1; /* This makes not much sense but avoids crashs for some unusual cases */
      *(pb += i) = '_';
      ++pb;
      if ( m ) strncpy(pb, bild1 + strlen(bild1) - m, m);
      pb += m;
   }
   *pb = 0;
   assert ( (pb = strdup(buf)) );
   return pb;
}


static int DifferenzBild(GList *piclist, OPTIONS *p)
/* Subtraktion der Grauwerte bild1 - bild2
 * --- Parameter: ---
 * PICTURE *bild1          : 1. Bild
 * PICTURE *bild2          : vom 1. Bild zu subtrahierendes Bild
 * int      flag           : Verhaltensflag
 * unsigned char eps       : Rauschgrenze
 * unsigned char offset    : Anheben der Helligkeit 
 * double   scale          : Faktor fr Grauwerte zur Kontrastverbesserung
 * --- Rueckgabe: ---
 * PICTURE *diff           : Differenzbild, NULL im Fehlerfall
 * int     Differenzbild() : 0 fr OK;
 */
{ 
   PICTURE                *bild1 = BILD(piclist), 
                          *bild2 = BILD(piclist->next);
   unsigned char           eps = p->eps, offset = p->offset;
   double                  scale = p->scale;
   register int            d;
   register unsigned char *a, *b, *u, *v, *fip;
   char                    was[256], *neuname;

   if ( bild1->W != bild2->W || 
        bild1->H != bild2->H ) {
      g_warning("%s and %s contain images of different size (%ix%i bzw. %ix%i).\n\
                 It makes no sense to build differnces from different sized images",
		bild1->file, bild2->file, bild1->W, bild1->H, bild2->W, bild2->H);
      return -1;
   }
   if ( !IsMonochrom(bild1) || !IsMonochrom(bild2) ) {
      g_warning("Differences are calculated only from monochrome images.");
      return -1;
   }
   
   sprintf(was, "%s - %s", bild1->file, bild2->file);
   CreateDescription(bild1, was);
   CopySpec(bild1->spec, ChunkNameTyp, TypDifferenzBild);

   neuname     = DiffBildName(bild1->file, bild2->file);
   free(bild1->file);
   bild1->file = neuname;
   bild1->spp  = 3;
   bild1->flag = DIFFERENZ;

   a = (bild1->DATA) + 1;
   b = (bild2->DATA) + 1;   
   for ( fip = a + 3 * (bild1->size); a < fip; 
         a += 3, b += 3) {
      if ( (d  = (int)*a - (int)*b) < 0 ) {
         u = a - 1;  /* bild1 < bild2 => speichere Differenz in Rot */
         v = a + 1;  /*               => setze blau 0               */
      } else {
         u = a + 1;
         v = a - 1;
      }
      if ( (d = abs(d)) < eps ) { /* Rauschen unterdrcken */
         *(a-1) = *a = *(a+1) = 0xFF;  /* Rest wei statt schwarz */
         continue;   
      }
      d  = d * scale + offset;     /* Kontrast und Helligkeit anheben */
      *u = (unsigned char)(d < 0x100 ? d : 0xFF);
      *a = *v = 0;
   }
   return 0;
}


int DifferenzBilder(PAUL *p)
/* Differences of a list of images
 * --- Parameter: ---
 * PAUL *p               : list of images, options
 *                       : used options:
 *                         eps    : noise border
 *                         offset : increase brightness of difference image  
 *                         scale  : increase contrast of difference image
 * --- return: ---
 * p->piclist            : new liste of images containing differences
 * int  DifferenzBilder(): 0 if OK
 */
{
   char    *typ;
   PICTURE *bild;
   GList   *pl;

   if ( NBILDER(p->piclist) < 2 ) return 0;

   for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
      if ( (typ = GetSpec(bild->spec, ChunkNameTyp)) &&
           !strcmp(TypDifferenzBild, typ) ) {
         g_warning("%s contains a formerly calculated diference image.\n\
                    Make sure that this is what you want", bild->file);
      }
   }

   for ( pl = p->piclist; pl->next; pl = pl->next ) 
      if ( DifferenzBild(pl, p->opt) ) return -1;
   FreeBild(g_list_last(p->piclist));
   g_list_remove_link(p->piclist, g_list_last(p->piclist));
   return 0;
}






