/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     histogrm.c                                                     *
 *            Funktionen zur Ermittlung von Histogrammen                     *
 * Autor:     Andreas Tille                                                  *
 * Datum:     18.05.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "paul.h"

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

const char *histfile = HISTFILE,
           *vprofile = VPROFILE;

void MakeHistogram(PICTURE *bild)
/* Erstellen eines Histogramms der Farbwerte eines Bildes
 * --- Parameter: ---
 * PICTURE *bild          : Bilddaten
 * --- Rueckgabe: ---
 * int     MakeHistogram(): 0 fr OK
 */
{
   register unsigned char *buf, *fip;
   register unsigned long *his;
   unsigned char          *anf, *end;

   if ( !bild->his )
      assert( (bild->his = calloc(bild->storepix * 0x100, sizeof(unsigned long))) != NULL );

   if ( IsMonochrom(bild) )
      for ( fip = (buf = bild->DATA + 1) + bild->storepix*bild->size; buf < fip; 
            buf += bild->storepix ) 
         ++*(bild->his + (*(int *)buf & 0xFF));
   else
      for ( end = (anf = bild->DATA) + 3, his = bild->his; anf < end; anf++, his += 0x100 )
         for ( fip = (buf = anf) + bild->spp*bild->size; buf < fip; buf += bild->spp ) 
            ++*(his + (*(int *)buf & 0xFF));
}

void MakeVerticalProfile(PICTURE *bild)
/* Erstellen eines vertikalen Profils der Grnwerte eines Bildes
 * --- Parameter: ---
 * PICTURE *bild                : Bilddaten
 * --- Rueckgabe: ---
 * int     MakeVerticalProfile(): 0 fr OK
 */
{
   register unsigned long *his;
   register int            storepix;
   register unsigned char *ap, *fap, *buf, *fip;
   unsigned long          *fis;

   storepix = bild->storepix;
   if ( bild->his ) free(bild->his);
   assert( (bild->his = calloc(bild->H, sizeof(unsigned long))) != NULL );

   for ( fip = (buf = bild->DATA + (storepix==3?1:0)) + storepix*bild->size, 
	    his = bild->his; buf < fip; buf = fap, his++ )
      for ( fap = (ap = buf) + storepix*bild->W; ap < fap; ap += storepix )
         *his += (unsigned long)*ap;
   storepix = bild->W >> 1;
   for ( fis = (his = bild->his) + bild->H; his < fis; his++ )
      *his = (*his + storepix) / bild->W;
}

void MakeHistograms(GList *piclist)
/* Erstellen der Histogramme der Farbwerte aller Bilder
 * --- Parameter: ---
 * GList *piclist          : Bilddaten
 * --- Rueckgabe: ---
 * int    MakeHistograms() : 0 fr OK
 */
{
   FILE     *fp;
   char     RGB[] = "RGB";
   int      j, l, len, rgb = 0;
   PICTURE *bild;
   GList   *pl;
      
   if ( !NBILDER(piclist) ) return;

   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) 
      MakeHistogram(bild);

   Backup(histfile);
   if ( (fp = fopen(histfile, "w")) == NULL ) {
      fprintf(stderr, "Histogrammfile %s konnte nicht geffnet werden.\n", histfile);
      for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
         FREE(bild->his);
      return;
   }
   fprintf(fp, "%3c", ' ');
   len = 1;
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      if ( (l = strlen(bild->file)) > len ) len = l;
      if ( !IsMonochrom(bild) )             rgb = 1;
   }
   len += 1 + rgb*4;  /* 4 Zeichen fr _[c], c aus [RGB] */
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) 
      if ( IsMonochrom(bild) )
         fprintf(fp, "%*s", len, bild->file);
      else
         for ( l = 0; l < 3; l++ )
            fprintf(fp, "%*s_[%c]", len-4, bild->file, RGB[l]);

   for ( j = 0; j < 0x100; j++ ) {
      fprintf(fp, "\n%3i", j);
      for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) 
         if ( IsMonochrom(bild) )
            fprintf(fp, "%*li", len, bild->his[j]);
         else
            for ( l = 0; l < 3; l++ )
               fprintf(fp, "%*li", len, bild->his[j+l*0x100]);
   }
   fclose(fp);
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      FREE(bild->his);
      MakeVerticalProfile(bild);
   }
   
   Backup(vprofile);
   if ( (fp = fopen(vprofile, "w")) == NULL ) {
      fprintf(stderr, "File fr vertikales Profil %s konnte nicht geffnet werden.\n", vprofile);
      for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
         FREE(bild->his);
      return;
   }
   fprintf(fp, "%3c", ' ');
   len = 1;
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
      if ( (l = strlen(bild->file)) > len ) len = l;
   len += 1; 
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
      fprintf(fp, "%*s", len, bild->file);

   for ( j = 0; /* Abbruch per break */ ; j++ ) {
      for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
	 if ( j < bild->H ) break;
      if ( !pl ) break;
      fprintf(fp, "\n%3i", j+1);
      for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
         if ( j < bild->H ) 
            fprintf(fp, "%*lu", len, bild->his[j]);
	 else
            fprintf(fp, "%*c", len, ' ');
   }
   fclose(fp);
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) )
      FREE(bild->his);
}

