/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     matchpic.c                                                     *
 *            Funktionen gegeneinander Verschieben von Bildern               *
 * Autor:     Andreas Tille                                                  *
 * Datum:     08.06.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>

#include "paul.h" 

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

const char SchiebKlein[] = "Schiebebild zu klein, um %s aufzunehmen";

int ShiftValue(int mod)
/* Berechnet Weite der Verschiebung
 * --- Parameter: ---
 * int mod          : Tastaturmodus
 * --- R"uckgabe: ---
 * int ShiftValue() : Verschiebungsfaktor
 */
{
   int i = 0;
   
   if ( mod & GDK_SHIFT_MASK   ) i++;
   if ( mod & GDK_LOCK_MASK    ) i++;
   if ( mod & GDK_CONTROL_MASK ) i += 2;
   return i;
}

static void ClearTarget(PICTURE *bild, int red, int green, int blue)
/* Lscht Rot/Grn/Blau-Anteil eines (Schiebe)Bildes
 * --- Parameter: ---
 * PICTURE *bild       : (Schiebe)Bild
 * int      red        : rot Lschen ?
 * int      green      : grn Lschen ?
 * int      blau       : blau Lschen ?
 * --- R"uckgabe: ---
 */
{
   register unsigned char *ap, *fip;
   register int           storepix = bild->storepix;

   if ( red && green && blue ) {
      memset(bild->DATA, 0, bild->storepix*bild->size);
      return;
   }
   if ( red )
      for ( fip = (ap = bild->DATA)   + bild->storepix*bild->size; 
            ap < fip; ap += storepix )
         *ap = 0;
   if ( green )
      for ( fip = (ap = bild->DATA+1)   + bild->storepix*bild->size; 
            ap < fip; ap += storepix )
         *ap = 0;
   if ( blue )
      for ( fip = (ap = bild->DATA+2) + bild->storepix*bild->size; 
            ap < fip; ap += storepix )
         *ap = 0;
}

static int GetMoveDimensions(PICTURE *target, PICTURE *src, int *width, int *height,
                             unsigned char **tptr, unsigned char **sptr, int storepix)
/* Obtain pointer to start of common area in each image and the dimensions of this common area
 * --- Parameter: ---
 * PICTURE        *target   : moving image
 * PICTURE        *src      : image to insert
 * int            *width    : width of moving image
 * int            *height   : height of moving image
 * unsigned char **tptr     : pointer to start of pixels to write in the moving image
 * unsigned char **sptr     : pointer to start of pixels to write from the source image
 * int             storepix : number of bytes per pixel
 * --- Return: ---
 * int GetMoveDimensions()  : 0 if OK
 * int            *width    : width of moving image
 * int            *height   : height of moving image
 * unsigned char **tptr     : pointer to start of pixels to write in the moving image
 * unsigned char **sptr     : pointer to start of pixels to write from the source image
 */
{
   register int xt0, yt0, /* linke obere Ecke */
                toffset;

   if ( !IS_PICTURE(target) || !IS_PICTURE(src) ) {
      g_warning("Illegal image pointer");
      return -1;
   }
   if ( (xt0 = (target->W - src->W) >> 1) < 0 ||
        (yt0 = (target->H - src->H) >> 1) < 0 ) {
      g_warning(SchiebKlein, src->file);
      return -1;
   }

   xt0 += target->roff_x;
   yt0 += target->roff_y;
   if ( xt0 > 0 ) *width  = MIN(target->W - xt0, src->W);
   else           *width  = MIN(src->W    + xt0, target->W);
   if ( yt0 > 0 ) *height = MIN(target->H - yt0, src->H);
   else           *height = MIN(src->H    + yt0, target->H);

   if ( *width <= 0 || *height <= 0 ) {
      gdk_beep();
      return 1;
   }
      
   if ( (toffset = yt0*target->W + (xt0<0?0:xt0)) <= 0 ) 
      *tptr = target->DATA + storepix*(xt0<0?0:xt0);
   else *tptr = target->DATA + storepix*toffset;

   *sptr  = src->DATA;
   if ( yt0 < 0 ) *sptr -= storepix * src->W * yt0;
   if ( xt0 < 0 ) *sptr -= storepix * xt0;

   return 0;
}

static int InsertPicture(PICTURE *target, PICTURE *src, int colshift)
/* Schreibt Bilddaten in Schiebebild
 * --- Parameter: ---
 * PICTURE *target     : Schiebebild
 * PICTURE *src        : einzufgendes Bild
 * int      colshift   : == 0 => festes Bild, benutze grn
 *                       == 1 => blau
 *                       ==-1 => rot
 * --- R"uckgabe: ---
 * PICTURE *target     : Schiebebild mit eingefgtem Bild
 * int InsertPicture() : 0 fr OK
 */
{
   register int           storepix;
   register unsigned char *tap, *sap, *fap;
   unsigned char          *ts, *ss, *fsp;
   int                    width, height, i;
  
   if ( (storepix = target->storepix) < 3 ||
        storepix !=    src->storepix ) {
      g_warning("Ungltige interne Speicherung");
      return -1;
   }

   if ( abs(colshift) > 1 ) {
      g_warning("Ungltige Farbverschiebung (%i)", colshift);
      return -1;
   }

   if ( (i = GetMoveDimensions(target, src, &width, &height, &ts, &ss, storepix)) )
      return i;
   
   ts += colshift + 1;  /* +1 == grn !! */
   ss++;                /* +1 == grn == eigentliche Information */

   ClearTarget(target, colshift>0?0:1, !colshift, colshift>0?1:0);

   for ( fsp = ss + storepix * src->W * height;
         ss < fsp; 
         ts += storepix * target->W,
	 ss += storepix *    src->W ) {
      for ( tap = ts, fap = (sap = ss) + storepix*width; sap < fap; 
            sap += storepix, tap += storepix ) {
         *tap = *sap;	 
      }
   }
   return 0;
}

static int DifferenzMovePicture(PICTURE *target, PICTURE *a, PICTURE *op)
/* write image data into difference move image
 * --- Parameter: ---
 * PICTURE *target            : difference image (op-a) of a moved against op
 * PICTURE *a                 : picture to move against reference picture (will be subtracted 
 *                              from referenz picture)
 * PICTURE *op                : reference picture (operation picture or the first of the picture list)
 * --- R"uckgabe: ---
 * PICTURE *target            : difference move image (op-a)
 * int DifferenzMovePicture() : 0 if OK
 */
{
   register int           d, storepix;
   register unsigned char *tap, *sap, *u, *v, *fap;
   unsigned char          *ts, *ss, *fsp;
   int                    width, height, sxoff, syoff;
   unsigned char          offset=10, eps=0;
   int                    scale=2.0;
  
   if ( (storepix = target->storepix) < 3 ) {
      g_warning("Ungltige interne Speicherung");
      return -1;
   }
   if ( ((target->W - op->W) >> 1) < 0 ||
        ((target->H - op->H) >> 1) < 0 ) {
      g_warning(SchiebKlein, op->file);
      return -1;
   }

   if ( (d = GetMoveDimensions(target, a, &width, &height, &ts, &ss, storepix)) )
      return d;

   ss++;  /* +1 == green == real information */
   ts++;  /* +1 == green !! */

   ClearTarget(target, 1, 1, 1);
   sxoff = target->roff_x; syoff = target->roff_y;
   target->roff_x = 0; target->roff_y = 0;
   InsertPicture(target, op, 0);   /* Referenzbild einschreiben */
   target->roff_x = sxoff; target->roff_y = syoff;
   
   for ( fsp = ss + storepix * a->W * height;
         ss < fsp; 
         ts += storepix * target->W,
	 ss += storepix *      a->W ) {
      for ( tap = ts, fap = (sap = ss) + storepix*width; sap < fap; 
            sap += storepix, tap += storepix ) {
         if ( (d  = (int)*tap - (int)*sap) < 0 ) {
            u = tap - 1;  /* bild1 < bild2 => speichere Differenz in Rot */
            v = tap + 1;  /*               => setze blau 0               */
         } else {
            u = tap + 1;
            v = tap - 1;
         }
         if ( (d = abs(d)) < eps ) { /* suppress noise */
            *(tap-1) = *tap = *(tap+1) = 0xFF;  /* remainder white in favour of black */
            continue;   
         }
         d    = d * scale + offset;     /* Kontrast und Helligkeit anheben */
         *u   = (unsigned char)(d < 0x100 ? d : 0xFF);
         *tap = *v = 0;
      }
   }
   return 0;
}

void SetMoveTitle(PAUL *p)
/* Set window title of pictures to move
 */
{
   char     buf[256];
   PICTURE *asrc, *target;
   
   target = BILD(p->activ);
   /* If there is an operation image take the image with the same index as p->activ  *
    * from the sources (p->src).  Else take the next one because the first source    *
    * works as operating image which increases all indexes in the sources            */
   asrc   = BILD(g_list_nth(p->src, GetListPosition(p->piclist, p->activ) + (p->op?0:1)));
   if ( !target || !asrc ) return;
   sprintf(buf, "%s [auto: %i,%i] + (%i,%i)", target->file, asrc->roff_x, asrc->roff_y, 
                target->roff_x, target->roff_y);
   gdk_window_set_title(p->show->window, buf);
}

int MovePic(PAUL *p, int v, int h, int mod)
/* Move images relative to another image.
 * The first image of p->src is used as reference image if there isnt any operation picture.
 * There are three types of moving which can be determined in p->opt->mov.
 * Type color-moving: reference image is stored in green-byte, the moving image is colored blue
 * Type difference:   display moving image - reference image
 * Type flash:        move image and have a short  flash of reference image 
 * --- Parameter: ---
 * PAUL *p         : list of images, options
 *                   used:
 *                   activ : image to move
 *                   src   : BILD(src) = reference image
 *                   used options:
 *                   mov   : type of moving
 * int      v      : index of vertical moving (-1, 0, 1)
 * int      h      : index of horizontale moving (-1, 0, 1)
 * int      mod    : amplitude of moving (determined by the state of <CTRL>, <Shift> and <CapsLock>
 * --- R"uckgabe: ---
 * BILD(p->activ)  : changed image data, which represents the moved image
 * int MovePic()   : 0 if OK, -1 otherwise
 */
{
   register PICTURE *target, *mov, *op;

   if ( !p->src ) {
      g_warning("Source images not available.");
      return 0;
   }

   target = BILD(p->activ);
   if ( p->op ) op = p->op;
   else         op = BILD(p->src);
   if ( BILD(p->activ) == op ) {
      if      ( (p->activ)->next ) p->activ = (p->activ)->next;
      else if ( (p->activ)->prev ) p->activ = (p->activ)->prev;
      else {
         g_warning("Confusion between active and operating picture.");
	 return 0;
      }
   }
   
   /* If there is an operation image take the image with the same index as p->activ  *
    * from the sources (p->src).  Else take the next one because the first source    *
    * works as operating image which increases all indexes in the sources            */
   mov = BILD(g_list_nth(p->src, GetListPosition(p->piclist, p->activ) + (p->op?0:1)));
   
   if ( !v && !h ) {
      if ( IsMatchFlash(p->opt->mov) ) {
         if ( ApplyPicture(p->show->window, op) )     return -1;
         if ( ApplyPicture(p->show->window, target) ) return -1;
      }
      return 0;
   }

   if ( mod ) {
      v <<= mod;
      h <<= mod;      
   }
   target->roff_x += v;
   target->roff_y += h;

   if ( ( IsMatchDif(p->opt->mov)  && DifferenzMovePicture(target, mov, op) )          ||
        ( !IsMatchDif(p->opt->mov) && InsertPicture(target, mov, (IsMatchFlash(p->opt->mov)?0:1)) ) ) { 
      target->roff_x -= v;
      target->roff_y -= h;
      return -1;            /* Das war nix */
   }
   
   SHOW_SLOW = 0; /* avoid displaying target->file as window title in favour of the title *
                   * created in SetMoveTitle                                              */
   SetMoveTitle(p);
   gdk_imlib_changed_image(target->im);
   if ( !IsMatchFlash(p->opt->mov) ) return 0;  /* Display image in calling function ... *
                                                * gtk_list_select_item() causes display */
   if ( ApplyPicture(p->show->window, target) ) return -1;
   if ( ApplyPicture(p->show->window, op)   ) return -1;
   return 0;
}


int CreateMatchPictures(PAUL *p)
/* Create images for further move-processing
 * If successful the original image list is stored in p->src and the created list
 * is in p->piclist. The list p->piclist depends from the type of moving visualisation.
 * The first image of p->src is used as reference image.
 * Type color-moving: store reference image in green-byte
 * Type difference:   use reference image as image to subtract from each image
 * Type flash:        use reference image as flashing image between each move
 * --- Parameter: ---
 * PAUL *p                    : source list of images, options
 *                              used options:
 *                              mod : type of moving visualisation
 * --- return: ---
 * p->piclist                 : new list of images prepared for moving
 * p->src                     : the list of input images
 * int   CreateMatchPictures(): 0 if OK, else -1
 */
{ 
   PICTURE          *fix;
   register PICTURE *bild, *ap ;
   GList            *pl, *piclist, *newlist = NULL;
   char              buf[256], *typ = "";

   piclist = p->piclist;

   if ( NBILDER(piclist) < 2 ) return 0;

   if ( p->op ) fix = p->op;
   else {
      fix     = BILD(piclist);
      piclist = piclist->next;
   }
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      ap           = InitSpec();
      ap->W        = MAX(fix->W, bild->W);
      ap->H        = MAX(fix->H, bild->H);
      ap->size     = ap->W * ap->H;
      ap->storepix = 3;
      assert( (ap->DATA = calloc(ap->size, ap->storepix)) );
      ap->roff_x   = 0;
      ap->roff_y   = 0;

      switch ( p->opt->mov ) {
      case MATCHCOL  : 
         InsertPicture(ap, fix, 0);
         InsertPicture(ap, bild, 1);
         typ = "<~~>";
         break;
   
      case MATCHFLASH:
         InsertPicture(ap, bild, 0);
         typ = "<__>";
	 break;

      case MATCHDIF  :  
         DifferenzMovePicture(ap, bild, fix);
         typ = "--";
	 break;

      default:
         g_warning("Unknown method to move images.");
         free(ap->DATA);
         return -1;
      }
      sprintf(buf, "%s%s%s", fix->file, typ, bild->file);
      assert ( (ap->file = strdup(buf)) );
      CopySpec(ap->spec, ChunkReferenceFile, fix->file);
      ap->flag |= MATCHPICTURES;
      newlist = g_list_append(newlist, ap);
   }

   p->opt->f |= MATCHPICTURES;
   p->src     = p->piclist;
   p->piclist = newlist;
   return 0;
}

static unsigned long BrightMatchPictures(PICTURE *ref, PICTURE *mov, int rx, int ry, unsigned long *match)
/* Schreibt Bilddaten in Schiebebild
 * --- Parameter: ---
 * PICTURE      *ref                   : Referenzbild
 * PICTURE      *mov                   : zu schiebendes Bild
 * int           rx                    : Rand in x-Richtung
 * int           ry                    : Rand in y-Richtung
 * int          *match                 : Anzahl der exakten bereinstimmungen
 * --- R"uckgabe: ---
 * unsigned long BrightMatchPictures() : Helligkeit der Differenz beider Bilder
 */
{
   register int            d, storepix = ref->storepix, m = 0;
   register unsigned long  b = 0;
   register unsigned char *rap, *map, *mfip;
   unsigned char          *rs, *ms, *fsp;
   int                     width, height;
  
   if ( (d = GetMoveDimensions(ref, mov, &width, &height, &rs, &ms, storepix)) )
      return 0;
   if ( (width  -= (rx<<1)) < 2 || (height -= (ry<<1)) < 2 ) {
      g_warning(SchiebKlein, mov->file);
      return 0;
   }

   rs += storepix*(rx + ry*ref->W);
   ms += storepix*(rx + ry*mov->W);

   for ( fsp = ms + storepix * mov->W * height;
         ms < fsp; 
         rs += storepix * ref->W,
	 ms += storepix * mov->W ) {
      for ( rap = rs, mfip = (map = ms) + storepix*width; map < mfip; 
            rap += 25, map += 25 ) { /* ALLE Byte auswerten, nicht nur eine Farbe!!, aber NICHT alle Pixel */
         if ( !(d  = (int)*rap - (int)*map) ) m++;
	 else if ( d < 0 ) d = -d;
         b += d;
      }
   }
   *match = (unsigned long)m;
   return b;
}

static int CheckMin(unsigned long *buf, int bufstep, int abs, int min)
/* Check for valid minimum
 * --- Parameter: ---
 * unsigned long *buf       : vector of brightnesses
 * int            bufstep   : step between the each brightness
 * int            abs       : min has to be between <-abs,abs>
 * int            min       : index of maximum brightness
 * --- R"uckgabe: ---
 * int            CheckMin(): 0 if OK
 */
{
   register int            i;
   register unsigned long *ap;
   register long           delta;
   
   if ( !abs || min == -abs || min == abs ) return -1;
   delta = (*(buf + (min+abs)*bufstep)) >> 7;  /* 0.8% vom Minimum */
   for ( i = 1 - abs, ap = buf + bufstep; i < abs; i++, ap += bufstep )
      if ( (long)*(ap-bufstep) - (long)*ap < delta ) break;
   if ( (i-1) == min ) {
      for ( ; i < abs; i++, ap += bufstep )
         if ( (long)*ap - (long)*(ap-bufstep) < delta ) break;
      if ( i == abs ) return 0;  /* soweit fr monotone Differenzhelligkeiten */
   }
   return -1;
}

static int CheckMax(unsigned long *buf, int bufstep, int abs, int max)
/* test for valid maximum
 * --- Parameter: ---
 * unsigned long *buf       : buffer with number of equal pixels
 * int            bufstep   : step between the numbers of equal pixels to test
 * int            abs       : max has to be between <-abs,abs>
 * int            max       : index of maximam number of equal pixels 
 * --- R"uckgabe: ---
 * int            CheckMax(): 0 if OK
 */
{
   register int            i;
   register unsigned long *ap;
   register long           delta;

   if ( !abs || max == -abs || max == abs ) return -1;
   delta = (*(buf + (max+abs)*bufstep)) >> 7;  /* 0.8% vom Maximum */ 
   for ( i = 1 - abs, ap = buf + bufstep; i < abs; i++, ap += bufstep )
      if ( (long)*ap - (long)*(ap-bufstep) < delta ) break;
   if ( (i-1) == max ) {
      for ( ; i < abs; i++, ap += bufstep )
         if ( (long)*(ap-bufstep) - (long)*ap < delta ) break;
      if ( i == abs ) return 0;  /* soweit fr monotone bereinstimmungen */
   }
   return -1;
}

static int AutoMatchOnePicture(PICTURE *ref, PICTURE *mov, int x, int y)
/* Tryes to move images automatically by minimizing the difference between the
 * first (reference and each other image
 * Todo: May be it is useful to calculate the number of equal pixelvalues
 *       to have an additional criterium if the difference method fails
 *
 * Eventuell sollte "exakte" bereinstimmung durch
 * " < eps"-bereinstimmung ersetzt werden, vielleicht sogar derart, da exakte bereinstimmung
 * hher bewertet wird, also:
 * if ( d == 0 ) ++m; if ( d < 2 ) ++m; if ( d < 3 ) ++m; ... if ( d < eps ) ++m;
 * Aber ob das was hilft??  Auerdem sind Videobilder zur Zeit nicht aktuell.
 * Das Problem bei Videobildern ist, da die Folge der Differenzen nicht monoton ist.
 * --- Parameter: ---
 * PICTURE *ref                   : referenze image (first one in list)
 * PICTURE *mov                   : image to move
 * int      x                     : maximum movement in x-direction
 * int      y                     : maximum movement in y-direction
 * --- Return: ---
 * int      AutoMatchOnePicture() : 0 fr OK
 */
{ 
   unsigned long  bmin = 0xFFFFFFFF,    /* minimale Differenz                 */
                 *xsum,                 /* alle Differenzsummen               */
                 *xmatch,               /* alle bereinstimmungen             */
                 *xb, 
                **yb,                   /* Zeilen der Differenzsummenmatrix   */
                **ayb, **fyb, *xm;
   int            xa, ya,               /* aktuelle Position                  */
                  xmin = -x, ymin = -y, /* Position minimaler Differenz       */
                  mmax = 0,             /* maximale bereinstimmung           */
                  xmax = -x, ymax = -y; /* Position maximaler bereinstimmung */
   PICTURE       *int_ref,              /* internal reference picture         */
                 *int_mov;              /* internal picture to move           */
   
   
   if ( ref->storepix != mov->storepix ) {
      g_warning("Different storage of %s and %s. No movement possible.", ref->file, mov->file);
      return -1;
   }
   assert ( (xsum = calloc(((x<<1)+1)*((y<<2)+2), sizeof(unsigned long))) );
   xmatch = xsum + ((x<<1)+1)*((y<<1)+1);
   assert ( (yb   = malloc(((y<<2)+2)*sizeof(*yb))) );
   
   for ( fyb = (ayb = yb) + ((y<<2)+2), xb = xsum; ayb < fyb; ayb++, xb += (x<<1)+1 ) *ayb = xb;
   if ( ( ref->W < mov->W ) || ( ref->H < mov->H ) ) {
      if ( ( ref->W < mov->W ) && ( ref->H < mov->H ) ) {
         int_ref     = mov;
         int_mov     = ref;
         ref->roff_x = mov->roff_x;
         ref->roff_y = mov->roff_y;	 
      } else {
         g_warning("Unable to automove pictures with strange size.");
	 return -1;
      }
   } else {
      int_ref = ref;
      int_mov = mov;
   }
   
   for ( ya = -y, xb = xsum, xm = xmatch; ya <= y; ya++ ) 
      for ( xa = -x; xa <= x; xa++, xb++, xm++ ) {
         int_ref->roff_x = int_mov->roff_x + xa;
         int_ref->roff_y = int_mov->roff_y + ya;
         if ( (*xb = BrightMatchPictures(int_ref, int_mov, x, y, xm)) < bmin ) {
            if ( !*xb ) {
               free(xsum);
               return -1;
	    }
	    bmin = *xb;
	    xmin = xa;
	    ymin = ya;
	 }
         if ( *xm > mmax ) {
	    mmax = *xm;
	    xmax = xa;
	    ymax = ya;
	 }
      }
   if ( !CheckMin(xsum+(x*ymin), 1, x, xmin) ) int_mov->roff_x = xmin;
   else if ( !CheckMax(xmatch+(x*ymax), 1, x, xmax) ) int_mov->roff_x = xmax;
   if ( !CheckMin(xsum+xmin, x, y, ymin)     ) int_mov->roff_y = ymin;
   else if ( !CheckMax(xmatch+xmax, x, y, ymax) ) int_mov->roff_y = ymax;
   if ( ref != int_ref ) { /* if pictures where exchanged due to bigger mov change back now */
      iswap(ref->roff_x, mov->roff_x);
      iswap(ref->roff_y, mov->roff_y);
   }
   ref->roff_x = ref->roff_y = 0;
   free(yb);
   free(xsum);
   return 0;
}


int AutoMatchPictures(PAUL *p)
/* Automatically moving pictures by minimizing image differences.
 * The first picture is used as reference to which the difference with
 * all other images will be minimized.
 * --- Parameter: ---
 * PAUL *p                   : image sources
 * --- R"uckgabe: ---
 * int   AutoMatchPictures() : 0 fr OK
 */
{ 
   GList   *pl, *piclist = p->piclist;
   PICTURE *bilder = BILD(piclist), *bild;
   char     buf[256], autostr[] = "utomatic movement of ", *sp;
   int      flag = 0;

   if ( NBILDER(p->piclist) < 2 ) return 0;

   for ( bild = BILD(pl = piclist->next); pl; bild = BILD(pl = pl->next) ) {
      if ( AutoMatchOnePicture(bilder, bild, 10,  0) ||
	   AutoMatchOnePicture(bilder, bild,  0, 10) ||
           (!bild->roff_x && !bild->roff_y)            ) {
         g_warning("No a%s%s against %s.", autostr, bild->file, bilder->file);
         continue; 
      }
      flag = 1;
      printf("A%s%s at (%i,%i)\n", autostr, bild->file, bild->roff_x, bild->roff_y);
      if ( (sp = strstr(strcpy(buf, bilder->file), APPMOVE)) ) *sp = 0;
      CopySpec(bild->spec, ChunkReferenceFile, bilder->file);
   }
   if ( flag ) EndMovedPictures(NULL, piclist, NULL, 1);
   if ( IsMatchAuto(p->opt->mov) ) p->opt->f &= ~MATCHPICTURES;
   return 0;
}

GList *EndMovedPictures(GList *piclist, GList *srclist, PICTURE *op, long flag)
/* prepare moved pictures for saving and free memory
 * --- Parameter: ---
 * GList *piclist            : pictures to move
 * GList *srclist            : sources of the images
 * long   flag               : use flag to determine whether save or not
 * --- R"uckgabe: ---
 * GList *EndMovedPictures() : moved pictures with apropriate set chunks
 */
{
   PICTURE      *bild, *sp, *zw, *fix;
   char          buf[256], *sptr;
   int           min_xoff = 0, max_xoff = 0, min_yoff = 0, max_yoff = 0,
                 dw, dh, free = 0, samesize = 1;
   BOX           cut[1];
   GList        *pl, *sl;
   
   if ( !piclist ) {
      piclist = srclist->next;
      free    = 1;
   }
   
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      if ( bild->roff_x < min_xoff ) min_xoff = bild->roff_x;
      if ( bild->roff_x > max_xoff ) max_xoff = bild->roff_x;
      if ( bild->roff_y < min_yoff ) min_yoff = bild->roff_y;
      if ( bild->roff_y > max_yoff ) max_yoff = bild->roff_y;
   }
   dw = abs(min_xoff) + abs(max_xoff);
   dh = abs(min_yoff) + abs(max_yoff);
   if ( op ) {
      fix = op;
      sl  = srclist;
   } else {
      fix = BILD(srclist);
      sl  = srclist->next;
   }
   fix->roff_x = max_xoff;
   fix->roff_y = max_yoff;
   fix->flag  |= MATCHPICTURES;
   for ( bild = BILD(pl = piclist), sp = BILD(sl); 
         pl; bild = BILD(pl = pl->next), sp = BILD(sl = sl->next) ) {
      sp->roff_x = fix->roff_x - bild->roff_x;
      sp->roff_y = fix->roff_y - bild->roff_y;
      sp->flag   = bild->flag;
      if ( fix->W != bild->W || fix->H != bild->H ) samesize = 0;
   }
   if ( !free ) FreeBild(piclist);

   SHOW_SLOW = 1;
   if (!dw && !dh && samesize ) return srclist;   
      /* there wasnt done any moving and all images have the same size*/
   
   /* step the picture list backwards to avoid to early free of the reference image */
   for ( bild = BILD(pl = g_list_last(srclist)); pl; bild = BILD(pl = pl->prev) ) {
      free = 0;
      if ( bild != fix ) {
         if      ( bild->W - dw >= fix->W )
            cut->x2 = (cut->x1 = ((bild->W - fix->W) >> 1) + bild->roff_x) + fix->W;
         else if ( fix->W - dw  > bild->W ) { /* image is smaller than fix image */
            cut->x1 = 0;
	    cut->x2 = bild->W - 1;
            free = 1;
         } else
            cut->x2 = (cut->x1 = bild->roff_x) + bild->W - dw;

         if      ( bild->H - dh >= fix->H ) 
            cut->y2 = (cut->y1 = ((bild->H - fix->H) >> 1) + bild->roff_y) + fix->H;
         else if ( fix->H - dh  > bild->H ) { /* image is smaller than fix image */      
            if ( free ) continue;
            cut->y1 = 0;
	    cut->y2 = bild->H - 1;
         } else
            cut->y2 = (cut->y1 = bild->roff_y) + bild->H - dh;
      } else { /* moving the fix image */
         PICTURE *zw = BILD(pl->next);
         if ( bild->W >= zw->W ) 
            if ( bild->W - dw > zw->W ) {
               cut->x2 = (cut->x1 = ((bild->W - zw->W) >> 1) + bild->roff_x - dw) + zw->W;
               if ( cut->x2 > bild->W ) cut->x2 = bild->W;
            } else
               cut->x2 = (cut->x1 = bild->roff_x) + bild->W - dw;
         else {
            cut->x2 = (cut->x1 = bild->roff_x) + bild->W;
            free = 1;
         }

         if ( bild->H >= zw->H ) 
            if ( bild->H - dh > zw->H ) {
               cut->y2 = (cut->y1 = ((bild->H - zw->H) >> 1) + bild->roff_y - dh) + zw->H;
               if ( cut->y2 > bild->H ) cut->y2 = bild->H;
            } else
               cut->y2 = (cut->y1 = bild->roff_y) + bild->H - dh;
         else {
            if ( free ) continue ;
            cut->y2 = (cut->y1 = bild->roff_y) + bild->H;
         }
      }

      if ( (sptr = strchr(bild->file, ' ')) ) *sptr = 0;
      sprintf(buf, "%s verschoben", bild->file);
      if ( !(zw = CutArea(bild, cut, 0, buf, (free?APPMOVE:""), 1)) ) continue;
      pl->data = (bild = zw);

      /* Store shift values of automatic movement in reference file name */
      if ( !(sptr = GetSpec(bild->spec, ChunkReferenceFile)) ||
           !(sptr = strchr(sptr, '['))                     ||
           2 != sscanf(sptr, "[%i,%i]", &min_xoff, &min_yoff) )
	 min_xoff = min_yoff = 0;
      sprintf(buf, "%s [%i,%i]", fix->file ? fix->file : "", 
                    bild->roff_x + min_xoff, bild->roff_y + min_yoff);
      CopySpec(bild->spec, ChunkReferenceFile, buf);
   }
   return srclist;
}


