/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Verwendet: GTK, Imlib, libpng, libtiff                                    *
 * Modul:     paul.h                                                         *
 *            Definition von Strukturen und Prototypen der Bild-Funktionen   *
 * Autor:     Andreas Tille                                                  *
 * Datum:     18.02.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#if !defined(_paul_h_)
#define _paul_h_
#include <gtk/gtk.h>
#include <gdk_imlib.h>
#include <png.h>

#define  PAUL_ID   4711

typedef struct { char *key, *value; } CHUNK;

/* Physikalische Daten */
typedef struct { 
   double         U;        /* Anregungsamplitude [V]                 */
   double         f;        /* Anregungsfrequenz  [Hz]                */
   double         T;        /* Temperatur [C]                        */
   double         L;        /* Induktivitt [H]                       */
   double         C_m;      /* Mekondensator [F]                     */
   double         R_m;      /* Mewiderstand  [Ohm]                   */
   double         f_s;      /* Stroboskopfrequenz                     */
   double         dt;       /* Blitzlnge                             */
   double         phi;      /* Phasenwinkel des Bildes                */
   double         t;        /* Zeitdifferenz zum ersten Bild          */
   char          *probe;    /* Name der Probe                         */
   time_t         exptime;  /* Zeit der Aufnahme                      */
   char          *bilder;   /* kanonischer Name der Bilder            */
   int            nbilder;  /* Anzahl der Bilder                      */
   int            ndata;    /* Anzahl der Datenpunkte                 */
   int           *D;        /* Daten proportional D                   */
   int           *Dt;       /* Daten proportional dD/dt               */
} PHYS;

typedef struct { 
   int            id;       /* identify stucture as PICTURE           */
   GdkImlibImage *im;       /* ImLib Bildstruktur                     */
   unsigned char *data;     /* RGB-Daten                              */
   int            width;    /* Bildbreite                             */
   int            height;   /* Bildhhe                               */
   char          *file;     /* Filename (without dir and extension)   */
   char          *ext;      /* Extension                              */
   char          *dir;      /* Directory                              */
   int            size;     /* = width * height                       */
   char           spp;      /* Samples per Pixel (monochrom = 1,      *
                             * sonst != 1 (== 3)                      */
   char           storepix; /* Monochrombild in 1 oder 3 Byte         */
   char           trans;    /* Transparenz                            */
   unsigned int   res;      /* Auflsung in DPI                       */
   unsigned long  x_offset, /* linke obere Ecke beim Scannen          */
                  y_offset; /* rechte obere Ecke beim Scannen         */
         /* SANE (Nikon Coolscan) arbeitet mit 65536 Punkten pro mm   */
   unsigned long *his;      /* Histogramm                             */
   int            n_gamma;  /* Anzahl der Elemente der Gamma-Tabelle  */
   png_color     *gamma;    /* Gamma Tabelle                          */
   time_t         zeit;     /* Zeit, zu der das Bild erzeugt wurde    */
   CHUNK         *spec;     /* Spezifikationsdaten                    */
   int            roff_x;   /* x-Offset zur Referenzdatei             */
   int            roff_y;   /* y-Offset zur Referenzdatei             */
   PHYS          *phys;     /* physikalische Daten                    */
   char          *physname; /* Name der Datei mit physikalische Daten */
   long           flag;     /* Flag for different purpose             */
   GtkWidget     *label;    /* Label of picture list in GTK-interface *
                             * or window of operating image           */
} PICTURE;

typedef struct { int x1, y1, x2, y2; } BOX;

typedef struct { 
   long           f;      /* flag what to do with this garbage        */
   BOX           *cut;    /* corners of area to cut                   */
   unsigned char  eps,    /* "noise step" for difference images       */
                  fordn,  /* order of filtering                       */
                  filter, /* type of filtering                        */
                  greps,  /* allowed difference between RGB values    */
                  mov,    /* which type of moving images              *
                           * while moving images == 255-mov           */
                  shr,    /* step of shrinkage                        */
                  terr,   /* allowed error when deleting borders      */
                  thresh; /* threshold for deleting borders           */
   int            offset, /* offset for enhancing brightness          */
                  xi, yi; /* coordinates for image to insert          */
   double         sek,    /* animation delay time between two images  */
                  scale;  /* factor for contrast                      */
} OPTIONS;

typedef struct {
   GList     *activ,   /* currently displayed image                   */
             *piclist, /* list of all images to display               */
             *src;     /* list of source images (when moving images)  */
   PICTURE   *op;      /* operation image                             */
   GtkList   *filelist;/* list of filenames (contain's p as user_data)*/
   GtkWidget *show;    /* window to show images                       */
   GtkWidget *w;       /* file list window (contain's info as user_d.)*/
   GtkWidget *ani_but; /* button to start animation                   */
   OPTIONS   *opt;     /* alle Mglichen Optionen                     */
} PAUL;

typedef enum {
   P_S_CHUNK = 0,
   P_I_CHUNK,
   P_INT,
   P_LONG,
   P_GAMMA,
   P_UCHAR,
   P_DOUBLE
} PARM_TYPE;

PICTURE *BILD(GList *l);
#define  IS_PICTURE(bild)   ( bild && bild->id == PAUL_ID )
/* eventuell einfach im-> davorsetzen */
#define  DATA       data               
#define  W          width
#define  H          height

#define iswap(x, y)  {register int zw = x; x = y; y = zw;}

extern char *ChunkNames[],
            ChunkNameSource[],        /* Quelldatei                           */
            ChunkNameSoftware[],      /* erzeugende Software (paul/paulscan)  */
            ChunkNameAuthor[],	      /* Wer hats erzeugt (aus UID)          */
            ChunkNameDescription[],   /* Differenzbild, Ausschnitt, ...       */
            ChunkNameTyp[],	      /* wasfr Ausschnitt, ...               */
            ChunkFilmType[],          /* Negativ oder Positiv                 */
            ChunkBrightness[],        /* Helligkeit beim Scannen              */
	    ChunkContrast[],          /* Kontrast beim Scannen                */
            ChunkReferenceFile[];     /* Referenzdatei (Scannen, Verschieben) */

/* Default values to delete border */
#define  DEFAULT_THRESHOLD       20
#define  DEFAULT_ERROR           50
#define  DEFAULT_SCAN_THRESHOLD  68   /* oben/unten:  56, (nach drehen) Seite: 63 *
                                       * PCD images from convert need at least 67 */
#define  DEFAULT_SCAN_ERROR      82   /*              81,                      92 */

extern  char TypAusschnitt[],
             TypDifferenzBild[],
             TypDelRand[],
             TypDelScanRand[],
             TypFilter[],
             TypGray[],
             TypKontrastBild[],
             TypNegativ[],
             TypMirror[],
             TypRotate[],
             TypScan[],
             TypShrink[];


extern  const char *exename; /* "paul" ... or how to call the program */
#define HISTFILE  "histogram"
#define VPROFILE  "vertical_profile"

#define DIFFERENZ          0x00000001
#define MAKEGRAY           0x00000002
#define HISTOGRAMM         0x00000004
#define INFO               0x00000008
#define SAVEMAXIMA         0x00000010
#define SAVEMINIMA         0x00000020
#define SAVEEXTREMA        0x00000030
#define BRIGHT             0x00000040
#define SHRINK             0x00000080
#define FILTER             0x00000100
#define MATCHPICTURES      0x00000200
#define QUIET              0x00000400
#define NOTMONOCHROM       0x00000800
#define DELBORDER          0x00001000
#define ONLYLEFTRIGHT      0x00002000
#define DELSCANBORDER      0x00004000
#define NEGATIV            0x00008000
#define PRINTGAMMA         0x00010000
#define SAFEFLAG           0x00020000
#define PNGORTIFF          0x00040000
#define MIRROR_H           0x00080000
#define MIRROR_V           0x00100000
#define ROT_90             0x00200000
#define ROT_180            0x00400000
#define FFT                0x00800000
#define HTMLOUTPUT         0x01000000

#define MATCHAUTO          0
#define MATCHCOL           1
#define MATCHDIF           2
#define MATCHFLASH         3

#define Differenz(flag)          ( flag & DIFFERENZ  )
#define DoMakeGray(flag)         ( flag & MAKEGRAY )
#define DoMatchPictures(flag)    ( flag & MATCHPICTURES )
#define DelBorder(flag)          ( flag & DELBORDER )
#define OnlyLeftRight(flag)      ( flag & ONLYLEFTRIGHT )
#define DelScanBorder(flag)      ( flag & DELSCANBORDER )
#define Bright(flag)             ( flag & BRIGHT )
#define Negativ(flag)            ( flag & NEGATIV )
#define HistogrammFlag(flag)     ( flag & HISTOGRAMM )
#define NotMonoChrom(flag)       ( flag & NOTMONOCHROM )
#define SaveExtrema(flag)        ( flag & SAVEEXTREMA )
#define SaveMinima(flag)         ( flag & SAVEMINIMA )
#define SaveMaxima(flag)         ( flag & SAVEMAXIMA )
#define Filter(flag)             ( flag & FILTER )
#define SaveBild(flag)           ( flag & SAFEFLAG )
#define SavePNG(flag)            ( SaveBild(flag) && !(flag & PNGORTIFF) ) 
#define SaveTIFF(flag)           ( SaveBild(flag) &&  (flag & PNGORTIFF) )
#define SetPNG(flag)             ( flag |= SAFEFLAG )
#define SetTIFF(flag)            ( flag |= SAFEFLAG + PNGORTIFF )
#define DontSaveBild(flag)       ( flag &= ~SAFEFLAG )
#define Info(flag)               ( flag & INFO )
#define Shrink(flag)             ( flag & SHRINK )
#define HMirror(flag)            ( flag & MIRROR_H )
#define VMirror(flag)            ( flag & MIRROR_V )
#define Rotate90(flag)           ( (flag & ROT_90)  && !(flag & ROT_180) )
#define Rotate180(flag)          ( (flag & ROT_180) && !(flag & ROT_90)  )
#define Rotate270(flag)          ( flag & (ROT_90 | ROT_180) )
#define Rotate(flag)             ( (flag & ROT_90) | (flag & ROT_180) )
#define Mirror(flag)             ( flag & (MIRROR_H | MIRROR_V) )
#define OnlyInfo(flag)           ( !((0xFFFFFFFF&(~INFO)&(~QUIET)) & flag) && (flag&QUIET) )
#define DoFFT(flag)              ( flag & FFT )
#define HtmlOutput(flag)         ( flag & HTMLOUTPUT )
#define IsAnimation(opt)         ( opt->sek > -1 )
#define IsMatchAuto(mov)         ( mov == MATCHAUTO  )
#define IsMatchCol(mov)          ( mov == MATCHCOL   )
#define IsMatchDif(mov)          ( mov == MATCHDIF   )
#define IsMatchFlash(mov)        ( mov == MATCHFLASH )

#define IsMonochrom(bild)        ( bild->spp == 1 )
#define RED    0
#define GREEN  1
#define BLUE   2

int     usage(void);

#define APPCUT           "_c"
#define APPFILTER        "_f"
#define APPGRAY          "_g"
#define APPINSERT        "_i"
#define APPKONTRAST      "_k"
#define APPMOVE          "_m"
#define APPMIN           "_min"
#define APPMAX           "_max"
#define APPNEGATIV       "_n"
#define APPROTATE        "_o"
#define APPMIRROR        "_p"
#define APPDELBORDER     "_r"
#define APPDELSCANBORDER "_rs"
#define APPSHRINK        "_s"    /* level of shrinkage appended */

#define FREE(ptr)  { if ( ptr ) { free(ptr); ptr = NULL; } }

extern     int             SHOW_SLOW;
extern     int             ANIMATION_SPEED;  
#define    SINGLE          0  /* no animation activ                                             */
#define    SLOW_ANIMATION  1  /* animation is slow enought to show name of picture in title bar */
#define    FAST_ANIMATION  2  /* suppress updating title bar                                    */

/* bright.c */
unsigned long brightness(unsigned char *buf, int x, int y, int step, int storepix);
unsigned long Xsum(unsigned char *buf, int len);
unsigned long Ysum(unsigned char *buf, int len, int step);
unsigned long Ysum3(unsigned char *buf, int len, int step);
void     Helligkeiten(PAUL *p, char *file);

/* cutarea.c */
PICTURE *CutArea(PICTURE *bild, BOX *cut, long flag, char *was, char *ext, int del_old);
int      CutAreas(GList *piclist, OPTIONS *p);

/* delbordr.c */
int      SelectPictureWithoutBorder(PAUL *p);

/* delscanb.c */
int      DeleteScannerBorder(PAUL *p);

/* difpic.c */
int      DifferenzBilder(PAUL *p);

/* extrema.c */
int      Extrema(PAUL *p);

/* fft.c */
int      FastFourierTransform(PAUL *p);

/* filesel.c */
void     FileSelectionLoadSave(PAUL *p, GtkSignalFunc func, char *name, char *title);
char    *FilenameExtractFromSelection(GtkFileSelection *fs, PICTURE *bild);

/* filter.c */
int      FilterBilder(PAUL *p);

/* gray.c */
int      MakeGray(PAUL *p);

/* gtkpaul.c */
void     GtkInterface(PAUL *p);

/* histogrm.c */
void     MakeHistogram(PICTURE *bild);
void     MakeHistograms(GList *piclist);

/* insert.c */
int      InsertImage(PAUL *p);

/* matchpic.c */
int      AutoMatchPictures(PAUL *p);
int      CreateMatchPictures(PAUL *p);
GList   *EndMovedPictures(GList *piclist, GList *srclist, PICTURE *op, long flag);
int      MovePic(PAUL *p, int v, int h, int mod);
void     SetMoveTitle(PAUL *p);
int      ShiftValue(int mod);

/* menufactory.c */
void     get_main_menu(GtkWidget **menubar, GtkAcceleratorTable **table, GtkMenuEntry *menu, int nmenu);

/* messages.c */
void     MessagesInit(void);

/* mirror.c */
int      MakeMirror(PAUL *p);

/* negativ.c */
void     NegativBuffer(unsigned char *buf, int size, int step);
int      MakeNegative(GList *piclist);

/* paulload.c */
PICTURE *LoadPicture(PICTURE *bild, char *file, long flag);
GdkImlibImage *at_imlib_set_image_from_data(unsigned char *data, gint w, gint h);

/* paulinfo.c */
void PaulInfo(GList *piclist, long flag);

/* paulspec.c */
char    *GetSpec(CHUNK *chunks, const char *spec);
int      SetSpec(CHUNK *chunks, const char *spec, char *value);
int      CopySpec(CHUNK *chunks, const char *spec, const char *value);
int      BinCopySpec(CHUNK *chunks, const char *spec, void *value, int size);
PICTURE *InitSpec(void);
void     GetPictureAuthor(PICTURE *bild);
void     GetPictureSpecs(PICTURE *bild, char *file);
time_t   convert_png_time_to_time_t(png_time *ptime);
void     CreateDescription(PICTURE *bild, char *was);

/* picuti.c */
void     Backup(const char *name);
int      CheckInBounds(PICTURE *bild, BOX *cut);
void     CreateShowPixmaps(GList *piclist);
void     FreeBild(GList *piclist);
void     FreeSpecs(CHUNK *spec);
int      GetListPosition(GList *list, GList *elem);
int      MakeHistogramm(PICTURE *bild);
int      NBILDER(GList *l);
char    *NewFileName(PICTURE *bild, char *appendix);
int      NewImage(PICTURE *bild, int w_new, int h_new, unsigned char *data);
BOX     *ReadBoxFromString(char *string);
PICTURE *ReadPic(char *file, OPTIONS *p);
GList   *ReadPicFiles(int nfiles, char **efiles, OPTIONS *p);

/* pngload.c */
int      PaulLoadPNG(PICTURE *bild, char *file, long flag);

/* pngsave.c */
int      WritePNGImage(PICTURE *bild, char *filename);

/* rotate.c */
int MakeRotate(PAUL *p);

/* savepics.c */
void     SavePictures(GList *piclist, long flag);

/* scale.c */
int      SkaliereBilder(PAUL *p);

/* schrink.c */
void ShrinkPicture(PICTURE *bild, unsigned char shr);

/* screen.c */
int      ApplyPicture(GdkWindow *win, PICTURE *bild);
void     GrowPic(GdkWindow *win, PICTURE *bild);
void     ShrinkPic(GdkWindow *win, PICTURE *bild);

/* tifload.c */
int      PaulLoadTIFF(PICTURE *bild, char *file, long flag);

/* tifsave.c */
int      GetTag(char *key);
int      WriteTIFFImage(PICTURE *bild, char *filename);

#endif


