/* $Id: mouse.c,v 1.4 1998/09/20 21:24:44 marcus Exp $
***************************************************************************

   Display-SUID: mouse handling

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ------------------------------------------------------------------------
	This code was derived from the following sources of information:

	[1] gpm-Linux 1.0 (general purpose mouse)
	Copyright 1993        ajh@gec-mrc.co.uk (Andrew Haylett)
	Copyright 1994,1995   rubini@ipvvis.unipv.it (Alessandro Rubini)

***************************************************************************
*/

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <termios.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <linux/kdev_t.h>   /* only needed for MAJOR() macro */
#include <linux/major.h>    /* only needed for MISC_MAJOR */

#include "debug.h"
#include <ggi/internal/ggi-dl.h>


extern int GGIevqueueadd(ggi_visual *vis, ggi_event *ev);  /* !!! */


#define DPRINT1  DPRINT
#define DPRINT2  DPRINT
#define DPRINT3  DPRINT


static int mouse_fd = -1;
static struct termios mouse_termios;


/* ---------------------------------------------------------------------- */


#define MAX_MOUSE_NAMES  8
#define MAX_MOUSE_TYPES  32

#define MAX_PACKET_BUF	128
#define MAX_USER_BUF	128

#define C_NORM	(CREAD | CLOCAL | HUPCL)
#define I_NORM	(IGNBRK)


typedef struct mouse_state
{
	int button_state;
	int parse_state;
	int left_hand;

	ggi_visual *vis;

} MouseState;


typedef struct mouse_type
{
	const char *names[MAX_MOUSE_NAMES];

	/* Serial parameters.  If the mouse is not a serial mouse
	 * (for example, a busmouse), then default_baud should be < 0.
	 */

	int default_baud;
	int cflag;  int iflag;
	int oflag;  int lflag;

	/* The parser function.
	 *
	 * This returns the number of bytes used by the packet.  It
	 * should return 0 when the packet is too short, and return 1
	 * when the packet is invalid (to resync).  len is guaranteed
	 * to be >= minimum_packet_size.
	 */

	int (* parse_func)(MouseState *mstate, uint8 *buf, int len);

	int minimum_packet_len;
	
} MouseType;


/* ---------------------------------------------------------------------- */


/**
 **  Event dispatching routines
 **/

static void mouse_send_movement(MouseState *mstate, int dx, int dy)
{
	ggi_event mouse_event;

	if (dx || dy) {

		mouse_event.pmove.size   = sizeof(ggi_pmove_event);
		mouse_event.pmove.type   = evPtrRelative;
		mouse_event.pmove.origin = EV_ORIGIN_NONE;
		mouse_event.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(& mouse_event);

		mouse_event.pmove.x = dx;
		mouse_event.pmove.y = dy;

		GGIevqueueadd(mstate->vis, &mouse_event);
	}
}

static void mouse_send_buttons(MouseState *mstate, int buttons, int last)
{
	ggi_event mouse_event;

	int mask;
	int changed = buttons ^ last;

	
	/* change in button state ? */

	for (mask = 1; mask > 0; mask <<= 1)

	if (changed & mask) {

		int state = buttons & mask;

		mouse_event.pbutton.size = sizeof(ggi_pbutton_event);
		mouse_event.pbutton.type = state ?
			evPtrButtonPress : evPtrButtonRelease;
		mouse_event.pmove.origin = EV_ORIGIN_NONE;
		mouse_event.pmove.target = EV_TARGET_NONE;

		EV_TIMESTAMP(& mouse_event);

		mouse_event.pbutton.button = mask;
		mouse_event.pbutton.state  = 0;	 /* deprecated */

		GGIevqueueadd(mstate->vis, &mouse_event);
	}
}


/**
 **  Mouse Packet Parsers
 **/

static int B_microsoft[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_microsoft(MouseState *mstate, uint8 *buf, int len)
{
	int dx, dy, buttons;
	int hand = mstate->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT3("Invalid microsoft packet\n");
		return 1;
	}

	buttons = B_microsoft[hand | ((buf[0] & 0x30) >> 4)];

	dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
	dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

	mouse_send_movement(mstate, dx, dy);
	mouse_send_buttons(mstate, buttons, mstate->button_state);

	mstate->button_state = buttons;

	DPRINT3("Got ms packet\n");

	return 3;
} 

static MouseType T_microsoft =
{
	{ "Microsoft", "ms", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_microsoft, 3 
};


static int B_mousesys[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_mousesys(MouseState *mstate, uint8 *buf, int len)
{
	int dx, dy, buttons;
	int hand = mstate->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT3("Invalid msc packet\n");
		return 1;
	}

	buttons = B_mousesys[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1] + (sint8) buf[3];
	dy = -(sint8) buf[2] - (sint8) buf[4];

	mouse_send_movement(mstate, dx, dy);
	mouse_send_buttons(mstate, buttons, mstate->button_state);

	mstate->button_state = buttons;

	DPRINT3("Got mousesys packet\n");

	return 5;
}

static MouseType T_mousesys =
{
	{ "MouseSystems", "mousesystem", "mousesys", "msc", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_mousesys, 5
};


static int B_logitech[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_logitech(MouseState *mstate, uint8 *buf, int len)
{
	int dx, dy, buttons;
	int hand = mstate->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0xe0) != 0x80) || ((buf[1] & 0x80) != 0x00)) {

		DPRINT3("Invalid logitech packet\n");
		return 1;
	}

	buttons = B_logitech[hand | (buf[0] & 0x07)];

	dx = (buf[0] & 0x10) ?	(sint8)buf[1] : -(sint8)buf[1];
	dy = (buf[0] & 0x08) ? -(sint8)buf[2] :  (sint8)buf[2];

	mouse_send_movement(mstate, dx, dy);
	mouse_send_buttons(mstate, buttons, mstate->button_state);

	mstate->button_state = buttons;

	DPRINT3("Got logitech packet\n");

	return 3;
}

static MouseType T_logitech = 
{
	{ "Logitech", "logi", "log", "Logitech", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_logitech, 3
};


static int B_sun[16] =
{	0x0, 0x2, 0x4, 0x6, 0x1, 0x3, 0x5, 0x7,
	0x0, 0x1, 0x4, 0x5, 0x2, 0x3, 0x6, 0x7
};

static int M_sun(MouseState *mstate, uint8 *buf, int len)
{
	int dx, dy, buttons;
	int hand = mstate->left_hand ? 8 : 0;

	/* check header */

	if ((buf[0] & 0xf8) != 0x80) {

		DPRINT3("Invalid sun packet\n");
		return 1;
	}

	buttons = B_sun[hand | (~buf[0] & 0x07)];

	dx =  (sint8) buf[1];
	dy = -(sint8) buf[2];

	mouse_send_movement(mstate, dx, dy);
	mouse_send_buttons(mstate, buttons, mstate->button_state);

	mstate->button_state = buttons;

	DPRINT3("Got sun packet\n");

	return 3;
}

static MouseType T_sun =
{
	{ "Sun", NULL, },
	B1200, C_NORM | CS8 | CSTOPB, I_NORM, 0, 0,
	M_sun, 3
};


static int B_mouseman[16] =
{	0x0, 0x2, 0x1, 0x3, 0x4, 0x6, 0x5, 0x7,
	0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7
};

static int M_mouseman(MouseState *mstate, uint8 *buf, int len)
{
	/*  The damned MouseMan has 3/4 bytes packets.	The extra byte
	 *  is only there if the middle button is active.
	 *
	 *  This is what we do: when we get the first 3 bytes, we parse
	 *  the info and send off the events, and set a flag to say we
	 *  have seen the first three bytes.
	 * 
	 *  When we get the fourth byte (maybe with the first three,
	 *  or maybe later on), we check if it is a header byte.
	 *  If so, we return 3, otherwise we parse the buttons in it,
	 *  send off the events, and return 4.
	 *
	 *  Note also that unlike the other mices, the mouseman parser
	 *  stores the RAW buttons in priv->button_state.
	 */

	int dx, dy, buttons;
	int hand = mstate->left_hand ? 8 : 0;

	/* check header */

	if (((buf[0] & 0x40) != 0x40) || ((buf[1] & 0x40) != 0x00)) {

		DPRINT3("Invalid mouseman packet\n");
		return 1;
	}

	/* handle the common 3 bytes */

	if (mstate->parse_state == 0) {

		buttons = (mstate->button_state & 0x4) |
			((buf[0] & 0x30) >> 4);

		dx = (sint8) (((buf[0] & 0x03) << 6) | (buf[1] & 0x3f));
		dy = (sint8) (((buf[0] & 0x0c) << 4) | (buf[2] & 0x3f));

		mouse_send_movement(mstate, dx, dy);
		mouse_send_buttons(mstate, B_mouseman[hand | buttons],
			B_mouseman[hand | mstate->button_state]);

		mstate->button_state = buttons;
		mstate->parse_state  = 1;

		DPRINT3("Got mouseman base packet\n");
	}


	/* now look for extension byte */

	if (len < 4) {
		return 0;
	}

	mstate->parse_state = 0;

	if ((buf[3] & 0xc0) != 0) {
	
		/* 4th byte must be a header byte */
		
		return 3;
	}

	/* handle the extension byte */

	buttons = (mstate->button_state & 0x3) | ((buf[3] & 0x20) >> 3);

	mouse_send_buttons(mstate, B_mouseman[hand | buttons],
		B_mouseman[hand | mstate->button_state]);

	mstate->button_state = buttons;

	DPRINT3("Got mouseman extension packet\n");

	return 4;
}

static MouseType T_mouseman =
{
	{  "MouseMan", "mman", NULL, },
	B1200, C_NORM | CS7, I_NORM, 0, 0,
	M_mouseman, 3
};


/* 
 * MMSeries mice use the same protocal as Logitech (and vice-versa)
 */

static MouseType T_mmseries =
{
	{ "MMSeries", "mm", NULL, },
	B1200, C_NORM | CS8 | PARENB | PARODD, I_NORM, 0, 0,
	M_logitech, 3
};
	

/*
 * BusMice use the same protocal as Sun
 */

static MouseType T_busmouse =
{
	{ "BusMouse", "bus", "bm", NULL, },
	-1, 0, 0, 0, 0,
	M_sun, 3
};


/*** PS/2 NOT IMPLEMENTED YET

static MouseType T_ps2 =
{
	{ "PS2", NULL, },
	-1, 0, 0, 0, 0,
	M_ps2, 3
};
***/


/* ---------------------------------------------------------------------- */


/* The Holy Table of Mouse Parsers
*/

static MouseType *mice_types[MAX_MOUSE_TYPES] =
{
	& T_microsoft, 
	& T_mousesys,
	& T_logitech,
	& T_sun,
	& T_mmseries,
	& T_mouseman,
	& T_busmouse,

	/* !!! & T_ps2, */
	
	NULL,
};

#define DEFAULT_MOUSE  0


typedef struct mouse_buffer 
{
	int packet_len;
	uint8 packet_buf[MAX_PACKET_BUF];

} MouseBuffer;


static MouseType  *motype;
static MouseState  mostate;
static MouseBuffer mobuf;


/* ---------------------------------------------------------------------- */


/* Parse one packet
 */
 
static int do_parse_packet(void)
{
	int used;

#if 0
	{
		int i;

		fprintf(stderr, "fbdev-mouse: do_parse_packet [");

		for (i=0; i < (mobuf.packet_len - 1); i++) {
			fprintf(stderr, "%02x ", mobuf.packet_buf[i]);
		}

		fprintf(stderr, "%02x]\n", mobuf.packet_buf[i]);
	}
#endif

	/* call parser function */
	
	used = motype->parse_func(&mostate, mobuf.packet_buf, 
				  mobuf.packet_len);

	DPRINT3("packet used %d bytes\n", used);

	return used;
}


static int find_mouse(char *name)
{
	int m, n;

	for (m=0; (m < MAX_MOUSE_TYPES) && 
	          (mice_types[m]->names[0] != NULL); m++) {

		MouseType *mtype = mice_types[m];

		for (n=0; (n < MAX_MOUSE_NAMES) && 
		          (mtype->names[n] != NULL); n++) {

			if (strcasecmp(mtype->names[n], name) == 0) {
			
				return m;  /* found it */
			}
		}
	}

	fprintf(stderr, "Unknown mouse type '%s' -- using the default.\n", 
		name);

	return DEFAULT_MOUSE;
}


static int find_baud(char *baudname)
{
	switch (atoi(baudname))
	{
		case 9600: return B9600;
		case 4800: return B4800;
		case 2400: return B2400;
		case 1200: return B1200;
	}

	fprintf(stderr, "Baud rate '%s' not supported\n", baudname);
	
	return B1200;  /* !!! */
}


static int do_mouse_open(MouseType *mtype, char *filename,
			 int dtr, int rts, int baud)
{
	int fd;

	struct termios new;


	fd = open(filename, O_RDWR | O_NOCTTY);

	if (fd < 0) {
		DPRINT("display-fbdev: Failed to open '%s'.\n", filename);
		return -1;
	}

	if (mtype->default_baud >= 0) {

		/* set up the termios state and baud rate */
	
		tcflush(fd, TCIOFLUSH);

		if (tcgetattr(fd, &mouse_termios) < 0) {
			DPRINT("tcgetattr failed.\n");
			close(fd);
			return -1;
		}

		new = mouse_termios;

		if (baud < 0) {
			baud = mtype->default_baud;
		}
		
		new.c_cflag = mtype->cflag | baud;
		new.c_iflag = mtype->iflag;
		new.c_oflag = mtype->oflag;
		new.c_lflag = mtype->lflag;
		new.c_cc[VMIN]  = 1;
		new.c_cc[VTIME] = 0;

		if (tcsetattr(fd, TCSANOW, &new) < 0) {
			DPRINT("tcsetattr failed.\n");
			close(fd);
			return -1;
		}

		/* set up RTS and DTR modem lines */

		if ((dtr >= 0) || (rts >= 0))
		{
			unsigned int modem_lines;

			if (ioctl(fd, TIOCMGET, &modem_lines) == 0) {

				if (dtr == 0) modem_lines &= ~TIOCM_DTR;
				if (rts == 0) modem_lines &= ~TIOCM_RTS;

				if (dtr > 0) modem_lines |= TIOCM_DTR;
				if (rts > 0) modem_lines |= TIOCM_RTS;
				
				ioctl(fd, TIOCMSET, &modem_lines);
			}
		}
	}

	mouse_fd = fd;

	return 0;
}


static void do_mouse_close(void)
{
	if (tcsetattr(mouse_fd, TCSANOW, &mouse_termios) < 0) {
		DPRINT("tcsetattr failed.\n");
	}

	close(mouse_fd);

	mouse_fd = -1;
}


static char *parse_field(char *dst, int max, char *src)
{
	int len=1;   /* includes trailing NUL */

	for (; *src && (*src != ':'); src++) {
		if (len < max) {
			*dst++ = *src;
			len++;
		}
	}

	*dst = 0;

	if (*src == ':') {
		src++;
	}
	return src;
}


static void parse_mouse_specifier(char *spec, char *protname,
				  char *devname, char *options)
{
	*protname = *devname = *options = 0;

	/* LISP haters should shut their eyes now :) */

	if (spec) 
	parse_field(options, 255,
		    parse_field(devname, 255,
				parse_field(protname, 255, spec)));

	/* supply defaults for missing bits */

	if (*devname == 0) {
		strcpy(devname, "/dev/mouse");
	}

	if (*protname == 0) {

		/* Protocol hasn't been specified. So try statting the
		 * file to see if it is a char device with Major == 10.
		 * In this case, the protocol is most likely BusMouse.
		 */

		struct stat m_stat;

		strcpy(protname, "microsoft");

		if ((stat(devname, &m_stat) == 0) &&
		    S_ISCHR(m_stat.st_mode) &&
		    (MAJOR(m_stat.st_rdev) == MISC_MAJOR)) {
		    
			strcpy(protname, "busmouse");
		}
	}
}


static inline char *parse_opt_int(char *opt, int *val)
{
	*val = 0;

	for (; *opt && isdigit(*opt); opt++) {
		*val = ((*val) * 10) + ((*opt) - '0');
	}

	return opt;
}


static void parse_options(char *opt, int *baud, int *dtr, int *rts)
{
	while (*opt)
	switch (*opt++) {

		case 'b': case 'B':    /* baud */
		{
			opt = parse_opt_int(opt, baud);
			break;
		}

		case 'd': case 'D':    /* dtr */
		{
			opt = parse_opt_int(opt, dtr);
			break;
		}

		case 'r': case 'R':    /* rts */
		{
			opt = parse_opt_int(opt, rts);
			break;
		}

		default:
		{
			fprintf(stderr, "Unknown mouse option "
				"'%c' -- rest ignored.\n", *opt);
			return;
		}
	}
}


/* ---------------------------------------------------------------------- */


int fbdev_init_mouse(ggi_visual *vis, char *typename)
{
	char protname[256];
	char devname[256];
	char options[256];

	int mindex;
	int dtr=-1, rts=-1, baud=-1;


	/* parse the mouse specifier (LIBGGI_MOUSE) */
	parse_mouse_specifier(typename, protname, devname, options);

	mindex = find_mouse(protname);

	parse_options(options, &baud, &dtr, &rts);
	

 	/* open mouse */

	motype = mice_types[mindex];

	if (do_mouse_open(motype, devname, dtr, rts, baud) < 0) {

		fprintf(stderr, "display-fbdev: Couldn't open mouse.\n");
		return -1;
	}

	memset(&mostate, 0, sizeof(MouseState));
	memset(&mobuf,   0, sizeof(MouseBuffer));

	mostate.vis = vis;

	
	DPRINT("display-fbdev: mouse init OK.\n");

	return mouse_fd;
}


void fbdev_exit_mouse(ggi_visual *vis)
{
	if (mouse_fd >= 0) {
		do_mouse_close();
	}

	DPRINT("display-fbdev: mouse exit OK.\n");
}


void fbdev_handle_mouse_data(ggi_visual *vis)
{
	int read_len;


	/* read the mouse data */

	read_len = MAX_PACKET_BUF - mobuf.packet_len - 1;

	/* ASSERT(read_len >= 1) */

	read_len = read(mouse_fd, mobuf.packet_buf + mobuf.packet_len,
			read_len);

	if (read_len < 1) {
		DPRINT("display-fbdev: Error reading mouse.\n");
		return;
	}

	mobuf.packet_len += read_len;
	

	/* parse any packets */

	while (mobuf.packet_len >= motype->minimum_packet_len) {

		int used;

		used = do_parse_packet();

		if (used <= 0) {
			break;	 /* not enough data yet */
		}

		mobuf.packet_len -= used;

		if (mobuf.packet_len > 0) {
		
			memmove(mobuf.packet_buf, mobuf.packet_buf + used,
				mobuf.packet_len);
		} else {
			mobuf.packet_len = 0;
		}
	}

	DPRINT3("fbdev_handle_mouse_data done.\n");
}
