/* $Id: mode.c,v 1.30 1998/10/29 01:56:35 marcus Exp $
***************************************************************************

   LibGGI Mode management.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
   Copyright (C) 1998 Hartmut Niemann

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ggi/internal/internal.h>


/* Static variables */
static ggi_mode _ggiDefaultMode =
{
	GGI_AUTO,               /* frames */
	{GGI_AUTO,GGI_AUTO},    /* visible size */
	{GGI_AUTO,GGI_AUTO},    /* virtual size */
	{GGI_AUTO,GGI_AUTO},    /* size in mm (don't care) */
	GT_AUTO,                /* graphtype */
	{GGI_AUTO,GGI_AUTO},    /* dots per pixel */
	/* 0 */
};


void _ggiSetDefaultMode(const char *str)
{
	ggiParseMode(str, &_ggiDefaultMode);
}


static void _ggiCheck4Defaults(ggi_mode *tm)
{
#define DOCHECK(what)  \
	if (tm->what == GGI_AUTO) tm->what=_ggiDefaultMode.what
	
	DOCHECK(frames);
	DOCHECK(visible.x);
	DOCHECK(visible.y);
	DOCHECK(virt.x);
	DOCHECK(virt.y);
	DOCHECK(dpp.x);
	DOCHECK(dpp.y);
	DOCHECK(graphtype);

#undef DOCHECK
}

/********************************/
/* set any mode (text/graphics) */
/********************************/

int ggiSetMode(ggi_visual *vis,ggi_mode *tm)
{ 
	int retval;
	ggi_mode oldmode;

	LIBGGI_APPASSERT(vis != NULL, "ggiSetMode: vis == NULL");
	LIBGGI_APPASSERT(tm != NULL,  "ggiSetMode: tm == NULL");

#ifdef DEBUG
	if ((_ggiDebugState & GGIDEBUG_CORE)
	    || (_ggiDebugState & GGIDEBUG_MODE)) {
		fprintf(stderr, "libggi: ggiSetMode(%p, ", vis);
		ggiFPrintMode(stderr, tm);
		fprintf(stderr, ") called\n");
	}
#endif
	_ggiLock(vis->mutex);

	DPRINT_MODE("ggiSetMode: trying (vis %dx%d virt %dx%d)\n",
		    tm->visible.x,tm->visible.y,tm->virt.x,tm->virt.y);
	_ggiCheck4Defaults(tm);
               
	memcpy(&oldmode, tm, sizeof(ggi_mode));
	DPRINT_MODE("ggiSetMode: trying2 (vis %dx%d virt %dx%d)\n",
		tm->visible.x,tm->visible.y,tm->virt.x,tm->virt.y);
	DPRINT_MODE("ggiSetMode: calling %p\n",vis->opdisplay->setmode);

	retval=vis->opdisplay->setmode(vis,tm);

	if (retval < 0) {
		fprintf(stderr, "LibGGI: Failed to set mode: ");
		ggiFPrintMode(stderr, &oldmode);
		fprintf(stderr, "\n");
	} else {
		DPRINT_CORE("ggiSetMode: setframes\n");
		ggiSetDisplayFrame(vis, 0);
		ggiSetReadFrame(vis, 0);
		ggiSetWriteFrame(vis, 0);
		DPRINT_CORE("ggiSetMode: setorigin\n");
		ggiSetOrigin(vis,0,0);
		DPRINT_CORE("ggiSetMode: set GC\n");
		ggiSetGCForeground(vis,0);
		ggiSetGCBackground(vis,0);
		ggiSetGCClipping(vis,0,0,tm->virt.x,tm->virt.y);

		DPRINT_CORE("ggiSetMode: fillscreen\n");
		ggiFillscreen(vis);
		DPRINT_CORE("ggiSetMode: success (vis %dx%d virt %dx%d)\n",
			    tm->visible.x,tm->visible.y,tm->virt.x,tm->virt.y);
	}

	DPRINT_CORE("ggiSetMode: done!\n");
	_ggiUnlock(vis->mutex);

	return retval;
}

/**********************************/
/* check any mode (text/graphics) */
/**********************************/

int ggiCheckMode(ggi_visual *vis,ggi_mode *tm)
{
	LIBGGI_APPASSERT(vis != NULL, "ggiCheckMode: vis == NULL");
	LIBGGI_APPASSERT(tm != NULL,  "ggiCheckMode: tm == NULL");

	DPRINT_CORE("ggiCheckMode(%p, %p) called\n", vis, tm);

	_ggiCheck4Defaults(tm);
	return vis->opdisplay->checkmode(vis,tm);
}

/************************/
/* get the current mode */
/************************/

int ggiGetMode(ggi_visual *vis,ggi_mode *tm)
{
	LIBGGI_APPASSERT(vis != NULL, "ggiGetMode: vis != NULL");
	LIBGGI_APPASSERT(tm != NULL,  "ggiGetMode: tm != NULL");

	DPRINT_CORE("ggiGetMode(%p, %p) called\n", vis, tm);

	return vis->opdisplay->getmode(vis,tm);
}

/******************/
/* set a textmode */
/******************/
int ggiSetTextMode(ggi_visual *vis,int cols,int rows,
				   int vcols,int vrows,
				   int fontsizex,int fontsizey,
				   ggi_graphtype mode)
{
	ggi_mode tim;
	
	DPRINT_CORE("ggiSetTextMode(%p, %d, %d, %d, %d, %d, %d, 0x%x) called\n",
		    cols, rows, vcols, vrows, fontsizex, fontsizey, mode);
	
	tim.frames    = GGI_AUTO;
	tim.visible.x = cols;
	tim.visible.y = rows;
	tim.virt.x    = vcols;
	tim.virt.y    = vrows;
	tim.size.x    = tim.size.y = 0;
	tim.graphtype = mode;
	tim.dpp.x     = fontsizex;
	tim.dpp.y     = fontsizey;
	
	return ggiSetMode(vis,&tim);
}

/*************************/
/* check a text mode */
/*************************/
int ggiCheckTextMode(ggi_visual *vis,int cols,int rows,
				     int vcols,int vrows,
				     int fontsizex,int fontsizey,
				     ggi_graphtype mode,
				     ggi_mode *md,...)
{
	int rc;
	ggi_mode tim;
        
	DPRINT_CORE("ggiCheckTextMode(%p, %d, %d, %d, %d, %d, %d, 0x%x, %p) called\n",
		    vis, cols, rows, vcols, vrows, fontsizex, fontsizey,
		    mode, md);
	
	tim.frames    = GGI_AUTO;
	tim.visible.x = cols;
	tim.visible.y = rows;
	tim.virt.x    = vcols;
	tim.virt.y    = vrows;
	tim.size.x    = tim.size.y = 0;
	tim.graphtype = mode;
	tim.dpp.x     = fontsizex;
	tim.dpp.y     = fontsizey;

	rc = ggiCheckMode(vis,&tim);
	if (md) *md = tim;	/* give back the mode if asked for. */
	return rc;
}

/***********************/
/* set a graphics mode */
/***********************/
int ggiSetGraphMode(ggi_visual *vis,int xsize,int ysize,
		    int xvirtual,int yvirtual,ggi_graphtype type)
{
	ggi_mode tim;
	DPRINT_CORE("ggiSetGraphMode(%p, %d, %d, %d, %d, 0x%x) called\n",
		    vis, xsize, ysize, xvirtual, yvirtual, type);
	
	tim.frames    = GGI_AUTO;
	tim.visible.x = xsize;
	tim.visible.y = ysize;
	tim.virt.x    = xvirtual;
	tim.virt.y    = yvirtual;
	tim.size.x    = tim.size.y = 0;
	tim.graphtype = type;
	tim.dpp.x     = tim.dpp.y = 1;
	
	return ggiSetMode(vis,&tim);
}

/*************************/
/* check a graphics mode */
/*************************/
int ggiCheckGraphMode(ggi_visual *visual,int xsize,int ysize,
		      int xvirtual,int yvirtual,ggi_graphtype type,
		      ggi_mode *md,...)
{
	int rc;
	ggi_mode tim;
	
	DPRINT_CORE("ggiCheckGraphMode(%p, %d, %d, %d, %d, 0x%x, %p) called\n",
		    visual, xsize, ysize, xvirtual, yvirtual, type, md);
	
	tim.frames    = GGI_AUTO;
	tim.visible.x = xsize;
	tim.visible.y = ysize;
	tim.virt.x    = xvirtual;
	tim.virt.y    = yvirtual;
	tim.size.x    = tim.size.y = 0;
	tim.graphtype = type;
	tim.dpp.x     = tim.dpp.y = 1;

	rc = ggiCheckMode(visual,&tim);
	if (md) *md = tim;	/* give back the mode if asked for. */
	return rc;
}


/*******************/
/* print mode      */
/*******************/

#define _SCHEMECHAR(s)  ("TCKP"[(((s) >> GT_SCHEME_SHIFT) - 1) & 0x03])

#define _MODESTRING "%dx%d#%dx%d D%dx%d F%d [%c%d/%d]",m->visible.x,m->visible.y,m->virt.x,m->virt.y,m->dpp.x,m->dpp.y,m->frames,_SCHEMECHAR(GT_SCHEME(m->graphtype)),GT_DEPTH(m->graphtype),GT_SIZE(m->graphtype)


int ggiSPrintMode(char * s, ggi_mode * m)
{
	return sprintf(s,_MODESTRING);
}

int ggiFPrintMode(FILE * s, ggi_mode * m)
{
	return fprintf(s,_MODESTRING);
}


/*******************/
/* parse mode      */
/*******************/

/*
 * format = size virt dpp frames graphtype position.
 *
 * size = X 'x' Y    virt = '#' X 'x' Y    dpp = 'D' X 'x' Y
 * frames = 'F' num   position = '+'|'-' XOFF '+'|'-' YOFF
 * graphtype = '[' scheme depth '/' size ']'
 * scheme = 'C' | 'P' | 'K' | 'T'.
 *
 * Anything and Everything (!) can be omitted, all ommitted values
 * default to GGI_AUTO (and GT_AUTO for the graphtype).  
 * Whitespace is ignored.  Case insensitive.
 *
 * Please give full specs until the drivers understand GGI_AUTO:
 *    320x200#320x200D1x1F1[8]
 *
 * This is is a full (AFAIK) superset of what the X geometry parser supports. 
 *
 * Examples include:
 * 640x480           just a visible size
 * 640x480#640x960   same size, but double-height virtual screen
 * #1024x768         only virtual size defined
 *
 * 80x40[T]          (default-bitsized) text mode with 80x40 characters
 * #x100[T]          text mode with 100 virtual lines
 * 640x400[8]        640x400 at 8 bits per pixel
 * 640x400[GT_8BIT]  same as above, but palettized
 *
 * 320x200[C15]      320x200 with 32768 colors (hicolor)
 * 320x200[C/16]     320x200 with 16 bit pixels (also hicolor)
 * 320x200[C24/32]   320x200, 32 bit pixels, 16777216 colors (truecolor)
 * 320x200[GT_32BIT] same as above
 *
 * Currently unsupported by GGI, but prepared in the parser:
 * 640x400+2+2       position window at upper left corner 
 *                   like X does it.
 *
 * The only way of specifying GGI_AUTO is omitting the parameter;
 *
 * (You can do: x#x[]+-6 to specify only y position :-)
 *
 * Returncode:
 *  0 on success, i.e. the string was correct.
 *    ignored characters, like GT_ and a position information
 *    do not make it fail, and a missing ] from the bitdepth field
 *    is ignored, no failure
 * -1 if there is text that can not be parsed. 
 *    This text is printed to stderr.
 *    All paramters parsed so far are written into m!
 *
 * So m contains all parameters that have been successfully parsed.
 * For most applications there will be no need for testing parsemode
 * for failure.
 */


int ggiParseMode(const char * s, ggi_mode * m)
{
	int bitdepth=0;
	int negative=0;  /* negative flag for positions */

	int xposition=0;
	int yposition=0;
	
	DPRINT_CORE("ggiParseMode(%p, %p) called\n", s, m);

	*m = _ggiDefaultMode;

#define SKIPSPACE   while ( (*s!='\000') && (isspace((int)*s)) ) s++;
 
/* scan the integer from the string pointer s */
#define SCANINT(x)  SKIPSPACE;                 \
		    if (isdigit((int)*s)){          \
		       x=*s-'0';               \
	               s++;                    \
		       while (isdigit((int)*s)){    \
			  x = x*10+ ((int)*s) -'0'; \
			  s++;                 \
		       }                       \
		    }                          \
		    SKIPSPACE

	/* first number is visible-x: */
	SCANINT(m->visible.x);
	if (tolower((int)*s) == 'x') { /* now for the y */
		s++;
		SCANINT(m->visible.y);
	}
	if (*s == '#'){ /* virtual starts here */
		s++;
		SCANINT(m->virt.x);
		if (tolower((int)*s) == 'x') { /* now for the y */
			s++;
			SCANINT(m->virt.y);
		}
	}
	if (tolower((int)*s) == 'd') { /* dpp starts here */
		s++;
		SCANINT(m->dpp.x);
		if (tolower((int)*s) == 'x') { /* now for the y */
			s++;
			SCANINT(m->dpp.y);
		}
	}
	if (tolower((int)*s) == 'f') { /* frames starts here */
		s++;
		SCANINT(m->frames);
	}

	if (*s == '[') { /* graphtype starts here */
		s++;

#define CHECKGTMODE(str,len,val)  \
	if (strncasecmp(s,(str),(len)) == 0)  \
		{ m->graphtype = (val); s += len; }

		CHECKGTMODE("GT_1BIT]",   8,  GT_1BIT)   else
		CHECKGTMODE("GT_2BIT]",   8,  GT_2BIT)   else
		CHECKGTMODE("GT_4BIT]",   8,  GT_4BIT)   else
		CHECKGTMODE("GT_8BIT]",   8,  GT_8BIT)   else
		CHECKGTMODE("GT_15BIT]",  9,  GT_15BIT)  else
		CHECKGTMODE("GT_16BIT]",  9,  GT_16BIT)  else
		CHECKGTMODE("GT_24BIT]",  9,  GT_24BIT)  else
		CHECKGTMODE("GT_32BIT]",  9,  GT_32BIT)  else
		CHECKGTMODE("GT_TEXT16]", 10, GT_TEXT16) else
		CHECKGTMODE("GT_TEXT32]", 10, GT_TEXT32) else
		{
			/* scheme */
			if (tolower((int)*s) == 't') {  /* text mode */
				GT_SETSCHEME(m->graphtype, GT_TEXT);
				s++;
			} else
			if (tolower((int)*s) == 'p') {  /* palette mode */
				GT_SETSCHEME(m->graphtype, GT_PALETTE);
				s++;
			} else
			if (tolower((int)*s) == 'c') {  /* truecolor mode */
				GT_SETSCHEME(m->graphtype, GT_TRUECOLOR);
				s++;
			} else
			if (tolower((int)*s) == 'k') {  /* greyscale mode */
				GT_SETSCHEME(m->graphtype, GT_GREYSCALE);
				s++;
			}
			
			bitdepth = GT_AUTO;
			SCANINT(bitdepth);
			GT_SETDEPTH(m->graphtype, bitdepth);

			if (*s == '/') {
				s++;
				bitdepth = GT_AUTO;
				SCANINT(bitdepth);
				GT_SETSIZE(m->graphtype, bitdepth);
			}

			if (*s == ']') {
				s++;
			} else {
				fprintf(stderr,"ggiParseMode: warning: ] "
					"missing or bad graphtype\n.");
			}
		}
#undef CHECKGTMODE
	}

	if ((*s=='-') || (*s=='+')){ /* x position starts */
		if (*s=='-'){
			negative=1;
		}
		s++;
		SCANINT(xposition);
		if (negative){
			negative=0;
			xposition = - xposition;
		}
		fprintf(stderr,"X position %d ignored.\n",xposition);
	}
       	if ((*s=='-') || (*s=='+')){ /* y position starts */
		if (*s=='-'){
			negative=1;
		}
		s++;
		SCANINT(yposition);
		if (negative){
			negative=0;
			yposition = - yposition;
		}
		fprintf(stderr,"Y position %d ignored.\n",yposition);
	}
	if (*s !='\000'){
		fprintf(stderr,"trailing text %s ignored.\n"
			"parsed mode is ",s);
		ggiFPrintMode(stderr,m);
		fprintf(stderr,"\n");
		return -1;
	}
#undef SCANINT
	
	return 0;
}
