/* $Id: tt_region.c,v 1.4 1998/09/02 02:21:48 tristan Exp $ */
/* 
 *  Rectangle and region functions
 *
 *  Copyright (C) 1998 by Thomas Tanner. 
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "tt_region.h"
#include <string.h>
#include <malloc.h>

#define min(a,b)      ( (a) < (b) ? (a) : (b) )
#define max(a,b)      ( (a) > (b) ? (a) : (b) )
#define xchg(a,b) do { typeof (a) tmp; tmp = (a); (a) = (b); (b) = (tmp); } while(0)

/*************************************************************************/

rect_t	emptyRect = { 0, 0, 0, 0};

void	rectAssign(rect_t *r, int x1, int y1, int x2, int y2)
{
	if (x1 > x2)	xchg(x1, x2);
	if (y1 > y2)	xchg(y1, y2);
	r->left   = x1;
	r->top    = y1;
	r->right  = x2+1;
	r->bottom = y2+1;
}

void	rectCopy(rect_t *dest, rect_t *src)
{
 	memcpy(dest, src, sizeof(rect_t));
}

int	rectEmpty(rect_t *r)
{
	return (r->left == r->right || r->top  == r->bottom);
}

int	rectArea(rect_t *r)
{
	return (r->right - r->left) * (r->bottom - r->top);
}

int	rectEqual(rect_t *r1, rect_t *r2)
{
	if (rectEmpty(r1))
		return rectEmpty(r2);
	return (r1->left  == r2->left  && r1->top   == r2->top   &&
		r1->right == r2->right && r1->bottom== r2->bottom);
}

int	rectCompare(rect_t *r1, rect_t *r2)
{
	if (rectEmpty(r1)) {
		if (rectEmpty(r2))
			return 0;
		else
			return -1;
	}
	if (r1->left  == r2->left  && r1->top   == r2->top   &&
	    r1->right == r2->right && r1->bottom== r2->bottom)
		return 0;
	if (rectContains(r1, r2))
		return	1;
	return	-1;
}

int	rectContainsPoint(rect_t *r, int x, int y)
{
	return (x >= r->left && x < r->right &&
		y >= r->top  && y < r->bottom);
}

int	rectContains(rect_t *r, rect_t *sub)
{
	return (sub->left  >= r->left  && sub->top    >= r->top  &&
		sub->right <= r->right && sub->bottom <= r->bottom);
}

int	rectIntersects(rect_t *r1, rect_t *r2)
{
	return (r1->left  < r2->right && r1->top    < r2->bottom &&
		r1->right > r2->left  && r1->bottom > r2->top);
}

void	rectMove(rect_t *r, int dx, int dy)
{
	r->left   += dx;	r->top    += dy;
	r->right  += dx;	r->bottom += dy;
}

void	rectMoveTo(rect_t *r, int left, int top)
{
	r->right  += left - r->left;	r->left   = left;
	r->bottom += top - r->top;	r->top    = top;
}

int	rectResize(rect_t *r, int dx, int dy)
{
	r->left   -= dx;	r->top    -= dy;
	r->right  += dx;	r->bottom += dy;
	if (r->left > r->right || r->top > r->bottom) {
		rectCopy(r, &emptyRect);
		return 0;
	}
	return 1;
}

int	rectIntersect(rect_t *r1, rect_t *r2, rect_t *result)
{
	if (!rectIntersects(r1, r2)) {
		rectCopy(result, &emptyRect);
		return 0;
	}
	result->left   = max(r1->left,   r2->left);
	result->top    = max(r1->top,    r2->top);
	result->right  = min(r1->right,  r2->right);
	result->bottom = min(r1->bottom, r2->bottom);
	return 1;
}

void	rectUnion(rect_t *r1, rect_t *r2, rect_t *result)
{
	if (rectEmpty(r1))
		rectCopy(result, r2);
	else if (rectEmpty(r2)) 
		rectCopy(result, r1);
	else {
		result->left   = min(r1->left,   r2->left);
		result->top    = min(r1->top,    r2->top);
		result->right  = max(r1->right,  r2->right);
		result->bottom = max(r1->bottom, r2->bottom);
	}
}

/*************************************************************************/

int	regionInit(region_t *r, unsigned limit)
{
	r->size  = 0;
	r->limit = limit;
	if (limit) {
		r->rect = (rect_t*) malloc(limit * sizeof(rect_t));
		if (!r->rect) 
			return 0;
	}
	return 1;
}

void	regionFree(region_t *r)
{
	if (r->limit)
		free(r->rect);
	r->size  = 0;
	r->limit = 0;
}

int	regionAdjustSize(region_t *r)
{
	if (r->size == r->limit)
	  	return 1;
	if (r->size) {
		rect_t	*rects = (rect_t*) malloc(r->size * sizeof(rect_t));
		if (!rects) 
			return 0;
		memcpy(rects, r->rect, r->size * sizeof(rect_t));
		free(r->rect);
		r->rect  = rects;
	} else 
		free(r->rect);
	r->limit = r->size;
	return 1;
}

int	regionCopy(region_t *dest, region_t *src)
{
	dest->size  = src->size;
	dest->limit = src->limit;
	if (src->limit) {
		dest->rect = (rect_t*) malloc(src->limit * sizeof(rect_t));
		if (!dest->rect)
			return 0;
		memcpy(dest->rect, src->rect, src->size * sizeof(rect_t));
	}
	return 1;
}

int	regionEmpty(region_t *r)
{
	return !(r->size);
}

int	regionArea(region_t *r)
{
 	int	i, area;

	area = 0;
	for (i = 0; i < r->size; i++)
		area += rectArea(&r->rect[i]);
	return area;
}

int	regionEqual(region_t *r1, region_t *r2)
{
	return (regionCompare(r1, r2) == 0);
}

int	regionCompare(region_t *r1, region_t *r2)
{
	int	result;
 	region_t tmp;

	regionCopy(&tmp, r1);
	result = regionSubtract(&tmp, r2);
	regionFree(&tmp);
	return result;
}

void	regionBoundingBox(region_t *r, rect_t *bbox)
{
 	int	i;

	rectCopy(bbox, &emptyRect);
	for (i = 0; i < r->size; i++)
		rectUnion(bbox, &r->rect[i], bbox);
}

int	regionContainsPoint(region_t *r, int x, int y)
{
 	int	i;

	for (i = 0; i < r->size; i++)
		if (rectContainsPoint(&r->rect[i], x, y))
		  	return 1;
        return 0;
}

int	regionContainsRect(region_t *r, rect_t *rect)
{
	int	result;
 	region_t tmp;

	regionCopy(&tmp, r);
	result = (regionSubtractRect(&tmp, rect) != -1);
	regionFree(&tmp);
	return result;
}

int	regionContains(region_t *r, region_t *sub)
{
	int	result;
 	region_t tmp;

	regionCopy(&tmp, r);
	result = (regionSubtract(&tmp, sub) != -1);
	regionFree(&tmp);
	return result;
}

int	regionIntersectsRect(region_t *r, rect_t *rect)
{
 	int	i;

	for (i = 0; i < r->size; i++)
		if (rectIntersects(&r->rect[i], rect))
		  	return 1;
        return 0;
}

int	regionIntersects(region_t *r, region_t *sub)
{
 	int	i, j;

	for (i = 0; i < r->size; i++)
		for (j = 0; j < sub->size; j++)
			if (rectIntersects(&r->rect[i], &sub->rect[j]))
		  		return 1;
        return 0;
}

void	regionMove(region_t *r, int dx, int dy)
{
 	int	i;

	for (i = 0; i < r->size; i++)
		rectMove(&r->rect[i], dx, dy);
}


#define DELTA	8

static void addRect(region_t *r, rect_t *rect)
{
	if (r->size == r->limit) {
		rect_t	*rects;

		r->limit += DELTA;
		rects = (rect_t*) malloc(r->limit * sizeof(rect_t));
		if (r->limit != DELTA) {
			memcpy(rects, r->rect, r->size * sizeof(rect_t));
			free(r->rect);
		}
		r->rect = rects;
	}
	rectCopy(&r->rect[r->size], rect);
	r->size++;
}

static int	rectAppend(rect_t *r, rect_t *other)
{
	if (rectEmpty(r))
		return 0;
	if (rectEmpty(other))
		return 1;
	if (rectContains(r, other)) 
		return 1;
	if (rectContains(other, r)) {
		return -1;	/* tw */
		rectCopy(r, other);
		return 1;
	}
	if ((r->left != other->left || r->right != other->right ||
	     r->top > other->bottom || r->bottom < other->top)  &&
	    (r->top != other->top   || r->bottom != other->bottom ||
	     r->left > other->right || r->right < other->left))
		return 0;
	rectUnion(r, other, other);	/* tw */
	return -1;			/* tw */
	rectUnion(r, other, r);
	return 1;
}

static void appendRect(region_t *r, rect_t *rect)
{
 	int	i, ratmp;

search:
	for (i = 0; i < r->size; i++) {
		ratmp=rectAppend(rect, &r->rect[i]);
		if (ratmp==0) break;
		if (ratmp==1) {
		        r->size--;
	    		if (r->size)
	    			rectCopy(&r->rect[i],&r->rect[r->size]);
			goto search;
		}
		if (ratmp==-1)
			return;
	}
	addRect(r, rect);
}


void	regionAddRect(region_t *r, rect_t *rect)
{
 	if (rectEmpty(rect))
		return;
	if (regionIntersectsRect(r, rect))
		regionSubtractRect(r, rect);
	appendRect(r, rect);
}

void	regionAdd(region_t *r, region_t *other)
{
 	int	i;

	for (i = 0; i < other->size; i++)
		regionAddRect(r, &other->rect[i]);
}


static void intersectRect(region_t *r, rect_t *rect,
			 region_t *dest, int *intersect)
{
 	int 	i;
	rect_t	tmp;

	for (i = 0; i < r->size; i++)
		if (rectIntersects(&r->rect[i], rect)) {
			rectIntersect(&r->rect[i], rect, &tmp);
			appendRect(dest, &tmp);
			*intersect = 1;
		}
}


int	regionIntersectRect(region_t *r, rect_t *other)
{
	int	intersect;
 	region_t result;

	regionInit(&result, 0);
	intersect = 0;
	intersectRect(r, other, &result, &intersect);
	regionFree(r);
	memcpy(r, &result, sizeof(region_t));
	return intersect;
}

int	regionIntersect(region_t *r, region_t *other)
{
	int	i, intersect;
 	region_t result;

	regionInit(&result, 0);
	intersect = 0;
	for (i = 0; i < other->size; i++)
		intersectRect(r, &other->rect[i], &result, &intersect);
	regionFree(r);
	memcpy(r, &result, sizeof(region_t));
	return intersect;
}

#if 0
static void subtractRect(rect_t *rect, rect_t *other, region_t *dest)
{
 	rect_t	tmp;

 	if (rectContains(other, rect))
		return;
	if (rect->left < other->left) {
		rectAssign(&tmp, rect->left, max(rect->top, other->top),
 			   other->left-1, min(rect->bottom, other->bottom)-1);
		appendRect(dest, &tmp);
	}
	if (rect->top < other->top) {
		rectAssign(&tmp, rect->left, rect->top,
 			   rect->right-1, other->top-1);
		appendRect(dest, &tmp);
	}
	if (rect->right > other->right) {
		rectAssign(&tmp, other->right, max(rect->top, other->top),
 			    rect->right-1, min(rect->bottom, other->bottom)-1);
		appendRect(dest, &tmp);
	}
	if (rect->bottom > other->bottom) {
		rectAssign(&tmp, rect->left, other->bottom,
			   rect->right-1, rect->bottom-1);
		appendRect(dest, &tmp);
	}
}
#else
static void subtractRect(rect_t *rect, rect_t *other, region_t *dest)
{
 	rect_t	tmp;

 	if (rectContains(other, rect))
		return;
	if (rect->top < other->top) {
		rectAssign(&tmp, rect->left, rect->top,
 			   rect->right-1, other->top-1);
		appendRect(dest, &tmp);
	}
	if (rect->bottom > other->bottom) {
		rectAssign(&tmp, rect->left, other->bottom,
 			   rect->right-1, rect->bottom-1);
		appendRect(dest, &tmp);
	}
	if (rect->left < other->left) {
		rectAssign(&tmp, rect->left, max(rect->top, other->top),
 			    other->left-1, min(rect->bottom-1, other->bottom-1));
		appendRect(dest, &tmp);
	}
	if (rect->right > other->right) {
		rectAssign(&tmp, other->right, max(rect->top, other->top),
			   rect->right-1, min(rect->bottom-1, other->bottom-1));
		appendRect(dest, &tmp);
	}
}
#endif

static int subtract(region_t *r, region_t *other)
{
	int	i, j, result;
 	region_t sub, rest;
	rect_t tmp;

	regionInit(&sub, 0);
	regionInit(&rest, 0);
	result = 0;

	for (j = 0; j < other->size; j++) {
		i = 0;
		while (i < r->size) {
			if (rectIntersects(&r->rect[i], &other->rect[j])) {
				tmp=r->rect[i];
				r->size--;
				r->rect[i]=r->rect[r->size];
				subtractRect(&tmp, &other->rect[j],
					     r);
				continue;
			}
			i++;
		}
	}
	if (!regionEmpty(other))
		result = -1;
	else
	  	result = 1;
	regionFree(other);
	return result;
}

int	regionSubtractRect(region_t *r, rect_t *other)
{
 	region_t rest;

	regionInit(&rest, 1);
	addRect(&rest, other);
	return subtract(r, &rest);
}

int	regionSubtract(region_t *r, region_t *other)
{
 	region_t rest;

	regionCopy(&rest, other);
	return subtract(r, &rest);
}

static void optimiseChunk(region_t *r, rect_t *rect)
{
	/* Find horizontal possiblities */
	
}

void regionDefrag(region_t *r)
{
	int i, j, aligned;
	region_t work;
	rect_t tmp;

	regionInit(&work, 0);

	i = 0;
	while (i < r->size) {
		aligned=0;
		j = i+1;
		while (j < r->size) {
			if ( (r->rect[i].top == r->rect[j].bottom) ||
					(r->rect[i].bottom == r->rect[j].top) ||
					(r->rect[i].left == r->rect[j].right) ||
					(r->rect[i].right == r->rect[j].left) ) {
				addRect(&work, &r->rect[j]);
				r->size--;
				r->rect[j]=r->rect[r->size];
				aligned=1;
				continue;
			}
			j++;
		}
		if (aligned) {
			tmp=r->rect[i];
			r->size--;
			r->rect[i]=r->rect[r->size];
			optimiseChunk(&work, &tmp);
			continue;
		}
		i++;
	}
}

/*************************************************************************/

