#! /users1/bert/bin/wbuild
#
# Contains the source for the Frame widget
# Version 2.2.1 for FWF V4.0
#
# $Id: Frame.w,v 1.1 1996-09-25 09:23:44+02 mho Exp $

@class XfwfFrame (XfwfCommon) @file=Frame

@ The Frame widget is a composite widget that accepts just one child.
Its only purpose is to draw a frame around widgets that do not have a
frame of their own. It always uses the size of its child, with a
little extra for the frame. There are several types of frames
available, selectable with a resource.

Widget writers can also use the Frame class as a superclass for new
widgets.  The frame is drawn by the |expose| method (which must
therefore be called by subclasses). Its width is given by
|XtNframeWidth|, the appearance by |XtNframeType|. The possible types
are:

\item{|XfwfRaised|} Gives a beveled look. The top and left borders will
be lighter, the bottom and right sides darker.

\item{|XfwfSunken|} Just the opposite.

\item{|XfwfChiseled|} The border will look as if it was made with a
chisel.

\item{|XfwfLedged|} The border will be a ledge that juts out of the
background.

\item{|XfwfNoFrame|} The border will be invisible.

\item{|XfwfForeground|} The frame will be drawn in the highlight color.

\item{|XfwfBackground|} The frame will be drawn in the background color.

@public

@ The cursor (when not |None|) is the mouse cursor that is displayed
when the mouse is over the Board widget. The default value |None|
causes the parent's cursor to be used.

        @var Cursor cursor = None

@ The |XtNframeType| determines how the border looks.

        @var FrameType frameType = XfwfRaised

@ |XtNframeWidth| gives the width of the border. The default value
of 0 shows no border at all. The border is drawn {\it inside\/} the
widget. (See also |XtNouterOffset|.)

        @var Dimension frameWidth = 0

@ Normally, the border is draw along the outer edge of the widget, but
it can be moved inward. |XtNouterOffset| is the number of pixels
between the edge and the frame.

        @var Dimension outerOffset = 0

@ Between the frame and whatever is inside the widget, there is also
margin. By default, however, it is 0.

        @var Dimension innerHOffset = 0
        @var Dimension innerVOffset = 0

@ The colors of the top and bottom shadows can be set with the
resources |topShadowColor| and |bottomShadowColor|, but it is also
possible to use a stiple of foreground and background colors. This may
be preferable on workstations with limited or no color capabilities.
However, the easiest way (which is also the default) is to let the
widget determine its own shadow colors or stipples, based on the
widget's background color and the color capabilities of the screen.

The resource |shadowScheme| can be set to |XfwfColor|, |XfwfStipple|
or |XfwfAuto|. The converter for the shadow pixmap accepts the strings
|"stipple0"| through |"stipple8"|, which create pixmaps of the current
background and foreground colors, with |"stipple0"| entirely
background and |"stipple8"| entirely foreground. Setting pixmaps or
colors is only useful when |shadowScheme| is set to |XfwfStipple| or
|XfwfColor| respectively. |beNiceToColormap| is a hint for XfwfAuto
to use stipple pixmaps instead of computing a new color.

        @var ShadowScheme shadowScheme     = XfwfAuto
	@var Boolean      beNiceToColormap = True

@ The values of |topShadowColor| and |bottomShadowColor| are ignored by
the Frame widget as long as |shadowScheme| is not |XfwfColor|, but the
default values are computed nevertheless, since they are useful, e.g.,
when an icon uses `topShadowColor' and `bottomShadowColor' as dynamic
colors.

        @var Pixel  topShadowColor      = NULL
        @var Bitmap topShadowStipple    = NULL
        @var Pixel  bottomShadowColor   = NULL
        @var Bitmap bottomShadowStipple = NULL

@ There is another computed color/pixmap pair that may be used for
sunken frames. It is slightly darker than the background but lighter
than the bottom shadow.

        @var Pixel  sunkenShadowColor   = NULL
        @var Bitmap sunkenShadowStipple = NULL

@ The inherited resource |borderWidth| is given a default value of 0,
instead of 1.

        @var border_width = 0

@exports

@incl "XfwfDraw3D.h"

@ A number of new types is introduced by the Common class.  The
possible types of borders are enumerated in |Frame3dType| (see the
introduction).

        @type FrameType = XfwfShadowType

@ The shadow scheme can be used to choose colors, pixmaps or automatic
shadows.

        @type ShadowScheme = XfwfShadowScheme

@ The type |Bitmap| is an alias for |Pixmap|, but it is meant to
contain only bitmaps, i.e., pixmaps of depth one.

        @type Bitmap = Pixmap

@ The routine that draws the border is generally useful, so it is
exported. |t| is the thickness of the frame. The frame is drawn inside
the rectangle |(x, y, x+w-1, y+h-1)|.

@proc XfwfDrawFrame($, int x, int y, int w, int h, FrameType tp, int t,
		    GC lightgc, GC darkgc)
{
    if (XtIsSubclass($, xfwfFrameWidgetClass))
	XfwfDrawRectangle($, XtWindow($), lightgc, darkgc, $bordergc, $backgroundgc,
			  x, y, w, h, t, tp);
    else
	XfwfDrawRectangle($, XtWindow($), lightgc, darkgc, (GC)0, (GC)0,
			  x, y, w, h, t, tp);
}

@ It is useful for decorations to use the same GCs as the widget itself

@proc Boolean XfwfGetFrameGCs($, GC* light, GC *dark, GC *fg, GC *bg)
{
    if (XtIsSubclass($, xfwfFrameWidgetClass)) {
	*light = $lightgc;
	*dark  = $darkgc;
	*fg    = $bordergc;
	*bg    = $backgroundgc;
	return TRUE;
    } else 
        XtWarning("XfwfGetFrameGCs only applies to subclasses of Frame");
    return FALSE;
}

@private

@ The GC for drawing the light parts of the frame:

        @var GC lightgc

@ The GC for drawing the dark parts of the frame:

        @var GC darkgc

@ The GC for sunken shapes:

	@var GC sunkengc

@ The GC to erase drawings:

	@var GC backgroundgc

@ The |gray| bitmap is used on screens with insufficient colors to
simulate light and dark shadows. It will be created by the
|initialize| method, whether or not it is needed. Since it is but a
small bitmap, this can't hurt much.

        @var Pixmap gray

@ The |old_frame_type| variable is used by the |set_shadow| action
function to store the original frame type, when it is temporarily
changed.

        @var FrameType old_frame_type


@methods

@ |class_initialize| installs the type converters. The type converters
back to String are installed as a convenience, so resources can be
retrieved in readable form with |XtVaGetValues|.

@proc class_initialize
{
    static XtConvertArgRec screenArg[] = {
    {XtBaseOffset, (XtPointer)XtOffset(Widget, core.screen), sizeof(Screen*)}};

    XtSetTypeConverter(XtRString, XtRFrameType, cvtStringToFrameType,
                       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRFrameType, XtRString, cvtFrameTypeToString,
                       NULL, 0, XtCacheNone, NULL);

    XtAddConverter(XtRString, XtRBitmap, XmuCvtStringToBitmap,
                       screenArg, XtNumber(screenArg));

    XtSetTypeConverter(XtRString, XtRShadowScheme, cvtStringToShadowScheme,
                       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRShadowScheme, XtRString, cvtShadowSchemeToString,
                       NULL, 0, XtCacheNone, NULL);
}

@ The |initialize| method creates the sets the sizesand creates the
GC's. The setting of the window cursor is delegated to the |realize| method.

@proc initialize
{
    Dimension h_frame, w_frame;

    $old_frame_type = $frameType;
    /* Make sure the width and height are at least as large as the frame */
    w_frame = $total_frame_width($);
    h_frame = $total_frame_height($);
    if ($width  < 2 * w_frame) $width  = 2 * w_frame;
    if ($height < 2 * h_frame) $height = 2 * h_frame;
    /* create stipple pixmaps */
    $gray = XCreateBitmapFromData(
	XtDisplay($), RootWindowOfScreen(XtScreen($)),
        gray_bits, gray_width, gray_height);
    /* create GCs */
    $lightgc = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
			       1.25, XfwfLighter, &($topShadowColor),
			       &($background_pixel), &($topShadowStipple));
    $darkgc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
			       0.6, XfwfDarker, &($bottomShadowColor),
			       &($background_pixel), &($bottomShadowStipple));
    $sunkengc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				 0.85, XfwfGray, &($sunkenShadowColor),
				 &($background_pixel), &($sunkenShadowStipple));
    AllocBackgroundGC($);
}

@ The |destroy| method should free the allocated resources.

@proc destroy
{
    XfwfFreeShadowGC($, $shadowScheme, $lightgc,
		     &($topShadowColor), &($topShadowStipple));
    XfwfFreeShadowGC($, $shadowScheme, $darkgc,
		     &($bottomShadowColor), &($bottomShadowStipple));
    XfwfFreeShadowGC($, $shadowScheme, $sunkengc,
		     &($sunkenShadowColor), &($sunkenShadowStipple));
    XtReleaseGC($, $backgroundgc);
    XFreePixmap(XtDisplay($), $gray); $gray = (Pixmap)NULL;
}

@ The |realize| method uses the inherited method, but adds the cursor
attribute.

@proc realize
{
    *mask |= CWCursor;
    attributes->cursor = $cursor;
    #realize($, mask, attributes);
}

@ The |set_values| method has to create new GC's if the resources
change. It also makes sure that |frameWidth| is even if the frame type
is chiseled or ledged.

If the frame width was and is zero, nothing needs to be drawn,
regardless of the changes in other resources. Therefore, at the end
|need_redisplay| is set to False.

When the cursor changes, the |set_values| method uses the
|XDefineCursor| routine to set the attribute on the widget's window,
provided the widget is realized.

@proc set_values
{
    Boolean need_redisplay = False;

    if ($cursor != $old$cursor && XtIsRealized($))
        XDefineCursor(XtDisplay($), XtWindow($), $cursor);

    /* has something affected the current GCs? */
    if ($beNiceToColormap    != $old$beNiceToColormap
    ||  $shadowScheme        != $old$shadowScheme
    ||  $background_pixel    != $old$background_pixel
    ||  $topShadowColor      != $old$topShadowColor
    ||  $topShadowStipple    != $old$topShadowStipple
    ||  $bottomShadowColor   != $old$bottomShadowColor
    ||  $bottomShadowStipple != $old$bottomShadowStipple) {
	/* free old GCs */
	XfwfFreeShadowGC($, $old$shadowScheme, $old$lightgc,
			 &($old$topShadowColor), &($old$topShadowStipple));
	XfwfFreeShadowGC($, $old$shadowScheme, $old$darkgc,
			 &($old$bottomShadowColor), &($old$bottomShadowStipple));
	XfwfFreeShadowGC($, $shadowScheme, $sunkengc,
			 &($sunkenShadowColor), &($sunkenShadowStipple));
	XtReleaseGC($, $old$backgroundgc);
	/* create new GCs */
	$lightgc = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				   1.20, XfwfLighter, &($topShadowColor),
				   &($background_pixel), &($topShadowStipple));
	$darkgc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				   0.6, XfwfDarker, &($bottomShadowColor),
				   &($background_pixel), &($bottomShadowStipple));
	$sunkengc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				     0.85, XfwfGray, &($sunkenShadowColor),
				     &($background_pixel), &($sunkenShadowStipple));
	AllocBackgroundGC($);
        need_redisplay = True;
    }

    if ($outerOffset  != $old$outerOffset
    ||  $innerHOffset != $old$innerHOffset
    ||  $innerVOffset != $old$innerVOffset
    ||  $frameWidth   != $old$frameWidth)
        need_redisplay = True;

    if ($frameType != $old$frameType) {
        $old_frame_type = $old$frameType;
        need_redisplay = True;
    }

    return need_redisplay;
}

@ The |expose| method draws the frame, for which it uses the
|XfwfDrawFrame| routine. Before it calls the routine, it sets the clip
region. Afterwards, the clip region is reset, because we don't know
which other widgets share the same GC's. As explained in {\em X
Toolkit Intrinsics Programming Manual} (Nye \& O'Reilly, Motif
Edition, 1990, p~223), the test for |XtIsRealized| is there for the
unlikely case when an expose event arrives after the widget has been
destroyed or unrealized.

@proc _expose
{
    Position x, y;
    Dimension w, h;

    if (! XtIsRealized($)) return;
    if (region != NULL) {
        XSetRegion(XtDisplay($), $lightgc, region);
        XSetRegion(XtDisplay($), $darkgc, region);
        XSetRegion(XtDisplay($), $backgroundgc, region);
        XSetRegion(XtDisplay($), $sunkengc, region);
    }
    #compute_inside($, &x, &y, &w, &h);
    XfwfDrawFrame($, x + $outerOffset, y + $outerOffset, w - 2*$outerOffset,
		  h - 2*$outerOffset, $frameType, $frameWidth, $lightgc, $darkgc);
    if (region != NULL) {
        XSetClipMask(XtDisplay($), $lightgc, None);
        XSetClipMask(XtDisplay($), $darkgc, None);
        XSetClipMask(XtDisplay($), $backgroundgc, None);
        XSetClipMask(XtDisplay($), $sunkengc, None);
    }
    #_expose($, event, region);
}

@ The method |compute_inside| is re-defined. The method now returns
the area inside the frame. It calls the superclass's method and then
decreases the area by the width of the frame.

@proc compute_inside
{
    #compute_inside($, x, y, w, h);
    *x += $outerOffset + $frameWidth + $innerHOffset;
    *y += $outerOffset + $frameWidth + $innerVOffset;
    *w -= 2 * ($outerOffset + $frameWidth + $innerHOffset);
    *h -= 2 * ($outerOffset + $frameWidth + $innerVOffset);
}

@ The methods |total_frame_width| and |total_frame_height| are overridden,
because the Frame widget draws an additional frame. The frame that the
Common widget draws is added to the shadow frame of the Frame widget.

@proc Dimension total_frame_width
{
    return #total_frame_width($) + $outerOffset + $frameWidth + $innerHOffset ;
}

@proc Dimension total_frame_height
{
    return #total_frame_height($) + $outerOffset + $frameWidth + $innerVOffset ;
}

@ A Frame widget passes its parent's inquiry on to its (presumably)
single child. If there is no child, the proposal is accepted.
The border and position proposals are always accepted, the stacking
order and size are left to the child to decide.

@proc query_geometry
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Dimension w, h;

    if ($num_children == 0) return XtGeometryYes;

    /* We're only interested in size and stacking order */
    reply->request_mode =
        (CWWidth | CWHeight | CWStackMode) & request->request_mode;

    /* If nothing of interest is left, we can return immediately */
    if (reply->request_mode == 0)
        return XtGeometryYes;

    /* Prepare a request to the child */
    w = 2 * ($outerOffset + $frameWidth + $innerHOffset);
    h = 2 * ($outerOffset + $frameWidth + $innerVOffset);
    request2.request_mode = reply->request_mode;
    request2.width = request->width - w;
    request2.height = request->height - h;
    request2.sibling = request->sibling;
    request2.stack_mode = request->stack_mode;

    result = XtQueryGeometry($children[0], &request2, &reply2);

    /* If the child accepted its proposal, we accept ours */
    if (result == XtGeometryYes) return XtGeometryYes;

    /* If the child doesn't want any change, we don't want any, either */
    if (result == XtGeometryNo) return XtGeometryNo;

    /* Otherwise, ignore everything but size and stacking order */
    reply->request_mode &= reply2.request_mode;
    if (reply->request_mode == 0) return XtGeometryYes;

    reply->width = reply2.width + w;
    reply->height = reply2.height + h;
    reply->sibling = reply2.sibling;
    reply->stack_mode = reply2.stack_mode;
    return XtGeometryAlmost;
}

@ Requests by the child to be resized are passed on to the parent. If
the parent replies with |XtGeometryYes|, the change is accepted and
(if not |XtCWQueryOnly|) already done. In that case the Frame widget
accepts its child's request. If the parent replies with
|XtGeometryNo|, the change is denied and the denial is passed on. If
the parent replies with a different geometry, the geometry is passed
on, after compensating for the frame width.

Requests for anything other than width or height are always granted.

@proc geometry_manager
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Position x, y;
    Dimension w, h, extraw, extrah;

    $compute_inside($, &x, &y, &w, &h);
    if (! (request->request_mode & (CWWidth|CWHeight))) return XtGeometryYes;
    extraw = $width - w;
    extrah = $height - h;
    request2.request_mode = request->request_mode & (CWWidth|CWHeight);
    request2.width = request->width + extraw;
    request2.height = request->height + extrah;
    result = XtMakeGeometryRequest($, &request2, &reply2);
    if (result == XtGeometryNo) return XtGeometryNo;
    if (result == XtGeometryYes) return XtGeometryYes;
    reply->request_mode = reply2.request_mode & (CWWidth|CWHeight);
    reply->width = reply2.width - extraw;
    reply->height = reply2.height - extrah;
    return XtGeometryAlmost;
}

@ The |resize| method doesn't have to recompute any private variables,
but it passes on the resize message to its child, after decreasing the
area by the amount needed for the frame.

@proc resize
{
    Position x, y;
    Dimension w, h;
    Widget child;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];
    w -= 2 * $child$border_width;
    h -= 2 * $child$border_width;
    XtConfigureWidget(child, x, y, w, h, $child$border_width);
}

@ The |change_managed| method is called when a child becomes managed
or unmanaged. The task of the routine is enforcing the layout policy,
which in this case consists of trying to take on the size of the child
or otherwise resize the child to fit inside the frame.
If the parent of the Frame widget doesn't allow the Frame widget to be
resized, the child of the Frame widget will be resized instead.

@proc change_managed
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Widget child;
    Position x, y;
    Dimension w, h;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];
    request2.request_mode = CWWidth | CWHeight;
    request2.width = $child$width + $width - w;
    request2.height = $child$height + $height - h;
    result = XtMakeGeometryRequest($, &request2, &reply2);
    $compute_inside($, &x, &y, &w, &h);
    w -= 2 * $child$border_width;
    h -= 2 * $child$border_width;
    XtConfigureWidget(child, x, y, w, h, $child$border_width);
}

@actions

@ Although the Frame widget has no translations, one action is
defined, that may be of use to subclasses. The action function
|set_shadow| can be used to change the shadow frame. It has zero or
one argument. Without an argument, it resets the shadow to its
original type; with an argument, it sets the shadow to the type given
in the argument.

Warning: the function uses the |XfwfDrawFrame| routine to draw the
frames directly, instead of calling the |expose| or even |set_values|
methods.  Any subclass that defines behaviour that depends on knowing
the frame type, will have to redefine the |set_shadow| action.

@proc set_shadow
{
    FrameType f = $old_frame_type; /* Reset to old style */

    if (*num_params != 0)
	switch (*(params[0])) {
	case 'f': f = XfwfForeground; break;
	case 'b': f = XfwfBackground; break;
	case 'r': f = XfwfRaised; break;
	case 's': f = XfwfSunken; break;
	case 'c': f = XfwfChiseled; break;
	case 'l': f = XfwfLedged; break;
	case 'n': f = XfwfNone; break;
	default:  XtWarning("Unknown frame type in set_shadow action");
	}
    if ($frameType != f) {
	$old_frame_type = $frameType;
	$frameType = f;
	$expose($, event, NULL);
    }
}

@utilities

@ The converters use the following macro.

@def done(type, value) =
  do {
      if (to->addr != NULL) {
          if (to->size < sizeof(type)) {
              to->size = sizeof(type);
              return False;
          }
          *(type*)(to->addr) = (value);
      } else {
          static type static_val;
          static_val = (value);
          to->addr = (XtPointer)&static_val;
      }
      to->size = sizeof(type);
      return True;
  } while (0)

@exports

@ |cvtStringToFrameType| converts the strings `raised', `sunken',
`chiseled' and `ledged'. Case doesn't matter.

@proc Boolean cvtStringToFrameType(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    String s = (String) from->addr;

    if (*num_args != 0)
        XtAppErrorMsg(XtDisplayToApplicationContext(display),
                      "cvtStringToFrameType", "wrongParameters",
                      "XtToolkitError",
                      "String to frame type conversion needs no arguments",
                      (String*) NULL, (Cardinal*) NULL);

    if (XmuCompareISOLatin1(s, "foreground")== 0) done(FrameType, XfwfForeground);
    if (XmuCompareISOLatin1(s, "background") == 0) done(FrameType,  XfwfBackground);
    if (XmuCompareISOLatin1(s, "raised") == 0) done(FrameType, XfwfRaised);
    if (XmuCompareISOLatin1(s, "sunken") == 0) done(FrameType, XfwfSunken);
    if (XmuCompareISOLatin1(s, "chiseled") == 0) done(FrameType, XfwfChiseled);
    if (XmuCompareISOLatin1(s, "ledged") == 0) done(FrameType, XfwfLedged);
    if (XmuCompareISOLatin1(s, "none") == 0) done(FrameType, XfwfNoFrame);
    XtDisplayStringConversionWarning(display, s, XtRFrameType);
    done(FrameType, XfwfRaised);
}

@proc Boolean cvtFrameTypeToString(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
        XtAppErrorMsg(XtDisplayToApplicationContext(display),
                      "cvtFrameTypeToString", "wrongParameters",
                      "XtToolkitError",
                      "Fframe type to String conversion needs no arguments",
                      (String*) NULL, (Cardinal*) NULL);
    switch (*(FrameType*)from->addr) {
    case XfwfForeground: done(String, "foreground");
    case XfwfBackground: done(String, "background");
    case XfwfRaised: done(String, "raised");
    case XfwfSunken: done(String, "sunken");
    case XfwfChiseled: done(String, "chiseled");
    case XfwfLedged: done(String, "ledged");
    case XfwfNoFrame: done(String, "none");
    default: XtError("Illegal FrameType");
    }
    return FALSE;
}

@ The converter |cvtStringToShadowScheme| converts strings `color',
`auto' and `stipple' to |XfwfColor|, |XfwfAuto| and |XfwfStipple|.

@proc Boolean cvtStringToShadowScheme(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    String s = (String) from->addr;

    if (*num_args != 0)
        XtAppErrorMsg(XtDisplayToApplicationContext(display),
                      "cvtStringToShadowScheme", "wrongParameters",
                      "XtToolkitError",
                      "String to shadow scheme conversion needs no arguments",
                      (String*) NULL, (Cardinal*) NULL);

    if (XmuCompareISOLatin1(s, "auto")==0) done(ShadowScheme, XfwfAuto);
    if (XmuCompareISOLatin1(s, "color")==0) done(ShadowScheme, XfwfColor);
    if (XmuCompareISOLatin1(s, "stipple")==0) done(ShadowScheme, XfwfStipple);
    XtDisplayStringConversionWarning(display, s, XtRShadowScheme);
    done(ShadowScheme, XfwfAuto);
}

@proc Boolean cvtShadowSchemeToString(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
        XtAppErrorMsg(XtDisplayToApplicationContext(display),
                      "cvtShadowSchemeToString", "wrongParameters",
                      "XtToolkitError",
                      "Shadow scheme to String conversion needs no arguments",
                      (String*) NULL, (Cardinal*) NULL);

    switch (*(ShadowScheme*)from->addr) {
    case XfwfAuto: done(String, "auto");
    case XfwfColor: done(String, "color");
    case XfwfStipple: done(String, "stipple");
    default: XtError("Illegal ShadowScheme");
    }
    return FALSE;
}

@utilities

@ A utility function to allocate a GC for the background color

@proc AllocBackgroundGC($)
{
    XtGCMask	valuemask;
    XGCValues	myXGCV;

    valuemask = GCForeground;
    myXGCV.foreground = $background_pixel;
    $backgroundgc = XtGetGC($, valuemask, &myXGCV);
}

@imports

@incl <string.h>
@incl <stdio.h>
@incl <X11/Xmu/Converters.h>
@incl <X11/Xmu/CharSet.h>

@ The stipple for the shadows is loaded from a bitmap file.

@incl <X11/bitmaps/gray>

