/*
 * Copyright (c) 1997,1998 Motonori Nakamura <motonori@econ.kyoto-u.ac.jp>
 * Copyright (c) 1997,1998 WIDE Project
 * Copyright (c) 1997,1998 Kyoto University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 4. Neither the name of the Project, the University nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char *_id_ = "$Id: smtp.c,v 1.117 1998/09/30 14:04:27 motonori Exp $";
#endif

# include "common.h"
# include "extern.h"
# include "smtp.h"

static	struct connection connwork[FD_SETSIZE];
static	int num_trans, num_sock, max_sock;
static	jmp_buf jmp_env;
static	time_t Now;
static	int FirstRead, FirstWrite;
static	int Terminate, NewTransCheck, NeedTrimDomain;

static void
sig_alarm()
{
	errno = ETIMEDOUT;
	if (cnf.debug & DEBUG_EVENT)
	log(LOG_DEBUG, "alarm clock");
}

static void
sig_int()
{
	log(LOG_INFO, "SIGINT received: internal states dumped");
	dump_internal();
}

static void
sig_hup()
{
	log(LOG_INFO, "SIGHUP received: graceful termination");
	forced_terminate();
}

static void
sig_term()
{
	log(LOG_INFO, "SIGTERM received: semi graceful termination");
	markstatus(env.rcpt_list, SMTP_TEMPFAIL(51),
		"Aborted by INT signal", 1);
	longjmp(jmp_env, 1);
}

void
deliver()
{
#if 0
	int s;

	while((s = dup(0)) < 253);
	log(LOG_DEBUG, "descriptor reserved till %d", s);
#endif

	log(LOG_INFO, "from=%s size=%d nrcpt=%d msgid=%s",
		env.sender, env.realsize, sti.nrcpt,
		(env.mid == NULL)?"none":env.mid);

	bzero(connwork, sizeof(connwork));

	sti.time_start = time(NULL); 
    	num_trans = 0;
    	num_sock = 0;
	max_sock = cnf.sd_max;

	resource_usage("lmtp data done");

	/* load status of transaction of previous delivery */

	/* pre-getting MX RRs in parallel */
	pregetmx(domain_list);
	resource_usage("pre-getting MX done");

	/* generate MX list for each domain */
	if (getmxlist(domain_list) < 0)
	{
		return;
	}
	resource_usage("get MX done");

	sti.time_mxgot = time(NULL); 

	/* mark recipient for unresolved MX */
	finalstatus();
	resource_usage("finalstatus done");

	if (cnf.debug & DEBUG_QUERYONLY)	/* no SMTP processing */
	{
		sti.time_almostfinish = sti.time_finish = time(NULL); 
		return;
	}

	Now = time(NULL);
	Terminate = 0;
	NewTransCheck = 1;
	NeedTrimDomain = 0;

	/* process delivery with SMTP */
	sti.ntrans = 0;
	sti.noktrans = 0;
	sti.max_delay = 0;
	signal(SIGALRM, sig_alarm);
	signal(SIGINT,  sig_int);
	signal(SIGTERM, sig_term);
	signal(SIGHUP,  sig_hup);
	signal(SIGPIPE, SIG_IGN);
	if (setjmp(jmp_env) == 0)
		smtp();
	signal(SIGALRM, SIG_IGN);
	signal(SIGINT,  SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	signal(SIGHUP,  SIG_IGN);

	sti.time_finish = time(NULL); 
	if (sti.time_almostfinish == 0)
		sti.time_almostfinish = sti.time_finish;
	resource_usage("smtp all done");

	/* set final status to recipients for response */
	finalstatus();

	/* store status of transaction for next delivery */

	resource_usage("finalstatus done");
}

static int smtp_connected(), smtp_greeted(), smtp_flush();
static int smtp_helo_r(), smtp_helo_r();
static int smtp_mail_r(), smtp_rcpt_r(), smtp_rcpt_s();
static int smtp_data_r(), smtp_data_body(), smtp_data_t();
static int smtp_rset_r(), smtp_quit_r();

static struct	{
	int (*recv)();
	int (*send)();
	time_t tout;
	char *name;
} state_tbl[] = {
      /* RECV FUNC	SEND FUNC	  TIMEOUT	STATE NAME */
	{NULL,		NULL,		 0 MINUTES,	"CLOSED"},	/* 0 */
	{NULL,		smtp_connected,  5 MINUTES,	"CONNECTING"},	/* 1 */
	{smtp_greeted,	smtp_flush,	 5 MINUTES,	"CONNECTED"},	/* 2 */
	{smtp_helo_r,	smtp_flush,	 5 MINUTES, 	"EHLOSENT"},	/* 3 */
	{smtp_helo_r,	smtp_flush,	 5 MINUTES,	"HELOSENT"},	/* 4 */
	{smtp_mail_r,	smtp_flush,	10 MINUTES,	"MAILSENT"},	/* 5 */
	{smtp_rcpt_r,	smtp_rcpt_s,	60 MINUTES,	"RCPTSENT"},	/* 6 */
	{smtp_data_r,	smtp_flush,	 5 MINUTES,	"DATASENT"},	/* 7 */
	{NULL,		smtp_data_body,	20 MINUTES,	"DATABODY"},	/* 8 */
	{smtp_data_t,	smtp_flush,	60 MINUTES,	"DTERMSENT"},	/* 9 */
	{NULL,		NULL,		 5 MINUTES,	"SENT"},	/* 10 */
	{smtp_rset_r,	smtp_flush,	 5 MINUTES,	"RSETSENT"},	/* 11 */
	{smtp_quit_r,	smtp_flush,	 2 MINUTES,	"QUITSENT"},	/* 12 */
	{smtp_quit_r,	smtp_flush,	 2 MINUTES,	"QUITREOPEN"},	/* 13 */
	{smtp_quit_r,	smtp_flush,	 2 MINUTES,	"FAILQUIT"},	/* 14 */
	{NULL,		NULL,		 0 MINUTES,	"FAIL"},	/* 15 */
};
#define STATE(n)	state_tbl[n].name

void
set_timeout(opt)
char *opt;
{
	char *p;
	time_t t;
	int minus = 0;

	if (opt == NULL)
		return;
	if ((p = strchr(opt, '=')) == NULL)
	{
		/* XXX LOG: bad format */
		return;
	}
	p++;
	if (*p == '-')
	{
		minus = 1;
		p++;
	}
	t = 0;
	while (isdigit(*p))
	{
		t = t * 10 + (*p++ - '0');
		if (t >= 10000)
		{
			/* XXX LOG: too big */
			return;
		}
	}
	/* unit selection */
	switch (*p)
	{
	    case 's':
	    case 'S':
		/* seconds, do nothing */
		break;
	    case '\0':	/* default is minutes */
	    case 'm':
	    case 'M':
		/* minutes */
		t *= 60;
		break;
	    case 'h':
	    case 'H':
		/* hours */
		t *= 3600;
		break;
	    default:
		/* XXX LOG: bad format */
		return;
	}
	if (minus)
		t = -t;
	
	if        (strncasecmp(opt, "total=", 6) == 0) {
		cnf.t_timeout = t;
	} else if (strncasecmp(opt, "connect=", 8) == 0) {
		state_tbl[STAT_CONNECTING].tout = t;
	} else if (strncasecmp(opt, "greet=", 6) == 0) {
		state_tbl[STAT_CONNECTED].tout = t;
	} else if (strncasecmp(opt, "helo=", 5) == 0) {
		state_tbl[STAT_EHLOSENT].tout = t;
		state_tbl[STAT_HELOSENT].tout = t;
	} else if (strncasecmp(opt, "mail=", 5) == 0) {
		state_tbl[STAT_MAILSENT].tout = t;
	} else if (strncasecmp(opt, "rcpt=", 5) == 0) {
		state_tbl[STAT_RCPTSENT].tout = t;
	} else if (strncasecmp(opt, "data=", 5) == 0) {
		state_tbl[STAT_DATASENT].tout = t;
	} else if (strncasecmp(opt, "body=", 5) == 0) {
		state_tbl[STAT_DATABODY].tout = t;
	} else if (strncasecmp(opt, "term=", 5) == 0) {
		state_tbl[STAT_DTERMSENT].tout = t;
	} else if (strncasecmp(opt, "cache=", 6) == 0) {
		state_tbl[STAT_SENT].tout = t;
	} else if (strncasecmp(opt, "rset=", 5) == 0) {
		state_tbl[STAT_RSETSENT].tout = t;
	} else if (strncasecmp(opt, "quit=", 5) == 0) {
		state_tbl[STAT_QUITSENT].tout = t;
		state_tbl[STAT_QUITREOPEN].tout = t;
		state_tbl[STAT_FAILQUIT].tout = t;
	} else {
		/* XXX LOG: no such category */
	}
}

static void
smtp()
{
	struct recipient *rcptp;
	struct domain *domp, *wdomp;
	struct timeval timeout;
	fd_set r_fds, w_fds;
	int i, fds, rc, in_use;
	int all_done = 0;
	int bad_state = 0;
	int true_terminate;
	char *msg;

	/* initialize work pointers */
	for (rcptp = env.rcpt_list; rcptp != NULL; rcptp = rcptp->next)
	{
		if (rcptp->stat == RCPT_DONE)
		{
			msg = NULL;
			if (rcptp->domain != NULL)
				msg = rcptp->domain->response;
			/* DNS lookup failure or something else */
			if (msg == NULL)
				msg = "may be DNS problem";
			markstatus(rcptp, SMTP_TEMPFAIL(52), msg, 0);
			continue;
		}
		if (rcptp->domain == NULL)
		{
			rcptp->stat = RCPT_DONE;
			log(LOG_NOTICE, "no domain for %s", rcptp->address);
			continue;
		}
		if (rcptp->domain->firstmx == NULL)
		{
			rcptp->stat = RCPT_DONE;
			log(LOG_NOTICE, "no MX for %s", rcptp->address);
			continue;
		}
	}
	env.work_dom_ptr = NULL;
	env.work_domain = NULL;
	wdomp = NULL;
	/* pick up domains which have recipients */
	for (domp = domain_list; domp != NULL; domp = domp->next)
	{
		if (domp->rcpt_top == NULL)
			continue;
		if (cnf.debug & DEBUG_TRANS)
		log(LOG_DEBUG, "pick up domain: %s", domp->name);
		if (wdomp == NULL)
			env.work_domain = domp;
		else
			wdomp->workchain = domp;
		domp->workchain = NULL;
		domp->curmx = domp->firstmx;
/*
		trim_rcpts(domp);
*/
		wdomp = domp;
	}

	/*** event loop ***/
	for (;;)
	{
		if (NeedTrimDomain && trim_domain())
		{
			/* all recipients had been processed */
			if (cnf.debug & DEBUG_CONNECT)
			log(LOG_DEBUG, "closing all sockets");
			all_done = 1;
			for (i = 0; i < cnf.sd_max; i++)
			{
				if (connwork[i].host == NULL)
					continue;
				if (connwork[i].host->state == STAT_SENT)
					smtp_quit_s(&connwork[i]);
			}
			goto closewait;
		}

		if (Terminate)
			goto closewait;

		/* close connections if no free slot for new transaction */
		if (cnf.smtp_reuse)
		{
			in_use = 0;
			for (i = 0; i < cnf.sd_max; i++)
			{
				if (connwork[i].host == NULL)	/* XXX */
					continue;
				if (connwork[i].host->state == STAT_CLOSED)
					continue;
				in_use++;
			}

			if (cnf.debug & DEBUG_TRANS)
			log(LOG_DEBUG, "in_use=%d, max_sock=%d, min_sock=%d",
				in_use, max_sock, sti.minsock);

			if (in_use > (max_sock - sti.minsock))
			{
				/* XXX LRU required ? */
				for (i = 0; i < cnf.sd_max; i++)
				{
					if (connwork[i].rcpt != NULL)
						continue;
					if (connwork[i].host == NULL)
						continue;
					if (connwork[i].host->state != STAT_SENT)
						continue;
					if (cnf.debug & DEBUG_CONNECT)
					log(LOG_DEBUG, "%d/%d: closing idle socket",
						connwork[i].seq,
						connwork[i].socket);
					smtp_quit_s(&connwork[i]);
					/* break; */
				}
			}
		}
		else if (num_sock > (max_sock - sti.minsock))
		{
			for (i = 0; i < cnf.sd_max; i++)
			{
				if (connwork[i].host == NULL)
					break;
				if (connwork[i].host->state != STAT_SENT)
					continue;
				if (!connwork[i].reuse)
					break;
				if (cnf.debug & DEBUG_CONNECT)
				log(LOG_DEBUG, "%d/%d: closing idle socket",
					connwork[i].seq,
					connwork[i].socket);
				smtp_quit_s(&connwork[i]);
			}
		}

		/* create new transaction if there is a free port */
		while (NewTransCheck
		    && num_trans <= (max_sock - sti.minsock)
		    && num_sock <= (max_sock - sti.minsock))
		{
#if 1
			if (cnf.debug & DEBUG_TRANS)
			log(LOG_DEBUG, "new transaction: trans=%d, sock=%d",
				num_trans, num_sock);
#endif

			if (new_transaction() < 0)
				break;
		}



  closewait:
		fds = -1;
		FD_ZERO(&r_fds);
		FD_ZERO(&w_fds);

		for (i = 0; i < cnf.sd_max; i++)
		{
			if (connwork[i].host == NULL)
				continue;
			switch (connwork[i].host->state)
			{
			  case STAT_CLOSED:
			  case STAT_FAIL:
				continue;
			}
			if (connwork[i].socket > fds)
				fds = connwork[i].socket;
			FD_SET(connwork[i].socket, &r_fds);
			if (connwork[i].host->state == STAT_CONNECTING
			 || connwork[i].outlen > 0)
				FD_SET(connwork[i].socket, &w_fds);
		}

		if (all_done && fds < 0)
		{
			if (cnf.debug & DEBUG_CONNECT)
			log(LOG_DEBUG, "all sockets are closed");
#if 0
			dump_internal();
#endif
			return;
		}

		if (fds < 0)	/* XXX */
		{
			log(LOG_INFO, "No active sockets");
			if (bad_state++ > 5)
			{
				markstatus(env.rcpt_list, SMTP_TEMPFAIL(51),
					"No active sockets", 1);
				dump_internal();
				return;
			}
			if (bad_state > 1)
				sleep(3);
			rc = 0;	/* XXX */
		}
		else
		{
			timeout.tv_usec = 0;
			timeout.tv_sec = 10;
			rc = select(fds+1, &r_fds, &w_fds, NULL, &timeout);
		}

		Now = time(NULL);

		true_terminate = 0;
		if (cnf.t_timeout > 0
		 && Now - sti.time_start
		  > cnf.t_timeout + state_tbl[STAT_QUITSENT].tout)
		{
			/* true time up */
			true_terminate = 1;
		}

		if (!Terminate && cnf.t_timeout > 0
		 && Now - sti.time_start > cnf.t_timeout)
		{
			/* time up */
			forced_terminate();
			log(LOG_INFO, "total processing timed out (%d sec)",
				cnf.t_timeout);
		}
		if (rc < 0)
		{
			log(LOG_INFO, "select returns %d (fds=%d, errno=%d)",
				rc, fds, errno);
			sleep(1);
		}

		process_connection(&r_fds, &w_fds, (rc<=0), true_terminate);

		if (all_done)
			goto closewait;
	}
}

static int
new_transaction()
{
	struct recipient *rcptp;
	struct domain *domp;
	struct mx *lastgoodmx;
	struct connection *conn;
	int rc;

	if (env.work_dom_ptr == NULL)
		env.work_dom_ptr = env.work_domain;
	for (domp = env.work_dom_ptr; domp != NULL; domp = domp->workchain)
	{ 
#if 1
		if (cnf.debug & DEBUG_TRANS)
		log(LOG_DEBUG, "checking domain: %s", domp->name);
#endif
		/* skip bad MXs */
		lastgoodmx = NULL;
		while (domp->curmx != NULL
		    && (domp->curmx->host == NULL
		     || (domp->curmx->host != NULL
		     && domp->curmx->host->state == STAT_FAIL)))
		{
			if (domp->curmx->host == NULL)
				log(LOG_INFO, "skipping bad MX: %s for %s",
					domp->curmx->name, domp->name);
			else
				lastgoodmx = domp->curmx;
			domp->curmx = domp->curmx->next;
		}
		if (domp->curmx == NULL)
		{
			char *msg = domp->response;

			if (msg == NULL)
				msg = "No MX available";
			rcptp = domp->rcpt_top;
			domp->curmx = lastgoodmx; /* just for markstatus */
			while (rcptp != NULL)
			{
				if (rcptp->stat == RCPT_NOOP
				 || rcptp->stat == RCPT_RETRY)
				{
					markstatus(rcptp, SMTP_TEMPFAIL(52),
						msg, 0);
					rcptp->stat = RCPT_RETRY;
				}
				rcptp = rcptp->dom_chain;
			}
			domp->curmx = NULL;
			/* no MX for the domain */
			trim_rcpts(domp);
			continue;
		}
		if (domp->curmx->host->state != STAT_CLOSED
		 && domp->curmx->host->state != STAT_SENT)
		{
			/* working connection to the host exists */
			continue;
		}

		rcptp = domp->rcpt_top;
		while (rcptp != NULL)
		{
			if (rcptp->stat == RCPT_WORK)
				break;	/* working; try next domain */
			if (rcptp->stat == RCPT_NOOP
			 || (rcptp->stat == RCPT_RETRY
			  && rcptp->domain->curmx != NULL))
			{
				/* found ready-to-process recipient */

				goto found;
			}
			rcptp = rcptp->dom_chain;
		}
	}
#if 1
	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "no new transaction");
#endif
	NewTransCheck = 0;
	return -1;	/* no new transaction */

  found:
#if 1
	env.work_dom_ptr = domp;		/* start point at next time */
#else
	env.work_dom_ptr = domp->workchain;	/* start point at next time */
#endif
	conn = open_transaction(rcptp);

	if (conn == NULL)
		return -1;	/* XXX */
#if 1
	if (conn->host->addr == NULL)
	{
		char msgbuf[MAXLINE];

		snprintf(msgbuf, sizeof(msgbuf),
			"No address known for %s", conn->host->name);
		log(LOG_INFO, msgbuf, conn->host->name);
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52), msgbuf, 1);
		close_transaction(conn, 1);
		return 0;
	}
#endif
	if (conn->host->state == STAT_SENT)
	{
		smtp_rset_s(conn);
	}
	else
	{
		rc = smtp_connect(conn);
		if (rc == -1)
		{
			/* failure, try next MX on next time */
			close_transaction(conn, 1);
			return 0;	/* try another transaction */
		}
		if (rc == -2)
		{
			/* resource problem: try same MX again */
			close_transaction(conn, 0);
			sleep(1);
			return -1;	/* do not try immediately */
		}
	}
	return 0;	/* try another transaction */
}

static void
process_connection(r_fds, w_fds, timeout_only, true_term)
fd_set *r_fds, *w_fds;
int timeout_only, true_term;
{
	struct host *hp;
	struct connection *conn;
	int i, st;

	for (i = 0; i < cnf.sd_max; i++)
	{
		conn = &connwork[i];
		hp = conn->host;
		if (hp == NULL)
			continue;
		st = hp->state;
		switch (st)
		{
		  case STAT_CLOSED:
		  case STAT_FAIL:
			continue;
		}

		if (true_term || (hp->time != 0 && state_tbl[st].tout != 0
		  && hp->time + state_tbl[st].tout <= Now))
		{
			/* timeout! */
			sti.ntimeout++;
			if (cnf.debug & DEBUG_CONNECT)
			log(LOG_DEBUG, "%d/%d: timeout (%s)", conn->seq,
				conn->socket, STATE(st));
			if (st == STAT_DTERMSENT)
			{
				/* XXX LOG: may be duplicated */
			}
			hp->time = Now;
			if (smtp_timeout(conn) < 0)
				close_transaction(conn, 1);
		}

		if (timeout_only)
			continue;
		if (FD_ISSET(conn->socket, w_fds))
		{
			if (cnf.debug & DEBUG_EVENT)
			log(LOG_DEBUG, "%d/%d: writable (%s)",
				conn->seq, conn->socket, STATE(st));
			if (state_tbl[st].send != NULL)
			{
				FirstWrite = 1;
				if ((*state_tbl[st].send)(conn) < 0)
					close_transaction(conn, 1);
			}
			if (conn->host == NULL)
			{
				if (cnf.debug & DEBUG_EVENT)
				log(LOG_DEBUG, "%d/%d: state %s -> NULL",
					conn->seq, conn->socket, STATE(st));
			}
			else if (st != conn->host->state)
			{
				conn->host->time = Now;
				if (cnf.debug & DEBUG_EVENT)
				log(LOG_DEBUG, "%d/%d: state %s -> %s",
					conn->seq, conn->socket, STATE(st),
					STATE(conn->host->state));
			}
		}

		if (FD_ISSET(conn->socket, r_fds))
		{
			if (cnf.debug & DEBUG_EVENT)
			log(LOG_DEBUG, "%d/%d: readable (%s)",
				conn->seq, conn->socket, STATE(st));
			if (state_tbl[st].recv != NULL)
			{
				FirstRead = 1;
				if ((*state_tbl[st].recv)(conn) < 0)
					close_transaction(conn, 1);
			}
			if (conn->host == NULL)
			{
				if (cnf.debug & DEBUG_EVENT)
				log(LOG_DEBUG, "%d/%d: state %s -> NULL",
					conn->seq, conn->socket, STATE(st));
			}
			else if (st != conn->host->state)
			{
				conn->host->time = Now;
				if (cnf.debug & DEBUG_EVENT)
				log(LOG_DEBUG, "%d/%d: state %s -> %s",
					conn->seq, conn->socket, STATE(st),
					STATE(conn->host->state));
			}
		}
	}
}

static void
markstatus(rcptp, rcode, response, tracechain)
struct recipient *rcptp;
int rcode;
char *response;
int tracechain;
{
	char hbuf[MAXLINE], rbuf[MAXLINE], pbuf[64];
	char addrbuf[SYSLOG_BUFSIZE];
	char *hostname, *proto, *responsep;
	struct host *hostp;
	int delay, addrlen, estconst, estlen, n;
	static int PrevPCT = -1;

	if (rcptp == NULL)
		return;

	proto = "unknown";
	delay = -1;

	if (rcptp->domain != NULL && rcptp->domain->curmx != NULL)
	{
		hostp = rcptp->domain->curmx->host;
		if (hostp != NULL)
		{
			if (hostp->start > 0)
				delay = Now - hostp->start;
			else
				delay = 0;
			if (sti.max_delay < delay)
				sti.max_delay = delay;
			pbuf[0] = '\0';
#ifdef INET6
			if (hostp->current != NULL
			 && hostp->current->domain == AF_INET6)
				strncat(pbuf, "v6", sizeof(pbuf));
			if (hostp->proto & PROTO_ESMTP)
				strncat(pbuf, (*pbuf != '\0')?",esmtp":"esmtp",
					sizeof(pbuf));
			else
				strncat(pbuf, (*pbuf != '\0')?",smtp":"smtp",
					sizeof(pbuf));
#else
			if (hostp->proto & PROTO_ESMTP)
				strncat(pbuf, "esmtp", sizeof(pbuf));
#endif
			if (hostp->proto & PROTO_SIZE)
				strncat(pbuf, (*pbuf != '\0')?",size":"size",
					sizeof(pbuf));
			if (hostp->proto & PROTO_8BITMIME)
				strncat(pbuf, (*pbuf != '\0')?",8bm":"8bm",
					sizeof(pbuf));
#if notdef
			if (hostp->proto & PROTO_EXPN)
				strncat(pbuf, (*pbuf != '\0')?",expn":"expn",
					sizeof(pbuf));
#endif
			if (hostp->proto & PROTO_DSN)
				strncat(pbuf, (*pbuf != '\0')?",dsn":"dsn",
					sizeof(pbuf));
			if (hostp->proto & PROTO_PIPELINING)
				strncat(pbuf, (*pbuf != '\0')?",pl":"pl",
					sizeof(pbuf));
			if (*pbuf != '\0')
				proto = pbuf;
			else
				proto = "smtp";
		}
		else
			proto = "smtp";

		if (hostp != NULL && hostp->current != NULL)
		{
#ifdef INET6
			char buf[64];
#endif
			char *p;
			struct in_addr in;

			switch (hostp->current->domain)
			{
#ifdef INET6
			  case AF_INET6:
				inet_ntop(AF_INET6,
					hostp->current->address,
					buf, sizeof(buf));
				snprintf(hbuf, sizeof(hbuf), "%s [IPv6:%s]",
					hostp->name, buf);
				snprintf(rbuf, sizeof(rbuf),
					"%s (%s [IPv6:%s])",
					response, hostp->name, buf);
				break;
#endif
			  case AF_INET:
				bcopy(hostp->current->address,
					&in, sizeof(struct in_addr));
				p = inet_ntoa(in);
				snprintf(hbuf, sizeof(hbuf), "%s [%s]",
					hostp->name, p);
				snprintf(rbuf, sizeof(rbuf), "%s (%s [%s])",
					response, hostp->name, p);
				break;
			  default:
				snprintf(hbuf, sizeof(hbuf),
					"%s [unknown domain]", hostp->name);
				snprintf(rbuf, sizeof(rbuf),
					"%s (%s [unknown domain])",
					response, hostp->name);
			}
		}
		else
		{
			snprintf(hbuf, sizeof(hbuf), "%s", hostp->name);
			snprintf(rbuf, sizeof(rbuf), "%s (%s)",
				response, hostp->name);
		}
		hostname = hbuf;
		/* for reuse of domain in previous transaction */
		if (rcptp->domain->response != NULL
		 && rcptp->domain->response == response)
			responsep = response;
		else
			responsep = newstr(rbuf);
	}
	else
	{
		hostname = "unknown";
		responsep = response;
	}
	rcptp->domain->response = responsep;

	addrbuf[0] = '\0';
	addrlen = 0;
	estconst = ((env.queueid!=NULL)?strlen(env.queueid):
		    ((env.envid!=NULL)?strlen(env.envid):4))
		+ strlen(hostname) + strlen(proto)
		+ ((response==NULL)?1:strlen(response));

	while (rcptp != NULL)
	{
		if (rcptp->result != 0 && !IS4xx(rcptp->result))
		{
			if (tracechain == 0)
				break;
			rcptp = rcptp->chain;
			continue;
		}
		rcptp->result = rcode;
		rcptp->response = responsep;

		if (IS2xx(rcode))
		{
			rcptp->stat = RCPT_DONE;
			sti.nsent++;
		}
		else if (IS4xx(rcode))
		{
			struct mx *mxp = rcptp->domain->curmx;

			if (!Terminate && mxp != NULL
			 && (mxp->next != NULL
			  || (mxp->host != NULL && mxp->host->current != NULL
			   && mxp->host->current->next != NULL)))
			{
				rcptp->stat = RCPT_RETRY;
				if (cnf.debug & DEBUG_TRANS)
				log(LOG_DEBUG, "tempfail; next avail: %s",
					rcptp->address);
			}
			else
			{
				sti.ndeferred++;
				rcptp->stat = RCPT_DONE;
				if (cnf.debug & DEBUG_TRANS)
				log(LOG_DEBUG, "tempfail; done: %s",
					rcptp->address);
			}
		}
		else if (IS5xx(rcode))
		{
			rcptp->stat = RCPT_DONE;
			sti.nsmtpfailed++;
		}

		if (addrlen == 0)
		{
			strcpy(addrbuf, rcptp->address);
			addrlen += strlen(rcptp->address);
		}
		else
		{
			addrbuf[addrlen++] = ',';
			addrbuf[addrlen] = '\0';
			strcat(addrbuf+addrlen, rcptp->address);
			addrlen += strlen(rcptp->address);
		}

		/* length estimation with next recipient */
		if (tracechain && rcptp->chain != NULL)
		{
			estlen = 48 /* time stamp, hostname, pid, etc. */
			       + 64 /* constant string size (relay= etc.) */
			       + estconst + addrlen
			       + strlen(rcptp->chain->address)
			       + ((rcptp->chain->chain != NULL)?10:0);
		}
		else
			estlen = SYSLOG_BUFSIZE;

		if (!tracechain || (SYSLOG_BUFSIZE - estlen) <= 0)
		{
			/* buffer may be full */
			if (tracechain && rcptp->chain != NULL)
				strcat(addrbuf+addrlen, ",(more...)");
			log(LOG_INFO, "(%d+%d+%d+%d/%d) relay=%s to=%s proto=%s delay=%d code=%d (%s)",
				sti.nsent, sti.ndeferred, sti.nnsfailed,
				sti.nsmtpfailed, sti.nrcpt, hostname,
				addrbuf, proto, delay, rcode,
				(response==NULL)?"?":response);
			addrlen = 0;
			addrbuf[0] = '\0';
		}
		if (tracechain == 0)
			break;
		rcptp = rcptp->chain;
	}
	/* record the time if the most of addresses are processed */
	if ((sti.time_almostfinish == 0)
	 && (sti.nsent + sti.ndeferred + sti.nnsfailed + sti.nsmtpfailed
	     >= sti.nrcpt * 9 / 10))
	{
		sti.time_almostfinish = Now;
		resource_usage("smtp 90% done");
	}
	n = (sti.nsent + sti.ndeferred + sti.nnsfailed + sti.nsmtpfailed)
	    * 100 / sti.nrcpt;
	if (n != PrevPCT)
	{
		PrevPCT = n;
		if (env.queueid != NULL)
			setproctitle("[%s] %d%% delivered", env.queueid, n);
		else
			setproctitle("%d%% delivered", n);
	}
}

static struct connection *
open_transaction(rcptp)
struct recipient *rcptp;
{
	int rcpts, i;
	struct recipient *rcptp2, *rcpt_tail;
	struct domain *domp;
	struct host *hostp;
	struct mx *mxp;
	int first_time;

#if 1
	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "num_trans=%d, num_sock=%d, max_sock=%d, min_sock=%d",
		num_trans, num_sock, max_sock, sti.minsock);
#endif
	if (num_trans > (max_sock - sti.minsock))
		return NULL;	/* no more free slot */
	if (num_sock > (max_sock - sti.minsock))
		return NULL;	/* no more free socket */

	/* XXX search a new recipient bound to higher response host */

	if (rcptp == NULL)
		return NULL;	/* working on all recipients */

#if 1
	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "open transaction for %s", rcptp->address);
#endif

	if (rcptp->domain->curmx->host->state == STAT_SENT)
	{
		/* keep connected, look up the slot */
		for (i = 0; i < cnf.sd_max; i++)
		{
			if (connwork[i].host == rcptp->domain->curmx->host)
				break;
		}
		if (cnf.debug & DEBUG_TRANS)
		log(LOG_DEBUG, "reusing: slot=%d", i);
	}
	else
	{
		/* look up a free socket slot */
		for (i = 0; i < cnf.sd_max; i++)
		{
			if (connwork[i].rcpt != NULL)
				continue;
			if (connwork[i].host == NULL)	/* XXX */
				break;
			if (connwork[i].host->state == STAT_CLOSED
			 || connwork[i].host->state == STAT_FAIL)
				break;
		}
	}
	if (i == cnf.sd_max)
	{
#if 0
		if (cnf.debug & DEBUG_TRANS)
		log(LOG_DEBUG, "no free slot");
#endif
		/* sleep (1); */
		return NULL;	/* XXX socket slot is full */
	}

	sti.ntrans++;

	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "%d/-: open transaction for %s, slot=%d",
		sti.ntrans, rcptp->address, i);

	/* initialize transaction */
	connwork[i].seq = sti.ntrans;
	connwork[i].rcpt = rcptp;
	connwork[i].crcpts = rcptp;
	connwork[i].crcptr = rcptp;
	connwork[i].goodmail = 0;
	connwork[i].goodrcpt = 0;
	rcptp->stat = RCPT_WORK;
	rcptp->chain = NULL;
	hostp = rcptp->domain->curmx->host;
	connwork[i].host = hostp;
	connwork[i].msg = env.msg;
	connwork[i].offset = 0;
	connwork[i].reuse = 0;
	hostp->start = Now;

	/* find recipients to the same MX */
	rcpts = 1;
	rcptp->stat = RCPT_WORK;
	rcptp->chain = NULL;
	rcpt_tail = rcptp;
	mxp = rcptp->domain->curmx;
	first_time = 1;
	while (mxp != NULL)
	{
		domp = mxp->domain;
		if (domp == NULL)
		{
			log(LOG_NOTICE, "Oops, mxp->domain for %s is NULL",
				mxp->name);
			goto next_domain;
		}
		if (domp->curmx == NULL || domp->rcpt_top == NULL)
			/* domains not to be processed */
			/* XXX clean up required ? */
			goto next_domain;
		if (domp->curmx->host != hostp)
			goto next_domain;
		/* this is the domain which points same MX to be tried */
		for (rcptp2 = domp->rcpt_top; rcptp2 != NULL;
			rcptp2 = rcptp2->dom_chain) 
		{
			if (rcptp2 == rcptp)
				continue;
			if (rcptp2->stat == RCPT_NOOP
			 || rcptp2->stat == RCPT_RETRY)
			{
				if (rcpts + 1 >= cnf.rcpts_trans)
				{
					/* limited by RFC821 */
					connwork[i].reuse = 1;
					goto rcpts_max;
				}
				rcpts++;
				if (cnf.debug & DEBUG_TRANS)
				log(LOG_DEBUG, "%d/-: with %s",
					sti.ntrans, rcptp2->address);
				rcptp2->stat = RCPT_WORK;
				rcptp2->chain = NULL;
				rcpt_tail->chain = rcptp2;
				rcpt_tail = rcptp2;
			}
		}
  next_domain:
		if (first_time)
		{
			first_time = 0;
			mxp = hostp->mx_ref;
		}
		else
		{
			mxp = mxp->mx_ref;
		}
		if (mxp == rcptp->domain->curmx)
			mxp = mxp->mx_ref;
	}

  rcpts_max:
	num_trans++;
	return &connwork[i];
}

static void
close_transaction(conn, next)
struct connection *conn;
int next;	/* true: forward MX for next trial */
{
	struct recipient *rcptp;
	struct domain *domp;
	struct mx *mxp;
#if 0
	time_t delay;
#endif

	if (cnf.debug & DEBUG_TRANS)
	log(LOG_DEBUG, "%d/%d: close transaction", conn->seq, conn->socket);

	NewTransCheck = 1;		/* do not skip new_transaction() */
	env.work_dom_ptr = NULL;	/* retry from the top of list */
#if 0
	if (conn->host->start > 0)
		delay = Now - conn->host->start;
	else
		delay = 0;
	if (sti.max_delay < delay)
		sti.max_delay = delay;
#endif
	conn->host->start = 0;		/* clear start time */

	/* initialize "checked" flag */
	for (rcptp = conn->rcpt; rcptp != NULL; rcptp = rcptp->chain)
	{
		rcptp->domain->checked = 0;
	}

	for (rcptp = conn->rcpt; rcptp != NULL; rcptp = rcptp->chain)
	{
		if (!Terminate && (rcptp->stat == RCPT_DONE))
			continue;
		rcptp->stat = RCPT_RETRY;
		domp = rcptp->domain;
		if (domp->checked)
			continue;
		domp->checked = 1;
		if (Terminate)
		{
			domp->curmx = NULL;
			if (domp->rcpt_top != NULL)
				trim_rcpts(domp);
			continue;
		}
		if (next)
		{
			/* forward MX pointer for next trial */
			domp->curmx = domp->curmx->next;
#if 0
			while (domp->curmx != NULL)
			{
				if (domp->curmx->host == NULL)
					break;
				if (domp->curmx->host->state != STAT_FAIL)
					break;
				domp->curmx = domp->curmx->next;
			}
#endif
			if (domp->curmx != NULL)
			{
				if (cnf.debug & DEBUG_TRANS)
				log(LOG_DEBUG, "%d/-: switching to next MX: %s",
					conn->seq, domp->curmx->name);
			}
		}
	}
	if (Terminate)
	{
		/* trimming is not required */
		conn->host = NULL;
		conn->rcpt = NULL;
		num_trans--;
		return;
	}
	for (mxp = conn->host->mx_ref; mxp != NULL; mxp = mxp->mx_ref)
	{
		if (mxp->domain == NULL)
			continue;
		if (mxp->domain->curmx == NULL || mxp->domain->rcpt_top == NULL)
			continue;
		if (mxp->domain->curmx->host != conn->host)
			continue;
		trim_rcpts(mxp->domain);
	}
#if 0
	switch (conn->host->state)
	{
	  case STAT_FAIL:
		conn->host = NULL;
	}
#endif
	if (!next)
	{
		/* clear host info. since target host is not connected
		(connection will be requested to the same host on next trial) */
		conn->host = NULL;
	}
	conn->rcpt = NULL;
	num_trans--;
}

void
trim_rcpts(domp)
struct domain *domp;
{
	struct recipient *rcptp, *rcpttmp;
	struct recipient *rcptpp = NULL;	/* for warning suppression */

	rcptp = domp->rcpt_top;
	while (rcptp != NULL)
	{
		if (rcptp->stat == RCPT_RETRY && domp->curmx == NULL)
			rcptp->stat = RCPT_DONE;
		rcpttmp = rcptp;
		if (rcptp->stat == RCPT_DONE)
		{
			/* remove recipient from working chain */
			if (cnf.debug & DEBUG_TRANS)
			log(LOG_DEBUG, "removing recipient: %s",
				rcptp->address);
			if (rcptp == domp->rcpt_top)
			{
				domp->rcpt_top = rcptp->dom_chain;
			}
			else
			{
				rcptpp->dom_chain = rcptp->dom_chain;
			}
			if (domp->rcpt_tail == rcptp)
				domp->rcpt_tail = rcptpp;
		} else
			rcptpp = rcptp;
		rcptp = rcptp->dom_chain;
		if (rcpttmp->stat == RCPT_DONE)	/* XXX */
			rcpttmp->dom_chain = NULL;
	}
	if (domp->rcpt_top == NULL)
		NeedTrimDomain = 1;
}

int
trim_domain()
{
	struct domain *domp, *domtmp;
	struct domain *dompp = NULL;	/* for warning suppression */

	domp = env.work_domain;
	while (domp != NULL)
	{
		domtmp = domp;
		if (domp->rcpt_top == NULL)
		{
			/* remove from working domain chain */
			if (cnf.debug & DEBUG_TRANS)
			log(LOG_DEBUG, "removing domain: %s",
				domp->name);
			if (domp == env.work_domain)
			{
				env.work_domain = domp->workchain;
				env.work_dom_ptr = NULL;
			}
			else
			{
				dompp->workchain = domp->workchain;
			}
		}
		else
			dompp = domp;
		domp = domp->workchain;
		if (domtmp->rcpt_top == NULL)	/* XXX */
			domtmp->workchain = NULL;
	}
	NeedTrimDomain = 0;
	if (env.work_domain == NULL)
	{
		/* all domains had been processed */
		return 1;
	}
	return 0;
}

void
forced_terminate()
{
	struct recipient *rcptp;

	if (Terminate)
		return;

	Terminate = 1;
#if 0
	for (rcptp = env.rcpt_list; rcptp != NULL; rcptp = rcptp->next)
	{
		rcptp->domain->curmx = NULL;
	}
#endif
}

static int
smtp_connect(conn)
struct connection *conn;
{
	SockAddr rsin;	/* remote side */
	SockAddr lsin;	/* local side */
	int sd, found, res;
	int socksize = 0;	/* dummy initialization */

	if (conn->host == NULL)
	{
		log(LOG_NOTICE, "smtp_connect called with NULL host");
		return -1;
	}

	if (conn->host->current == NULL)
	{
		/* no more addresses on the host */
		conn->host->state = STAT_FAIL;
		conn->host->stat = SMTP_TEMPFAIL(52);
		/* no markstatus */
		return -1;
	}

	if (cnf.debug & DEBUG_CONNECT)
	log(LOG_DEBUG, "%d/-: connecting to %s",
		conn->seq, conn->host->name);

	bzero((char *)&rsin, sizeof(rsin));
	bzero((char *)&lsin, sizeof(lsin));

	found = 0;
	while (conn->host->current != NULL)
	{
		if (conn->host->current->stat != EX_OK)
		{
			conn->host->current = conn->host->current->next;
			continue;
		}
		switch (conn->host->current->domain)
		{
		  case AF_INET:
			rsin.in.sin_family = AF_INET;
			bcopy(conn->host->current->address, &rsin.in.sin_addr,
				conn->host->current->len);
#if !defined(MISSING_SIN_LEN)
			rsin.in.sin_len = conn->host->current->len;
#endif
			rsin.in.sin_port = htons(cnf.dst_port);
			lsin.in.sin_family = AF_INET;
			lsin.in.sin_addr.s_addr = INADDR_ANY;
#if !defined(MISSING_SIN_LEN)
			lsin.in.sin_len = conn->host->current->len;
#endif
			lsin.in.sin_port = 0;
			socksize = sizeof(struct sockaddr_in);

			if (cnf.debug & DEBUG_CONNECT)
			log(LOG_DEBUG, "%d/-: destination=%s/%d", conn->seq,
				inet_ntoa(rsin.in.sin_addr), cnf.dst_port);
			found = 1;
			break;
#ifdef INET6
		  case AF_INET6:
			rsin.in6.sin6_family = AF_INET6;
			bcopy(conn->host->current->address, &rsin.in6.sin6_addr,
				conn->host->current->len);
#if !defined(MISSING_SIN_LEN)
			rsin.in6.sin6_len = conn->host->current->len;
#endif
			rsin.in6.sin6_port = htons(cnf.dst_port);
			lsin.in6.sin6_family = AF_INET6;
			lsin.in6.sin6_addr = in6addr_any;
#if !defined(MISSING_SIN_LEN)
			lsin.in6.sin6_len = conn->host->current->len;
#endif
			lsin.in6.sin6_port = 0;
			socksize = sizeof(struct sockaddr_in6);

			if (cnf.debug & DEBUG_CONNECT)
			{
				char buf[64];

				inet_ntop(AF_INET6, &rsin.in6.sin6_addr,
					buf, sizeof(buf));
				log(LOG_DEBUG, "%d/-: destination=IPv6:%s/%d",
					conn->seq, buf, cnf.dst_port);
			}
			found = 1;
			break;
#endif
		  default:
			/* protocol is not supported */
			conn->host->current->stat = EX_TEMPFAIL;
			conn->host->current = conn->host->current->next;
			continue;
		}
		if (found)
			break;
	}
	if (!found)
	{
		/* XXX no more address, try next MX */
		conn->host->state = STAT_FAIL;
		conn->host->stat = SMTP_TEMPFAIL(52);
		conn->host->response = "Connection was not established";
		markstatus(conn->rcpt, conn->host->stat,
			conn->host->response, 1);
		return -1;
	}

	if (cnf.src_port < 0)
	{
		int rp = IPPORT_RESERVED - 1;

		sd = rresvport(&rp);	/* XXX */
	}
	else
		sd = socket(conn->host->current->domain, SOCK_STREAM, 0);

	if (sd > MAX_SOCK)
	{
		log(LOG_NOTICE, "%d/%d: can not manage such a big file descriptor number with select()",
			conn->seq, sd);
		close(sd);
		if (max_sock > MAX_SOCK)
			max_sock = MAX_SOCK;
		conn->host->state = STAT_CLOSED;
		/* conn->host = NULL; NULLed in close_transaction */
		/* no markstatus */
		return -2;
	}

	if (sd < 0){
		/* no more socket resource */
		log(LOG_NOTICE, "socket allocation failed for %s: %s (may be max=%d, current=%d/%d/%d)",
			conn->host->name, smtpstrerror(errno), sti.maxsock,
			max_sock, num_sock, sti.minsock);
		if (max_sock > NOFILE)
			max_sock = sti.maxsock;
		/* XXX give up all new transaction */
		/* no markstatus */
		conn->host->state = STAT_CLOSED;
		/* conn->host = NULL; NULLed in close_transaction */
		return -2;
	}

	if (cnf.debug & DEBUG_CONNECT)
	log(LOG_DEBUG, "%d/%d: socket allocated", conn->seq, sd);

#ifdef SO_SNDBUF
	if (cnf.sendbufsize > 0)
	{
		if (setsockopt(sd, SOL_SOCKET, SO_SNDBUF,
		    (char *)&cnf.sendbufsize, sizeof(cnf.sendbufsize)) < 0)
			log(LOG_NOTICE, "%d/%d: setsockopt(SO_SNDBUF) failed for %s: %s",
				conn->seq, sd, conn->host->name,
				smtpstrerror(errno));
		/* ignore error */
	}
#endif

	if (sourceIP[0] != '\0')
	{
		/* bind source address */
		switch (conn->host->current->domain)
		{
		    case AF_INET:
			lsin.in.sin_addr.s_addr = inet_addr(sourceIP);
			break;
#ifdef INET6
		    case AF_INET6:
			inet_pton(INET6, sourceIP, &lsin.in6.sin6_addr);
			break;
#endif
		}
	}
	else
	{
		switch (conn->host->current->domain)
		{
		    case AF_INET:
			lsin.in.sin_addr.s_addr = INADDR_ANY;
			break;
#ifdef INET6
		    case AF_INET6:
			lsin.in6.sin6_addr = in6addr_any;
			break;
#endif
		}
	}

	if (cnf.src_port > 0)
	{
		switch (conn->host->current->domain)
		{
		    case AF_INET:
			lsin.in.sin_port = htons(cnf.src_port);
			break;
#ifdef INET6
		    case AF_INET6:
			lsin.in6.sin6_port = htons(cnf.src_port);
			break;
#endif
		}
	}
	else
	{
		switch (conn->host->current->domain)
		{
		    case AF_INET:
			lsin.in.sin_port = 0;
			break;
#ifdef INET6
		    case AF_INET6:
			lsin.in6.sin6_port = 0;
			break;
#endif
		}
	}

	if (sourceIP[0] != '\0' || cnf.src_port > 0)
	{
		if (bind(sd, (struct sockaddr *)&lsin, socksize) < 0)
		{
			log(LOG_NOTICE, "%d/%d: bind failed for %s: %s", 
				conn->seq, sd, conn->host->name,
				smtpstrerror(errno));
			conn->host->state = STAT_FAIL;
			close (sd);
			markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
				"source port binding failure", 1);
			return -1;
		}
	}

	/* set non-blocking */
#ifdef FIONBIO
	res = 1;

	if (ioctl (sd, FIONBIO, &res) < 0)
	{
		log(LOG_NOTICE, "%d/%d: FIONBIO failed for %s: %s",
			conn->seq, sd, conn->host->name, smtpstrerror(errno));
		conn->host->state = STAT_FAIL;
		close (sd);
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
			"FIONBIO failed", 1);
		return -1;
	}
#else
	if ((res = fcntl(sd, F_GETFL, 0)) == -1)
	{
		log(LOG_NOTICE, "%d/%d: F_GETFL failed for %s: %s",
			conn->seq, sd, conn->host->name, smtpstrerror(errno));
		conn->host->state = STAT_FAIL;
		close (sd);
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
			"F_GETFL failed", 1);
		return -1;
	}
# ifdef O_NONBLOCK
	res |= O_NONBLOCK;
# else	/* expect defined O_NDELAY */
	res |= O_NDELAY;
# endif
	if (fcntl(sd, F_SETFL, res) == -1)
	{
		log(LOG_NOTICE, "%d/%d: F_SETFL failed for %s: %s",
			conn->seq, sd, conn->host->name, smtpstrerror(errno));
		conn->host->state = STAT_FAIL;
		close (sd);
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
			"F_SETFL failed", 1);
		return -1;
	}
#endif

	conn->host->time = Now;
	alarm(3);
	if (connect(sd, (struct sockaddr *)&rsin, socksize) < 0
	 && errno != EINPROGRESS && errno != EINTR)	/* XXX */
	{
		alarm(0);
		log(LOG_DEBUG, "%d/%d: connect failed for %s: %s",
			conn->seq, sd, conn->host->name, smtpstrerror(errno));
		conn->host->current->stat = EX_TEMPFAIL;
		close (sd);
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
			"connect() failed", 1);
		/* try next address of the same MX */
		conn->host->current = conn->host->current->next;
#if 0
		conn->host->state = STAT_CLOSED;
#endif
		/* conn->host = NULL; NULLed in close_transaction */
		return -2;
	}
	alarm(0);
	sti.nconnect++;

	conn->socket = sd;
	conn->host->state = STAT_CONNECTING;
	if (sd > sti.maxsock)
		sti.maxsock = sd;
	if (sti.minsock == 0 || sd < sti.minsock)
	{
		if (sti.minsock == 0 && sd > cnf.sd_max)
		{
			/* max is too low,
			   rais it to number of first allocated socket */
			cnf.sd_max += sd - 1;
			max_sock = cnf.sd_max;
		}
		sti.minsock = sd;
		if (cnf.debug & DEBUG_CONNECT)
		log(LOG_DEBUG, "minsock=%d", sti.minsock);
	}
	num_sock++;
	return 0;
}

void
smtp_close(sd)
int sd;
{
	num_sock--;
	close(sd);
}

static int
smtp_connected(conn)
struct connection *conn;
{
	SockAddr cin;
	int len = sizeof(SockAddr);
	const char *errmsg;

	if (getpeername(conn->socket, (struct sockaddr *)&cin, &len) < 0)
	{
		log(LOG_INFO, "%d/%d: getpeername failed for %s: %s",
			conn->seq, conn->socket, conn->host->name,
			smtpstrerror(errno));
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		conn->host->state = STAT_CLOSED;
		switch (errno)
		{
		  case ENOTCONN:
			if (conn->host->time + CONN_TIMEOUT < Now)
				errmsg = "Host is down";
			else
				errmsg = "Connection refused";
			break;
		  case EHOSTDOWN:
		  case EHOSTUNREACH:
			errmsg = "Host is down";
			break;
		  case ENETDOWN:
		  case ENETUNREACH:
		  case ENETRESET:
			errmsg = "Network is unreachable";
			break;
		  case ECONNREFUSED:
		  case ECONNRESET:
			errmsg = "Connection refused";
			break;
		  default:
			errmsg = smtpstrerror(errno);
		}
		markstatus(conn->rcpt, SMTP_TEMPFAIL(52), errmsg, 1);
		conn->host->current = conn->host->current->next;
		if (Terminate)
			return -1;
		return smtp_connect(conn);
	}

	/* create buffer */
	conn->bufsize = SMTPBUFSIZE;
	if (conn->inbuf == NULL)
		conn->inbuf = (char *)malloc(conn->bufsize);
	conn->inlen = 0;
	if (conn->inbuf != NULL && conn->outbuf == NULL)
		conn->outbuf = (char *)malloc(conn->bufsize);
	conn->outlen = 0;
	if (conn->inbuf == NULL || conn->outbuf == NULL)
	{
		log(LOG_NOTICE, "out of memory (SMTP buffer allocation)");
		conn->host->current->stat = EX_TEMPFAIL;
		conn->host->current = conn->host->current->next;
		smtp_close(conn->socket);
		if (conn->host->current == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		forced_terminate();
		return -1;
	}
	conn->host->state = STAT_CONNECTED;
	return 0;
}

static int
smtp_read(conn, buf, size, rest)
struct connection *conn;
char *buf;
int size;
int *rest;
{
	int nl_found, n, n2;

	nl_found = 0;
	if (conn->inlen > 0)
	{
		for (n = 0; n < conn->inlen; n++)
			if (conn->inbuf[n] == '\n')
				nl_found = 1;
	}
	if (nl_found == 0)
	{
		n = read(conn->socket, conn->inbuf + conn->inlen,
			conn->bufsize - conn->inlen);
		if (FirstRead)
		{
			FirstRead = 0;
			if (n == 0)
			{
				if (cnf.debug & DEBUG_CONNECT)
				log(LOG_DEBUG, "%d/%d: closed by foreign host",
					conn->seq, conn->socket);
				errno = E_SF_IO;
				return -1;
			}
		}
		if (n == 0)
			return 0;
		if (n < 0)
		{
			switch (errno)
			{
			  case EAGAIN:
#if EAGAIN != EWOULDBLOCK
			  case EWOULDBLOCK:
#endif
				return 0;
			  default:
				log(LOG_DEBUG, "%d/%d: read failed for %s: ret=%d, errno=%d",
					conn->seq, conn->socket,
					conn->host->name, n, errno);
				return n;
			}
		}
		conn->inlen += n;
		sti.inbounds += n;
	}
	for (n = 0; n < conn->inlen; n++)
	{
		if (conn->inbuf[n] == '\n')
		{
			/* return a buffered line */
			n++;
			if (size < n)
				n = size;	/* XXX */
			bcopy(conn->inbuf, buf, n);
			/* buf[n] = '\0'; */
			conn->inlen -= n;
			/* if (conn->inlen > 0) */
			bcopy(conn->inbuf + n, conn->inbuf, conn->inlen);
			if (rest != NULL)
			{
				nl_found = 0;
				if (conn->inlen > 0)
				{
					for (n2 = 0; n2 < conn->inlen; n2++)
						if (conn->inbuf[n2] == '\n')
							nl_found = 1;
				}
				if (nl_found)
					*rest = 1;
				else
					*rest = 0;
			}
			return n;
		}
	}
	/* no line to be returned */
	if (rest != NULL)
	{
		*rest = 0;
	}
	return 0;
}

static int
smtp_write(conn, buf, size, flush)
struct connection *conn;
char *buf;
int size;
int flush;
{
	int n;

	if (conn->outlen > 0 && conn->outlen + size > conn->bufsize)
	{
		/* flush buffer */
		n = write(conn->socket, conn->outbuf, conn->outlen);
		if (FirstWrite)
		{
			FirstWrite = 0;
			if (n == 0)
			{
				if (cnf.debug & DEBUG_CONNECT)
				log(LOG_DEBUG,
					"%d/%d: closed by foreign host (%s)",
					conn->seq, conn->socket,
					conn->host->name);
				errno = E_SF_IO;
				return -1;
			}
		}
		if (n == 0)
			return 0;
		if (n < 0)
		{
			switch (errno)
			{
			  case EAGAIN:
#if EAGAIN != EWOULDBLOCK
			  case EWOULDBLOCK:
#endif
				return 0;
			  default:
				log(LOG_DEBUG,
					"%d/%d: write failed for %s: ret=%d, errno=%d",
					conn->seq, conn->socket,
					conn->host->name, n, errno);
				return n;
			}
		}
		if (n == conn->outlen)
			conn->outlen = 0;
		else
		{
			conn->outlen -= n;
			bcopy(conn->outbuf + n, conn->outbuf, conn->outlen);
		}
		sti.outbounds += n;
	}
	if (conn->outlen > 0 && conn->outlen + size > conn->bufsize)
	{
		/* still no space to append */
		return 0;
	}
	if (buf != NULL)
	{
		bcopy(buf, conn->outbuf + conn->outlen, size);
		conn->outlen += size;
	}
	if (conn->outlen > 0 && flush)
	{
		n = write(conn->socket, conn->outbuf, conn->outlen);
		if (FirstWrite)
		{
			FirstWrite = 0;
			if (n == 0)
			{
				if (cnf.debug & DEBUG_CONNECT)
				log(LOG_DEBUG,
					"%d/%d: closed by foreign host (%s)",
					conn->seq, conn->socket,
					conn->host->name);
				errno = E_SF_IO;
				return -1;
			}
		}
		if (n < 0)
		{
			switch (errno)
			{
			  case EAGAIN:
#if EAGAIN != EWOULDBLOCK
			  case EWOULDBLOCK:
#endif
				return size;
			  default:
				log(LOG_DEBUG,
					"%d/%d: write failed for %s: ret=%d, errno=%d",
					conn->seq, conn->socket,
					conn->host->name, n, errno);
				return n;
			}
		}
		if (n == 0)
			return size;
		if (n == conn->outlen)
			conn->outlen = 0;
		else
		{
			conn->outlen -= n;
			bcopy(conn->outbuf + n, conn->outbuf, conn->outlen);
		}
		sti.outbounds += n;
	}
	return size;
}

static int
smtp_flush(conn)
struct connection *conn;
{
	int rcode, savestate;

	savestate = conn->host->state;
	rcode = smtp_write(conn, NULL, 0, 1);
	if (rcode < 0)
	{
		char msgbuf[MAXLINE];

		snprintf(msgbuf, sizeof(msgbuf),
			"Flushing write error (%s) in state %s",
			smtpstrerror(errno), STATE(savestate));
		conn->host->response = newstr(msgbuf);
		conn->host->state = STAT_FAIL;
		conn->host->stat = SMTP_TEMPFAIL(52);
		markstatus(conn->rcpt, conn->host->stat,
			conn->host->response, 1);
		return -1;
	}
	return 0;
}

#define	NUM(x)	((x) - '0')

static int
smtp_get_reply(conn, func, firstline, size, rest)
struct connection *conn;
void (*func)();
char firstline[];
int size;
int *rest;
{
	char *cbuf = firstline;
	int buflen = size;
	char contbuf[MAXLINE];
	char savec;
	int len, rcode, savestate;
	int stat = 0;

	cbuf[0] = '\0';
	savestate = conn->host->state;
	for (;;)
	{
		rcode = smtp_read(conn, cbuf, buflen - 1, rest);
		if (rcode < 0)
		{
			char msgbuf[MAXLINE];

			if (conn->host->state == STAT_CONNECTED)
				/* managed in the parent */
				return -1;

			snprintf(msgbuf, sizeof(msgbuf),
				"Read error (%s) in state %s",
				smtpstrerror(errno), STATE(savestate));
			conn->host->response = newstr(msgbuf);
			conn->host->state = STAT_FAIL;
			conn->host->stat = SMTP_TEMPFAIL(52);
			markstatus(conn->rcpt, conn->host->stat,
				conn->host->response, 1);
			return -1;
		}
		if (rcode == 0)
			return 0;
		len = rcode;
		cbuf[len] = '\0';
		while (len > 0 && (cbuf[len-1] == '\n' || cbuf[len-1] == '\r'
				|| isspace(cbuf[len-1])))
		{
			cbuf[--len] = '\0';
		}
		if (cnf.debug & DEBUG_SMTP)
		log(LOG_DEBUG, "%d/%d: >>> %s", conn->seq, conn->socket, cbuf);
		if (func != NULL)
			(*func)(conn, cbuf);
		if (!isdigit(cbuf[0]) || !isdigit(cbuf[1]) || !isdigit(cbuf[2])
		 || (cbuf[3] != '\0' && !isspace(cbuf[3]) && cbuf[3] != '-'))
			continue;
		savec = cbuf[3];
		if (cbuf[0] >= '1' && cbuf[0] <= '5')
			stat = NUM(cbuf[0])*100 + NUM(cbuf[1])*10
			     + NUM(cbuf[2]);
		if (cbuf[3] != '-')
			break;
		cbuf = contbuf;
		buflen = sizeof(contbuf);
	}
	if (stat < 100 || stat >= 600 || stat == 421)
	{
		char msgbuf[MAXLINE];

		if (stat == 421)
		{
			/* Service shutting down */
			snprintf(msgbuf, sizeof(msgbuf),
				"Service shutting down in state %s: %s",
				STATE(savestate), firstline);
			conn->host->stat = SMTP_TEMPFAIL(52);
			errno = E_SF_ABORT;
		}
		else
		{
			/* Unknown response */
			snprintf(msgbuf, sizeof(msgbuf),
				"Remote protocol error in state %s: %s",
				STATE(savestate), firstline);
			conn->host->stat = SMTP_ERROR(54);
			errno = E_SF_PROTO;
		}
		conn->host->response = newstr(msgbuf);
		conn->host->state = STAT_FAIL;
		markstatus(conn->rcpt, conn->host->stat,
			conn->host->response, 1);
		return -1;
	}
	return stat;
}

static int
smtp_send_command(conn, buf, size, flush)
struct connection *conn;
char *buf;
int size;
int flush;
{
	int rcode;
	int len = strlen(buf);

	if (cnf.debug & DEBUG_SMTP)
	log(LOG_DEBUG, "%d/%d: <<< %s", conn->seq, conn->socket, buf);
	if (len + 2 < size)
	{
		strcat(buf, CRLF);
		rcode = smtp_write(conn, buf, len + 2, flush);
	}
	else
	{
		char *p;
		p = (char *)malloc(len + 3);
		if (p == NULL)
		{
			forced_terminate();
			conn->host->state = STAT_FAIL;
			conn->host->stat = SMTP_TEMPFAIL(52);
			conn->host->response
				= "out of memory for sending buffer";
			markstatus(conn->rcpt, conn->host->stat,
				conn->host->response, 1);
			return -1;
		}
		strcpy(p, buf);
		strcat(p, CRLF);
		rcode = smtp_write(conn, p, len + 2, flush);
		free(p);
	}
	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		char msgbuf[MAXLINE];

		buf[len] = '\0';
		snprintf(msgbuf, sizeof(msgbuf),
			"Write error (%s) on writing \"%s\"",
			smtpstrerror(errno), buf);
		conn->host->response = newstr(msgbuf);
		conn->host->state = STAT_FAIL;
		conn->host->stat = SMTP_TEMPFAIL(52);
		markstatus(conn->rcpt, conn->host->stat,
			conn->host->response, 1);
		return -1;
	}
	return rcode;
}

static void
esmtp_check(conn, buf)
struct connection *conn;
char *buf;
{
	char *p = buf;

	while (p != NULL)
	{
		if ((p = strstr(p, "ESMTP")) != NULL)
		{
			if ((p > buf && isalnum(p[-1])) || isalnum(p[5]))
			{
				p += 5;
				continue;
			}
			conn->host->proto |= PROTO_ESMTP;
			break;
		}
	}
}

static int
smtp_greeted(conn)
struct connection *conn;
{
	char buf[MAXLINE];
	int rcode, savestate;
	const char *errmsg;

	errno = 0;
	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, esmtp_check, buf, sizeof(buf), NULL);

	if (rcode <= 0)
	{
		switch (errno)
		{
		  case 0:
			return 0;	/* XXX */
		  case EAGAIN:
#if EAGAIN != EWOULDBLOCK
		  case EWOULDBLOCK:
#endif
			return 0;	/* retry later */
		  case ENETDOWN:
		  case ENETUNREACH:
		  case ENETRESET:
		  case ECONNABORTED:
		  case ECONNRESET:
		  case ENOTCONN:
		  case ETIMEDOUT:
		  case ECONNREFUSED:
		  case EHOSTDOWN:
		  case EHOSTUNREACH:
		  case EBADF:
			errmsg = "Connection was not established";
			break;
		  case E_SF_IO:
			errmsg = "Unexpected close by foreign host";
			break;
		  case E_SF_PROTO:
		  case E_SF_ABORT:
			errmsg = NULL;	/* already marked */
			break;
		  default:
			errmsg = "Unexpected Error";
			break;
		}
		log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
			conn->seq, conn->socket, conn->host->name,
			smtpstrerror(errno), STATE(savestate));
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		conn->host->state = STAT_CLOSED;
		if (errmsg != NULL)
		{
			markstatus(conn->rcpt, SMTP_TEMPFAIL(52), errmsg, 1);
			if (conn->host->current->next == NULL
			 && conn->host->response == NULL)
				conn->host->response = (char *)errmsg;
		}
		conn->host->current = conn->host->current->next;
		if (Terminate)
			return -1;
		return smtp_connect(conn);
	}

	if (IS2xx(rcode))
	{
		if (Terminate)
		{
			sti.ntimeout++;
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				"Processing timed out (greeting)", 1);
			smtp_quit_s(conn);
			return -1;
		}
		smtp_helo_s(conn);
		return 0;
	}

	if (!(IS4xx(rcode) || IS5xx(rcode)))
	{
		char tmpbuf[MAXLINE];

		rcode = SMTP_ERROR(54);
		snprintf(tmpbuf, sizeof(tmpbuf),
			"Remote protocol error in state %s: %s",
			STATE(savestate), buf);
		conn->host->response = newstr(tmpbuf);
	}
	else
		conn->host->response = newstr(buf);
	conn->host->state = STAT_FAIL;
	conn->host->stat = rcode;
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_helo_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];

	if (conn->host->state == STAT_CONNECTED
	 && conn->host->proto & PROTO_ESMTP)
	{
		conn->host->state = STAT_EHLOSENT;
		strcpy(cmdbuf, "EHLO ");
	}
	else
	{	/* STAT_EHLOSENT */
		conn->host->state = STAT_HELOSENT;
		strcpy(cmdbuf, "HELO ");
	}
	strncat(cmdbuf, myname, sizeof(cmdbuf) - 1);

	if (smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	return 0;
}

static void
ehlo_opt(conn, buf)
struct connection *conn;
char *buf;
{
	if (strlen(buf) < 5)
		return;
	buf += 4;

	switch (buf[0])
	{
	  case 's':
	  case 'S':
		if (strcasecmp(buf, "size") == 0)
			conn->host->proto |= PROTO_SIZE;
		break;
	  case '8':
		if (strcasecmp(buf, "8bitmime") == 0)
			conn->host->proto |= PROTO_8BITMIME;
		break;
	  case 'e':
	  case 'E':
		if (strcasecmp(buf, "expn") == 0)
			conn->host->proto |= PROTO_EXPN;
		break;
	  case 'd':
	  case 'D':
		if (strcasecmp(buf, "dsn") == 0)
			conn->host->proto |= PROTO_DSN;
		break;
	  case 'p':
	  case 'P':
		if (strcasecmp(buf, "pipelining") == 0)
			if (!cnf.nopipelining)
				conn->host->proto |= PROTO_PIPELINING;
		break;
	}
}

static int
smtp_helo_r(conn)
struct connection *conn;
{
	char buf[MAXLINE], *p;
	int rcode, savestate;
	void (*func)();

	if (conn->host->state == STAT_EHLOSENT)
		func = ehlo_opt;
	else
		func = NULL;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, func, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, smtpstrerror(errno),
				conn->host->name, STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (conn->host->state == STAT_EHLOSENT && !IS2xx(rcode))
	{
		conn->host->state = STAT_HELOSENT;
		strcpy(buf, "HELO ");
		strncat(buf, myname, sizeof(buf) - 1);
		if (smtp_send_command(conn, buf, sizeof(buf), 1) < 0)
		{
			/* XXX */
			smtp_close(conn->socket);
			if (conn->host->current->next == NULL)
				conn->host->state = STAT_FAIL;
			else
				conn->host->state = STAT_CLOSED;
			return -1;
		}
		return 0;
	}

	if (IS2xx(rcode))
	{
		if (strlen(buf) > 4)
		{
			/* loop check */
			p = buf + 4;
			while (*p != '\0' && !isspace(*p))
				p++;
			*p = '\0';
			if (isamyalias(buf + 4))
			{
				/* conn->host->state = STAT_FAIL; */
				conn->host->stat = SMTP_ERROR(53);
				conn->host->response = "Config error: mail loops back to me (MX problem?)";
				markstatus(conn->rcpt, conn->host->stat,
					conn->host->response, 1);
				smtp_quit_s(conn);
				return -1;
			}
		}
		if (Terminate)
		{
			sti.ntimeout++;
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				"Processing timed out (HELO)", 1);
			smtp_quit_s(conn);
			return -1;
		}
		if (cnf.debug & DEBUG_NODELIVERY)
		{
			/* for debugging without delivery */
			conn->host->state = STAT_FAIL;
			conn->host->stat = SMTP_TEMPFAIL(51);
			conn->host->response = "Forced temporary failure";
			markstatus(conn->rcpt, conn->host->stat,
				 conn->host->response, 1);
			if (conn->reuse)
				return -1;	/* without QUIT */
			if (cnf.smtp_reuse && may_be_reused(conn))
				return -1;	/* without QUIT */
			smtp_quit_s(conn);
			return -1;
		}
		else
			return smtp_mail_s(conn);
	}

	if (!(IS4xx(rcode) || IS5xx(rcode)))
	{
		char tmpbuf[MAXLINE];

		rcode = SMTP_ERROR(54);
		snprintf(tmpbuf, sizeof(tmpbuf),
			"Remote protocol error in state %s: %s",
			STATE(savestate), buf);
		conn->host->response = newstr(tmpbuf);
	}
	else
		conn->host->response = newstr(buf);
	conn->host->state = STAT_FAIL;
	conn->host->stat = rcode;
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_mail_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];
	int len = sizeof(cmdbuf) - 1;

	strcpy(cmdbuf, "MAIL FROM:");
	len -= 10;
	if (len < strlen(env.sender))
		goto mail_err;
	strcat(cmdbuf, env.sender);
	len -= strlen(env.sender);

	if (env.envid != NULL && conn->host->proto & PROTO_DSN)
	{
		if (len < strlen(env.envid) + 7)
			goto mail_err;
		strcat(cmdbuf, " ENVID=");
		strcat(cmdbuf, env.envid);
		len -= strlen(env.envid) + 7;
	}
	if (env.size != NULL && conn->host->proto & PROTO_SIZE)
	{
		if (len < strlen(env.size) + 6)
			goto mail_err;
		/* XXX should send REAL message size */
		strcat(cmdbuf, " SIZE=");
		strcat(cmdbuf, env.size);
		len -= strlen(env.size) + 6;
	}
	if (env.body != NULL && conn->host->proto & PROTO_8BITMIME)
	{
		if (len < strlen(env.body) + 6)
			goto mail_err;
		strcat(cmdbuf, " BODY=");
		strcat(cmdbuf, env.body);
		len -= strlen(env.body) + 6;
	}
	if (env.ret != NULL && conn->host->proto & PROTO_DSN)
	{
		if (len < strlen(env.ret) + 5)
			goto mail_err;
		strcat(cmdbuf, " RET=");
		strcat(cmdbuf, env.ret);
		len -= strlen(env.ret) + 5;
	}

	if (smtp_send_command(conn, cmdbuf, sizeof(cmdbuf),
		conn->host->proto & PROTO_PIPELINING?0:1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	conn->host->state = STAT_MAILSENT;

	if (conn->host->proto & PROTO_PIPELINING)
		return smtp_rcpt_s(conn);

	return 0;

  mail_err:
	conn->host->state = STAT_FAIL;
	conn->host->stat = SMTP_ERROR(54);
	conn->host->response = "Sender address too long";
	markstatus(conn->rcpt, conn->host->stat, conn->host->response, 1);
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_mail_r(conn)
struct connection *conn;
{
	char buf[MAXLINE];
	char tmpbuf[MAXLINE], *rstr;
	int rcode, savestate;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (conn->host->proto & PROTO_PIPELINING)
		conn->host->state = STAT_RCPTSENT;

	if (IS2xx(rcode))
	{
		if (conn->host->proto & PROTO_PIPELINING)
			conn->goodmail = 1;
		if (Terminate)
		{
			sti.ntimeout++;
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				"Processing timed out (MAIL)", 1);
			smtp_quit_s(conn);
			return -1;
		}
		if (conn->host->proto & PROTO_PIPELINING)
			return smtp_rcpt_r(conn);
		else
			return smtp_rcpt_s(conn);
	}

	if (!(IS4xx(rcode) || IS5xx(rcode)))
	{
		rcode = SMTP_ERROR(54);
		snprintf(tmpbuf, sizeof(tmpbuf),
			"Remote protocol error in state %s: %s",
			STATE(savestate), buf);
		rstr = newstr(tmpbuf);
	}
	else
	{
		if (buf[4] == '\0')
			rstr = "Error by unknown reason";
		else
			rstr = newstr(buf + 4);
	}
	markstatus(conn->rcpt, rcode, rstr, 1);	/* XXX */
	conn->host->response = rstr;
	conn->host->stat = rcode;

	if (conn->host->proto & PROTO_PIPELINING)
		return smtp_rcpt_r(conn);
	else
		/* XXX should be cleared for next trans. */
		conn->host->state = STAT_FAIL;

	smtp_quit_s(conn);
	return -1;
}

static int
smtp_rcpt_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];
	int rcode, len;

	if (conn->crcpts == NULL)
	{
		if (conn->host->proto & PROTO_PIPELINING && !conn->datasent)
			return smtp_data_s(conn);
		return 0;
	}

  pipelining:
	len = sizeof(cmdbuf) - 1;
	strcpy(cmdbuf, "RCPT TO:");
	len -= 8;
	if (len < strlen(conn->crcpts->address))
		goto rcpt_err;
	strcat(cmdbuf, conn->crcpts->address);
	len -= strlen(conn->crcpts->address);

	if (conn->crcpts->notify != NULL && conn->host->proto & PROTO_DSN)
	{
		if (len < strlen(conn->crcpts->notify) + 8)
			goto rcpt_err;
		strcat(cmdbuf, " NOTIFY=");
		strcat(cmdbuf, conn->crcpts->notify);
		len -= strlen(conn->crcpts->notify) + 8;
	}
	if (conn->crcpts->orcpt != NULL && conn->host->proto & PROTO_DSN)
	{
		if (len < strlen(conn->crcpts->orcpt) + 7)
			goto rcpt_err;
		strcat(cmdbuf, " ORCPT=");
		strcat(cmdbuf, conn->crcpts->orcpt);
		len -= strlen(conn->crcpts->orcpt) + 7;
	}

	if (conn->host->proto & PROTO_PIPELINING)
		rcode = smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 0);
	else
		rcode = smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 1);
	if (rcode < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	if (rcode == 0)
		return 0;

	if ((conn->host->proto & PROTO_PIPELINING) == 0
	 && conn->host->state != STAT_RCPTSENT)
		conn->host->state = STAT_RCPTSENT;

	conn->crcpts = conn->crcpts->chain;

	if (conn->host->proto & PROTO_PIPELINING)
	{
		if (conn->crcpts != NULL)
			goto pipelining;
		return smtp_data_s(conn);
	}

	return 0;

  rcpt_err:
	conn->host->state = STAT_FAIL;
	conn->host->stat = SMTP_ERROR(54);
	conn->host->response = "Recipient address too long";
	markstatus(conn->rcpt, conn->host->stat, conn->host->response, 1);
	smtp_quit_s(conn);	/* XXX */
	return -1;
}

static char *
skipaddr(str, addr)
char *str, *addr;
{
	int len = strlen(addr);

	if (strncasecmp(str, addr, len) == 0)
	{
		char *p = &str[len];

		while (*p != '\0' && (isspace(*p) || *p == '.'))
			p++;
		return p;
	}
	return str;
}

static int
smtp_rcpt_r(conn)
struct connection *conn;
{
	char buf[MAXLINE];
	int rcode, rest, savestate;

  pipelining:
	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), &rest);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (IS2xx(rcode))
		conn->goodrcpt = 1;
	else if ((conn->host->proto & PROTO_PIPELINING) == 0
	      || conn->goodmail == 1)
	{
		char tmpbuf[MAXLINE], *rstr;

		if (!(IS4xx(rcode) || IS5xx(rcode)))
		{
			rcode = SMTP_ERROR(54);
			snprintf(tmpbuf, sizeof(tmpbuf),
				"Remote protocol error in state %s: %s",
				STATE(savestate), buf);
			rstr = newstr(tmpbuf);
		}
		else
		{
			if (buf[4] == '\0')
				rstr = "Error by unknown reason";
			else
				rstr = newstr(skipaddr(buf + 4,
					conn->crcptr->address));
		}
		markstatus(conn->crcptr, rcode, rstr, 0);
	}
	conn->crcptr = conn->crcptr->chain;

	if (Terminate)
	{
		sti.ntimeout++;
		markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
			"Processing timed out (RCPT)", 1);
		smtp_quit_s(conn);
		return -1;
	}
	if (conn->crcptr == NULL)
	{
		if (conn->host->proto & PROTO_PIPELINING)
		{
			/* DATA already sent */
			conn->host->state = STAT_DATASENT;
			return smtp_data_r(conn);
		}

		if (conn->goodrcpt)
			return smtp_data_s(conn);
		else
		{
			/* failure on all recipients */
			/* conn->host->state = STAT_FAIL; */
			smtp_quit_s(conn);	/* XXX */
			return -1;
		}
	}
	else
	{
		if (conn->host->proto & PROTO_PIPELINING && rest)
			goto pipelining;
		return smtp_rcpt_s(conn);
	}
	/* not reached */
	return 0;
}

static int
smtp_data_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];

	if (conn->host->proto & PROTO_PIPELINING)
		conn->datasent = 1;
	strcpy(cmdbuf, "DATA");
	if (smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	if ((conn->host->proto & PROTO_PIPELINING) == 0)
		conn->host->state = STAT_DATASENT;
	return 0;
}

static int
smtp_data_r(conn)
struct connection *conn;
{
	char buf[MAXLINE], tmpbuf[MAXLINE], *rstr;
	int rcode, savestate;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (IS3xx(rcode))
	{
		if (Terminate)
		{
			sti.ntimeout++;
			conn->host->state = STAT_FAIL;
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				"Processing timed out (DATA)", 1);
			/* XXX how can i terminate here ? */
			smtp_close(conn->socket);
			if (conn->host->current->next == NULL)
				conn->host->state = STAT_FAIL;
			else
				conn->host->state = STAT_CLOSED;
			return -1;
		}
		conn->host->state = STAT_DATABODY;
		return smtp_data_body(conn);
	}

	if (!(IS4xx(rcode) || IS5xx(rcode)))
	{
		rcode = SMTP_ERROR(54);
		snprintf(tmpbuf, sizeof(tmpbuf),
			"Remote protocol error in state %s: %s",
			STATE(savestate), buf);
		rstr = newstr(tmpbuf);
	}
	else
	{
		if (buf[4] == '\0')
			rstr = "Error by unknown reason";
		else
			rstr = newstr(buf + 4);
	}

	if ((conn->host->proto & PROTO_PIPELINING) == 0 || conn->goodmail == 1)
	{
		markstatus(conn->rcpt, rcode, rstr, 1);	/* XXX */
		conn->host->stat = rcode;
		conn->host->response = rstr;
	}
	conn->host->state = STAT_FAIL;
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_data_body(conn)
struct connection *conn;
{
	char tb[MAXLINE];
	char *datap, *p, savec;
	int rcode, len;

	if (cnf.showrecipient && conn->rcpt->chain == NULL /* only one rcpt. */
	 && conn->msg->data == env.msg->data && conn->offset == 0)
	{
		char altbuf[MAXLINE];

		snprintf(altbuf, MAXLINE,
			"To: (original recipient in envelope at %s) %s\r\n",
			myname, conn->rcpt->address);
		len = strlen(altbuf);
		rcode = smtp_write(conn, altbuf, len, 0);
		if (rcode != len)	/* XXX */
			log(LOG_NOTICE,
			    "Header of sent message will be corrupted: %s",
			    altbuf);
	}

	errno = 0;
	while (conn->msg != NULL)
	{
		while (1)
		{
			datap = &conn->msg->data[conn->offset];
			if ((p = strchr(datap, '\n')) == NULL)
				break;
			len = ++p - datap;
			savec = *p;
			*p = '\0';
			rcode = smtp_write(conn, datap, len, 0);
			*p = savec;
			if (rcode == 0)
			{
				/* sleep(2); */
				return 0;
			}
			if (rcode < 0)
			{
				conn->host->state = STAT_FAIL;
				conn->host->stat = SMTP_TEMPFAIL(51);
				conn->host->response = "lost connection";
				markstatus(conn->rcpt, conn->host->stat,
					conn->host->response, 1);
				/* smtp_quit_s(conn); */
				return -1;
			}
			conn->offset += rcode;
			if (rcode != len)
			{
				/* wait for send buffer ready */
				return 0;
			}
		}
		conn->msg = conn->msg->next;
		conn->offset = 0;
	}
	/* completed */
	strcpy(tb, ".");
	if (smtp_send_command(conn, tb, sizeof(tb), 1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	conn->host->state = STAT_DTERMSENT;
	return 0;
}

static int
may_be_reused(conn)
struct connection *conn;
{
	struct domain *domp;
	struct recipient *rcptp;
	struct mx *mxp, *rmxp;
	int reused;

	if (Terminate)
		return 0;

	if (cnf.debug & DEBUG_CONNECT)
	log(LOG_DEBUG, "reusability check: %s", conn->host->name);

	/* check possibility of reuse connection */
	for (rmxp = conn->host->mx_ref; rmxp != NULL; rmxp = rmxp->mx_ref)
	{
		domp = rmxp->domain;
		reused = 0;
		for (rcptp = domp->rcpt_top; rcptp != NULL;
			rcptp = rcptp->dom_chain)
		{
			if (rcptp->stat == RCPT_DONE)
				continue;
			if (rcptp->stat == RCPT_WORK)
				reused = 1;
			else if (rcptp->stat == RCPT_RETRY
			      || rcptp->stat == RCPT_NOOP)
			{
				reused = 2;
				break;
			}
		}
		if (cnf.debug & DEBUG_CONNECT)
		log(LOG_DEBUG, "on domain: %s = %d", domp->name, reused);

		if (reused == 0)
			continue;
		mxp = domp->curmx;
		if (reused == 1)
			mxp = mxp->next;
		for ( ; mxp != NULL; mxp = mxp->next)
		{
			if (mxp == rmxp)
			{
				if (cnf.debug & DEBUG_CONNECT)
				log(LOG_DEBUG, "%d/%d: may be reused: %s",
					conn->seq, conn->socket,
					conn->host->name);
				return 1;
			}
		}
	}
	return 0;
}

static int
smtp_data_t(conn)
struct connection *conn;
{
	char buf[MAXLINE], tmpbuf[MAXLINE], *rstr;
	int rcode, savestate;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (!(IS2xx(rcode) || IS4xx(rcode) || IS5xx(rcode)))
	{
		rcode = SMTP_ERROR(54);
		snprintf(tmpbuf, sizeof(tmpbuf),
			"Remote protocol error in state %s: %s",
			STATE(savestate), buf);
		rstr = newstr(tmpbuf);
	}
	else
	{
		if (buf[4] == '\0')
			rstr = "Error by unknown reason";
		else
			rstr = newstr(buf + 4);
	}

	markstatus(conn->rcpt, rcode, rstr, 1);

	if (IS2xx(rcode))
	{
		conn->host->state = STAT_SENT;
		sti.noktrans++;
		if (conn->reuse)
			return -1;	/* without QUIT */
		if (cnf.smtp_reuse && may_be_reused(conn))
			return -1;	/* without QUIT */
		smtp_quit_s(conn);
		return -1;	/* to close transaction */
	}

	conn->host->state = STAT_FAIL;
	conn->host->stat = rcode;
	conn->host->response = rstr;
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_rset_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];

	strcpy(cmdbuf, "RSET");
	if (smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	conn->host->state = STAT_RSETSENT;
	return 0;
}

static int
smtp_rset_r(conn)
struct connection *conn;
{
	char buf[MAXLINE];
	int rcode, savestate;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (IS2xx(rcode))
	{
		if (Terminate)
		{
			sti.ntimeout++;
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				"Processing timed out (RSET)", 1);
			smtp_quit_s(conn);
			return -1;
		}
		return smtp_mail_s(conn);
	}

	/* when RSET is not supported at server side */
	smtp_quit_s(conn);
	return -1;
}

static int
smtp_quit_s(conn)
struct connection *conn;
{
	char cmdbuf[MAXLINE];

	strcpy(cmdbuf, "QUIT");
	if (smtp_send_command(conn, cmdbuf, sizeof(cmdbuf), 1) < 0)
	{
		/* XXX */
		smtp_close(conn->socket);
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}
	if (conn->host->state == STAT_FAIL)
		conn->host->state = STAT_FAILQUIT;
	else if (conn->host->state == STAT_RSETSENT)
		conn->host->state = STAT_QUITREOPEN;
	else
		conn->host->state = STAT_QUITSENT;
	return 0;
}

static int
smtp_quit_r(conn)
struct connection *conn;
{
	char buf[MAXLINE];
	int rcode, savestate, prevstate;

	savestate = conn->host->state;
	rcode = smtp_get_reply(conn, NULL, buf, sizeof(buf), NULL);

	if (rcode == 0)
		return 0;
	NewTransCheck = 1;
	smtp_close(conn->socket);
	if (rcode < 0)
	{
		if (errno > 0)
		{
			log(LOG_INFO, "%d/%d: failed for %s: %s (%s)",
				conn->seq, conn->socket, conn->host->name,
				smtpstrerror(errno), STATE(savestate));
		}
		conn->host->current->stat = EX_TEMPFAIL;
		if (conn->host->current->next == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		return -1;
	}

	if (IS2xx(rcode))
	{
		sti.nquitok++;
		if (conn->host->state == STAT_FAILQUIT)
		{
			conn->host->state = STAT_FAIL;
			return -1;
		}
		prevstate = conn->host->state;
		conn->host->state = STAT_CLOSED;
		if (Terminate)
			return 0;
		if (prevstate == STAT_QUITREOPEN)
			return smtp_connect(conn);
		conn->host = NULL;
		return 0;
	}

	conn->host->state = STAT_FAIL;
	conn->host->stat = rcode;
	conn->host->response = newstr(buf);
	return -1;
}

static int
smtp_timeout(conn)
struct connection *conn;
{
	char buf[MAXLINE];

	snprintf(buf, sizeof(buf), "Connection timed out (in state %s)",
		STATE(conn->host->state));
	switch (conn->host->state)
	{
	  case STAT_CONNECTING:
		conn->host->current->stat = EX_TEMPFAIL;
		if (conn->host->current == NULL)
			conn->host->state = STAT_FAIL;
		else
			conn->host->state = STAT_CLOSED;
		if (Terminate || conn->host->current->next == NULL)
			markstatus(conn->rcpt, SMTP_TEMPFAIL(52),
				newstr(buf), 1);
		conn->host->current = conn->host->current->next;
		if (Terminate)
			return -1;
		return smtp_connect(conn);

	  case STAT_CONNECTED:
	  case STAT_EHLOSENT:
	  case STAT_HELOSENT:
	  case STAT_MAILSENT:
	  case STAT_RCPTSENT:
	  case STAT_DATASENT:
	  case STAT_DATABODY:
	  case STAT_DTERMSENT:
	  case STAT_RSETSENT:
	  case STAT_QUITSENT:
	  case STAT_QUITREOPEN:
	  case STAT_FAILQUIT:
		conn->host->state = STAT_FAIL;
		conn->host->stat = EX_TEMPFAIL;
		if (conn->rcpt != NULL)
			markstatus(conn->rcpt, SMTP_TEMPFAIL(51),
				newstr(buf), 1);
		smtp_close(conn->socket);
		return -1;

	  case STAT_SENT:
		smtp_quit_s(conn);
		break;
	}
	return 0;
}

static void
dump_internal()
{
	struct recipient *rcptp;
	int i;

	log(LOG_INFO, "=== dumping internal information ===");

	for (rcptp = env.rcpt_list; rcptp != NULL; rcptp = rcptp->next)
	{
		switch (rcptp->stat)
		{
		    case RCPT_NOOP:
			log(LOG_INFO, "rcpt=%s stat=noop", rcptp->address);
			break;
		    case RCPT_WORK:
			log(LOG_INFO, "rcpt=%s stat=working mx=%s st=%s",
				rcptp->address,
				(rcptp->domain->curmx == NULL)?"none":
				rcptp->domain->curmx->name,
				(rcptp->domain->curmx == NULL)?"?":
				STATE(rcptp->domain->curmx->host->state));
			break;
		    case RCPT_RETRY:
			if (rcptp->domain->curmx != NULL)
				log(LOG_INFO,
				    "rcpt=%s stat=retry code=%d mx=%s st=%s",
				    rcptp->address, rcptp->result,
				    rcptp->domain->curmx->name,
				    STATE(rcptp->domain->curmx->host->state));
			else
				log(LOG_INFO,
					"rcpt=%s stat=retry code=%d mx=none",
					rcptp->address, rcptp->result);
			break;
		    case RCPT_DONE:
			log(LOG_INFO, "rcpt=%s stat=done code=%d",
				rcptp->address, rcptp->result);
			break;
		}
	}

	for (i = 0; i < cnf.sd_max; i++)
	{
		if (connwork[i].host == NULL)
			continue;
		log(LOG_INFO, "slot=%d host=%s stat=%s", i,
			connwork[i].host->name,
			STATE(connwork[i].host->state));
	}

	log(LOG_INFO, "=== end of internal information ===");
}

static char *
smtpstrerror(eno)
int eno;
{
	switch (eno)
	{
	  case E_SF_IO:
		return "Unexpected close by foreign host";
	  case E_SF_PROTO:
		return "Remote protocol error";
	  case E_SF_ABORT:
		return "Service shutting down";
	}
	return strerror(eno);
}
