/*
 * ratStdFolder.c --
 *
 *      This file contains code which implements standard c-client folders.
 *	This means ONLY filefolders at this moment.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratStdFolder.h"


/*
 * This is the private part of a std folder info structure.
 */

typedef struct StdFolderInfo {
    MAILSTREAM *stream;		/* Handler to c-client entity */
    int referenceCount;		/* Number of entities referencing ths entry */
    int exists;			/* Number of messages which actually exists
				   in this folder */
    int isNotYet;		/* True if the mailbox file doesn't exist yet */
    char *origName;		/* The original name of this folder */
    RatStdFolderType type;	/* The exact type of this folder */
    char *host;			/* The host it resides on (if networked) */
    char *user;			/* The username (if networked) */
} StdFolderInfo;

/*
 * We use this structure to keep a list of open connections
 */

typedef struct Connection {
    MAILSTREAM *stream;		/* Handler to c-client entity */
    RatStdFolderType type;	/* Type of folder */
    char *host;			/* Host we are connected to */
    char *user;			/* User we are connected as */
    int closing;		/* True if this connection isunused and
				   waiting to be closed */
    Tcl_TimerToken token;	/* Timer token for closing timer */
    struct Connection *next;	/* Struct linkage */
} Connection;

/*
 * Use this list to remember passwords
 */
typedef struct MMPasswd {
    char *host;
    char *user;
    char *password;
    RatStdFolderType type;
    struct MMPasswd *next;
    Tcl_TimerToken token;
} MMPasswd;

/*
 * Remember if we must initialize the package
 */
static int initialize = 1;

/*
 * List of open connections
 */
static Connection *connListPtr = NULL;

/*
 * List of remembered passwords
 */
static MMPasswd *passwdListPtr = NULL;
static MMPasswd *passwdCandidate = NULL;

/*
 * Interpreter used for notifying the user (calls to RatLog).
 */
static Tcl_Interp *stdInterp;

/*
 * Global table of folders.
 */
static Tcl_HashTable folderTable;

/*
 * The values below are used to catch calls to mm_log. That is when you
 * want to handle the message internally.
 */
RatLogLevel logLevel;
char *logMessage = NULL;

/*
 * These variables are used by mm_login
 */
static char *loginUser = NULL;

/*
 * List of flag names
 */
static char *stdFlagNames[] = {
    RAT_SEEN_STR,
    RAT_DELETED_STR,
    RAT_FLAGGED_STR,
    RAT_ANSWERED_STR
};

/*
 * The types of folders. This list is closely related to RatStdFolderType
 * in ratStdFolder.h
 */
static char *ratStdTypeNames[] = {"berkely", "imap", "pop3", "mh"};

/*
 * This is used to build a list of found mailboxes when listing
 */
typedef struct Mailbox {
    char *name;			/* The nameof this mailbox */
    char *spec;			/* The specificaation to send to c-client */
    long attributes;		/* The attrubutes from c-client */
    struct Mailbox *children;	/* Pointer to the list of children */
    struct Mailbox *next;	/* Pointer to the next mailbox on this level */
} Mailbox;
static Mailbox *mailboxListPtr = NULL;
static void BuildIMAPList(Mailbox *mPtr, Tcl_DString *dsptr);

/*
 * Procedures private to this module.
 */
static RatCloseProc Std_CloseProc;
static RatUpdateProc Std_UpdateProc;
static RatInsertProc Std_InsertProc;
static RatSetFlagProc Std_SetFlagProc;
static RatGetFlagProc Std_GetFlagProc;
static RatInfoProc Std_InfoProc;
static RatCreateProc Std_CreateProc;
static MAILSTREAM *OpenAndConvert(Tcl_Interp *interp, RatStdFolderType type,
	char *host, char *name, char *user);
static Tcl_CmdProc StdListIMAPFolders;
static Tcl_TimerProc CloseConnection;
static Tcl_TimerProc ErasePasswd;
static Tcl_CmdProc StdJudgeFolder;


/*
 *----------------------------------------------------------------------
 *
 * RatStdFolderInit --
 *
 *      Initializes the file folder command.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	The C-client library is initialized and the apropriate mail drivers
 *	are linked.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatStdFolderInit(Tcl_Interp *interp)
{
    stdInterp = interp;
    Tcl_InitHashTable(&folderTable, TCL_ONE_WORD_KEYS);
    mail_link(&unixdriver);
    mail_link(&mmdfdriver);
    mail_link(&dummydriver);
    mail_link(&imapdriver);
    mail_link(&pop3driver);
    mail_link(&mhdriver);
    auth_link(&auth_log);
    Tcl_CreateCommand(interp, "RatListIMAP", StdListIMAPFolders, NULL, NULL);
    Tcl_CreateCommand(interp, "RatImapCreateFolder", StdJudgeFolder,
	    (ClientData)1, NULL);
    Tcl_CreateCommand(interp, "RatImapDeleteFolder", StdJudgeFolder,
	    (ClientData)0, NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * OpenAndConvert --
 *
 *      Opens a standard c-client folder and if it is a filefolder and
 *	is of an incompatible format (unfortunately generated by an older
 *	version of this program) we convert it.
 *
 * Results:
 *	The mail stream.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static MAILSTREAM*
OpenAndConvert(Tcl_Interp *interp, RatStdFolderType type, char *host,
	char *name, char *user)
{
    MAILSTREAM *stream = NULL;
    Connection *connPtr;
    int doCache = 0;

    if ('{' == name[0]) {
	for (connPtr = connListPtr; connPtr; connPtr = connPtr->next) {
	    if (connPtr->closing && !strcmp(host, connPtr->host)
		    && !strcmp(connPtr->user, user)) {
		break;
	    }
	}
	if (connPtr) {
	    stream = connPtr->stream;
	    if (connPtr->token) {
		Tcl_DeleteTimerHandler(connPtr->token);
	    }
	    connPtr->closing = 0;
	} else {
	    Tcl_GetBoolean(stdInterp,
		    Tcl_GetVar2(stdInterp, "option", "cache_passwd",
		    TCL_GLOBAL_ONLY), &doCache);
	    if (doCache) {
		passwdCandidate = (MMPasswd*)malloc(sizeof(MMPasswd)+
			strlen(host)+1);
		passwdCandidate->host = (char*)passwdCandidate+sizeof(MMPasswd);
		strcpy(passwdCandidate->host, host);
		passwdCandidate->type = type;
		passwdCandidate->password = NULL;
	    }
	}
    }
    stream = mail_open(stream, name, NIL);
    if (NIL == stream && name[0] == '/') {
	FILE *fpIn = NULL, *fpOut;
	struct stat statbuf;
	char tmp[1024], buf[1024];
	char *re = "^From ([^ @]+)(@[^ @]+)? (Mon|Tue|Wed|Thu|Fri|Sat|Sun), ([0-9]+) (Jan|Feb|Mar|Jun|Jul|Aug|Sep|Okt|Nov|Dec) (19[0-9][0-9]) ([0-9][0-9]:[0-9][0-9]:[0-9][0-9]) ([-+0-9]*)\n";
	char *startPtr, *endPtr, c;
	Tcl_RegExp regexp;
	int converted = 0;

	sprintf(tmp, "%s.tmp", name);
	if ((fpIn = fopen(name, "r"))
		&& !fstat(fileno(fpIn), &statbuf)
		&& S_ISREG(statbuf.st_mode)
		&& statbuf.st_size > 0
		&& stat(tmp, &statbuf)
		&& (fpOut = fopen(tmp, "w"))) {
	    regexp = Tcl_RegExpCompile(interp, re);
	    while (fgets(buf, sizeof(buf), fpIn), !feof(fpIn)) {
		if (Tcl_RegExpExec(interp, regexp, buf, buf)) {
		    converted = 1;
		    Tcl_RegExpRange(regexp, 1, &startPtr, &endPtr);
		    c = *endPtr; *endPtr = '\0';
		    fprintf(fpOut, "From %s", startPtr);
		    *endPtr = c;
		    Tcl_RegExpRange(regexp, 2, &startPtr, &endPtr);
		    if (startPtr) {
			*endPtr = '\0';
			fprintf(fpOut, startPtr);
		    }
		    Tcl_RegExpRange(regexp, 3, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s", startPtr);
		    Tcl_RegExpRange(regexp, 5, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s", startPtr);
		    Tcl_RegExpRange(regexp, 4, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s", startPtr);
		    Tcl_RegExpRange(regexp, 7, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s", startPtr);
		    Tcl_RegExpRange(regexp, 6, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s", startPtr);
		    Tcl_RegExpRange(regexp, 8, &startPtr, &endPtr);
		    *endPtr = '\0';
		    fprintf(fpOut, " %s\n", startPtr);
		} else {
		    if (!converted) {
			break;
		    }
		    fputs(buf, fpOut);
		}
	    }
	    fclose(fpIn);
	    if (0 == fclose(fpOut) && converted) {
		logLevel = RAT_BABBLE;
		rename(tmp, name);
		stream = mail_open(NULL, name, 0);
	    }
	} else {
	    if (fpIn) {
		fclose(fpIn);
	    }
	}
    }
    if (RAT_IMAP == type && NIL != stream) {
	Tcl_GetBoolean(stdInterp,
		Tcl_GetVar2(stdInterp, "option", "cache_conn",
		TCL_GLOBAL_ONLY), &doCache);
	if (doCache) {
	    connPtr = (Connection*)malloc(sizeof(Connection));
	    connPtr->stream = stream;
	    connPtr->type = type;
	    connPtr->host = cpystr(host);
	    connPtr->user = cpystr(user);
	    connPtr->closing = 0;
	    connPtr->next = connListPtr;
	    connListPtr = connPtr;
	}
    }
    if (passwdCandidate) {
	if (passwdCandidate->password && NIL != stream) {
	    int timeout;

	    passwdCandidate->next = passwdListPtr;
	    passwdListPtr = passwdCandidate;

	    Tcl_GetInt(interp,
		       Tcl_GetVar2(interp, "option", "cache_passwd_timeout",
		       TCL_GLOBAL_ONLY), &timeout);

	    if (timeout) {
		passwdCandidate->token = Tcl_CreateTimerHandler(timeout*1000,
			ErasePasswd, (ClientData)passwdCandidate);
	    } else {
		passwdCandidate->token = NULL;
	    }
	} else {
	    free(passwdCandidate);
	}
	passwdCandidate = NULL;
    }
    return stream;
}


/*
 *----------------------------------------------------------------------
 *
 * OpenStdFolder --
 *
 *      Opens a standard c-client folder and if it is a filefolder and
 *	is of an incompatible format (unfortunately generated by an older
 *	version of this program) we convert it.
 *
 * Results:
 *	The mail stream.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

MAILSTREAM*
OpenStdFolder(Tcl_Interp *interp, char *name, char *prot, char *user,
	void *voidPtr)
{
    MAILSTREAM *stream = NULL;
    RatStdFolderType type;
    StdFolderInfo *stdPtr = (StdFolderInfo*)voidPtr;
    char *host, *cPtr;
    struct stat sbuf;
    int i, isNotYet = 0;

    loginUser = user;
    if ( '{' == name[0] ) {
	for (i=1; '}' != name[i] && '/' != name[i]; i++);
	host = (char*)malloc(i);
	strncpy(host, &name[1], i-1);
	host[i-1] = '\0';

	if (!strcasecmp(prot, "pop3")) {
	    type = RAT_POP;
	} else {
	    type = RAT_IMAP;
	}
    } else {
	host = NULL;
	if ('#' == name[0]) {
	    type = RAT_MH;
	} else {
	    type = RAT_BERKELY;
	}
    }
    if ('/' == name[0] && stat(name, &sbuf) && ENOENT == errno) {
	/*
	 * This entry points to a file which doesn't exist. Check if the
	 * directory exists and in that case we accept.
	 */
	for (cPtr = name+strlen(name); '/' != *cPtr; cPtr--);
	*cPtr = '\0';
	if (stat(name, &sbuf) || !S_ISDIR(sbuf.st_mode)) {
	    *cPtr = '/';
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    name, "\"", (char *) NULL);
	    return NULL;
	}
	isNotYet = 1;
	*cPtr = '/';
    } else {
	logLevel = RAT_BABBLE;
	stream = OpenAndConvert(interp, type, host, name, user);
	if (logLevel > RAT_WARN) {
	    if (host) {
		free(host);
	    }
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return NULL;
	}
	if (NIL == stream) {
	    if (host) {
		free(host);
	    }
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    name, "\"", (char *) NULL);
	    return NULL;
	}
    }
    if (stdPtr) {
	stdPtr->stream = stream;
	stdPtr->referenceCount = 1;
	stdPtr->exists = isNotYet ? 0 : stream->nmsgs;
	stdPtr->isNotYet = isNotYet;
	stdPtr->origName = cpystr(name);
	stdPtr->type = type;
	stdPtr->host = host;
	stdPtr->user = loginUser ? cpystr(loginUser) : NULL;
    } else if (host && *host) {
	free(host);
    }
    return stream;
}


/*
 *----------------------------------------------------------------------
 *
 * RatStdFolderCreate --
 *
 *      Creates a std folder entity.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	A std folder is created.
 *
 *
 *----------------------------------------------------------------------
 */

RatFolderInfo*
RatStdFolderCreate(Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_HashEntry *entryPtr;
    RatFolderInfo *infoPtr;
    StdFolderInfo *stdPtr;
    MAILSTREAM *stream = NULL;
    int new;

    /*
     * Now it is time to initialize things
     */
    if (initialize) {
	env_parameters(SET_LOCALHOST, currentHost);
	initialize = 0;
    }
    if (argc != 3 && argc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" ", argv[1], " name [user prot]\"", (char *) NULL);
	return (RatFolderInfo *) NULL;
    }

    stdPtr = (StdFolderInfo *) ckalloc(sizeof(*stdPtr));

    if (5 == argc) {
	stream = OpenStdFolder(interp, argv[2], argv[4], argv[3], stdPtr);
    } else {
	stream = OpenStdFolder(interp, argv[2], NULL, NULL, stdPtr);
    }
    if (!stream && !stdPtr->isNotYet) {
	free(stdPtr);
	return NULL;
    }

    infoPtr = (RatFolderInfo *) ckalloc(sizeof(*infoPtr)); 

    infoPtr->name = cpystr(argv[2]);
    infoPtr->type = "std";
    infoPtr->size = -1;
    if (stdPtr->isNotYet) {
	infoPtr->number = 0;
    } else {
	infoPtr->number = stream->nmsgs;
	entryPtr = Tcl_CreateHashEntry(&folderTable, (char*)stream, &new);
	Tcl_SetHashValue(entryPtr, (ClientData) stdPtr);
    }
    infoPtr->closeProc = Std_CloseProc;
    infoPtr->updateProc = Std_UpdateProc;
    infoPtr->insertProc = Std_InsertProc;
    infoPtr->setFlagProc = Std_SetFlagProc;
    infoPtr->getFlagProc = Std_GetFlagProc;
    infoPtr->infoProc = Std_InfoProc;
    infoPtr->createProc = Std_CreateProc;
    infoPtr->private = (ClientData) stdPtr;

    loginUser = NULL;
    return infoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_CloseProc --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
void
CloseStdFolder(Tcl_Interp *interp, MAILSTREAM *stream)
{
    Connection **connPtrPtr;

    for (connPtrPtr = &connListPtr;
	    *connPtrPtr && stream != (*connPtrPtr)->stream;
	    connPtrPtr = &(*connPtrPtr)->next);
    if (*connPtrPtr) {
	int timeout;

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "cache_conn_timeout",
		TCL_GLOBAL_ONLY), &timeout);

	(*connPtrPtr)->closing = 1;
	if (timeout) {
	    (*connPtrPtr)->token = Tcl_CreateTimerHandler(timeout*1000,
		    CloseConnection, (ClientData)*connPtrPtr);
	} else {
	    (*connPtrPtr)->token = NULL;
	}
    } else {
	mail_close_full(stream, NIL);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Std_CloseProc --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_CloseProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    int expunge;

    if (stdPtr->isNotYet) {
	ckfree(stdPtr);
	return TCL_OK;
    }
    Tcl_GetBoolean(interp,
	   Tcl_GetVar2(interp, "option", "expunge_on_close", TCL_GLOBAL_ONLY),
	   &expunge);
    if (expunge) {
	mail_expunge(stdPtr->stream);
    }
    CloseStdFolder(interp, stdPtr->stream);
    if (0 == --stdPtr->referenceCount) {
	Tcl_HashEntry *entryPtr;
	if ((entryPtr=Tcl_FindHashEntry(&folderTable, (char*)stdPtr->stream))) {
	    Tcl_DeleteHashEntry(entryPtr);
	}
	if (stdPtr->host) {
	    ckfree(stdPtr->host);
	    ckfree(stdPtr->user);
	}
	ckfree(stdPtr->origName);
	ckfree(stdPtr);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_UpdateProc --
 *
 *      See the documentation for updateProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for updateProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_UpdateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int expunge)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;

    if (stdPtr->isNotYet) {
	Tcl_HashEntry *entryPtr;
	struct stat sbuf;
	int new;

	if (stat(stdPtr->origName, &sbuf)) {
	    return 0;
	}
	logLevel = RAT_BABBLE;
	stdPtr->stream = OpenAndConvert(interp, stdPtr->type, stdPtr->host,
		stdPtr->origName, NULL);
	if (logLevel > RAT_WARN) {
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return 0;
	}
	if (NIL == stdPtr->stream) {
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    stdPtr->origName, "\"", (char *) NULL);
	    return 0;
	}
	stdPtr->exists = stdPtr->stream->nmsgs;
	entryPtr = Tcl_CreateHashEntry(&folderTable,(char*)stdPtr->stream,&new);
	Tcl_SetHashValue(entryPtr, (ClientData) stdPtr);
	stdPtr->isNotYet = 0;
    } else {
	if (expunge) {
	    MESSAGECACHE *cachePtr;
	    char sequence[16];
	    int i, offset = 0;

	    if (infoPtr->number) {
		sprintf(sequence, "1:%d", infoPtr->number);
		mail_fetchfast_full(stdPtr->stream, sequence, NIL);
		for (i=0; i<infoPtr->number; i++) {
		    cachePtr = mail_elt(stdPtr->stream, i+1);
		    if (cachePtr->deleted) {
			if (-1 != infoPtr->size) {
			    infoPtr->size -= cachePtr->rfc822_size;
			}
			if (infoPtr->msgCmdPtr[i]) {
			    RatMessageDelete(interp, infoPtr->msgCmdPtr[i]);
			}
			offset++;
		    } else if (offset) {
			infoPtr->msgCmdPtr[i-offset] = infoPtr->msgCmdPtr[i];
			infoPtr->privatePtr[i-offset] = infoPtr->privatePtr[i];
			if (infoPtr->msgCmdPtr[i]) {
			    RatMessageSetIndex(interp, infoPtr->msgCmdPtr[i],
					       i - offset);
			}
		    }
		}
		for (i=infoPtr->number-offset; i<infoPtr->number; i++) {
		    infoPtr->msgCmdPtr[i] = NULL;
		    infoPtr->privatePtr[i] = NULL;
		}
	    }
	    mail_expunge(stdPtr->stream);
	    infoPtr->number = stdPtr->exists;
	} else {
	    if (T != mail_ping(stdPtr->stream)) {
		RatLog(interp, RAT_FATAL, "Mailbox stream died", 0);
		exit(1);
	    }
	}
    }
    return stdPtr->exists;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_InsertProc --
 *
 *      See the documentation for insertProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for insertProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_InsertProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int argc,
	char *argv[])
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    char flags[128], date[128];
    Tcl_CmdInfo cmdInfo;
    Tcl_DString ds;
    STRING string;
    int i;

    if (stdPtr->isNotYet) {
	Tcl_HashEntry *entryPtr;
	int new, perm;

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		TCL_GLOBAL_ONLY), &perm);
	close(open(stdPtr->origName, O_CREAT, perm));
	logLevel = RAT_BABBLE;
	stdPtr->stream = OpenAndConvert(interp, stdPtr->type, stdPtr->host,
		stdPtr->origName, NULL);
	if (logLevel > RAT_WARN) {
	    Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	    return TCL_ERROR;
	}
	if (NIL == stdPtr->stream) {
	    Tcl_AppendResult(interp, "Failed to open std mailbox \"",
		    argv[2], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	stdPtr->exists = stdPtr->stream->nmsgs;
	entryPtr = Tcl_CreateHashEntry(&folderTable,(char*)stdPtr->stream,&new);
	Tcl_SetHashValue(entryPtr, (ClientData) stdPtr);
	stdPtr->isNotYet = 0;
    }
    Tcl_DStringInit(&ds);
    for (i=0; i<argc; i++) {
	Tcl_GetCommandInfo(interp, argv[i], &cmdInfo);
	RatMessageGet(interp, (MessageInfo*)cmdInfo.clientData, &ds,flags,date);
	INIT(&string,mail_string,Tcl_DStringValue(&ds),Tcl_DStringLength(&ds));
	if (!mail_append_full(stdPtr->stream, stdPtr->origName, flags, date,
			      &string)){
	    fprintf(stderr, "mail_append failed\n");
	}
	Tcl_DStringSetLength(&ds, 0);
	if (!stdPtr->exists) {
	    mail_ping(stdPtr->stream);
	}
    }
    Tcl_DStringFree(&ds);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_SetFlagProc --
 *
 *      See the documentation for setFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for setFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_SetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag, int value)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    char sequence[8];

    if (stdPtr->stream->rdonly) {
	return TCL_OK;
    }

    sprintf(sequence, "%d", index+1);
    if (value) {
	mail_setflag_full(stdPtr->stream, sequence, stdFlagNames[flag], NIL);
    } else {
	mail_clearflag_full(stdPtr->stream, sequence, stdFlagNames[flag], NIL);
    }
    if (logLevel > RAT_WARN) {
	Tcl_SetResult(interp, logMessage, TCL_VOLATILE);
	return TCL_ERROR;
    } else {
	return TCL_OK;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Std_GetFlagProc --
 *
 *      See the documentation for getFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for getFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Std_GetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;
    MESSAGECACHE *cachePtr;
    char sequence[8];
    int value = 0;

    sprintf(sequence, "%d", index+1);
    logLevel = RAT_BABBLE;
    (void)mail_fetchstructure_full(stdPtr->stream, index+1, NIL, NIL);
    cachePtr = mail_elt(stdPtr->stream, index+1);
    switch (flag) {
    case RAT_SEEN:	value = cachePtr->seen; break;
    case RAT_DELETED:	value = cachePtr->deleted; break;
    case RAT_FLAGGED:	value = cachePtr->flagged; break;
    case RAT_ANSWERED:	value = cachePtr->answered; break;
    }
    return value;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_InfoProc --
 *
 *      See the documentation for infoProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for infoProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */

static char*
Std_InfoProc(Tcl_Interp *interp, ClientData clientData, RatFolderInfoType type,
	int index)
{
    RatFolderInfo *infoPtr = (RatFolderInfo*)clientData;
    int i = infoPtr->presentationOrder[index];

    if (NULL == infoPtr->privatePtr[i]) {
	infoPtr->msgCmdPtr[i] = Std_CreateProc(infoPtr, interp, i);
    }
    return Std_GetInfoProc(interp, (ClientData)infoPtr->privatePtr[i], type, 0);
}

/*
 *----------------------------------------------------------------------
 *
 * Std_CreateProc --
 *
 *      See the documentation for createProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for createProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static char*
Std_CreateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    StdFolderInfo *stdPtr = (StdFolderInfo *) infoPtr->private;

    return RatStdMessageCreate(interp, infoPtr, stdPtr->stream, index,
	    stdPtr->type, stdPtr->host, stdPtr->user);
}


/*
 *----------------------------------------------------------------------
 *
 * StdListIMAPFolders --
 *
 *      List the IMAP folders on an remote server
 *
 * Results:
 *	The folders found are returned as a list in interp->result;
 *
 * Side effects:
 *	RatLogin may be called.
 *
 *
 *----------------------------------------------------------------------
 */
static int
StdListIMAPFolders(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    Connection *connPtr;
    MAILSTREAM *stream = NIL;
    int doClose = 0;
    Tcl_DString ds;
    char *spec;

    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" host user pattern\"", (char *) NULL);
	return TCL_ERROR;
    }
    spec = (char*)malloc(strlen(argv[1])+strlen(argv[3])+3);

    /*
     * Find any open stream that matches, if not then we have to open one.
     */
    for (connPtr = connListPtr; !stream && connPtr; connPtr = connPtr->next) {
	if (!strcmp(connPtr->host, argv[1]) && !strcmp(connPtr->user, argv[2])){
	    stream = connPtr->stream;
	}
    }
    if (!stream) {
	loginUser = argv[2];
	sprintf(spec, "{%s}", argv[1]);
	stream = mail_open(NIL, spec, NIL);
	loginUser = NULL;
	doClose = 1;
    }
    if (!stream) {
	Tcl_SetResult(interp, "Failed to open connection", TCL_STATIC);
	return TCL_ERROR;
    }

    /*
     * Do the listing
     */
    sprintf(spec, "{%s}%s", argv[1], argv[3]);
    mail_list(stream, NIL, spec);
    free(spec);
    if (doClose) {
	mail_close_full(stream, NIL);
    }

    /*
     * Build the return string
     */
    Tcl_DStringInit(&ds);
    BuildIMAPList(mailboxListPtr, &ds);
    mailboxListPtr = NULL;
    Tcl_DStringResult(interp, &ds);

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * BuildIMAPList --
 *
 *      This function calls itself recursively to build the list of
 *	folders found.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given dstring is modified.
 *
 *
 *----------------------------------------------------------------------
 */
static void
BuildIMAPList(Mailbox *mPtr, Tcl_DString *dsPtr)
{
    Mailbox *nPtr;

    for (; mPtr; mPtr = nPtr) {
	Tcl_DStringStartSublist(dsPtr);
	Tcl_DStringAppendElement(dsPtr, mPtr->name);
	if (mPtr->children) {
	    Tcl_DStringAppendElement(dsPtr, "0");
	    Tcl_DStringStartSublist(dsPtr);
	    if (!(LATT_NOSELECT & mPtr->attributes)) {
		Tcl_DStringStartSublist(dsPtr);
		Tcl_DStringAppendElement(dsPtr, mPtr->name);
		Tcl_DStringAppendElement(dsPtr, "1");
		Tcl_DStringStartSublist(dsPtr);
		Tcl_DStringAppendElement(dsPtr, mPtr->spec);
		Tcl_DStringEndSublist(dsPtr);
		Tcl_DStringEndSublist(dsPtr);
	    }
	    BuildIMAPList(mPtr->children, dsPtr);
	    Tcl_DStringEndSublist(dsPtr);
	} else {
	    Tcl_DStringAppendElement(dsPtr, "1");
	    Tcl_DStringStartSublist(dsPtr);
	    Tcl_DStringAppendElement(dsPtr, mPtr->spec);
	    Tcl_DStringEndSublist(dsPtr);
	}
	Tcl_DStringEndSublist(dsPtr);
	nPtr = mPtr->next;
	free(mPtr->name);
	if (mPtr->spec) {
	    free(mPtr->spec);
	}
	free(mPtr);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * ErasePasswd --
 *
 *      Removes a passwd from the cache
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The connection list is modified.
 *
 *
 *----------------------------------------------------------------------
 */
static void
ErasePasswd(ClientData clientData)
{
    MMPasswd **pwdPtrPtr, *pwdPtr = (MMPasswd*)clientData;

    for (pwdPtrPtr = &passwdListPtr; *pwdPtrPtr != pwdPtr;
	    pwdPtrPtr = &(*pwdPtrPtr)->next);
    *pwdPtrPtr = pwdPtr->next;
    free(pwdPtr->user);
    memset(pwdPtr->password, '\0', strlen(pwdPtr->password));
    free(pwdPtr->password);
    free(pwdPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * CloseConnection --
 *
 *      Closes a connection.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The connection list is modified.
 *
 *
 *----------------------------------------------------------------------
 */
static void
CloseConnection(ClientData clientData)
{
    Connection **connPtrPtr, *connPtr = (Connection*)clientData;

    mail_close_full(connPtr->stream, NIL);
    for (connPtrPtr = &connListPtr; *connPtrPtr != connPtr;
	    connPtrPtr = &(*connPtrPtr)->next);
    *connPtrPtr = connPtr->next;
    free(connPtr->host);
    free(connPtr->user);
    free(connPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * StdJudgeFolder --
 *
 *      Create or delete c-client folders
 *
 * Results:
 *	A standard Tcl-result
 *
 * Side effects:
 *	A folder may be created or deleted
 *
 *
 *----------------------------------------------------------------------
 */
static int
StdJudgeFolder(ClientData op, Tcl_Interp *interp, int argc, char *argv[])
{
    MAILSTREAM *stream = NULL;
    Connection *connPtr;
    int doCachePw, doCacheConn;
    char *host, *cPtr;

    if (3 != argc) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" folder user\"", (char *) NULL);
	return TCL_ERROR;
    }
    if ('{' != argv[1][0]) {
	Tcl_SetResult(interp, "You can only use this on IMAP-folders",
		TCL_STATIC);
	return TCL_ERROR;
    }

    /*
     * Look for reusable stream
     */
    for (cPtr = host = cpystr(argv[1]);
	    *cPtr && '/' != *cPtr && ':' != *cPtr && '}' != *cPtr;
	    cPtr++);
    *cPtr++ = '\0';
    for (connPtr = connListPtr; connPtr; connPtr = connPtr->next) {
	if (connPtr->closing && !strcmp(host, connPtr->host)
		&& !strcmp(connPtr->user, argv[2])) {
	    break;
	}
    }
    if (connPtr) {
	stream = connPtr->stream;
	if (connPtr->token) {
	    Tcl_DeleteTimerHandler(connPtr->token);
	}
	connPtr->closing = 0;
    } else {
	Tcl_GetBoolean(stdInterp,
		Tcl_GetVar2(stdInterp, "option", "cache_passwd",
		TCL_GLOBAL_ONLY), &doCachePw);
	if (doCachePw) {
	    passwdCandidate = (MMPasswd*)malloc(sizeof(MMPasswd)+
		    strlen(host)+1);
	    passwdCandidate->host = (char*)passwdCandidate+sizeof(MMPasswd);
	    strcpy(passwdCandidate->host, host);
	    passwdCandidate->type = RAT_IMAP;
	    passwdCandidate->password = NULL;
	}
    }

    /*
     * Open stream (if needed)
     */
    loginUser = argv[2];
    if (!stream) {
	if ( NULL == (stream = mail_open(NULL, argv[1], OP_HALFOPEN))) {
	    Tcl_SetResult(interp, "Failed to open stream", TCL_STATIC);
	    free(host);
	    if (passwdCandidate) {
		free(passwdCandidate);
		passwdCandidate = NULL;
	    }
	    return TCL_ERROR;
	}
    }

    /*
     * Do operation
     */
    if (op) {
	(void)mail_create(stream, argv[1]);
    } else {
	(void)mail_delete(stream, argv[1]);
    }

    Tcl_GetBoolean(stdInterp,
	    Tcl_GetVar2(stdInterp, "option", "cache_conn",
	    TCL_GLOBAL_ONLY), &doCacheConn);
    if (doCacheConn) {
	int timeout;

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "cache_conn_timeout",
		TCL_GLOBAL_ONLY), &timeout);

	connPtr = (Connection*)malloc(sizeof(Connection));
	connPtr->stream = stream;
	connPtr->type = RAT_IMAP;
	connPtr->host = cpystr(host);
	connPtr->user = cpystr(loginUser);
	connPtr->closing = 1;
	connPtr->next = connListPtr;
	if (timeout) {
	    connPtr->token = Tcl_CreateTimerHandler(timeout*1000,
		    CloseConnection, (ClientData)connPtr);
	} else {
	    connPtr->token = NULL;
	}
	connListPtr = connPtr;
    } else {
	mail_close_full(stream, NIL);
    }
    if (passwdCandidate) {
	if (passwdCandidate->password) {
	    int timeout;

	    passwdCandidate->next = passwdListPtr;
	    passwdListPtr = passwdCandidate;

	    Tcl_GetInt(interp,
		       Tcl_GetVar2(interp, "option", "cache_passwd_timeout",
		       TCL_GLOBAL_ONLY), &timeout);

	    if (timeout) {
		passwdCandidate->token = Tcl_CreateTimerHandler(timeout*1000,
			ErasePasswd, (ClientData)passwdCandidate);
	    } else {
		passwdCandidate->token = NULL;
	    }
	} else {
	    free(passwdCandidate);
	}
	passwdCandidate = NULL;
    }

    free(host);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * AppendToIMAP --
 *
 *      Append the given message to an IMAP folder
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The specified folder will be modified
 *
 *
 *----------------------------------------------------------------------
 */

void
AppendToIMAP(Tcl_Interp *interp, char *mailboxSpec, char *user, char *flags,
	     char *date, char *msg, int length)
{
    char buf[1024], *mailbox, *crlfMsg;
    Connection *connPtr;
    MAILSTREAM *stream;
    int i, doClose;
    STRING msgString;
    unsigned long crlfLength, buflen = length*1.05;

    crlfMsg = (char*)malloc(buflen);

    mailbox = RatLindex(interp, mailboxSpec, 0);
    for (i=0; mailbox[i+1] != '}'; i++) {
	buf[i] = mailbox[i+1];
    }
    buf[i] = '\0';
    for (connPtr = connListPtr; connPtr; connPtr = connPtr->next) {
	if (!strcmp(buf, connPtr->host) && !strcmp(connPtr->user, user)) {
	    break;
	}
    }
    if (connPtr) {
	stream = connPtr->stream;
	doClose = 0;
    } else {
	loginUser = user;
	stream = OpenAndConvert(interp, RAT_IMAP, buf, mailbox, user);
	if (NULL == stream) {
	    return;
	}
	doClose = 1;
    }

    crlfLength = strcrlfcpy(&crlfMsg, &buflen, msg, length);
    INIT(&msgString, mail_string, crlfMsg, crlfLength);
    mail_append_full(stream, mailbox, flags, date, &msgString);
    free(crlfMsg);

    if (doClose) {
	Connection **connPtrPtr;

	for (connPtrPtr = &connListPtr;
		*connPtrPtr && stream != (*connPtrPtr)->stream;
		connPtrPtr = &(*connPtrPtr)->next);
	if (*connPtrPtr) {
	    int timeout;
	    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option",
		    "cache_conn_timeout", TCL_GLOBAL_ONLY), &timeout);
	    (*connPtrPtr)->closing = 1;
	    (*connPtrPtr)->token = Tcl_CreateTimerHandler(timeout*1000,
		    CloseConnection, (ClientData)*connPtrPtr);
	} else {
	    mail_close_full(stream, NIL);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * ClearStdPasswds --
 *
 *      Clear the standard passwords from memory. If freethem is non
 *	null we also try to deallocate the memory.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The specified folder will be modified
 *
 *
 *----------------------------------------------------------------------
 */

void
ClearStdPasswds(int freethem)
{
    MMPasswd *pwPtr, *nextPtr;

    for (pwPtr = passwdListPtr; pwPtr; pwPtr = nextPtr) {
	memset(pwPtr->password, '\0', strlen(pwPtr->password));
	nextPtr = pwPtr->next;
	if (freethem) {
	    free(pwPtr->user);
	    free(pwPtr->password);
	    free(pwPtr);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * mm_*
 *
 *	The functions below are called from the C-client library. They
 *	are docuemnted in Internal.DOC.
 *
 *----------------------------------------------------------------------
 */
void mm_searched (MAILSTREAM *stream,unsigned long number)
{
}


void mm_exists (MAILSTREAM *stream,unsigned long number)
{
    Tcl_HashEntry *entryPtr;
    StdFolderInfo *stdPtr;

    if ((entryPtr = Tcl_FindHashEntry(&folderTable, (char*)stream))) {
	stdPtr = (StdFolderInfo *)Tcl_GetHashValue(entryPtr);
	stdPtr->exists = (int)number;
    }
}


void mm_expunged (MAILSTREAM *stream,unsigned long number)
{
    Tcl_HashEntry *entryPtr;
    StdFolderInfo *stdPtr;

    if ((entryPtr = Tcl_FindHashEntry(&folderTable, (char*)stream))) {
	stdPtr = (StdFolderInfo *)Tcl_GetHashValue(entryPtr);
	stdPtr->exists -= 1;
    }
}


void mm_mailbox (char *string)
{
}


void mm_bboard (char *string)
{
}


void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
    long flag = errflg;

    if (flag == BYE) {
	flag = NIL;
    }
    mm_log(string, flag);
}


void mm_log (char *string,long errflg)
{
    switch(errflg) {
    case NIL:	logLevel = RAT_BABBLE; break;
    case PARSE:	logLevel = RAT_PARSE; break;
    case WARN:	logLevel = RAT_WARN; break;
    case BYE:	logLevel = RAT_FATAL; break;
    case ERROR:	/* fallthrough */
    default:	logLevel = RAT_ERROR; break;
    }

    if (logMessage) {
	free(logMessage);
    }
    logMessage = cpystr(string);
    RatLog(stdInterp, logLevel, string, 0);
    if (logLevel > RAT_ERROR) {
	fprintf(stderr, "c-client error. TkRat is aborting...\n");
	RatDbClose();
	exit(1);
    }
}


void mm_dlog (char *string)
{
    RatLog(stdInterp, RAT_BABBLE, string, 0);
}


void mm_login (NETMBX *mbPtr, char *user, char *pwd, long trial)
{
    char **largv, buf[1024];
    MMPasswd *pwPtr;
    int largc, timeout;

    /*
     * Check for cached entry
     */
    sprintf(buf, "%s:%ld", mbPtr->host, mbPtr->port);
    for (pwPtr = passwdListPtr; pwPtr; pwPtr = pwPtr->next) {
	if ((!strcmp(pwPtr->host, mbPtr->host) || !strcmp(pwPtr->host, buf))
		&& !strcmp(pwPtr->user, loginUser)
		&& !strcasecmp(ratStdTypeNames[pwPtr->type], mbPtr->service)) {
	    strcpy(user, pwPtr->user);
	    strcpy(pwd, pwPtr->password);
	    if (pwPtr->token) {
		Tcl_DeleteTimerHandler(pwPtr->token);
	    }
	    Tcl_GetInt(stdInterp,
		       Tcl_GetVar2(stdInterp, "option", "cache_passwd_timeout",
		       TCL_GLOBAL_ONLY), &timeout);

	    if (timeout) {
		pwPtr->token = Tcl_CreateTimerHandler(timeout*1000,
			ErasePasswd, (ClientData)pwPtr);
	    } else {
		pwPtr->token = NULL;
	    }
	    return;
	}
    }
    sprintf(buf, " %lx ", trial);
    if (TCL_OK != Tcl_VarEval(stdInterp, "RatLogin ", mbPtr->host, buf,
			      loginUser?loginUser:"{}", " ",
			      mbPtr->service?mbPtr->service:"", NULL)
	    || TCL_OK != Tcl_SplitList(stdInterp, stdInterp->result,
		    &largc,&largv)
	    || 2 != largc) {
	return;
    }
    strcpy(user, largv[0]);
    loginUser = cpystr(largv[0]);
    strcpy(pwd, largv[1]);

    if (passwdCandidate) {
	passwdCandidate->password = cpystr(largv[1]);
	passwdCandidate->user = loginUser;
    }
    ckfree(largv);
}


void mm_critical (MAILSTREAM *stream)
{
}


void mm_nocritical (MAILSTREAM *stream)
{
}


long mm_diskerror (MAILSTREAM *stream,long errcode,long serious)
{
    char buf[64];

    sprintf(buf, "Disk error: %ld", errcode);
    RatLog(stdInterp, RAT_FATAL, buf, 0);
    return 1;
}


void mm_fatal (char *string)
{
    RatLog(stdInterp, RAT_FATAL, string, 0);
}

void mm_flags (MAILSTREAM *stream,unsigned long number)
{
}


void
mm_list(MAILSTREAM *stream, int delimiter, char *name,long attributes)
{
    Mailbox **mPtrPtr = &mailboxListPtr;
    char buf[1024], *cPtr, *elem;
    int found;

    strcpy(buf, strchr(name, '}')+1);

    /*
     * Teoretically this code could fail to import a folder if
     * it has the same name as a previous found structure. I do
     * not think this can happen... (famous last words?)
     */
    elem = buf;
    if (delimiter) {
	cPtr = strchr(buf, delimiter);
    } else {
	cPtr = NULL;
    }
    while (1) {
	found = 0;
	if (cPtr) {
	    *cPtr = '\0';
	}
	for (; *mPtrPtr; mPtrPtr = &(*mPtrPtr)->next) {
	    if (!strcmp((*mPtrPtr)->name, elem)) {
		found = 1;
		break;
	    }
	}
	if (!found) {
	    *mPtrPtr = (Mailbox*)malloc(sizeof(Mailbox));
	    (*mPtrPtr)->name = cpystr(elem);
	    if (cPtr) {
		(*mPtrPtr)->attributes = LATT_NOSELECT;
		(*mPtrPtr)->spec = NULL;
	    } else {
		(*mPtrPtr)->attributes = attributes;
		if (attributes & LATT_NOSELECT) {
		    (*mPtrPtr)->spec = NULL;
		} else {
		    (*mPtrPtr)->spec = cpystr(name);
		}
	    }
	    (*mPtrPtr)->children = NULL;
	    (*mPtrPtr)->next = NULL;
		
	}
	mPtrPtr = &(*mPtrPtr)->children;
	if (!cPtr) {
	    break;
	}
	elem = cPtr+1;
	cPtr = strchr(elem, delimiter);
    }
}


void
mm_lsub (MAILSTREAM *stream, int delimiter, char *name, long attributes)
{
    mm_list(stream, delimiter, name, attributes);
}


void mm_status (MAILSTREAM *stream, char *mailbox, MAILSTATUS *status)
{
}
