/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * central_def.c
 */

#include <model_headers.h>

/* central force problem */
/*
        This vector field describes of a particle moving under a
	central force potential in the plane.

	x' = vx
	y' = vy
	vx' = - alpha x / r3
	vy' = - alpha y / r3

	where
	rsqr = x x + y y
	r3 = rsqr sqrt(rsqr)

	Translation table:

		x[0] <--> x
		x[1] <--> y
		x[2] <--> vx
		x[3] <--> vy
		x[4] <--> time
		p[0] <--> alpha  (gravitational attraction)

*/

static void central_install(void);

/* ------------------------------------------------------------------------
   proc used to define the vector field 
   ------------------------------------------------------------------------ */
int 
central( double* f, double* x, double* p)
{
  double rsqr = x[0]*x[0]+x[1]*x[1];
  double r3 = rsqr * sqrt(rsqr);

  f[0] = x[2];
  f[1] = x[3];
  f[2] = - p[0] * x[0] / r3;
  f[3] = - p[0] * x[1] / r3;

  return(0);

}

/* ------------------------------------------------------------------------
   proc used to define functions of the variable, parameters or derivatives
   ------------------------------------------------------------------------ */
int 
central_func(double* f, double* x, double* p)
{  double T, U;

   T = (x[2]*x[2]+x[3]*x[3])/2.0;
   U = -p[0] / (x[0]*x[0]+x[1]*x[1]);
   f[0] = T + U;

   return 0;
}
/* ------------------------------------------------------------------------
   proc used to define jacobian
   ------------------------------------------------------------------------ */
int 
central_jac( double** m, double* x, double* p)
{

}

/* ---------------------------------
  proc for geomview
  ---------------------------------  */
int 
central_gv( double* f, double* x, double* p )
{
  int i;
  double c,s;
  double rsqr = x[0]*x[0]+x[1]*x[1];
  double r = sqrt(rsqr);

  /* c,s are cosine and sine for rotation in x[0],x[1] plane */
  
  c = x[0]/r;
  s = x[1]/r;

  f[0] = f[1] = f[3] = f[6] = f[7] = f[10] = f[11] = 0.0;
  f[2] = -1.0;
  f[4] = -s;
  f[5] = c;
  f[8] = c;
  f[9] = s;
  f[12] = x[0];
  f[13] = x[1];
  f[14] = 0;
  f[15] = 1.0;

  return 0;
}

/* ------------------------------------------------------------------------
   proc to define the default data for the dynamical system
   Note: You may change the entries for each variable but
	 DO NOT change the list of items.  If a variable is
	 unused, NULL or zero the entry, as appropriate.
   ------------------------------------------------------------------------ */
int 
central_init()
{

  /* define the dynamical system in this segment 
     ------------------------------------------------------------------------------------------- */
  int	 n_varb=4;					  /* dim of the phase spase */
  static char	*variable_names[]={"x","y", "vx", "vy"};       	  /* list of phase varb names */
  static double	variables[]={0.5,0.5,-0.5,0.45};			  /* initial conditions for the variables */
  static double	variable_min[]={-5.0,-5.0,-5.0,5.0};		  /* default varb min for display */
  static double	variable_max[]={5.0,5.0,5.0,5.0};		  /* default varb max for display */
  static char   *indep_varb_name="time";		  /* name of indep variable  */
  double indep_varb_min=0.;				  /* default indep varb min r display */
  double indep_varb_max=100.;				  /* default indep varb max r display */

  int	 n_param=1;					  /* dim of the parameter space */
  static char	*parameter_names[]={"gravity"};  /* list of param names */
  static double	parameters[]={0.5};		  /* initial parameter values */
  static double	parameter_min[]={0.0};		  /* default param min for display */
  static double	parameter_max[]={5.0};		  /* default param max for display */

  int	 n_funct=1;					  /* number of user-defined functions */
  static char	*funct_names[]={"Energy"};		  /* list of param names */
  static double	funct_min[]={-1};			  /* default varb max for display */
  static double	funct_max[]={1};			  /* default varb max for display */

  int	 manifold_type = EUCLIDEAN;			  /* EUCLIDEAN or PERIODIC (periodic variables) */
  static int	periodic_varb[] = {FALSE, FALSE};	  /* if PERIODIC, which variables are periodic */
  static double period_start[] = {-PI,0.};		  /* if PERIODIC, beginning of fundamental domain */
  static double period_end[] = {PI, 0.};		  /* if PERIODIC, end of fundamental domain */

  int 	 mapping_toggle=FALSE;				  /* is this a map? */
  int	 inverse_toggle=FALSE;				  /* if so, is inverse FALSE, APPROX_INV, or EXPLICIT_INV?*/

  /*  In this section, input NULL or the name of the function which contains... */
  int           (*def_name)()=central;		  /* the eqns of motion */
  int           (*jac_name)()=NULL;		  /* the jacobian (deriv w.r.t. space) */
  int           (*aux_func_name)()=central_func;	  /* the auxiliary functions */
  int           (*inv_name)()=NULL;			  /* the inverse or approx inverse */
  int           (*dfdt_name)()=NULL;			  /* the deriv w.r.t time */
  int           (*dfdparam_name)()=NULL;		  /* the derivs w.r.t. parameters */

#define GV
  int (*gv_funct)()=central_gv;   /* transformations for geomview */
  int (*gv_custom_funct)()=NULL;    /* nontransformation geomview commands */
  int gv_n=1;                       /* number of transformations */
  static char *gv_filename = "central.oogl";

  c_filename = __FILE__;  /* display this file for help with this system */

/* ------------------ end of dynamical system definition ------------------ */
#include <ds_define.c>
  pm (PUT, "Model.Install", central_install, NULL);

  return 0;
}


/* display a large cross at the origin */
static void
central_install(void)
{
  static double origin[] = {0.0, 0.0, 0.0, 0.0}; /* values to save */
  static double	variables[]={0.5,0.5,-0.5,0.45}; /* values to restore */

  /* change selected point and save it */
  pm(PUT_LIST, "Selected.Varb_Ic", 0, 3, origin,
     PUT, "Defaults.Symbol_Index", 5,
     EXEC, "Selected.Save_Point", NULL);
   
  /* reset stuff we changed */
  pm(PUT, "Defaults.Symbol_Index", 0,
     PUT_LIST, "Selected.Varb_Ic", 0, 3, variables, NULL);

  /* load rkqc integrator */
  pm(PUT, "Prop.Int_Num", 2,
     EXEC, "Prop.Load_Int", NULL);
}

