/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * plate_def.c
 */

#include <model_headers.h>

/* oscillating plate */
/*

   A plate is defined on [0,1]x[0,1].
   Its height is given by z = a(t) sin(2 pi x) + b(t) sin(pi y)
   where the amplitudes a(t),b(t) evolve according to:

   a' = eps_a b
   b' = -eps_b a

		x[0] <--> a
		x[1] <--> b
		x[2] <--> t
		p[0] <--> eps_a
		p[1] <--> eps_b
*/
/* ------------------------------------------------------------------------
   proc used to define the vector field 
   ------------------------------------------------------------------------ */
int 
plate( double* f, double* x, double* p )
{
  f[0] = p[0]*x[1];
  f[1] = -p[1]*x[0];
  return(0);

}

/* ------------------------------------------------------------------------
   proc used to define functions of the variable, parameters or derivatives
   ------------------------------------------------------------------------ */
int 
plate_func( double* f, double* x, double* p )
{
  return(0);
}
/* ------------------------------------------------------------------------
   proc used to define jacobian
   ------------------------------------------------------------------------ */
int 
plate_jac( double** m, double* x, double* p )
{
  m[0][0] = 0.0;
  m[0][1] = p[0];
  m[1][0] = -p[1];
  m[1][1] = 0.0;

  return 0;
}

/* ---------------------------------
  proc for geomview
  ---------------------------------  */
#define NX 20
#define NY 10
int 
plate_OFF( double* x, double* p )
{
  int i,j, faces, vertices, edges;
  double xi, yj, z;
  char buf[256];

  geomview_send("(progn\n");
  geomview_send("(read geometry {define plate_h { OFF\n");
  vertices = NX*NY;
  faces = (NX-1)*(NY-1);
  edges = (NX-1)*NY+NX*(NY-1);
  sprintf(buf, "%d %d %d\n", vertices, faces, edges);
  geomview_send(buf);
  /* specify the vertices */
  for (i=0; i<NX; i++)
    for(j=0; j<NY; j++)
      {
	xi = 4.0*i/NX-2.0;
	yj = 2.0*j/NY-1.0;
	z = x[0]*sin(PI*xi) + x[1]*cos(PI*yj/2.0);
	sprintf(buf,"%g %g %g\n",xi,yj,z);
	geomview_send(buf);
      }
  /* specify the faces */
  for (i=0; i<NX-1; i++)
    for (j=0; j<NY-1; j++)
      {
	sprintf(buf,"4 %d %d %d %d 0.0 0.0 1.0\n",
		i*NY+j, i*NY+j+1, i*NY+j+NY+1, i*NY+j+NY);
	geomview_send(buf);
      }
  geomview_send("} } )\n");
  geomview_send(")\n");

  return 0;
}

/* ------------------------------------------------------------------------
   proc to define the default data for the dynamical system
   Note: You may change the entries for each variable but
	 DO NOT change the list of items.  If a variable is
	 unused, NULL or zero the entry, as appropriate.
   ------------------------------------------------------------------------ */
int 
plate_init(void)
{

  /* define the dynamical system in this segment 
     ------------------------------------------------------------------------------------------- */
  int	 n_varb=2;					  /* dim of the phase spase */
  static char	*variable_names[]={"a","b"};       	  /* list of phase varb names */
  static double	variables[]={0.3,0.1};			  /* initial conditions for the variables */
  static double	variable_min[]={-1.0,-1.0};		  /* default varb min for display */
  static double	variable_max[]={1.0,1.0};		  /* default varb max for display */
  static char   *indep_varb_name="time";		  /* name of indep variable  */
  double indep_varb_min=0.;				  /* default indep varb min r display */
  double indep_varb_max=100.;				  /* default indep varb max r display */

  int	 n_param=2;					  /* dim of the parameter space */
  static char	*parameter_names[]={"eps_a","eps_b"};  /* list of param names */
  static double	parameters[]={1.0,1.0};		  /* initial parameter values */
  static double	parameter_min[]={0.0,0.0};		  /* default param min for display */
  static double	parameter_max[]={1.0,1.0};		  /* default param max for display */

  int	 n_funct=0;					  /* number of user-defined functions */
  static char	*funct_names[]={"Period"};		  /* list of param names */
  static double	funct_min[]={-1};			  /* default varb max for display */
  static double	funct_max[]={1};			  /* default varb max for display */

  int	 manifold_type = EUCLIDEAN;			  /* EUCLIDEAN or PERIODIC (periodic variables) */
  static int	periodic_varb[] = {TRUE, FALSE};	  /* if PERIODIC, which variables are periodic */
  static double period_start[] = {-PI,0.};		  /* if PERIODIC, beginning of fundamental domain */
  static double period_end[] = {PI, 0.};		  /* if PERIODIC, end of fundamental domain */

  int 	 mapping_toggle=FALSE;				  /* is this a map? */
  int	 inverse_toggle=FALSE;				  /* if so, is inverse FALSE, APPROX_INV, or EXPLICIT_INV?*/

  /*  In this section, input NULL or the name of the function which contains... */
  int           (*def_name)()=plate;		  /* the eqns of motion */
  int           (*jac_name)()=plate_jac;		  /* the jacobian (deriv w.r.t. space) */
  int           (*aux_func_name)()=plate_func;	  /* the auxiliary functions */
  int           (*inv_name)()=NULL;			  /* the inverse or approx inverse */
  int           (*dfdt_name)()=NULL;			  /* the deriv w.r.t time */
  int           (*dfdparam_name)()=NULL;		  /* the derivs w.r.t. parameters */

#define GV
  int (*gv_funct)()=NULL;       /* transformations for geomview */
  int (*gv_custom_funct)()=plate_OFF;    /* nontransformation geomview commands */
  int gv_n=0;                       /* number of transformations */
  static char *gv_filename = "plate.oogl";

  c_filename = __FILE__;

/* ------------------ end of dynamical system definition ------------------ */
#include <ds_define.c>

  return 0;
}

