/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/*
 * gv.c
 */

#include <stdio.h>
#include <string.h>
#include <limits.h>

#include <pm.h>
#include <constants.h>
#include <memory.h>
#include <view.h>
#include <symbols.h>
#include "dscolor.h"

extern int geomview_send(char *buf);

/* internal declarations */

static void gv_mem(memory m, char *name, int xtypes[], int xindices[]);
static void gv_fixed_mem(memory m, char *name, int xtypes[], int xindices[]);
static void geomview_bbox(double mins[], double maxs[]);
static void geomview_vbbox(double mins[], double maxs[]);

static DsColor* gvcolor;

void
gv_dismiss(int n)
{
#ifdef DEBUG
  fprintf(stdout,"Deregistering GV %d\n",n);
#endif
  dismiss_view(GEOMVIEW,n);
}

void
gv_config(int n, int hortype, int horindex, double hormin, double hormax,
	  int vertype, int verindex, double vermin, double vermax)
{
#ifdef DEBUG
  fprintf(stdout, "Setting range for GV %d\n",n);
#endif
}

void
gv_activate(int n)
{
#ifdef DEBUG
  fprintf(stdout, "Activating GV %d\n",n);
#endif
  add_view(GEOMVIEW, n);
}

void
gv_size(int n)
{
#ifdef DEBUG
  fprintf(stdout, "Setting size for GV %d\n",n);
#endif
}

void
gv_allmem(int xtypes[], int xindices[], double mins[], double maxs[])
{
  char *name;
  memory m;
  int i=0;

  if (geomview_send("(progn\n") == -1) return;
  geomview_send("(bbox-draw allgeoms no)\n(normalization allgeoms none)\n");
  geomview_vbbox(mins,maxs);
  geomview_send("(look-recenter dstool_bbox Camera)\n");
  while ( (name = (char *) pm(QUERY, "Memory", MEMRY, i, NULL)) )
    {
      i++;
      m = (memory) pm(GET, name, NULL);
      if (strcmp(name,"Memory.Fixed") == 0)
	gv_fixed_mem(m, strrchr(name,'.')+1, xtypes, xindices);
      else
	gv_mem(m, strrchr(name,'.')+1, xtypes, xindices);
    }
  geomview_send(")\n");
}

static void
gv_mem(memory m, char *name, int xtypes[], int xindices[])
{
  int fn=FALSE, flow_counter=0, traj_counter=0, i, j, npoints, *color;
  double x[3], *points, *params;
  char buf[256];
  XColor *xcolor;
  int mapping_flag = *((int *) pm(GET, "Model.Mapping_Flag", NULL));
  int pickcolor_flag = *((int *) pm(GET, "Geomview.PickColor", NULL));
  int pcolor;

  if ( (xtypes[0] == FUNCTION_VARB) || 
      (xtypes[1] == FUNCTION_VARB) || 
      (xtypes[2] == FUNCTION_VARB) )
    fn = TRUE;

  if (memory_reset_read(m) == 0) 
    {
      while (memory_read_next_flow(m, NULL, NULL, NULL, NULL, NULL) == 0)  
	{
	  flow_counter++;
	  traj_counter = 0;
	  while (memory_read_next_traj(m, &npoints, NULL, NULL) == 0)
	    {
	      traj_counter++;
	      /* start geomview VECT obj */
	      if (npoints > 0)
		{
		  sprintf(buf,"(geometry %s.%d.%d {appearance {linewidth 2} VECT\n",
			  name, flow_counter, traj_counter);
		  geomview_send(buf);
		  if (mapping_flag) {
		    /* don't connect the dots */
		    sprintf(buf, "%d %d 1\n", npoints, npoints);
		    geomview_send(buf);
		    for (i=0; i<npoints; i += 100) {
		      for (j=0; (j<100) && (i+j<npoints); j++)
			sprintf(buf+2*j, "1 ");
		      geomview_send(buf);
		    }
		    for (i=0; i<npoints; i += 100) {
		      for (j=0; (j<100) && (i+j<npoints); j++)
			if ((i==0) && (j==0))
			  sprintf(buf, "1 ");
			else sprintf(buf+2*j, "0 ");
		      geomview_send(buf);
		    }
		  } else {
		    /* connect the dots */
		    sprintf(buf, "1 %d 1\n%d\n1\n", npoints, npoints);
		    geomview_send(buf);
		  }
		}
	      while (memory_read_next_point(m, &points, &params, &color, 
					    NULL, NULL) == 0)
		{
		  if (fn == TRUE)
		     view_auxf(view_f, points, params);
		  for (i=0; i<3; i++)
		    {
		      switch(xtypes[i])
			{
			case PHASE_SPACE_VARB:
			  x[i] = points[xindices[i]];
			  break;
			case PARAMETER_VARB:
			  x[i] = params[xindices[i]];
			  break;
			case FUNCTION_VARB:
			  x[i] = view_f[xindices[i]];
			  break;
			default:
			  x[i] = 0.0;
			  break;
			}
		    }
		  /* add point to VECT object */
		  sprintf(buf, "%f %f %f\n", x[0], x[1], x[2]);
		  geomview_send(buf);
		}
	      /* end geomview vect obj
		 - these four numbers specify the color (RGBA).*/
 
              pcolor = DsColor_get_index( pickcolor_flag, color[0], color[1] );
              gvcolor = DsColor_get_rgb(pcolor);
 
	      sprintf(buf,"%f %f %f 1})\n",(float)(gvcolor->r/255.0),
		                           (float)(gvcolor->g/255.0), 
                                           (float)(gvcolor->b/255.0) );

	      geomview_send(buf);
 	    }
	}
    }
  
}

static void
gv_fixed_mem(memory m, char *name, int xtypes[], int xindices[])
{
  int fn=FALSE, flow_counter=0, traj_counter=0, i, j, npoints, *color;
  double x[3], *points, *params;
  char buf[256];
  XColor *xcolor;
  int pickcolor_flag = *((int *) pm(GET, "Geomview.PickColor", NULL));
  int mapping_flag = *((int *) pm(GET, "Model.Mapping_Flag", NULL));
  int pcolor;

  if ( (xtypes[0] == FUNCT) || (xtypes[1] == FUNCT) || (xtypes[2] == FUNCT) )
    fn = TRUE;

  if (memory_reset_read(m) == 0) 
    {
      while (memory_read_next_flow(m, NULL, NULL, NULL, NULL, NULL) == 0)  
	{
	  flow_counter++;
	  traj_counter = 0;
	  while (memory_read_next_traj(m, &npoints, NULL, NULL) == 0)
	    {
	      traj_counter++;
	      /* start geomview VECT obj */
	      if (npoints > 0)
		{
		  sprintf(buf,"(geometry %s.%d.%d {appearance {linewidth 2} VECT\n",
			  name, flow_counter, traj_counter);
		  geomview_send(buf);
		  if (mapping_flag) {		/* don't connect the dots */
		    sprintf(buf, "%d %d 1\n", npoints, npoints);
		    geomview_send(buf);
		    for (i=0;i<npoints; i += 100) {
                      for (j=0; (j<100) && (i+j<npoints); j++)
                        sprintf(buf+2*j, "1 ");
                      geomview_send(buf);
                    }
                    for (i=0; i<npoints; i += 100) {
                      for (j=0; (j<100) && (i+j<npoints); j++)
                        if ((i==0) && (j==0))
                          sprintf(buf, "1 ");
                        else sprintf(buf+2*j, "0 ");
                      geomview_send(buf);
                    }
		  } else {	/* connect the dots */
		    sprintf(buf,"1 %d 1\n%d\n1\n",npoints, npoints);
		    geomview_send(buf);
		  }
		}
	      while (memory_read_next_point(m, &points, &params, &color, 
					    NULL, NULL) == 0)
		{
		  if (fn == TRUE)
		     view_auxf(view_f, points, params);
		  for (i=0; i<3; i++)
		    {
		      switch(xtypes[i])
			{
			case PHASE_SPACE_VARB:
			  x[i] = points[xindices[i]];
			  break;
			case PARAMETER_VARB:
			  x[i] = params[xindices[i]];
			  break;
			case FUNCTION_VARB:
			  x[i] = view_f[xindices[i]];
			  break;
			default:
			  x[i] = 0.0;
			  break;
			}
		    }
		  /* add point to VECT object */
		  sprintf(buf, "%f %f %f\n", x[0], x[1], x[2]);
		  geomview_send(buf);
		}
	      /* end geomview vect obj - these four numbers specify the color (RGBA).*/

              pcolor = DsColor_get_index( pickcolor_flag, color[0], color[1] );
              gvcolor = DsColor_get_rgb(pcolor);
 
              sprintf(buf,"%f %f %f 1})\n",(float)(gvcolor->r/255.0),
                                           (float)(gvcolor->g/255.0),
                                           (float)(gvcolor->b/255.0) );

	      geomview_send(buf);
	    }
	}
    }
  
}

static void
geomview_bbox(double mins[], double maxs[])
{
  char buf[256];
  geomview_send("(geometry dstool_bbox { appearance {-face +edge} OFF\n8 6 12\n");
  
  /* vertices */
  sprintf(buf,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  mins[0], mins[1], mins[2],
	  mins[0], mins[1], maxs[2],
	  mins[0], maxs[1], maxs[2],
	  mins[0], maxs[1], mins[2]);
  geomview_send(buf);
  sprintf(buf,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  maxs[0], mins[1], mins[2],
	  maxs[0], mins[1], maxs[2],
	  maxs[0], maxs[1], maxs[2],
	  maxs[0], maxs[1], mins[2]);
  geomview_send(buf);

  /* faces */
  geomview_send("4 0 1 2 3\n4 0 1 5 4\n4 1 2 6 5\n4 2 3 7 6\n4 3 0 4 7\n4 4 5 6 7\n");
  geomview_send("} )\n");
  
}


/*
 * this is a bounding box made of a VECT object which works
 * better in geomview for picking objects inside the box, etc.
 */
static void
geomview_vbbox(double mins[], double maxs[])
{
  char buf[512];

  geomview_send("(geometry dstool_bbox {VECT\n4 16 1\n6 6 2 2\n1 0 0 0\n");

  /* polyline 1 */
  sprintf(buf,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  maxs[0], mins[1], mins[2],
	  mins[0], mins[1], mins[2],
	  mins[0], mins[1], maxs[2],
	  mins[0], maxs[1], maxs[2],
	  mins[0], maxs[1], mins[2],
	  mins[0], mins[1], mins[2]);
  geomview_send(buf);
  /* polyline 2 */
  sprintf(buf,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  mins[0], mins[1], maxs[2],
	  maxs[0], mins[1], maxs[2],
	  maxs[0], maxs[1], maxs[2],
	  maxs[0], maxs[1], mins[2],
	  maxs[0], mins[1], mins[2],
	  maxs[0], mins[1], maxs[2]);
  geomview_send(buf);
  /* polyline 3 */
  sprintf(buf,"%f %f %f\n%f %f %f\n",
	  mins[0], maxs[1], maxs[2],
	  maxs[0], maxs[1], maxs[2]);
  geomview_send(buf);
  /* polyline 4 */
  sprintf(buf,"%f %f %f\n%f %f %f\n",
	  mins[0], maxs[1], mins[2],
	  maxs[0], maxs[1], mins[2]);
  geomview_send(buf);

  /* color */
  geomview_send("0 0 0 0\n");

  geomview_send("} )\n");
  
}
