/* AMCL - A simple Mud CLient
 * Copyright (C) 1998 Robin Ericsson <lobbin@lobbin.ml.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "amcl.h"

SYSTEM_DATA prefs;
GtkWidget   *prefs_window;

int check_amcl_dir (gchar *dirname)
{
    struct stat file_stat;
    int return_val = 0;
    gchar buf[256];

    if ( stat (dirname, &file_stat) == 0) /* can we stat ~/.amcl? */
    {
        if ( !(S_ISDIR(file_stat.st_mode))) /* if it's not a directory */
        {
            snprintf (buf, 256, "%s already exists and is not a directory!", dirname);
            popup_window (buf);
            return_val = -1;
        }
    }
    else /* it must not exist */
    {
        snprintf (buf, 256, "%s does not exist, Creating it as a directory.", dirname);
        popup_window (buf);

        if ((mkdir (dirname, 0777)) == 0) /* this isn't dangerous, umask modifies it */
        {
            snprintf (buf, 256, "%s created.", dirname);
            popup_window (buf);
        }
        else
        {
            snprintf (buf, 256, "%s NOT created: %s", dirname, strerror (errno));
            popup_window (buf);
            return_val = errno;
        }
    }

    return (return_val);
}

void load_prefs ( )
{
    FILE *fp;
    gchar *home, filename[255] = "";
    gchar line[255];

    prefs.EchoText = prefs.KeepText = TRUE;
    
    home = getenv ("HOME");

    snprintf (filename, 255, "%s%s", home, "/.amcl");
    if (check_amcl_dir (filename) != 0)
        return;

    snprintf (filename, 254, "%s%s", home, "/.amcl/preferences");
    
    fp = fopen (filename, "r");

    if ( fp == NULL )
        return;

    while ( fgets (line, 80, fp) != NULL )
    {
        gchar pref[25];
        gchar value[10];

        sscanf (line, "%s %[^\n]", pref, value);

        if ( !strcmp (pref, "EchoText") )
        {
            if ( !strcmp (value, "On") )
                prefs.EchoText = TRUE;
            else
                prefs.EchoText = FALSE;
        }

        if ( !strcmp (pref, "KeepText") )
        {
            if ( !strcmp (value, "On") )
                prefs.KeepText = TRUE;
            else
                prefs.KeepText = FALSE;
        }
    }

    fclose (fp);
}

void save_prefs (GtkWidget *button, gpointer data)
{
    gchar *home, filename[256] = "";
    FILE *fp;
    gchar buf[256];

    home = getenv ("HOME");

    snprintf (filename, 255, "%s%s", home, "/.amcl");
    if (check_amcl_dir (filename) != 0)
        return;

    snprintf (filename, 255, "%s%s", home, "/.amcl/preferences");

    fp = fopen (filename, "w");

    if ( fp == NULL )
    {
        sprintf (buf, "You must create the directory %s/.amcl before you can save.",
                 home);
        popup_window (buf);
        return;
    }

    if ( prefs.EchoText )
        fprintf (fp, "EchoText On\n");
    else
        fprintf (fp, "EchoText Off\n");

    if ( prefs.KeepText )
        fprintf (fp, "KeepText On\n");
    else
        fprintf (fp, "KeepText Off\n");
    
    fclose (fp);
}

void check_text_toggle (GtkWidget *widget, GtkWidget *button)
{
    if ( GTK_TOGGLE_BUTTON (button)->active )
        prefs.KeepText = TRUE;
    else
        prefs.KeepText = FALSE;
}

void check_callback (GtkWidget *widget, GtkWidget *check_button)
{
    if ( GTK_TOGGLE_BUTTON (check_button)->active )
        prefs.EchoText = TRUE;
    else
        prefs.EchoText = FALSE;
}

void prefs_close_window ()
{
    gtk_widget_set_sensitive (menu_option_prefs, TRUE);
    gtk_widget_destroy (prefs_window);
}

void window_prefs (GtkWidget *widget, gpointer data)
{
    GtkWidget *check_text;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *button_save;
    GtkWidget *button_close;
    GtkWidget *separator;
    GtkWidget *check_button;

    GtkTooltips *tooltip;

    gtk_widget_set_sensitive (menu_option_prefs, FALSE);
    
    prefs_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (prefs_window), "Amcl - Preferences");
    gtk_signal_connect_object (GTK_OBJECT (prefs_window), "destroy",
                               GTK_SIGNAL_FUNC(prefs_close_window), NULL );
    gtk_widget_set_usize (prefs_window,250,110);

    vbox = gtk_vbox_new (FALSE, 5);
    gtk_container_border_width (GTK_CONTAINER (vbox), 5);
    gtk_container_add (GTK_CONTAINER (prefs_window), vbox);
    gtk_widget_show (vbox);

    tooltip = gtk_tooltips_new ();
    gtk_tooltips_set_colors (tooltip, &color_yellow, &color_black);

    check_button = gtk_check_button_new_with_label ("Echo the text sent?");
    gtk_box_pack_start (GTK_BOX (vbox), check_button, FALSE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (check_button), "toggled",
                        GTK_SIGNAL_FUNC (check_callback), check_button);
    gtk_tooltips_set_tip (tooltip, check_button,
                          "With this toggled on, all the text you type and "
                          "enter will be echoed on the connection so you can "
                          "control what you are sending."
                          "\nSome people think this is annoying, and therefor this "
                          "is an options.",
                          NULL);
    GTK_WIDGET_UNSET_FLAGS (check_button, GTK_CAN_FOCUS);
    gtk_widget_show (check_button);
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (check_button), prefs.EchoText);

    check_text = gtk_check_button_new_with_label ("Keep the text entered?");
    gtk_box_pack_start (GTK_BOX (vbox), check_text, FALSE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (check_text), "toggled",
                        GTK_SIGNAL_FUNC (check_text_toggle),
                        check_text);
    gtk_tooltips_set_tip (tooltip, check_text,
                          "With this toggled on, the text you have entered and sent "
                          "to the connection, will be left in the entry box but "
                          "seleceted."
                          "\nTurn this off to remove the text after it has been sent.",
                          NULL);
    GTK_WIDGET_UNSET_FLAGS (check_text, GTK_CAN_FOCUS);
    gtk_widget_show (check_text);
    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (check_text), prefs.KeepText);

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 5);
    gtk_widget_show (separator);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);

    button_save  = gtk_button_new_with_label ( " save ");
    button_close = gtk_button_new_with_label ( " close ");
    gtk_signal_connect_object (GTK_OBJECT (button_save), "clicked",
                               GTK_SIGNAL_FUNC (save_prefs),
                               NULL);
    gtk_signal_connect_object (GTK_OBJECT (button_close), "clicked",
                               GTK_SIGNAL_FUNC (prefs_close_window),
                               NULL);
    gtk_box_pack_start (GTK_BOX (hbox), button_save,  TRUE, TRUE, 15);
    gtk_box_pack_start (GTK_BOX (hbox), button_close, TRUE, TRUE, 15);
    gtk_widget_show (button_save);
    gtk_widget_show (button_close);
    
    gtk_widget_show (prefs_window);
}
