/*
fetch -- post articles to and get news from upstream server(s)

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <rskerka@metronet.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998.

See file COPYING for restrictions on the use of this software.
*/

#include <sys/types.h>
#ifdef BSD
#include <sys/errno.h>
#endif
#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <pwd.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <syslog.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <utime.h>

#include "leafnode.h"

int verbose = 0;
int debug = 1;

int artno;

time_t now;

struct newsgroup *cg;

int reals, fakes;

/* Variables set by command-line options which are specific for fetch */
int extraarticles = 0;
int usesupplement = 0;
int noexpire = 0;	/* if 1, don't automatically unsubscribe newsgroups */
int forceactive = 0;	/* if 1, reread complete active file */

int isinteresting( char * groupname );
void store( const char * filename,
	    FILE * filehandle,
	    const size_t bytes,
	    char * newsgroups,
	    const char * subject,
	    const char * from,
	    const char * date,
	    const char * msgid,
	    const char * references,
	    const char * lines);

int age( const char * date );
int getgroup ( struct newsgroup *g , int server );
int postarticles( void );
void usage( void );
void fixxover( void );

void usage( void ) {
   fprintf( stderr, "Usage: fetch [-v] [-x #] [-l] [-n]\n"
                    "  -v: more verbose (may be repeated)\n"
                    "  -x: check for # extra articles in each group\n"
                    "  -l: don't use supplementary servers\n"
                    "  -n: don't automatically expire unread groups\n");
}

/*
 * check whether group is represented in interesting.groups
 */
int isinteresting( char * groupname ) {
    struct stat st;

    sprintf( s, "%s/interesting.groups/%s", spooldir, groupname );
    return(stat( s, &st ) + 1);
}

void  store( const char * filename,
	     FILE * filehandle,
	     size_t bytes,
	     char * newsgroups,
	     const char * subject,
	     const char * from,
	     const char * date,
	     const char * msgid,
	     const char * references,
	     const char * lines)
{
    char tmp[10];
    char xrefincase[4096]; /* 1024 for newsgroups, plus article numbers */
    char * p;
    char * q;
    char * x;
    int n;

    x = xrefincase;
    n = 0;

    if ( verbose > 2 )
	printf( "storing %s: %s\n", msgid, newsgroups );

    p = newsgroups;
    while (p && *p) {
	n++;
	q = strchr( p, ',' );
	if (q)
	    *q++ = '\0';
	if ( *p ) {
	    if (!cg || strcmp(cg->name, p)) {
		cg = findgroup( p );
		if ( cg ) {
		    if ( scarce_links == 0 )
		        chdirgroup( p );
		    else if ( isinteresting(cg->name) )
		        chdirgroup( p );
		    else
		    	cg = NULL;
		}
	    }
	    if ( cg ) {
		do {
		    sprintf(tmp, "%d", ++cg->last);
		    errno = 0;
		    if ( verbose > 2 )
			printf( "..as article %d in %s\n",
			       cg->last, cg->name );
		} while ( link( filename, tmp )<0 && errno==EEXIST );
		if ( errno )
		    syslog( LOG_ERR, "error linking %s into %s: %m",
			    filename, p );
		else {
		    sprintf( x, " %s:%d", cg->name, cg->last );
		    x += strlen( x );
		}
	    } else {
		if ( verbose > 2 )
		    printf( ".. discarding unknown group %s\n", p );
	    }
	}
	p = q;
    }
    if ( n > 1 )
	fprintf( filehandle, "Xref: %s%s\n", fqdn, xrefincase );
}



int age( const char * date ) {
    char monthname[4];
    int month;
    int year;
    int day;
    const char * d;
    time_t tmp;
    struct tm gmt;

    if ( !date )
	return 1000; /* large number: OLD */
    d = date;
    d += 5; /* "Date:" */
    while( isspace( *d ) )
	d++;

    if ( isalpha( d[0] ) && isalpha( d[0] ) && isalpha( d[2] ) && 
	 d[3] == ',' && isspace( d[4] ) )
	d += 5; /* skip Mon, */

    monthname[0] = '\0';

    if ( sscanf( d, "%d %s %d", &day, monthname, &year ) < 3 )
	return 1003;
    else if ( ! ((year > 95 && year < 100) || (year > 1995)) )
	return 1004;
    else if ( strlen( monthname ) != 3 )
	return 1005;
    else if ( !(day > 0 && day < 32) )
	return 1006;
    else {
	if ( !strcasecmp( monthname, "jan" ) )
	    month = 0;
	else if ( !strcasecmp( monthname, "feb" ) )
	    month = 1;
	else if ( !strcasecmp( monthname, "mar" ) )
	    month = 2;
	else if ( !strcasecmp( monthname, "apr" ) )
	    month = 3;
	else if ( !strcasecmp( monthname, "may" ) )
	    month = 4;
	else if ( !strcasecmp( monthname, "jun" ) )
	    month = 5;
	else if ( !strcasecmp( monthname, "jul" ) )
	    month = 6;
	else if ( !strcasecmp( monthname, "aug" ) )
	    month = 7;
	else if ( !strcasecmp( monthname, "sep" ) )
	    month = 8;
	else if ( !strcasecmp( monthname, "oct" ) )
	    month = 9;
	else if ( !strcasecmp( monthname, "nov" ) )
	    month = 10;
	else if ( !strcasecmp( monthname, "dec" ) )
	    month = 11;
	else {
	    return 1001;
	}
	if ( year > 100 )
	    year -= 1900; /* *sigh */
	tmp = time(0);
	gmt   = *(gmtime(&tmp));
	return ((gmt.tm_year - year)*365 +
		(gmt.tm_mon - month)*28 +
		(gmt.tm_mday - day));
    }
}		       

/*
 * Get body of a single message of which the header has already been
 * downloaded and append it to the file with the header.
 */
static void getbody( struct newsgroup* group, int id ) {
    const char *c ;
    char *l;
    char *p, *q;
    char messageid[ 1024 ];
    char newsgroups[ 1024 ];	/* I hope this is enough */
    char xref[ 4096 ];
    FILE *f, *g;

    chdirgroup( group->name );

    /* extract message-id: and xref: headers */
    snprintf( s, 1024, "%d", id );
    if ( ! ( f = fopen( s, "r" ) ) ) {
        syslog( LOG_ERR, "%s: cannot open %s for reading", group->name, s );
        return;
    }
    strcpy( messageid, "" );
    strcpy( xref, "" );
    strcpy( newsgroups, "" );
    debug = 0;
    while ( ( l = getaline( f ) ) != NULL ) {
	if ( ! strncmp( l, "Newsgroups: ", 12 ) )
            strcpy( newsgroups, l + 12 );
        if ( ! strncmp( l, "Message-ID: ", 12 ) )
            strcpy( messageid, l + 12 );
	if ( ! strncmp( l, "Xref: ", 6 ) )
	    strcpy( xref, l + 6 );
    }
    debug = 1;
    fclose( f );

    /* check whether we can retrieve the body */
    if ( verbose > 2 )
        printf( "%s: BODY %s\n", group->name, messageid );
    sprintf( lineout, "BODY %s\r\n", messageid );
    putaline();

    if ( nntpreply() != 222 ) {
        syslog( LOG_ERR, "%s: Retrieving body %s failed. No response",
                group->name, messageid );
        return;
    }

    snprintf( s, 1024, "%d", id );
    c = lookup( messageid );
    unlink( c );	/* make space for new file */
    
    if ( ! ( f = fopen( c, "w" ) ) ) {
	syslog( LOG_ERR, "%s: cannot open %s for writing", group->name, l );
	link( s, c );	/* if we can't open new file restore old one */
	return;
    }

    /* copy all headers except Xref: into new file */
    g = fopen( s, "r" );
    debug = 0;
    while ( ( l = getaline( g ) ) != NULL ) {
	if ( strncmp( l, "Xref: ", 6 ) ) 
	    fprintf( f, "%s\n", l );
    }
    debug = 1;
    fclose( g );

    /* create a whole bunch of new hardlinks */
    store( c, f, 0, newsgroups, "", "", "", messageid, "", "" );

    /* retrieve body */
    fprintf( f, "\n" ); /* Empty line between header and body. */
    debug = 0;
    while ( ( ( l = getaline( nntpin ) ) != NULL ) && strcmp( l, "." ) ) {
        if ( *l == '.' )
            ++l;
        fprintf( f, "%s\n", l );
    }
    debug = 1;
    if ( ! l ) {
        syslog( LOG_ERR, "%s: Retrieving body %s failed. "
                "Transmission interrupted.",
                group->name, messageid );
        fprintf( f, "\n\t[ Leafnode: ]\n"
                    "\t[ An error occured while "
                    "retrieving this message. ]\n" );
    }
    fclose( f );

    /* Remove old article files since we don't need them anymore.
       This is done by evaluating the old Xref: header.
     */
    if ( xref[0] == '\0' )	/* no Xref: header --> make a fake one */
	sprintf( xref, "%s %s:%d", fqdn, group->name, id );

    syslog( LOG_DEBUG, "xref: %s", xref );
    c = strchr( xref, ' ' );
    while ( ( c++ ) && ( *c ) && ( q = strchr( c, ':' ) ) != NULL ) {
	*q++ = '\0';
	if ( ( p = strchr( q, ' ' ) ) != NULL )
	    *p = '\0';
	
	/* c points to the newsgroup, q to the article number */
	chdirgroup( c );
	if ( unlink( q ) )
	    syslog( LOG_NOTICE, "unlinking %s/%s failed", c, q );
	else
	    syslog( LOG_DEBUG, "unlinking %s/%s", c, q );

	c = strchr( q, ' ' );
    }
}



/*
 * Get bodies of messages that have marked for download.
 * The group must already be selected at the remote server and
 * the current directory must be the one of the group.
 */
static void getmarked( struct newsgroup* group ) {
    int n, i;
    FILE *f;
    char filename[ 1024 ];
    int id[ BODY_DOWNLOAD_LIMIT ];

    if ( verbose )
        printf( "Getting bodies of marked messages for group %s ...\n",
                group->name );
    n = 0;
    snprintf( filename, 1024,
              "%s/interesting.groups/%s", spooldir, group->name );
    if ( ! ( f = fopen( filename, "r+" ) ) )
        syslog( LOG_ERR, "Cannot open %s", filename );
    else {
        while ( fgets( s, 1024, f ) ) {
            if ( sscanf( s, "%d", &id[ n ] ) )
                ++n;
        }
        fclose( f );
        if ( n )
            truncate( filename, 0 );
    }
    syslog( LOG_INFO, "%s: marked bodies %d", group->name, n );
    for ( i = 0; i < n; ++i )
        getbody( group, id[ i ] );
    if ( verbose )
        printf( "Done.\n" );
}

/*
 * get newsgroup from a server. "server" is the last article that
 * was previously read from this group on that server
 */
int getgroup ( struct newsgroup * g, int server ) {
    static char *hd[10];
    char *hnames[10] = {"Path: ", "Message-ID: ", "From: ", "Newsgroups: ",
			"Subject: ", "Date: ", "References: ", "Lines: ", 
			"Xref: ", ""};
    int h;
    int n;
    int error;
    int last;
    int window; /* last ARTICLE n command sent */
    char *l;
    FILE *f;
    const char *c;
    struct stat st;
    int outstanding = 0, i;
    static int * stufftoget;

    if ( !g )
	return server;

    if (g->first > g->last)
	g->last = g->first;

    reals = fakes = 0;
    chdirgroup( g->name );

    sprintf( lineout, "GROUP %s\r\n", g->name );
    putaline();

    l = getaline( nntpin );
    if ( !l )
	return server;

    if (sscanf(l, "%3d %d %d %d ", &n, &h, &window, &last) < 4
	|| n != 211 )
	return server;

    if ( extraarticles ) {
	i = server - extraarticles;
	if ( i < window )
	    i = window;
	if ( verbose > 1 )
	    printf( "backing up from %d to %d\n",
		    server, i );
	server = i;
    }

    if ( server > last+1 ) {
	syslog( LOG_INFO, "last seen article was %d, server now has %d-%d",
		server, window, last );
	if ( server > last+5 ) {
	    if ( verbose )
		printf( "switched upstream servers? %d > %d\n",
			server-1, last );
	    server = window; /* insane - recover thoroughly */
	} else {
	    if ( verbose )
		printf( "rampant spam cancel? %d > %d\n",
			server-1, last );
	    server = last - 5; /* a little bit too much */
	}
    }
    if ( artlimit && last-server > artlimit ) {
	if ( verbose > 1 )
	    printf( "skipping articles %d-%d inclusive (article limit)\n", 
		    server, last-artlimit-1 );
	syslog( LOG_INFO, "skipping articles %d-%d inclusive (article limit)",
		server, last-artlimit-1 );
	server = last-artlimit;
    }

    if ( initiallimit && (g->first >= g->last) && 
	 last-server > initiallimit ) {
	if ( verbose > 1 )
	    printf( "skipping articles %d-%d inclusive (initial limit)\n", 
		    server, last-initiallimit-1 );
	syslog( LOG_INFO, "skipping articles %d-%d inclusive (initial limit)",
		server, last-initiallimit-1 );
	server = last-initiallimit;
    }

    sprintf( s, "%s/interesting.groups/%s", spooldir, g->name );
    if ( stat( s, &st ) )
	return last; /* race condition: I hope this is proper */

    if ( noexpire == 0 ) {
	if ( ((st.st_mtime==st.st_ctime) && (now-st.st_ctime > TIMEOUT_SHORT) &&
	     (server <= last) )
	     || (now-st.st_mtime > TIMEOUT_LONG) ) {
	    if ( verbose > 1 )
		printf("skipping %s from now on\n", g->name);
	    syslog( LOG_INFO, "skip %s: %d, %d", g->name, server, last );
	    return server ;
       }
   }

    if ( window < server )
	window = server;
    if ( window < 1 )
	window = 1;
    server = window;

    if ( server <= last ) {
	if ( verbose > 1 )
	    printf( "%s: considering articles %d-%d\n", 
		    g->name, server, last );
	sprintf( lineout, "XHDR Message-ID %d-%d\r\n",
		 server, last );
        putaline();
	if ( nntpreply() == 221 ) {
	    stufftoget = realloc(stufftoget, sizeof(int)*(last + 1 - server) );
	    if ( stufftoget )
	    	memset( stufftoget, 0, last + 1 - server );
	    else {
	    	syslog( LOG_ERR, "not enough memory for XHDRs" );
		return server;
	    }
	    while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
		int art;
		char * t;
		art = strtol(l, &t, 10);
		if (t && isspace( *t )) {
		    while ( isspace( *t ) )
			t++;
		    if ( art >= server && art <= last &&
			 stat( lookup( t ), &st ) != 0 ) {
			stufftoget[outstanding] = art;
			outstanding++;
			if ( verbose > 2 )
			    printf( "%s: will fetch %d (%s)\n",
			            g->name, art, t );
		    }
		}
	    }
	    if ( !l )
		return server;
	}
	else
	   realloc( stufftoget, 0 );
    } else if ( verbose > 1 ) {
	printf( "%s: no new articles\n", g->name );
	return server;
    }
    syslog( LOG_INFO, "%s: will fetch %d articles", g->name, outstanding );

    /* now we have a list of articles in stufftoget[] */
    /* let's get the header and possibly bodies of these */
    while ( outstanding > 0 ) {
        syslog( LOG_DEBUG, "%s %d: %d", g->name, outstanding, stufftoget[outstanding]);
        i = outstanding-1;
        error = 0;
	if ( stufftoget[i] == 0 )
	    error = 1;
	if ( !error && ( stufftoget[i] < server ) ) {
	    if ( verbose > 2 )
		printf( "%s: skipping %d - not available or too old\n",
			g->name, stufftoget[i] );
	    syslog( LOG_INFO, "%s: skipping %d - not available or too old",
	    	    g->name, stufftoget[i] );
	    error = 1;
	}
	if ( !error ) {
	    sprintf(lineout, "HEAD %d\r\n", stufftoget[i]);
	    putaline();
	    outstanding--;
	    l = getaline( nntpin );
	    if ( !l )
		return server;
	    if (sscanf(l, "%3d %d", &n, &h) < 2 || n != 221 ) {
		if ( verbose )
		    printf( "%s %d: reply %s (%d more up in the air)\n",
			    g->name, stufftoget[i], l, outstanding );
		syslog( LOG_INFO, "%s %d: reply %s (%d more up in the air)",
			g->name, stufftoget[i], l, outstanding );
		error = 1;
	    }
	}
	if ( !error ) {
	    debug = 0;
	    if ( verbose > 2 )
		printf( "%s: receiving article %d (%d more up in the air)\n",
			g->name, stufftoget[i], outstanding );

	    for ( h=0; h<10; h++ ) {
		if (hd[h])
		    free(hd[h]);
		hd[h] = strdup("");
	    }
	    c = NULL;
	    while ( (l = getaline( nntpin )) && strcmp( l, "." ) ) {
		/* allow for multiline headers */
		if ( !isspace(l[0]) ) {
		    n = 0;
		    while ( strncasecmp( l, hnames[n], strlen(hnames[n]) ) )
		    	n++;
		    if ( n<9 && hd[n] && *(hd[n]) )
		    	/* second occurance is "other header" */
		    	n = 9;
		} 
		hd[n] = critrealloc( hd[n], strlen(hd[n])+strlen(l)+2,
				     "Fetching article header" );
		strcat( hd[n], l );
		strcat( hd[n], "\n" );
		if ( verbose > 3 && hnames[n] && *hnames[n] )
		    printf( "...saw header %s\n", hnames[n] );
	    }
	    debug = 1;
	}

	/* check headers for legality - one could do much more parsing here */
        if ( !error ) {
	    for ( h=0; hd[h] && *(hd[h]) && h<4; h++ )
                ;
	    if ( h != 4 ) {
		if ( verbose )
		    printf( "Discarding article %d - no message-id found\n",
			     stufftoget[i] );
		syslog( LOG_INFO,
			"Discarding article %d %s - no message-id found\n", i,
			hd[4] );
		error = 1;
	    }
	    if ( age( hd[5] ) > maxage ) {
		if ( verbose > 2 )
		    printf( "Discarding article %d - older than %d days\n",
			    stufftoget[i], maxage );
		syslog( LOG_INFO,
			"Discarding article %d %s - older than %d days",
			stufftoget[i], hd[4], maxage );
		error = 1;
	    }
	    if ( maxlines ) {
	        char * p;
		p = strchr( hd[7], ' ' );
		if ( p && ( atoi(p) > maxlines )) {
		    error = 1;
		    if ( verbose > 2 )
			printf("Discarding article %d - longer than %d lines",
				stufftoget[i], maxlines );
		    syslog( LOG_INFO,
			    "Discarding article %d %s - longer than %d lines",
			    stufftoget[i], hd[4], maxlines );
		}
	    }
	    if ( crosspostlimit ) {
	        char * t;
		t = hd[3];
		n = 1;	/* number of groups the article is posted to */
	        while ( ( t = strchr( t, ',' ) ) != NULL ) {
		    t++ ;
		    n++ ;
		}
		if ( crosspostlimit < n ) {
		    if ( verbose > 2 )
		        printf( "Discarding article %d - posted to %d groups "
				"(max. %d)\n",
				stufftoget[i], n, crosspostlimit );
		    syslog( LOG_INFO,
			    "Discarding article %d %s - posted to %d groups "
			    "(max. %d)", stufftoget[i], hd[4], n,
			    crosspostlimit );
		    error = 1;
		}
	    }
	}

	/* store articles */
	if ( !error ) {
	    c = lookup(strchr( hd[1], '<'));
	    if ( stat( c, &st ) && errno == ENOENT )
		f = fopen( c, "w" );
	    else {
	        syslog( LOG_ERR, "unable to store article %s: %m", c );
	        if ( verbose )
		    printf( "unable to store article %s\n", c );
	    }
	    if (f) {
		for ( h=0; h<10; h++ )
		    if ( h!=8 && hd[h] && *(hd[h]))
			fprintf( f, "%s", hd[h] );
		h = 0;
		/* replace tabs and other odd signs with spaces */
		while ( h<8 ) {
		    char * p1;
		    char * p2;
		    p1 = p2 = hd[h];
		    while (p1 && *p1) {
			if ( isspace(*p1) ) {
			    *p2 = ' ';
			    do {
				p1++;
			    } while ( isspace(*p1) );
			} else {
			    *p2 = *p1++;
			}
			if (*p1)
			    p2++;
			else
			    *p2 = '\0';
		    }
		    h++;
		}
		/* generate hardlinks; this procedure also increments g->last */
		store( c, f, st.st_size,
		       *hd[3] ? hd[3]+strlen(hnames[3]) : "", 
		       *hd[4] ? hd[4]+strlen(hnames[4]) : "", 
		       *hd[2] ? hd[2]+strlen(hnames[2]) : "", 
		       *hd[5] ? hd[5]+strlen(hnames[5]) : "", 
		       *hd[1] ? hd[1]+strlen(hnames[1]) : "", 
		       *hd[6] ? hd[6]+strlen(hnames[6]) : "", 
		       *hd[7] ? hd[7]+strlen(hnames[7]) : "");
	    }
	}

	if ( !error && delaybody ) {
	    fclose( f );
	    error = 1;	/* just to avoid the rest of the code */
	}

	if ( !error ) {
	    sprintf(lineout, "BODY %d\r\n", stufftoget[i]);
	    putaline();
	    l = getaline(nntpin);
	    if ( sscanf( l, "%3d", &n ) != 1 || n != 222 ) {
	       syslog( LOG_INFO, "%d: reply %s", stufftoget[i], l );
	       error = 1;
	    }
	}
	if ( !error ) {
	    debug = 0;
	    fprintf( f, "\n" ); /* empty line between header and body */
	    while ( ((l=getaline(nntpin)) != NULL) && strcmp(l, ".") ) {
	        if ( l && *l != '.' && f )
		    fprintf(f, "%s\n", l);
	    }
	    debug = 1;
	    if ( f )
		fclose( f );
	    if ( !l )
		unlink( c );
	    else {
		/* one might add handling for additional headers here.
		   Since "Supersedes:" and similar headers are often
		   ignored, this (pseudo-)code is ignored
		 */
#ifdef NOTYET
		if ( hd[7] && 
		     ((c=strstr(hd[7], "\nSupersedes:" )) != NULL) ) {
		    /* that isn't strictly correct, it's case sensitive
		       and lacks trailing space */
		    c += strlen( "\nSupersedes:" );
		    while ( c && isspace( *c ) )
			c++;
		    if ( *c == '<' && ( (c=lookup( c )) != NULL ))
			/* supersede( c ) */ ;
		}
#endif
	    }
	}
    }

    syslog( LOG_INFO, "%s fetched to %d", g->name, g->last );
    return last+1;
}

/*
 * get active file from remote server
 */
static void nntpactive( void ) {
    char * l, * p;
    struct serverlist * a, * b, *c;
    			/* not for servers but we need a list of strings */
    char timestr[64];
    time_t update, mtime;
    struct stat st;
    struct utimbuf buf;
    int n, error;
    FILE * f;

    sprintf( s, "%s/active.read", spooldir );
    n = stat( s, &st );
    update = st.st_atime;
    mtime  = st.st_mtime;
    if ( active && ( forceactive == 0 ) && ( n == 0 ) &&
	( now-mtime < READ_INTERVALL ) ) {
	if ( verbose )
	    printf( "LIST ACTIVE done %d seconds ago, issuing NEWGROUPS\n",
		    (int)(now-mtime) );
	strftime( timestr, 64, "%y%m%d %H%M00", gmtime( &update ) );
	sprintf( lineout, "NEWGROUPS %s GMT\r\n", timestr );
	putaline();
	if ( nntpreply() != 231 ) {
	    syslog( LOG_ERR, "Reading new newsgroups failed" );
	    return;
	}
	a = NULL;
	while ( (l=getaline(nntpin)) && ( *l != '.' ) ) {
	    p = l;
	    while (!isspace(*p) )
		p++;
	    if (*p)
		*p = '\0';
	    b = (struct serverlist *)critmalloc( sizeof(struct serverlist)
		 + strlen(l), "allocating space for newsgroup name");
	    strcpy( b->name, l );
	    b->next = NULL;
	    if ( a == NULL )
		a = b;
	    else
		c->next = b;
	    c = b;
	}
	b = a;
	while ( b != NULL ) {
	    error = 0;
	    sprintf( lineout, "XGTITLE %s\r\n", b->name );
	    putaline();
	    if ( nntpreply() != 282 ) {
		sprintf( lineout, "LIST NEWSGROUPS %s\r\n", b->name );
		putaline();
		if ( nntpreply() != 215 )
		    error = 1;
	    }
	    if ( !error ) {
		l = getaline(nntpin);
		if ( l && *l && *l != '.' ) {
		    p = l;
		    while ( !isspace(*p) )
			p++;
		    while ( isspace(*p) ) {
		        *p = '\0';
		    	p++;
		    }
		    insertgroup( l, 0, 0, *p ? p : NULL , update );
		    do {
		        l = getaline(nntpin);
		    } while ( l && *l && *l != '.' );
		}
		else
		    insertgroup( b->name, 0, 0, NULL, update );
	    }
	    b = b->next;
	}
	b = a;
	while ( b != NULL ) {
	   c = b->next;
	   free(b);
	   b = c;
	}
    } else {
        update = 0;
	if ( verbose )
	    printf( "LIST ACTIVE done %d seconds ago, getting all groups\n",
	    	(int)(now-mtime) );
	sprintf(lineout, "LIST ACTIVE\r\n" );
	putaline();
	if ( nntpreply() != 215 ) {
	    syslog( LOG_ERR, "Reading all newsgroups failed" );
	    return;
	}
	debug = 0;
        while ( (l=getaline(nntpin)) && ( *l != '.' ) ) {
            p = l;
            while (!isspace(*p) )
                p++;
            while ( isspace(*p) ) {
                *p = '\0';
                p++;
            }
            insertgroup( l, 0, 0, NULL, update );
        }
	debug = 1;
	sprintf(lineout, "LIST NEWSGROUPS\r\n" );
	putaline();
	if ( nntpreply() != 215 ) {
	    syslog( LOG_ERR, "Reading newsgroups descriptions failed" );
	    return;
	}
	debug = 0;
	while ( (l=getaline(nntpin)) && ( *l != '.' ) ) {
	    p = l;
	    while (!isspace(*p) )
		p++;
	    while ( isspace(*p) ) {
                *p = '\0';
                p++;
            }
	    insertgroup( l, 0, 0, *p ? p : NULL, update );
	}
	debug = 1;
    }

    sprintf( s, "%s/active.read", spooldir );
    buf.actime = now;
    buf.modtime = update ? mtime : now;
    if ( utime( s, &buf ) < 0 ) {
	/* active.read doesn't exist yet */
	if ( ( f = fopen( s, "w" ) ) > 0 )
	    fclose( f );
    }
}


/*
 * post all spooled articles
 *
 * if all postings succeed, returns 1
 * if there are no postings to post, returns 1
 * if a posting is strange for some reason, closes the nntp connection
 *  and returns 0.  this is to recover from unknown states
 *
 * a posting is deleted once it has been posted, whether it succeeded
 * or not: we don't want to re-do an error.
 *
 */

int postarticles( void ) {
    char * line;
    DIR * d;
    struct dirent * de;
    FILE * f;
    struct stat st;
    int r;

    if ( chdir( spooldir ) || chdir ( "out.going" ) ) {
	syslog( LOG_ERR, "Unable to cd to outgoing directory: %m" );
	return 1;
    }

    d = opendir( "." );
    if ( !d ) {
	syslog( LOG_ERR, "Unable to opendir out.going: %m" );
	return 1;
    }

    while ( (de=readdir( d )) != NULL ) {
	if ( !stat(de->d_name, &st) &&
	     S_ISREG( st.st_mode ) &&
	     (f=fopen( de->d_name, "r" )) != NULL ) {
	    if ( verbose > 2 )
		printf( "Posting %s...\n", de->d_name );
	    sprintf( lineout, "POST\r\n" );
	    putaline();
	    r = nntpreply();
	    if ( r == 340 ) {
		do {
		    line = getaline( f );
		    if ( line ) {
			sprintf( lineout, "%s\r\n", line );
			putaline();
		    }
		} while ( line && strcmp( line, "." ) );
		if ( !line ) {
		    if ( verbose )
			printf( "Posting %s failed: %03d reply to POST\n",
				de->d_name, r );
		    sprintf( s, "%s/failed.postings", spooldir );
		    mkdir( s, 0775 );
		    sprintf( s, "%s/failed.postings/%s", 
			     spooldir, de->d_name );
		    syslog( LOG_ERR, 
			    "unable to post (%s), moving to %s",
			    line, s );
		    if ( rename( de->d_name, s ) )
			syslog( LOG_ERR,
				"unable to move failed posting to %s: %m",
				s );
		    closedir( d );
		    fclose( nntpout );
		    return 0; /* strange state, so re-connect */
		}
		line = getaline( nntpin );
		if ( line && !strncmp( line, "240", 3 ) ) {
		    if ( verbose > 2 )
			printf( " - OK\n" );
		    unlink( de->d_name);
		} else {
		    if ( line && !strncmp( line, "441 435", 7 ) ) {
			if ( verbose ) {
			    syslog( LOG_INFO,
			    	    "Message-ID %s already in use upstream -- "
				    "article discarded\n", 
				    de->d_name );
			    printf(" - upstream server had that message-id\n");
			}
			unlink( de->d_name );
		    } else {
			if ( verbose )
			    printf( "Article %s rejected: %s\n",
			            de->d_name, line );
			sprintf( s, "%s/failed.postings", spooldir );
			mkdir( s, 0775 );
			sprintf( s, "%s/failed.postings/%s", 
				 spooldir, de->d_name );
			syslog( LOG_ERR, 
				"Article %s rejected (%s), moving to %s",
				de->d_name, line, s );
			if ( rename( de->d_name, s ) )
			    syslog( LOG_ERR,
				    "unable to move failed posting to %s: %m",
				    s );
			closedir( d );
			fclose( nntpout );
			return 0;
		    }
		}
	    } else
		syslog( LOG_ERR, "unable to post: %03d", r );
	    fclose( f );
	}
    }
    closedir( d );
    return 1;
}


static void processupstream( const char * server ) {
    FILE * f;
    DIR * d;
    struct dirent * de;
    struct newsgroup * g;
    struct stat st;
    int fd;
    int havefile = 0;
    int newserver;
    char * l;
    char * oldfile ;

    static char * stuff ;

    /* read server info */
    sprintf( s, "%s/leaf.node/%s", spooldir, server );
    oldfile = strdup( s );
    if ( stat( s, &st ) == 0 ) {
        if ( verbose > 1 )
            printf( "Read server info from %s\n", s );
	syslog( LOG_INFO, "Read server info from %s", s );
        stuff = critmalloc( st.st_size+1, "Reading server info" );
        if ( (fd=open( s, O_RDONLY))<0 ||
             (read( fd, stuff, st.st_size ) < st.st_size) ) {
            syslog( LOG_ERR, "can't open/read %s: %m", s );
	    havefile = 0;
            free( stuff );
        } else {
	    havefile = 1;
            close( fd );
            stuff[st.st_size] = '\0'; /* 0-terminate string */
	}
    }

    sprintf( s, "%s/interesting.groups", 
	     spooldir );
    d = opendir( s );
    if ( !d ) {
	syslog( LOG_ERR, "opendir %s: %m", s );
	return;
    }

    sprintf( s, "%s/leaf.node/%s~", spooldir, server );
    f = fopen( s, "w" );
    if ( ( f == NULL ) && verbose ) {
	syslog( LOG_ERR, "Couldn't open %s for writing.", s );
        printf( "Couldn't open %s for writing.\n", s );
    }
    while ( (de=readdir(d)) ) {
	if ( isalnum( *(de->d_name) ) ) {
	    g = findgroup( de->d_name );
	    if ( g ) {
                g->alive = 1;
		l = havefile ? strstr( stuff, g->name ) : 0;
		if ( l && *l ) {
		    l = strchr( l, ' ' );
		    newserver = getgroup( g, strtol( l, NULL, 10 ) );
		}
		else
		    newserver = getgroup( g, 1 );
		if ( delaybody )
		    getmarked( g );
		if ( f != NULL )
                    fprintf( f, "%s %d\n", g->name, newserver );
            }
	}
    }
    closedir( d );
    if ( f != NULL )
        fclose(f);

    sprintf( s, "%s/leaf.node/%s~", spooldir, server );
    rename( s, oldfile );

    free( oldfile );
    free( stuff );
}

void fixxover( void ) {
    DIR * d;
    struct dirent * de;

    sprintf( s, "%s/interesting.groups",
             spooldir );
    d = opendir( s );
    if ( !d ) {
        syslog( LOG_ERR, "opendir %s: %m", s );
        return;
    }

    while ( (de=readdir(d)) ) {
        if ( isalnum( *(de->d_name) ) && findgroup( de->d_name ) ) {
            chdirgroup( de->d_name );
            getxover();
        }
    }
    closedir( d );
}

int main( int argc, char ** argv ) {
    struct passwd * pw;
    int option;
    FILE * lf;

    verbose = 0;
    usesupplement = 0;

    pw = getpwnam( "news" );
    if ( !pw ) {
	fprintf( stderr, "no such user: news\n" );
	exit( 1 );
    }

    setregid( pw->pw_gid, pw->pw_gid );
    setreuid( pw->pw_uid, pw->pw_uid );

    if ( getuid() != pw->pw_uid || getgid() != pw->pw_gid ) {
	fprintf( stderr, "%s: must be run as news or root\n", argv[0] );
	exit( 1 );
    }

    openlog( "fetch", LOG_PID|LOG_CONS, LOG_NEWS );

    while ( (option=getopt( argc, argv, "flnvx:" )) != -1 ) {
	if ( option == 'v' ) {
	    verbose++;
	} else if ( option == 'x' ) {
	    char *nptr, *endptr;
	    nptr = optarg;
	    endptr = NULL;
	    extraarticles = strtol( nptr, &endptr, 0 );
	    if ( !nptr || !*nptr || !endptr || *endptr || !extraarticles ) {
	        usage();
		fprintf( stderr, "Usage: fetch [-v] [-x #] [-l] [-n] [-f]\n"
			 "  -v: more verbose (may be repeated)\n"
			 "  -x: check for # extra articles in each group\n"
			 "  -l: don't use supplementary servers\n"
			 "  -n: don't automatically expire unread groups\n"
			 "  -f: force reread of active file\n" );
		exit( 1 );
	    }
	} else if ( option == 'l' ) {
	    usesupplement = -1; /* don't use supplementary servers */
        } else if ( option == 'n' ) {
            noexpire = 1;
	} else if ( option == 'f' ) {
	    forceactive = 1;
	} else {
	    usage();
	    exit( 1 );
	}
    }

    if ( fopen(lockfile, "r+") != NULL ) {
        syslog( LOG_ERR, "lockfile %s exists, abort fetch ...", lockfile);
	if ( verbose )
	    printf("lockfile %s exists, abort fetch ...\n", lockfile);
        exit( 1 );
    }
    lf = fopen( lockfile, "w" );
    fprintf( lf, "%d\n", getpid() );
    fclose( lf );

    if ( verbose ) {
	printf( "%s: verbosity level is %d\n", version, verbose );
	if ( verbose > 1 ) {
	    printf("Read active groups every %d days\n",
		    READ_INTERVALL/3600/24);
	    if ( noexpire == 0 ) {
		printf("Unsubscribe unread groups after %d days\n",
			TIMEOUT_LONG/3600/24);
		printf("Unsubscribe groups that have been read once after %d days\n",
			TIMEOUT_SHORT/3600/24);
	    } else if ( noexpire == 1 ) {
		printf("Don't automatically unsubscribe unread newsgroups.\n");
	    }
        }
    }

    whoami();

    now = time(NULL);

    umask(2);

    if ( !readconfig() ) {
        unlink( lockfile );
	exit( 2 );
    }

    if ( readserver == NULL ) {
	if ( verbose )
	    printf( "can't find an upstream server\n" );
	syslog( LOG_ERR, 
		"no server name in %s", s );
	unlink( lockfile );
	exit( 2 );
    }

    if ( forceactive )
	fakeactive();
    else
	readactive();

    if ( postserver ) {
        if ( nntpout )
            fclose( nntpout );
        if ( nntpin )
            fclose( nntpin );
        nntpin = nntpout = NULL;
	if ( verbose )
	    printf( "Trying to connect to %s ... ", postserver->name );
        if ( nntpconnect( postserver->name ) ) {
            if ( verbose )
                printf("connected.\n");
            sprintf( lineout, "MODE READER\r\n" );
	    putaline();
            fflush( nntpout );
            if ( nntpreply() != 498 )
                (void) postarticles();
            sprintf( lineout, "QUIT\r\n"); /* say it, then just exit :) */
            putaline();
	}
        else {
            if ( verbose )
                printf("failed.\n");
        }
    }

    while ( readserver ) {
	if ( nntpout )
	    fclose( nntpout );
	if ( nntpin )
	    fclose( nntpin );
	nntpin = nntpout = NULL;
	if ( verbose )
	    printf("Trying to connect to %s ... ", readserver->name);
	if ( nntpconnect( readserver->name ) ) {
	    if ( verbose )
	        printf("connected.\n");
	    sprintf( lineout, "MODE READER\r\n" );
	    putaline();
	    fflush( nntpout );
	    if ( nntpreply() != 498 ) {
		nntpactive();	/* get list of newsgroups or new newsgroups */
		processupstream( readserver->name );
	    } else
	        syslog( LOG_INFO, "Connection refused by ", readserver->name );
	    sprintf( lineout, "QUIT\r\n"); /* say it, then just exit :) */
	    putaline();
	}
	else {
	    if ( verbose )
	        printf("failed.\n");
	}
	if ( usesupplement < 0 )
	    readserver = NULL;
			/* small memory leak but we are soon finished anyway */
	else
	    readserver = readserver->next;
    }
	
    if ( fork() <= 0 ) {
	syslog( LOG_DEBUG, "Process forked." );
	writeactive();
	fixxover();
	unlink( lockfile );
    }

    exit(0);
}
