/*
libutil -- handling xover records

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <rskerka@metronet.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998.

See file COPYING for restrictions on the use of this software.
*/

#include <fcntl.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>

#include "leafnode.h"

extern int errno;
extern int h_errno;
extern struct state _res;

/* return 1 if xover is a legal overview line, 0 else */
int legalxoverline ( char * xover ) {
    char * p;
    char * q;

    if ( !xover )
	return 0;

    /* anything that isn't tab, printable ascii, or latin-* ? then killit */

    p = xover;
    while ( *p ) {
	int c = (unsigned char)*p++;

	if ( ( c != '\t' && c < ' ' ) || ( c > 126 && c < 160 ) ) {
	    syslog( LOG_INFO, "xover error: non-printable chars.");
	    return 0;
	}
    }

    p = xover;
    q = strchr( p, '\t' );
    if ( !q ) {
	syslog( LOG_INFO, "xover error: no Subject: header.");
	return 0;
    }

    /* article number */

    while( p != q ) {
	if ( !isdigit( *p ) ) {
            syslog( LOG_INFO, "xover error: "
                              "article number must consists of digits.");
	    return 0;
	}
	p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	syslog( LOG_INFO, "xover error: no From: header.");
	return 0;
    }

    /* subject: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
        syslog( LOG_INFO, "xover error: no Date: header.");
	return 0;
    }

    /* from: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
        syslog( LOG_INFO, "xover error: no Message-ID: header.");
	return 0;
    }

    /* date: no limitations */

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	syslog( LOG_INFO, "xover error: no References: or Bytes: header.");
	return 0;
    }

    /* message-id: <*@*> */

    if ( *p != '<' ) {
        syslog( LOG_INFO, "xover error: Message-ID does not start with <.");
	return 0;
    }
    while ( p != q && *p != '@' && *p != '>' && *p != ' ' )
	p++;
    if ( *p != '@' ) {
        syslog( LOG_INFO, "xover error: Message-ID does not contain @.");
	return 0;
    }
    while ( p != q && *p != '>' && *p != ' ' )
	p++;
    if ( *p != '>' ) {
        syslog( LOG_INFO, "xover error: Message-ID does not end with >.");
	return 0;
    }
    if ( ++p != q ) {
        syslog( LOG_INFO, "xover error: Message-ID does not end with >.");
	return 0;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
	syslog( LOG_INFO, "xover error: no Bytes: header.");
	return 0;
    }

    /* references: a series of <*@*> separated by space */

    while ( p != q ) {
	if ( *p != '<' ) {
            syslog( LOG_INFO, "xover error: Reference does not start with <.");
	    return 0;
	}
	while ( p != q && *p != '@' && *p != '>' && *p != ' ' )
	    p++;
	if ( *p != '@' ) {
            syslog( LOG_INFO, "xover error: Reference does not contain @.");
	    return 0;
	}
	while ( p != q && *p != '>' && *p != ' ' )
	    p++;
	if ( *p++ != '>' ) {
            syslog( LOG_INFO, "xover error: Reference does not end with >.");
	    return 0;
	}
	while ( p != q && *p == ' ' )
	    p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( !q ) {
        syslog( LOG_INFO, "xover error: no Lines: header.");
	return 0;
    }

    /* byte count */

    while( p != q ) {
	if ( !isdigit( *p ) ) {
	    syslog( LOG_INFO, "xover error: illegal digit in Bytes: header.");
	    return 0;
	}
	p++;
    }

    p = q+1;
    q = strchr( p, '\t' );
    if ( q )
        *q = '\0'; /* kill any extra fields */

 /* line count */

    while( p && *p && p != q ) {
	if ( !isdigit( *p ) ) {
            syslog( LOG_INFO, "xover error: illegal digit in Lines: header.");
	    return 0;
	}
	p++;
    }

    return 1;
}



void stripspace( char * p );
void stripspace( char * p ) {
    char * p1;
    char * p2;

    p1 = p2 = p;
    while ( *p1 ) {
	if ( isspace(*p1) ) {
	    *p2 = ' ';
	    do {
		p1++;
	    } while ( isspace(*p1) );
	    if ( *p1 )
		p2++;
	} else {
	    *p2++ = *p1++;
	}
    }
    *p2 = '\0';
}




char * getxoverline( const char * filename );
char * getxoverline( const char * filename ) {
    char * l;
    FILE * f;

    if ( (f=fopen( filename, "r" )) ) {
	char * from;
	char * subject;
	char * date;
	char * msgid;
	char * references;
	int bytes;
	int lines, linecount;
	char * xref;
	char ** h;
	int body;

	from = subject = date = msgid = references = xref = NULL;
	bytes = lines = linecount = 0;
	h = NULL;
	body = 0;

	do {
	    debug = 0;
	    l = getaline( f );
	    debug = 1;
	    if ( l ) {
		linecount++; /* most luxurious - but it doesn't happen often */
		bytes += strlen( l ) + 2;
	    }
	    if ( body || !l ) {
		/* nothing */
	    } else if ( !body && !*l ) {
		linecount = 0;
		body = 1;
	    } else if ( h && isspace( *l ) ) {
		stripspace( l );
		if ( *l ) {
		    (*h) = critrealloc( *h, strlen( *h ) + strlen( l ) + 1,
					"extending header" );
		    strcat( *h, l );
		}
	    } else if ( !from && !strncasecmp( "From: ", l, 6 ) ) {
		l += 6;
		stripspace( l );
		if ( *l ) {
		    from = strdup( l );
		    h = &from;
		}
	    } else if ( !subject && !strncasecmp( "Subject: ", l, 9 ) ) {
		l += 9;
		stripspace( l );
		if ( *l ) {
		    subject = strdup( l );
		    h = &subject;
		}
	    } else if ( !date && !strncasecmp( "Date: ", l, 6 ) ) {
		l += 6;
		stripspace( l );
		if ( *l ) {
		    date = strdup( l );
		    h = &date;
		}
	    } else if ( !msgid && !strncasecmp( "Message-ID: ", l, 12 ) ) {
		l += 12;
		stripspace( l );
		if ( *l ) {
		    msgid = strdup( l );
		    h = &msgid;
		}
	    } else if ( !references &&
			!strncasecmp( "References: ", l, 12 ) ) {
		l += 12;
		stripspace( l );
		if ( *l ) {
		    references = strdup( l );
		    h = &references;
		}
	    } else if ( !lines && !strncasecmp( "Lines: ", l, 7 ) ) {
		l += 7;
		stripspace( l );
		if ( *l ) {
		    lines = atoi( l );
/* not strictly correct, but there are probably no multiline Line: headers
		    h = &lines;
*/
		}
	    } else if ( !xref && !strncasecmp( "Xref: ", l, 6 ) ) {
		l += 6;
		stripspace( l );
		if ( *l ) {
		    xref = strdup( l );
		    h = &xref;
		}
	    } else {
		h = NULL;
	    }
	} while ( l && !feof(f) );
	if ( from && date && subject && msgid && bytes && lines ) {
	    char * result;

	    result = critmalloc( strlen( from ) + strlen( date ) +
				 strlen( subject ) + strlen( msgid ) +
				 (references ? strlen( references ) : 0 ) +
				 100 + ( xref ? strlen( xref) : 0 ),
				 "computing overview line" );
	    sprintf( result, "%s\t%s\t%s\t%s\t%s\t%s\t%d\t%d",
		     filename, subject, from, date, msgid,
		     references ? references : "" ,
		     bytes,
		     lines ? lines : linecount );
	    if ( xref ) {
		strcat( result, "\tXref: " );
		strcat( result, xref );
	    }
	    free ( from );
	    free ( date );
	    free ( subject );
	    free ( msgid );
	    free ( references );
	    free ( xref );
	    fclose ( f );
	    return result;
	} else {
	    fclose ( f );
	}
	free ( from );
	free ( date );
	free ( subject );
	free ( msgid );
	free ( references );
	free ( xref );
    }
    return NULL;
}


/* utility routine to pull the xover info into memory
   returns 0 if there's some error, non-zero else */

struct xoverinfo * xoverinfo;
int xfirst, xlast;

int getxover( void ) {
    DIR *d;
    struct dirent * de;
    int fd;
    struct stat st;
    int art;
    static char * overview;
    int error;
    char * p, * q;

    /* free any memory left over from last time */

    error = 0;

    if ( xoverinfo ) {
	for( art=xfirst; art<=xlast; art++ ) {
	    if ( xoverinfo[art-xfirst].mallocd &&
		 xoverinfo[art-xfirst].text ) {
		free( xoverinfo[art-xfirst].text );
		xoverinfo[art-xfirst].text = NULL;
	    }
	}
    }

    if ( overview ) {
	free( overview );
	overview = NULL;
    }

    /* find article range, without locking problems */

    d = opendir( "." );
    if ( !d ) {
	syslog( LOG_ERR, "opendir: %m" );
	return 0;
    }

    xfirst = INT_MAX;
    xlast = 0;
    while ( (de=readdir(d)) ) {
	art = strtol( de->d_name, &p, 10 );
	if (p && !*p) {
	    if ( art < xfirst )
		xfirst = art;
	    if ( art > xlast )
		xlast = art;
	}
    }

    if ( xlast < xfirst ) {
	closedir( d );
	return 0;
    }

    /* next, read .overview, correct it if it seems too different from
       what the directory implies, and write the result back */

    rewinddir( d );

    xoverinfo = (struct xoverinfo *)
		critrealloc( (char*)xoverinfo,
			     sizeof(struct xoverinfo) * (xlast+1-xfirst),
			     "allocating overview array" );
    memset( xoverinfo, 0, sizeof(struct xoverinfo) * (xlast+1-xfirst) );

	if ( (fd=open( ".overview", O_RDONLY)) >= 0 &&
	     fstat( fd, &st )==0 &&
	     (overview=(char*)malloc( st.st_size + 1 )) != NULL &&
	 read( fd, overview, st.st_size ) == st.st_size ) {

	close( fd );
	overview[st.st_size] = '\0';

	/* okay, we have the content, so let's parse it roughly */

	p = overview;
	while ( p && *p ) {
	    while ( p && isspace( *p ) )
		p++;
	    q = strchr( p, '\n' );
	    if ( q )
		*q++ = '\0';

	    art = strtol( p, NULL, 10 );
	    if ( art > xlast || art < xfirst ) {
		error++;
	    } else if ( xoverinfo[art-xfirst].text ) {
		error++;
		xoverinfo[art-xfirst].text = NULL;
	    } else {
		xoverinfo[art-xfirst].text = p;
	    }

	    p = q;
	}
    }

    /* so, what was missing? */

    while ( (de=readdir(d)) ) {
	art = strtol( de->d_name, &p, 10 );
	if (p && !*p && art>=xfirst && art<=xlast) {
	    if ( !xoverinfo[art-xfirst].text ) {
		syslog( LOG_DEBUG, "reading XOVER info from %s/%s",
			getcwd( s, 1024 ), de->d_name );
		if ( verbose > 2 )
		    printf( "reading XOVER info from %s/%s\n",
			    getcwd( s, 1024 ), de->d_name );
		error++;
		if ( ( xoverinfo[art-xfirst].text =
		      getxoverline( de->d_name )) != NULL ) {
		    xoverinfo[art-xfirst].mallocd = 1;
		} else if ( delaybody ) {
		    xoverinfo[art-xfirst].mallocd = 1;
		} else {
		    syslog( LOG_INFO, "illegal article: %s/%s",
			    getcwd( s, 1024 ), de->d_name );
		    printf( "illegal article: %s/%s\n",
			    getcwd( s, 1024 ), de->d_name );
		    (void) unlink( de->d_name ); /* most dubious, this */
		}
	    }
	}
	if ( art >= xfirst && art <= xlast && xoverinfo[art-xfirst].text ) {
	    xoverinfo[art-xfirst].exists = 1;
	} else if ( verbose ) {
	    if ( art == 0 && strcmp( de->d_name, ".overview" ) &&
		 strcmp( de->d_name, "." ) && strcmp( de->d_name, ".." ) ) {
		if ( verbose && unlink( de->d_name ) &&
		     errno != EISDIR && errno != EPERM )
		    printf( "attempted to delete %s in case it's junk: %s\n",
			    de->d_name, errno ? strerror(errno) : "OK" );
	    } else if ( art > 0 && art < xfirst )
		printf( "article %d is below the low-water mark (%d)\n",
			art, xfirst );
	    else if ( art > xlast )
		printf( "article %d is above the high-water mark (%d)\n",
			art, xlast );
	    else if ( p && !*p )
		printf( "article %d contained illegal headers\n",
			art );
	}
    }

    /* if something had to be fixed, write a better file to disk for
       next time - race conditions here, but none dangerous */

    if ( error ) {
	int wfd;
	char newfile[20];

	if ( verbose )
	    printf( "corrected %d lines in %s/.overview\n",
		    error, getcwd( s, 1024 ) );

	strcpy( newfile, ".overview.XXXXXX" );
	if ( (wfd=open( mktemp(newfile), O_WRONLY|O_CREAT|O_EXCL, 0664 )) ) {
	    struct iovec oooh[UIO_MAXIOV];
	    int vi, vc, va;

	    vi = vc = va = 0;
	    for ( art=xfirst; art<=xlast; art++ ) {
		if ( xoverinfo[art-xfirst].exists &&
		     xoverinfo[art-xfirst].text ) {
		    oooh[vi].iov_base = xoverinfo[art-xfirst].text;
		    oooh[vi].iov_len = strlen( xoverinfo[art-xfirst].text );
		    vc += oooh[vi++].iov_len + 1;
		    oooh[vi].iov_base = "\n";
		    oooh[vi++].iov_len = 1;
		    if ( vi >= (UIO_MAXIOV - 1) ) {
			if ( writev( wfd, oooh, vi ) != vc ) {
			    syslog( LOG_ERR,
				    "writev() for .overview failed: %m" );
			    art = xlast+1; /* so the loop will stop */
			}
			vi = vc = 0;
			va = 1;
		    }
		}
	    }
	    if ( vi ) {
		if ( writev( wfd, oooh, vi ) != vc ) {
		    syslog( LOG_ERR, "writev() for .overview failed: %m" );
		} else {
		    va = 1;
		}
	    }
	    fchmod( wfd, 0664 );
	    close( wfd );
	    if ( va ) {
		if ( rename( newfile, ".overview" ) ) {
		    if ( unlink( newfile ) )
			syslog( LOG_ERR,
				"rename() and unlink() both failed: %m" );
		    else
			syslog( LOG_ERR,
				"rename(%s/%s, .overview) failed: %m",
				getcwd(s, 1024), newfile );
		} else {
		    if ( verbose )
			printf( "wrote %s/.overview\n",
				getcwd( s, 1024 ) );
		}
	    } else {
		unlink( newfile );
		/* the group must be newly empty: I want to keep the old
		   .overview file I think */
	    }
	} else {
	    syslog( LOG_ERR, "open(O_WRONLY|O_CREAT|O_EXCL) of new "
		    ".overview failed: %m" );
	}
    }
    return 1;
}
