/*
 *  Driver for generic CS4235/CS4236B/CS4237B/CS4238B/CS4239 chips
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"
#include "mpu401.h"
#include "opl3.h"
#include "initval.h"

int snd_index[ SND_CARDS ] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[ SND_CARDS ] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[ SND_CARDS ] = SND_DEFAULT_PORT;	/* PnP setup */
int snd_cport[ SND_CARDS ] = SND_DEFAULT_PORT;	/* PnP setup */
int snd_mpu_port[ SND_CARDS ] = SND_DEFAULT_PORT; /* PnP setup */
int snd_fm_port[ SND_CARDS ] = SND_DEFAULT_PORT; /* PnP setup */
int snd_jport[ SND_CARDS ] = SND_DEFAULT_PORT;	/* joystick port - PnP setup */
int snd_irq[ SND_CARDS ] = SND_DEFAULT_IRQ;	/* 5,7,9,11,12,15 */
int snd_mpu_irq[ SND_CARDS ] = SND_DEFAULT_IRQ;	/* 9,11,12,15 */
int snd_dma1[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 0,1,3,5,6,7 */
int snd_dma1_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
int snd_dma2[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 0,1,3,5,6,7 */
int snd_dma2_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
#ifdef MODULE_PARM
MODULE_PARM( snd_index, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_index, "Index value for CS4236+ soundcard." );
MODULE_PARM( snd_id, "1-" __MODULE_STRING(SND_CARDS) "s" );
MODULE_PARM_DESC( snd_id, "ID string for CS4236+ soundcard." );
MODULE_PARM( snd_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_port, "Port # for CS4236+ driver." );
MODULE_PARM( snd_cport, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_cport, "Control port # for CS4236+ driver." );
MODULE_PARM( snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_mpu_port, "MPU-401 port # for CS4236+ driver." );
MODULE_PARM( snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_fm_port, "FM port # for CS4236+ driver." );
MODULE_PARM( snd_jport, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_jport, "Joystick port # for CS4236+ driver." );
MODULE_PARM( snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_irq, "IRQ # for CS4236+ driver." );
MODULE_PARM( snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma1, "DMA1 # for CS4236+ driver." );
MODULE_PARM( snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma1_size, "Size of DMA1 # for CS4236+ driver." );
MODULE_PARM( snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma2, "DMA2 # for CS4236+ driver." );
MODULE_PARM( snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma2_size, "Size of DMA2 # for CS4236+ driver." );
#endif

static struct snd_card_cs4236 {
  int irqnum;
  int mpuirqnum;
  int dma1num;
  int dma2num;
  snd_card_t *card;
  snd_pcm_t *pcm;
  snd_kmixer_t *mixer;
  snd_rawmidi_t *rmidi;
  snd_synth_t *synth;
  cs4231_t *codec;
  unsigned short pcm_status_reg;
} *snd_card_cs4236_cards[ SND_CARDS ] = SND_DEFAULT_PTR;

static void snd_card_cs4236_use_inc( snd_card_t *card )
{
  MOD_INC_USE_COUNT;
}

static void snd_card_cs4236_use_dec( snd_card_t *card )
{
  MOD_DEC_USE_COUNT;
}

static snd_pcm_t *snd_card_cs4236_detect( snd_card_t *card,
                                          unsigned short port,
                                          unsigned short cport,
                                          unsigned short mpu_port,
                                          unsigned short fm_port,
                                          unsigned short jport,
                                          unsigned short irqnum,
                                          unsigned short dma1num,
                                          unsigned short dma2num )
{
  snd_pcm_t *pcm;

  if ( snd_register_ioport( card, port, 4, "CS4236+" ) < 0 )
    return NULL;
  if ( snd_register_ioport( card, cport, 8, "CS4236+ - control" ) < 0 ) {
    snd_unregister_ioports( card );
    return NULL;
  }
  if ( mpu_port != SND_AUTO_PORT ) {
    if ( snd_register_ioport( card, mpu_port, 2, "CS4236+ - MPU-401" ) < 0 ) {
      snd_unregister_ioports( card );
      snd_printk( "ports for CS4236+ MPU-401 are already used, try another\n" );
      return NULL;
    }
  }
  if ( fm_port != SND_AUTO_PORT ) {
    if ( snd_register_ioport( card, fm_port, 4, "CS4236+ - FM" ) < 0 ) {
      snd_unregister_ioports( card );
      snd_printk( "ports for CS4236+ FM are already used, try another\n" );
      return NULL;
    }
  }
  if ( jport != SND_AUTO_PORT ) {
    if ( snd_register_ioport( card, jport, 8, "CS4236+ - joystick" ) < 0 ) {
      snd_unregister_ioports( card );
      snd_printk( "ports for CS4236+ joystick are already used, try another\n" );
      return NULL;
    }
  }
  pcm = snd_cs4236_new_device( card, port, cport, jport, irqnum, dma1num, dma2num, CS4231_HW_DETECT3 );
  if ( !pcm ) {
    snd_unregister_ioports( card );
    return NULL;
  }
  return pcm;
}

static void snd_card_cs4236_interrupt( int irq, void *dev_id, struct pt_regs *regs )
{
  struct snd_card_cs4236 *acard;
  register unsigned char status;
  
  acard = (struct snd_card_cs4236 *)dev_id;
  if ( acard == NULL || acard -> pcm == NULL ) return; 
  status = inb( acard -> pcm_status_reg );
  if ( status & 0x01 )
    snd_cs4231_interrupt( acard -> pcm, status );
}

static void snd_card_cs4236_midi_interrupt( int irq, void *dev_id, struct pt_regs *regs )
{
  struct snd_card_cs4236 *acard;
  
  acard = (struct snd_card_cs4236 *)dev_id;
  if ( acard == NULL || acard -> rmidi == NULL ) return; 
  snd_mpu401_uart_interrupt( acard -> rmidi );
}

static int snd_card_cs4236_resources( int dev, struct snd_card_cs4236 *acard, snd_card_t *card )
{
  static int possible_irqs[] = { -1 };
  static int possible_dmas[] = { -1 };

  if ( (acard -> irqnum = snd_register_interrupt( card, "CS4236+", snd_irq[ dev ], SND_IRQ_TYPE_ISA, snd_card_cs4236_interrupt, acard, possible_irqs )) < 0 )
    return acard -> irqnum;
  acard -> mpuirqnum = SND_IRQ_DISABLE;
  if ( snd_mpu_port[ dev ] != SND_AUTO_PORT && snd_mpu_irq[ dev ] >= 0 ) {
    if ( (acard -> mpuirqnum = snd_register_interrupt( card, "CS4236+ - MPU401", snd_mpu_irq[ dev ], SND_IRQ_TYPE_ISA, snd_card_cs4236_midi_interrupt, acard, possible_irqs )) < 0 ) {
      acard -> mpuirqnum = SND_IRQ_DISABLE;
      snd_mpu_port[ dev ] = SND_AUTO_PORT;
      snd_printk( "warning: can't allocate IRQ for MPU401, midi port is disabled\n" );
    }
  } else {
    snd_mpu_port[ dev ] = SND_AUTO_PORT;
    acard -> mpuirqnum = SND_IRQ_DISABLE;
  }
  if ( (acard -> dma1num = snd_register_dma_channel( card, "CS4236+ - DMA1", snd_dma1[ dev ], SND_DMA_TYPE_ISA, snd_dma1_size[ dev ], possible_dmas )) < 0 ) {
    return acard -> dma1num;
  }
  if ( snd_dma2[ dev ] >= 0 ) {
    if ( (acard -> dma2num = snd_register_dma_channel( card, "CS4236+ - DMA2", snd_dma2[ dev ], SND_DMA_TYPE_ISA, snd_dma2_size[ dev ], possible_dmas )) < 0 ) {
      return acard -> dma2num;
    }
  } else {
    acard -> dma2num = SND_DMA_DISABLE;
  }
  return 0;
}

static int snd_card_cs4236_probe( int dev, struct snd_card_cs4236 *acard )
{
  snd_card_t *card;
  snd_pcm_t *pcm = NULL;
  snd_pcm1_t *pcm1;
  snd_kmixer_t *mixer = NULL;
  snd_rawmidi_t *rmidi = NULL;
  snd_synth_t *synth = NULL;
  cs4231_t *codec;

  if ( snd_port[ dev ] == SND_AUTO_PORT ||
       snd_cport[ dev ] == SND_AUTO_PORT ) {
    snd_printk( "probing for CS4236+ isn't supported\n" );
    snd_printk( "port = 0x%x, cport = 0x%x\n", snd_port[ dev ], snd_cport[ dev ] );
    return -ENODEV;
  }
  card = snd_card_new( snd_index[ dev ], snd_id[ dev ],
  		       snd_card_cs4236_use_inc, snd_card_cs4236_use_dec );
  if ( !card ) return -ENOMEM;
  card -> type = SND_CARD_TYPE_CS4236;
  if ( snd_card_cs4236_resources( dev, acard, card ) < 0 ) {
    snd_card_free( card );
    return -EBUSY;
  }
  pcm = snd_card_cs4236_detect( card, snd_port[ dev ],
  				snd_cport[ dev ],
  				snd_mpu_port[ dev ],
  				snd_fm_port[ dev ],
  				snd_jport[ dev ],
  				acard -> irqnum, acard -> dma1num,
  				acard -> dma2num == SND_DMA_DISABLE ? acard -> dma1num : acard -> dma2num );
  if ( !pcm ) {
    snd_card_free( card );
    return -ENODEV;
  }

  pcm1 = (snd_pcm1_t *)pcm -> private_data;
  codec = (cs4231_t *)pcm1 -> private_data;
  acard -> codec = codec;
  acard -> pcm_status_reg = codec -> port + 2;

  mixer = snd_cs4236_new_mixer( pcm );
  if ( !mixer ) goto __nodev;

  if ( snd_fm_port[ dev ] != SND_AUTO_PORT )
    synth = snd_opl3_new_device( card, snd_fm_port[ dev ], snd_fm_port[ dev ] + 2, OPL3_HW_OPL3_CS, 0 );
  
  if ( acard -> mpuirqnum != SND_IRQ_DISABLE ) {
    rmidi = snd_mpu401_uart_new_device( card, MPU401_HW_CS4232, snd_mpu_port[ dev ], acard -> mpuirqnum );
    if ( !rmidi ) goto __nodev;
    if ( snd_rawmidi_register( rmidi, 0 ) < 0 ) goto __nodev;
  }
  if ( synth && snd_synth_register( synth ) < 0 ) {
    if ( rmidi ) snd_rawmidi_unregister( rmidi ); rmidi = NULL;
    goto __nodev;
  }
  if ( snd_mixer_register( mixer, 0 ) < 0 ) {
    if ( synth ) snd_synth_unregister( synth ); synth = NULL;
    if ( rmidi ) snd_rawmidi_unregister( rmidi ); rmidi = NULL;
    goto __nodev;
  }
  if ( snd_pcm_register( pcm, 0 ) ) {
    if ( synth ) snd_synth_unregister( synth ); synth = NULL;
    if ( rmidi ) snd_rawmidi_unregister( rmidi ); rmidi = NULL;
    snd_mixer_unregister( mixer ); mixer = NULL;
    goto __nodev;
  }
  snd_enable_irq( card, acard -> irqnum );
  if ( acard -> mpuirqnum != SND_IRQ_DISABLE )
    snd_enable_irq( card, acard -> mpuirqnum );
  strcpy( card -> abbreviation, "CS4236" );
  strcpy( card -> shortname, pcm -> name );
  sprintf( card -> longname, "%s at 0x%x, irq %i, dma %i",
  	pcm -> name,
  	codec -> port,
  	card -> irqs[ acard -> irqnum ] -> irq,
  	card -> dmas[ acard -> dma1num ] -> dma );
  if ( acard -> dma2num != SND_DMA_DISABLE ) {
    sprintf( card -> longname + strlen( card -> longname ), "&%i",
    	card -> dmas[ acard -> dma2num ] -> dma );
  }
  if ( !snd_card_register( card ) ) {
    acard -> card = card;
    acard -> pcm = pcm;
    acard -> mixer = mixer;
    acard -> rmidi = rmidi;
    acard -> synth = synth;
    return 0;
  }
  snd_synth_unregister( synth );
  snd_mixer_unregister( mixer );
  snd_pcm_unregister( pcm );
  snd_card_free( card );
  return -ENOMEM;
  
  __nodev:
  if ( synth ) snd_synth_free( synth );
  if ( rmidi ) snd_rawmidi_free( rmidi );
  if ( mixer ) snd_mixer_free( mixer );
  if ( pcm ) snd_pcm_free( pcm );
  snd_card_free( card );
  return -ENXIO;
}

int init_module( void )
{
  int dev, cards;
  struct snd_card_cs4236 *acard;

#ifndef LINUX_2_1
  register_symtab( NULL );
#endif
  for ( dev = cards = 0; dev < SND_CARDS && snd_port[ dev ] > 0; dev++ ) {
    acard = (struct snd_card_cs4236 *)snd_malloc( sizeof( struct snd_card_cs4236 ) );
    if ( !acard ) continue;
    memset( acard, 0, sizeof( struct snd_card_cs4236 ) );
    if ( snd_card_cs4236_probe( dev, acard ) < 0 ) {
      snd_free( acard, sizeof( struct snd_card_cs4236 ) );
      if ( snd_port[ dev ] == SND_AUTO_PORT ) break;
      snd_printk( "CS4236+ soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[ dev ] );
      continue;
    }
    snd_card_cs4236_cards[ dev ] = acard;
    cards++;
  }
  if ( !cards ) { 
    snd_printk( "CS4236+ soundcard #%i not found or device busy\n", dev + 1 );
    return -ENODEV;
  }
  return 0;
}

void cleanup_module( void )
{
  int idx;
  struct snd_card_cs4236 *acard;

  for ( idx = 0; idx < SND_CARDS; idx++ ) {
    acard = snd_card_cs4236_cards[ idx ];
    if ( acard ) {
      snd_card_unregister( acard -> card );
      if ( acard -> synth )
        snd_synth_unregister( acard -> synth );
      if ( acard -> rmidi )
        snd_rawmidi_unregister( acard -> rmidi );
      if ( acard -> mixer )
        snd_mixer_unregister( acard -> mixer );
      if ( acard -> pcm ) {
        snd_pcm_unregister( acard -> pcm );
      }
      snd_card_free( acard -> card );
      snd_free( acard, sizeof( struct snd_card_cs4236 ) );
    }
  }
}
