/*
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *  Routines for control of CS4232/4232A/4235/4236B/4237B/4238B/4239 chips
 *
 *  Note:
 *     Code for CS4235/CS4239 code isn't tested very much and may fail.
 *
 *  TODO:
 *     CS4235/CS4239 doesn't support special PCM formats - remove.
 *
 *  Bugs:
 *     -----
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
  
/*
 *  Indirect control registers (CS4236B+)
 * 
 *  C0
 *     D8: WSS reset (all chips)
 *
 *  C1 (all chips except CS4236)
 *     D7-D5: version 
 *     D4-D0: chip id
 *	       11101 - CS4235
 *             01011 - CS4236B
 *             01000 - CS4237B
 *             01001 - CS4238B
 *             11110 - CS4239
 *
 *  C2
 *     D7-D4: 3D Space (CS4235,CS4237B,CS4238B,CS4239)
 *     D3-D0: 3D Center (CS4237B); 3D Volume (CS4238B)
 * 
 *  C3
 *     D7: 3D Enable (CS4237B)
 *     D6: 3D Mono Enable (CS4237B)
 *     D5: 3D Serial Ouput (CS4237B,CS4238B)
 *     D4: 3D Enable (CS4235,CS4238B,CS4239)
 *
 *  C4
 *     D7: consumer serial port enable (CS4237B,CS4238B)
 *     D6: channels status block reset (CS4237B,CS4238B)
 *     D5: user bit in sub-frame of digital audio data (CS4237B,CS4238B)
 *     D4: validity bit bit in sub-frame of digital audio data (CS4237B,CS4238B)
 * 
 *  C5  lower channel status (digital serial data description) (CS4237B,CS4238B)
 *     D7-D6: first two bits of category code
 *     D5: lock
 *     D4-D3: pre-emphasis (0 = none, 1 = 50/15us)
 *     D2: copy/copyright (0 = copy inhibited)
 *     D1: 0 = digital audio / 1 = non-digital audio
 *     
 *  C6  upper channel status (digital serial data description) (CS4237B,CS4238B)
 *     D7-D6: sample frequency (0 = 44.1kHz)
 *     D5: generation status (0 = no indication, 1 = original/commercially prerecorded data)
 *     D4-D0: category code (upper bits)
 *
 *  C7  wavetable control
 *     D7: volume control interrupt enable (CS4235,CS4239)
 *     D6: hardware volume control format (CS4235,CS4239)
 *     D3: wavetable serial port enable (all chips)
 *     D2: DSP serial port switch (all chips)
 *     D1: disable MCLK (all chips)
 *     D0: force BRESET low (all chips)
 *
 */
  
#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"

/*
 *
 */

static void snd_cs4236_ext_outm( cs4231_t *codec, unsigned char reg, unsigned char mask, unsigned char val )
{
#if 1
  outb( codec -> mce_bit | 0x17, CS4231P( codec, REGSEL ) );
  outb( reg | (codec -> image.ra3ic & 0x01), CS4231P( codec, REG ) );
  val |= inb( CS4231P( codec, REG ) ) & mask;
  outb( codec -> mce_bit | 0x17, CS4231P( codec, REGSEL ) );
  outb( reg | (codec -> image.ra3ic & 0x01), CS4231P( codec, REG ) );
  outb( val, CS4231P( codec, REG ) );
#else
  unsigned char res, val1;
  outb( codec -> mce_bit | 0x17, CS4231P( codec, REGSEL ) );
  outb( reg | (codec -> image.ra3ic & 0x01), CS4231P( codec, REG ) );
  res = inb( CS4231P( codec, REG ) );
  val1 = val;
  val |= res & mask;
  outb( codec -> mce_bit | 0x17, CS4231P( codec, REGSEL ) );
  outb( reg | (codec -> image.ra3ic & 0x01), CS4231P( codec, REG ) );
  outb( val, CS4231P( codec, REG ) );
  printk( "ext outm : reg = 0x%x, mask = 0x%x, val = 0x%x, in = 0x%x, out = 0x%x\n", reg, mask, val1, res, val );
#endif
}

static void snd_cs4236_ctrl_out( cs4231_t *codec, unsigned char reg, unsigned char val )
{
  outb( reg, codec -> cport + 3 );
  outb( val, codec -> cport + 4 );
}

static unsigned char snd_cs4236_ctrl_in( cs4231_t *codec, unsigned char reg )
{
  outb( reg, codec -> cport + 3 );
  return inb( codec -> cport + 4 );
}

static void snd_cs4236_ctrl_outm( cs4231_t *codec, unsigned char reg, unsigned char mask, unsigned char val )
{
  unsigned char res;

  outb( reg, codec -> cport + 3 );
  res = inb( codec -> cport + 4 );
  outb( reg, codec -> cport + 3 );
  outb( val | (res & mask), codec -> cport + 4 );
}

/*
 *  PCM
 */

static unsigned char snd_cs4236_rate( unsigned int rate, unsigned int *real_rate )
{
  static struct {
    unsigned char val;
    unsigned int divider;
    unsigned int rate;
  } dividers[] = {
    { 1,   353,  48000 },
    { 2,   529,  32000 },
    { 3,   617,  27420 },
    { 4,   1058, 16000 },
    { 5,   1764, 9600 },
    { 6,   2117, 8000 },
    { 7,   2558, 6620 },
  };
  int idx;
  unsigned int divider;
  unsigned char val = 21;

  divider = 16934400U / rate;
  for ( idx = 0; idx < 7; idx++ ) {
    if ( dividers[idx].divider == divider || dividers[idx].rate == rate ) {
      if ( real_rate )
        *real_rate = dividers[idx].rate;
      return dividers[idx].val;
    }
  }
  
  if ( divider > 3072 ) {
    val = 192;
  } else {
    if ( divider < 336 ) {
      val = 21;
    } else {
      val = divider >> 4;
    }
  }
  if ( real_rate )
    *real_rate = 16934400U / ( (unsigned int)val << 4 );
  return val;
}

static unsigned int snd_cs4236_xrate( snd_pcm1_t *pcm1, cs4231_t *codec, unsigned int rate )
{
  unsigned int rrate;

  snd_cs4236_rate( rate, &rrate ); 
  return rrate;
}

static void snd_cs4236_playback_format( snd_pcm1_t *pcm1, cs4231_t *codec, unsigned char pdfr )
{
  unsigned long flags;
  
  snd_spin_lock( codec, reg, &flags );
  /* set fast playback format change and clean playback FIFO */
  snd_cs4231_out( codec, CS4231_ALT_FEATURE_1, codec -> image.afei | 0x10 );
  snd_cs4231_out( codec, CS4231_PLAYBK_FORMAT, codec -> image.pdfr = (pdfr & 0xf0) );
  snd_cs4231_out( codec, CS4231_ALT_FEATURE_1, codec -> image.afei );
  snd_cs4236_ext_out( codec, CS4236_DAC_RATE, snd_cs4236_rate( pcm1 -> playback.real_rate, NULL ) );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_record_format( snd_pcm1_t *pcm1, cs4231_t *codec, unsigned char cdfr )
{
  unsigned long flags;
  
  snd_spin_lock( codec, reg, &flags );
  /* set fast record format change and clean playback FIFO */
  snd_cs4231_out( codec, CS4231_ALT_FEATURE_1, codec -> image.afei | 0x20 );
  snd_cs4231_out( codec, CS4231_REC_FORMAT, codec -> image.cdfr = (cdfr & 0xf0) );
  snd_cs4231_out( codec, CS4231_ALT_FEATURE_1, codec -> image.afei );
  snd_cs4236_ext_out( codec, CS4236_ADC_RATE, snd_cs4236_rate( pcm1 -> record.real_rate, NULL ) );
  snd_spin_unlock( codec, reg, &flags );
}

snd_pcm_t *snd_cs4236_new_device( snd_card_t *card,
				  unsigned short port,
				  unsigned short cport,	/* control port */
				  unsigned short jport,	/* joystick port */
				  unsigned short irqnum,
				  unsigned short dmanum1,
				  unsigned short dmanum2,
				  unsigned short hardware )
{
  snd_pcm_t *pcm;
  snd_pcm1_t *pcm1;
  cs4231_t *codec;
  unsigned char ver;
  
  pcm = snd_cs4231_new_device( card, port, irqnum, dmanum1, dmanum2, hardware );
  if ( !pcm ) return NULL;
  pcm1 = (snd_pcm1_t *)pcm -> private_data;
  codec = (cs4231_t *)pcm1 -> private_data;
  if ( !(codec -> hardware & CS4231_HW_CS4236_MASK) ) {
    snd_pcm_free( pcm );
    return NULL;
  }
  strcpy( pcm -> id, "CS4232" );
  codec -> cport = cport;
  codec -> jport = jport;
  if ( codec -> hardware != CS4231_HW_CS4236 ) {
#if 0
    int idx;
    for ( idx = 0; idx < 8; idx++ )
      snd_printk( "CD%i = 0x%x\n", idx, inb( codec -> cport + idx ) );
    for ( idx = 0; idx < 9; idx++ )
      snd_printk( "C%i = 0x%x\n", idx, snd_cs4236_ctrl_in( codec, idx ) );
#endif
    ver = snd_cs4236_ctrl_in( codec, 1 );
    snd_printdd( "CS4236: [0x%x] C1 (version) = 0x%x\n", cport, ver );
    if ( ver == 0xff || (ver & 0x1f) == 0 ) {
      snd_printk( "CS4236+ chip detected, but control port 0x%x is bad\n", cport );
      snd_pcm_free( pcm );
      return NULL;
    }
    snd_cs4236_ctrl_out( codec, 0, 0x00 );
    snd_cs4236_ctrl_out( codec, 2, 0x00 );
    snd_cs4236_ctrl_out( codec, 3, 0x00 );
    snd_cs4236_ctrl_out( codec, 4, 0x00 );
    snd_cs4236_ctrl_out( codec, 5, 0x00 );
    snd_cs4236_ctrl_out( codec, 6, 0x00 );
    snd_cs4236_ctrl_out( codec, 7, 0x00 );
    snd_cs4236_ctrl_out( codec, 8, 0x03 );
    if ( codec -> hardware == CS4231_HW_CS4235 ||
         codec -> hardware == CS4231_HW_CS4239 ) {
      snd_printk( "%s chip isn't fully supported yet (code isn't tested)...\n", pcm -> name );
      /* don't fail, some code may work */
      pcm1 -> playback.hw.formats = 
      pcm1 -> record.hw.formats = SND_PCM_FMT_MU_LAW | SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE;
      pcm1 -> playback.hw.hw_formats = 
      pcm1 -> record.hw.hw_formats = SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE;
      pcm1 -> playback.hw.dma = snd_pcm1_playback_dma_ulaw_loud;
      pcm1 -> record.hw.dma = snd_pcm1_record_dma_ulaw_loud;
    }
  }
  pcm -> info_flags &= ~SND_PCM_INFO_DUPLEX_LIMIT;
  codec -> set_playback_rate =
  codec -> set_record_rate = snd_cs4236_xrate;
  codec -> set_playback_format = snd_cs4236_playback_format;
  codec -> set_record_format = snd_cs4236_record_format;
  snd_cs4236_ext_out( codec, CS4236_DAC_MUTE, 0xe0 );	/* IFSE enable, digital master mute */
  return pcm;
}

/*
 *  MIXER
 */

static void snd_cs4236_record_source( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int enable )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;
  unsigned char regl, regr;
  int ext = 0;

  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  switch ( channel -> hw.priority ) {
    case SND_MIXER_PRI_MIC:
      ext = 1;
      break;
    case SND_MIXER_PRI_MASTERD:
      snd_spin_lock( codec, reg, &flags );
      snd_cs4236_ext_outm( codec, CS4236_DAC_MUTE, 0x3f, !enable ? 0xc0 : 0 );
      snd_spin_unlock( codec, reg, &flags );
      return;
  }
  snd_spin_lock( codec, reg, &flags );
  if ( ext ) {
    snd_cs4236_ext_outm( codec, regl, 0x7f, !enable ? 0x80 : 0 );
    snd_cs4236_ext_outm( codec, regr, 0x7f, !enable ? 0x80 : 0 );
  } else {
    snd_cs4231_outm( codec, regl, 0xbf, !enable ? 0x40 : 0 );
    snd_cs4231_outm( codec, regr, 0xbf, !enable ? 0x40 : 0 );
  }
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;
  unsigned char regl, regr, mask;

  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  mask = (unsigned char)channel -> hw.private_value;
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ext_outm( codec, regl, ~mask, (mute & SND_MIX_MUTE_LEFT) ? mask : 0 );
  snd_cs4236_ext_outm( codec, regr, ~mask, (mute & SND_MIX_MUTE_RIGHT) ? mask : 0 );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;
  unsigned char regl, regr, mask;

  regl = (unsigned char)(channel -> hw.private_value >> 24);
  regr = (unsigned char)(channel -> hw.private_value >> 16);
  if ( channel -> hw.priority == SND_MIXER_PRI_MASTERD ) {
    if ( left < 64 ) left = 63 - left; else left = 64 + (71 - left);
    if ( right < 64 ) right = 63 - right; else right = 64 + (71 - right);
    mask = 0x7f;
  } else {
    mask = channel -> hw.max;
    left = (mask - left);
    right = (mask - right);
  }
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ext_outm( codec, regl, ~mask, left );
  if ( channel -> hw.stereo )
    snd_cs4236_ext_outm( codec, regr, ~mask, right );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_loopback_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, CS4231_LOOPBACK, 0xfe, !(mute & SND_MIX_MUTE_LEFT) ? 0x01 : 0 );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_loopback_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  left = (63 - left);
  right = (63 - right);
  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, CS4231_LOOPBACK, 0x03, left << 2 );
  snd_cs4236_ext_outm( codec, CS4236_RIGHT_LOOPBACK, 0xc0, right );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_mono_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, CS4231_MONO_CTRL, 0x7f, (mute & SND_MIX_MUTE) ? 0x80 : 0 );
  snd_cs4236_ext_outm( codec, CS4236_LEFT_MIX_CTRL, 0x7f, (mute & SND_MIX_MUTE) ? 0x80 : 0 );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_igain_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  left = (3 - left);
  right = (3 - right);
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ext_outm( codec, CS4236_LEFT_MIX_CTRL, ~0x60, left << 5 );
  snd_cs4236_ext_outm( codec, CS4236_RIGHT_MIX_CTRL, ~0x60, right << 5 );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_3d_mute( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, unsigned int mute )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;
  unsigned char mask = 0x10;
  
  if ( codec -> hardware == CS4231_HW_CS4237B ) mask = 0x80;
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ctrl_outm( codec, 3, ~mask, !(mute & SND_MIX_MUTE) ? mask : 0 );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4236_3d_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;
  unsigned char mask = 0xf0, shift;

  shift = (unsigned char)channel -> hw.private_value;  
  mask >>= shift;
  left = (15 - left);
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ctrl_outm( codec, 2, mask, left << shift );
  snd_spin_unlock( codec, reg, &flags );
}

static void snd_cs4235_master1_volume_level( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel, int left, int right )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  switch ( left ) {
    case 0: left = 3; break;
    case 1: left = 0; break;
    case 2: left = 2; break;
    default: left = 1; break;
  }
  switch ( right ) {
    case 0: right = 3; break;
    case 1: right = 0; break;
    case 2: right = 2; break;
    default: right = 1; break;
  }
  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, 27, 0x9f, left << 5 );
  snd_cs4231_outm( codec, 29, 0x9f, right << 5 );
  snd_spin_unlock( codec, reg, &flags );
}

#define CS4236_MIXS (sizeof(snd_cs4236_mixs)/sizeof(struct snd_stru_mixer_channel_hw))
#define CS4236_PRIVATE( left, right, mask ) ((left << 24)|(right << 16)|(mask))

static struct snd_stru_mixer_channel_hw snd_cs4236_mixs[] = {
  {
    SND_MIXER_PRI_MIC,          /* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_MIC,           /* device name */
    SND_MIXER_OSS_MIC,          /* OSS device # */
    0, 1, 0, 1,                 /* mmute/stereo/digital/input */
    0, 31,                      /* min, max value */
    -2250, 2250, 150,		/* min, max, step - dB */
    CS4236_PRIVATE( CS4236_LEFT_MIC, CS4236_RIGHT_MIC, 0x40 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    snd_cs4236_record_source,	/* record source */
    snd_cs4236_mute,		/* set mute */
    snd_cs4236_volume_level,    /* set volume level */
  },
  {
    SND_MIXER_PRI_FM,           /* priority */
    SND_MIXER_PRI_MASTERD,      /* parent priority */
    SND_MIXER_ID_FM,            /* device name */
    SND_MIXER_OSS_SYNTH,	/* OSS device # */
    0, 1, 1, 0,                 /* mmute/stereo/digital/input */
    0, 63,                      /* min, max value */
    -9450, 0, 150,		/* min, max, step - dB */
    CS4236_PRIVATE( CS4236_LEFT_FM, CS4236_RIGHT_FM, 0x80 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    snd_cs4236_mute,		/* set mute */
    snd_cs4236_volume_level,    /* set volume level */
  },
  {
    SND_MIXER_PRI_DSP,		/* priority */
    SND_MIXER_PRI_MASTERD,      /* parent priority */
    SND_MIXER_ID_DSP,		/* device name */
    SND_MIXER_OSS_DIGITAL1,	/* OSS device # */
    0, 1, 1, 0,                 /* mmute/stereo/digital/input */
    0, 63,                      /* min, max value */
    -9450, 0, 150,		/* min, max, step - dB */
    CS4236_PRIVATE( CS4236_LEFT_DSP, CS4236_RIGHT_DSP, 0x80 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    snd_cs4236_mute,		/* set mute */
    snd_cs4236_volume_level,    /* set volume level */
  },
  {
    SND_MIXER_PRI_SYNTHESIZER,	/* priority */
    SND_MIXER_PRI_MASTERD,      /* parent priority */
    SND_MIXER_ID_SYNTHESIZER,	/* device name */
    SND_MIXER_OSS_VIDEO,	/* OSS device # */
    0, 1, 1, 0,                 /* mmute/stereo/digital/input */
    0, 63,                      /* min, max value */
    -9450, 0, 150,		/* min, max, step - dB */
    CS4236_PRIVATE( CS4236_LEFT_WAVE, CS4236_RIGHT_WAVE, 0x80 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    snd_cs4236_mute,		/* set mute */
    snd_cs4236_volume_level,    /* set volume level */
  },
  {
    SND_MIXER_PRI_MASTERD,	/* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_MASTERD,	/* device name */
    SND_MIXER_OSS_VOLUME,	/* OSS device # */
    0, 1, 0, 0,                 /* mmute/stereo/digital/input */
    0, 71,                      /* min, max value */
    -9450, 1200, 150,		/* min, max, step - dB */
    CS4236_PRIVATE( CS4236_LEFT_MASTER, CS4236_RIGHT_MASTER, 0x80 ),
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    snd_cs4236_mute,		/* set mute */
    snd_cs4236_volume_level,    /* set volume level */
  },
  {
    SND_MIXER_PRI_IGAIN,	/* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_IGAIN,		/* device name */
    SND_MIXER_OSS_IGAIN,	/* OSS device # */
    0, 1, 0, 0,                 /* mmute/stereo/digital/input */
    0, 3,                       /* min, max value */
    -1800, 0, 600,		/* min, max, step - dB */
    0,
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_cs4236_igain_volume_level, /* set volume level */
  },
  {
    SND_MIXER_PRI_3D_CENTER,	/* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_3D_CENTER,	/* device name */
    SND_MIXER_OSS_DIGITAL2,	/* OSS device # */
    0, 0, 0, 0,                 /* mmute/stereo/digital/input */
    0, 15,                      /* min, max value */
    -2250, 0, 150,		/* min, max, step - dB */
    0,
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    NULL,			/* set mute */
    snd_cs4236_3d_volume_level, /* set volume level */
  },
  {
    SND_MIXER_PRI_3D_SPACE,	/* priority */
    SND_MIXER_PRI_PARENT,       /* parent priority */
    SND_MIXER_ID_3D_SPACE,	/* device name */
    SND_MIXER_OSS_DIGITAL3,	/* OSS device # */
    0, 0, 0, 0,                 /* mmute/stereo/digital/input */
    0, 15,                      /* min, max value */
    -2250, 0, 150,		/* min, max, step - dB */
    4,
    NULL,                       /* compute dB -> linear */
    NULL,                       /* compute linear -> dB */
    NULL,			/* record source */
    snd_cs4236_3d_mute,		/* set mute */
    snd_cs4236_3d_volume_level, /* set volume level */
  },
};

static int snd_cs4236_mic_gain_get( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  uswitch -> type = SND_MIXER_SW_TYPE_BOOLEAN;
  snd_spin_lock( codec, reg, &flags );
  uswitch -> value.enable = snd_cs4236_ext_in( codec, CS4236_LEFT_MIC ) & 0x20 ? 1 : 0;
  snd_spin_unlock( codec, reg, &flags );
  return 0;
}

static int snd_cs4236_mic_gain_set( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  if ( uswitch -> type != SND_MIXER_SW_TYPE_BOOLEAN ) return -EINVAL;
  snd_spin_lock( codec, reg, &flags );
  snd_cs4236_ext_outm( codec, CS4236_LEFT_MIC, ~0x20, uswitch -> value.enable ? 0x20 : 0 );
  snd_cs4236_ext_outm( codec, CS4236_RIGHT_MIC, ~0x20, uswitch -> value.enable ? 0x20 : 0 );
  snd_spin_unlock( codec, reg, &flags );
  return 0;
}

static snd_kmixer_switch_t snd_cs4236_mic_gain = {
  SND_MIXER_SW_MIC_GAIN,
  snd_cs4236_mic_gain_get,
  snd_cs4236_mic_gain_set,
  0,
  NULL
};

static int snd_cs4236_mono_output_get( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  uswitch -> type = SND_MIXER_SW_TYPE_BOOLEAN;
  snd_spin_lock( codec, reg, &flags );
  uswitch -> value.enable = snd_cs4231_in( codec, CS4231_MONO_CTRL ) & 0x40 ? 1 : 0;
  snd_spin_unlock( codec, reg, &flags );
  return 0;
}

static int snd_cs4236_mono_output_set( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  if ( uswitch -> type != SND_MIXER_SW_TYPE_BOOLEAN ) return -EINVAL;
  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, CS4231_MONO_CTRL, ~0x40, uswitch -> value.enable ? 0x40 : 0 );
  snd_cs4236_ext_outm( codec, CS4236_RIGHT_MIX_CTRL, 0x7f, uswitch -> value.enable ? 0x80 : 0 );
  snd_spin_unlock( codec, reg, &flags );
  return 0;
}

static snd_kmixer_switch_t snd_cs4236_mono_output = {
  "Mono Output",
  snd_cs4236_mono_output_get,
  snd_cs4236_mono_output_set,
  0,
  NULL
};

static int snd_cs4236_iec958_get( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  uswitch -> type = SND_MIXER_SW_TYPE_USER;
  snd_spin_lock( codec, reg, &flags );
  uswitch -> value.enable = snd_cs4231_in( codec, CS4231_ALT_FEATURE_1 ) & 0x02 ? 1 : 0;
  uswitch -> value.data8[4] = snd_cs4236_ctrl_in( codec, 5 );
  uswitch -> value.data8[5] = snd_cs4236_ctrl_in( codec, 6 );
  /* 3D serial output */
  uswitch -> value.data8[8] = snd_cs4236_ctrl_in( codec, 3 ) & 0x20 ? 1 : 0;
  snd_spin_unlock( codec, reg, &flags );
  return 0;
}

static int snd_cs4236_iec958_set( snd_kmixer_t *mixer, snd_kmixer_switch_t *kswitch, snd_mixer_switch_t *uswitch )
{
  unsigned long flags;
  cs4231_t *codec = (cs4231_t *)mixer -> private_data;

  if ( uswitch -> type != SND_MIXER_SW_TYPE_USER ) return -EINVAL;
  snd_mutex_down( codec, mce );
  snd_cs4231_mce_up( codec );
  snd_spin_lock( codec, reg, &flags );
  snd_cs4231_outm( codec, CS4231_ALT_FEATURE_1, ~0x02, uswitch -> value.enable ? 0x02 : 0 );
  snd_cs4236_ctrl_out( codec, 5, uswitch -> value.data8[4] );
  snd_cs4236_ctrl_out( codec, 6, uswitch -> value.data8[5] );
  snd_cs4236_ctrl_outm( codec, 3, ~0x20, uswitch -> value.data8[8] ? 0x20 : 0 );
  snd_spin_unlock( codec, reg, &flags );
  snd_cs4231_mce_down( codec );
  snd_mutex_up( codec, mce );
  return 0;
}

static snd_kmixer_switch_t snd_cs4236_iec958 = {
  "IEC-958 (S/PDIF) Output",
  snd_cs4236_iec958_get,
  snd_cs4236_iec958_set,
  0,
  NULL
};

snd_kmixer_t *snd_cs4236_new_mixer( snd_pcm_t *pcm )
{
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel, *channel1;
  snd_pcm1_t *pcm1;
  cs4231_t *codec;
  int idx;

  mixer = snd_cs4231_new_mixer( pcm );
  if ( !mixer ) return NULL;
  pcm1 = (snd_pcm1_t *)pcm -> private_data;
  codec = (cs4231_t *)pcm1 -> private_data;
  /* initialize some mixer registers */
  snd_cs4236_ext_out( codec, CS4236_LEFT_MIX_CTRL, 0x80 | 0x18 );
  snd_cs4236_ext_out( codec, CS4236_RIGHT_MIX_CTRL, 0x80 );
  snd_cs4236_ext_out( codec, CS4236_RIGHT_LOOPBACK, 0xbf );
  /* change some current channels */
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.set_record_source )
      channel -> hw.set_record_source = snd_cs4236_record_source;
  }
  snd_cs4231_outm( codec, CS4231_LEFT_INPUT, 0x3f, 0xc0 );
  snd_cs4231_outm( codec, CS4231_RIGHT_INPUT, 0x3d, 0xc0 );
  /* create new channels */
  for ( idx = 0; idx < CS4236_MIXS; idx++ ) {
    switch ( snd_cs4236_mixs[ idx ].priority ) {
      case SND_MIXER_PRI_3D_SPACE:
        if ( codec -> hardware == CS4231_HW_CS4236 ||
             codec -> hardware == CS4231_HW_CS4236B ) goto __skip;
        break;
      case SND_MIXER_PRI_3D_CENTER:
        if ( codec -> hardware == CS4231_HW_CS4236 ||
             codec -> hardware == CS4231_HW_CS4236B ||
             codec -> hardware == CS4231_HW_CS4235 ||
             codec -> hardware == CS4231_HW_CS4239 ) goto __skip;
        break;
    }      
    channel = snd_mixer_new_channel( mixer, &snd_cs4236_mixs[ idx ] );
    if ( !channel ) {
      snd_mixer_free( mixer );
      return NULL;
    }
    if ( channel -> hw.priority == SND_MIXER_PRI_3D_CENTER &&
         codec -> hardware == CS4231_HW_CS4238B ) {
      channel -> hw.priority = SND_MIXER_PRI_3D_VOLUME;
      strcpy( channel -> hw.name, SND_MIXER_ID_3D_VOLUME );
      snd_mixer_reorder_channel( mixer, channel );
    }
    __skip:
  }
  /* update current channels */
  channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_LOOPBACK );
  channel -> hw.mmute = 1;
  channel -> hw.stereo = 1;
  channel -> hw.set_mute = snd_cs4236_loopback_mute;
  channel -> hw.set_volume_level = snd_cs4236_loopback_volume_level;
  channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_PCM );
  channel -> hw.digital = 1;
  channel -> hw.parent_priority = SND_MIXER_PRI_MASTERD;
  channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_MONO );
  channel -> hw.set_mute = snd_cs4236_mono_mute;
  snd_mixer_new_switch( mixer, &snd_cs4236_mic_gain );

  if ( codec -> hardware == CS4231_HW_CS4235 ||
       codec -> hardware == CS4231_HW_CS4239 ) {

    channel1 = snd_mixer_find_channel( mixer, SND_MIXER_PRI_LINE );
    channel1 -> hw.priority = SND_MIXER_PRI_MASTERD1;
    channel1 -> hw.parent_priority = SND_MIXER_PRI_MASTER;
    strcpy( channel1 -> hw.name, SND_MIXER_ID_MASTERD1 );
    snd_mixer_reorder_channel( mixer, channel1 );

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_FM );
    channel -> hw.parent_priority = SND_MIXER_PRI_MASTERD1;

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_SYNTHESIZER );
    channel -> hw.priority = SND_MIXER_PRI_MASTER;
    channel -> hw.parent_priority = SND_MIXER_PRI_PARENT;
    strcpy( channel -> hw.name, SND_MIXER_ID_MASTER );
    channel -> hw.stereo = 1;
    channel -> hw.digital = 0;
    channel -> hw.min = 0;
    channel -> hw.max = 31;
    channel -> hw.min_dB = -5600;
    channel -> hw.max_dB = 600;
    channel -> hw.step_dB = 200;
    channel -> hw.private_value = ((27<<24)|(29<<16)|0x80);
    channel -> hw.set_record_source = NULL;
    channel -> hw.set_mute = channel1 -> hw.set_mute;
    channel -> hw.set_volume_level = channel1 -> hw.set_volume_level;

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_DSP );
    channel -> hw.priority = SND_MIXER_PRI_MASTER1;
    channel -> hw.parent_priority = SND_MIXER_PRI_MASTER;
    strcpy( channel -> hw.name, SND_MIXER_ID_MASTER1 );
    channel -> hw.stereo = 0;
    channel -> hw.digital = 0;
    channel -> hw.min = 0;
    channel -> hw.max = 3;
    channel -> hw.min_dB = -2400;
    channel -> hw.max_dB = 0;
    channel -> hw.step_dB = 800;
    channel -> hw.private_value = 0;
    channel -> hw.set_record_source = NULL;
    channel -> hw.set_mute = NULL;
    channel -> hw.set_volume_level = snd_cs4235_master1_volume_level;    

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXA );
    channel -> hw.parent_priority = SND_MIXER_PRI_MASTER;

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXB );
    channel -> hw.parent_priority = SND_MIXER_PRI_MASTER;

    channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_MIC );
    channel -> hw.parent_priority = SND_MIXER_PRI_MASTER;
    channel -> hw.stereo = 0;
    channel -> hw.mmute = 1;

    /* enable DSP input to second digital master */
    snd_cs4236_ext_outm( codec, CS4236_LEFT_DSP, 0x7f, 0x00 );
    snd_cs4236_ext_outm( codec, CS4236_RIGHT_DSP, 0x7f, 0x00 );
    /* mute wave */
    snd_cs4236_ext_outm( codec, CS4236_LEFT_WAVE, 0x7f, 0x80 );
    snd_cs4236_ext_outm( codec, CS4236_RIGHT_WAVE, 0x7f, 0x80 );
  } else {
    snd_mixer_new_switch( mixer, &snd_cs4236_mono_output );
  }
#if 1
  /* change settings to recommended */
  channel1 = snd_mixer_find_channel( mixer, SND_MIXER_PRI_LINE );

  channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXA );
  if ( channel ) {
    channel -> hw.priority = SND_MIXER_PRI_LINE;
    strcpy( channel -> hw.name, SND_MIXER_ID_LINE );
    channel -> hw.ossdev = SND_MIXER_OSS_LINE;
    snd_mixer_reorder_channel( mixer, channel );
  }

  channel = snd_mixer_find_channel( mixer, SND_MIXER_PRI_AUXB );
  if ( channel ) {
    channel -> hw.priority = SND_MIXER_PRI_CD;
    strcpy( channel -> hw.name, SND_MIXER_ID_CD );
    channel -> hw.ossdev = SND_MIXER_OSS_CD;
    snd_mixer_reorder_channel( mixer, channel );
  }

  if ( channel1 ) {
    channel1 -> hw.priority = SND_MIXER_PRI_AUXA;
    strcpy( channel1 -> hw.name, SND_MIXER_ID_AUXA );
    channel1 -> hw.ossdev = SND_MIXER_OSS_LINE1;
    snd_mixer_reorder_channel( mixer, channel1 );
  }
#endif
  /* --- */
  mixer -> hw.caps &= ~SND_MIXER_INFO_CAP_EXCL_RECORD;
  /* --- */
  if ( codec -> hardware == CS4231_HW_CS4237B ||
       codec -> hardware == CS4231_HW_CS4238B ) {
    snd_mixer_new_switch( mixer, &snd_cs4236_iec958 );
  }
  return mixer;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_cs4236_export;
#endif 

int init_module( void )
{
#ifndef LINUX_2_1
  if ( register_symtab( &snd_symbol_table_cs4236_export ) < 0 ) 
    return -ENOMEM;
#endif
  return 0;
}

void cleanup_module( void )
{
}
