/*
   aumix.c:  adjust audio mixer
   copyright (c) 1993, 1996-1998 the authors--see AUTHORS file

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "aumix.h"

extern void RefreshAllSettings(void);
void ExitIfError(int error);
int InitializeMixer(char *device_file);
int MixerStatus(void);
#if 0				/* not yet used */
int ReadWriteMixer(int device, char *rw, int *left, int *right, char *rp);
#endif
int SetShowNoninter(int dev);
int LoadSettings(void);
int SaveSettings(void);
void Usage(void);
void i18n_initialize(void);
#ifdef HAVE_LIBNCURSES
void ReadInteractiveKeys(void);
#endif
#ifdef HAVE_ALSA
void Alsa_Anty_Mute(void);
#endif				/* HAVE_ALSA */
FILE *OpenDefaultFile(char *mode);

FILE *setfile;
char setfile_name[256];		/* name of file for saved settings */
char *device_filename = "/dev/mixer";	/* name of mixer device file */
unsigned short setfile_opened = FALSE, setfile_write_perm = FALSE, setfile_read_perm = FALSE;
int mixer_fd = -1, mutelevel[SOUND_MIXER_NRDEVICES], devmask = 0, recmask = 0,
 recsrc = 0, stereodevs = 0, mutestate = FALSE, interactive = FALSE;
char *dev_name[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_NAMES;
char *dev_label[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_LABELS;
/*
   An index into this array gives the device number for the
   corresponding command-line option.
 */
char *moptindx = "vbtswplmcxWrio123";
char *mopt;
extern char *optarg;
int usage_ok = 1;

int main(int argc, char *const argv[], char *optstring)
{

#if HAVE_LIBGPM
	Gpm_Connect conn;
#endif
	int optn, i;
	if (argc <= 1) {
#ifdef HAVE_LIBNCURSES
		interactive = TRUE;
#else
		interactive = FALSE;
#endif
	}
/* Internationalization */
#ifdef HAVE_NLS
	i18n_initialize();
#endif
#ifdef HAVE_LIBNCURSES
	ReadInteractiveKeys();
#endif

/* Get options from the command line.  Using numbers as options is deprecated,
   but we do it anyway because it makes sense for line1, line2 and line3.
 */
	while (TRUE) {
		optn = getopt(argc, argv, "hI:LqSd:v:b:t:s:w:P:p:l:m:c:x:W:R:r:i:o:1:2:3:");
		if (optn == -1)
			break;
		if ((mopt = strchr(moptindx, optn))) {
			usage_ok = 0;
			if (mixer_fd == -1)
				ExitIfError(InitializeMixer(device_filename));
			ExitIfError(SetShowNoninter(mopt - moptindx));
		} else {
			usage_ok = 0;
			switch (optn) {
			case 'q':
				optarg = "q";
				if (mixer_fd == -1)
					ExitIfError(InitializeMixer(device_filename));
				for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
					if ((1 << i) & devmask)
						ExitIfError(SetShowNoninter(i));
				}
				break;
			case 'd':	/* User specified the device file. */
				device_filename = malloc(strlen(optarg));
				sprintf(device_filename, optarg);
				ExitIfError(InitializeMixer(device_filename));
				break;
			case 'S':	/* Save to file. */
				if (mixer_fd == -1)
					ExitIfError(InitializeMixer(device_filename));
				ExitIfError(SaveSettings());
				break;
			case 'L':	/* Load from file. */
				if (mixer_fd == -1)
					ExitIfError(InitializeMixer(device_filename));
				ExitIfError(LoadSettings());
				break;
#ifdef HAVE_LIBNCURSES
			case 'I':
				interactive = TRUE;
				break;
#endif
			default:
				Usage();
			}
		}
	}
	if (!interactive) {
		if (usage_ok && optn == -1)
			Usage();
		exit(0);
	}
	if (mixer_fd == -1)
		ExitIfError(InitializeMixer(device_filename));
#if HAVE_LIBNCURSES
	initscr();
	leaveok(stdscr, TRUE);
	keypad(stdscr, TRUE);
	cbreak();
	noecho();
	timeout(0);
	InitColors();
	InitScreen();
#if HAVE_LIBGPM
	conn.eventMask = ~0;
	conn.defaultMask = GPM_MOVE | GPM_HARD;
	conn.maxMod = ~0;
	conn.minMod = 0;
	Gpm_Open(&conn, 0);
	gpm_handler = MouseHandler;
#endif
	Inter();
#if HAVE_LIBGPM
	Gpm_Close();
#endif
#endif
	close(mixer_fd);
#if HAVE_LIBNCURSES
	CloseScreen();
#endif
	if (setfile_opened)
		fclose(setfile);
	exit(0);
}

void ExitIfError(int error)
{
/* Print error messages and bail out. */
	char string[80];
	const char *errorlist[] =
	{LOCAL_TEXT("aumix:  no device found"), "aumix:  SOUND_MIXER_READ_DEVMASK", "aumix:  SOUND_MIXER_READ_RECMASK", "aumix:  SOUND_MIXER_READ_RECSRC", "aumix:  SOUND_MIXER_READ_STEREODEVS", "aumix:  SOUND_MIXER_WRITE_RECSRC", "aumix:  MIXER_READ", "aumix:  MIXER_WRITE", LOCAL_TEXT("aumix:  mixer not open"), LOCAL_TEXT("aumix:  unable to open settings file")};
	if (!error)
		return;
#if HAVE_LIBNCURSES
	CloseScreen();
#endif
	switch (error) {
	case 1:
		sprintf(string, LOCAL_TEXT("aumix:  error opening %s"), device_filename);
		perror(string);
		exit(1);
	}
	if (error < 12) {
		sprintf(string, LOCAL_TEXT("aumix:  unknown error %i"), error);
		perror(string);
		exit(-1);
	}
	if (error > 1) {
		perror(errorlist[error + 2, 0]);
		exit(1);
	}
	return;
}

int LoadSettings(void)
{
/* Read settings from file.
   We depend on the 'mixer' file descriptor having been set
   (and not clobbered) before entering this routine.
 */

	char tmpstring[80], recplay;
	int tmp, i, j = TRUE, left, right;
	setfile = OpenDefaultFile("r");
	if (setfile == NULL)
		return (11);
	for (j = TRUE; (!(j == EOF) && !(j == 0)); j = fscanf(setfile, "%[^:;,]%*[:;,]%3u%*[:;,]%3u%*[:;,]%1c\n", tmpstring, &left, &right, &recplay)) {
		for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
/* Cycle through names of channels, looking for matches. */
			if (!strcmp(tmpstring, dev_name[i])) {
				left = (left > 100) ? 100 : left;
				left = (left < 0) ? 0 : left;
				right = (right > 100) ? 100 : right;
				right = (right < 0) ? 0 : right;
				tmp = left + (right << 8);
				if ((ioctl(mixer_fd, MIXER_WRITE(i), &tmp) == -1) && !interactive) {
					printf("%s %s\n", dev_name[i], LOCAL_TEXT("not set"));
				} else {
					if (!interactive)
						printf("%s %s %i, %i", dev_name[i], LOCAL_TEXT("set to"), left, right);
					if ((1 << i) & recmask) {
						recsrc = (recplay == 'R') ? recsrc | (1 << i) : recsrc &
						    ~(1 << i);
						if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc) == -1)
							return (7);
						if (!interactive)
							printf(", %c", recplay);
					}
					if (!interactive)
						printf("\n");
				}
			}
		}
	}
	fclose(setfile);
#if HAVE_LIBNCURSES
	if (interactive)
		RefreshAllSettings();
#endif
}

int SaveSettings(void)
{
/* Write settings to file.
   We depend on the 'mixer' file descriptor having been set
   (and not clobbered) before entering this routine.
 */
	int tmp, i;
	setfile = OpenDefaultFile("w");
	if (setfile == NULL)
		return (11);
	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
		perror("SOUND_MIXER_READ_RECSRC");
		return (5);
	}
	for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
		if ((1 << i) & devmask) {
			if (ioctl(mixer_fd, MIXER_READ(i), &tmp) == -1)
				return (8);
			fprintf(setfile, "%s:%i:%i:%c\n", dev_name[i], (tmp & 0xFF), ((tmp >> 8) & 0xFF), ((1 << i) & recsrc ? 'R' : 'P'));
		}
	}
	fclose(setfile);
}

int InitializeMixer(char *device_filename)
{
/* Initialize the mixer.

   Global:

   mixer_fd   = mixer file descriptor reference number

   Return values:

   Success:   = 0
   Failure:   = 1     Unable to open specified device.
   Failure:   = 2     Specified device not found.
   Failure:   = 3     SOUND_MIXER_READ_DEVMASK
   Failure:   = 4     SOUND_MIXER_READ_RECMASK
   Failure:   = 5     SOUND_MIXER_READ_RECSRC
   Failure:   = 6     SOUND_MIXER_READ_STEREODEVS
 */
	int result;
#ifdef HAVE_ALSA
	Alsa_Anty_Mute();
#endif				/* HAVE_ALSA */
	if ((mixer_fd = open(device_filename, O_RDWR)) < 0)
		return (1);
	result = MixerStatus();
	if (!devmask)
		return (2);
	return (result);
}

int MixerStatus(void)
{
/* Get status of the mixer.

   Global:

   mixer_fd   = mixer file descriptor reference number

   Bit masks indicating:

   devmask    = valid devices
   recmask    = valid input devices
   recsrc     = devices currently selected for input
   stereodevs = stereo devices
 */
	if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &devmask))
		return (3);
	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECMASK, &recmask))
		return (4);
	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc))
		return (5);
	if (ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs))
		return (6);
	return (0);
}

#if 0				/* We aren't using this routine yet. */
int ReadWriteMixer(int device, char *rw, int *left, int *right, char *rp)
{
/* Read or write settings.

 * Global:
 *
 *   mixer_fd   = mixer file descriptor reference number
 *
 *   Bit masks indicating:
 *
 *   devmask    = valid devices
 *   recmask    = valid input devices
 *   recsrc     = devices currently selected for input
 *   stereodevs = stereo devices
 *
 * Input:
 *
 *   device     = index into the array of mixer devices
 *   rw         = r - read : w - write
 *   left       = left channel setting
 *   right      = right channel setting
 *   rp         = r - record : p - play
 *
 *
 * Output:
 *
 *   left       = left channel setting
 *   right      = right channel setting
 *   rp         = r - record : p - play
 *
 * Return:
 *
 *   success:   = 0
 *   failure:   = 7     SOUND_MIXER_WRITE_RECSRC
 *   failure:   = 8     MIXER_READ
 *   failure:   = 9     MIXER_WRITE
 *   failure:   = 10    mixer not open
 */
	int rightleft, result;
	if (mixer_fd == -1)	/* We haven't opened the mixer. */
		return (10);
	result = MixerStatus();
	if (result)
		return (result);
/* Is the given device index valid? */
	if (!((1 << device) & devmask))
		return (0);
	if (*rw == 'r') {	/* Read settings. */
		if (ioctl(mixer_fd, MIXER_READ(device), &rightleft))
			return (8);
/* Unpack left and right settings. */
		if ((1 << device) & stereodevs) {
			*right = ((rightleft >> 8) & 0xFF);
			*left = rightleft & 0xFF;
		} else {
			*right = ((rightleft >> 8) & 0xFF);
			*left = rightleft & 0xFF;
		}
/* Can the current device be a recording source? */
		if ((1 << device) & recmask) {
			*rp = (1 << device) & recsrc ? 'R' : 'P';
		} else {
			*rp = '\0';
		}
	} else {
/* "But it's got 11, so it *must* be louder." */
		*left = (*left > 100) ? 100 : *left;
		*left = (*left < 0) ? 0 : *left;
		*right = (*right > 100) ? 100 : *right;
		*right = (*right < 0) ? 0 : *right;
/* Pack left and right settings for writing. */
		rightleft = *left + (*right << 8);
/* Write settings. */
		if (ioctl(mixer_fd, MIXER_WRITE(device), &rightleft))
			return (9);
/* Is the current device capable of being a recording source? */
		if ((1 << device) & recmask) {
			*rp = *rp & 0xDF;
			if (*rp == 'R') {
				recsrc = recsrc | (1 << device);
			} else {
				recsrc = recsrc & ~(1 << device);
				*rp = 'P';
			}
/* Set recording or playing mode. */
			if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc))
				return (7);
		} else {
			*rp = '\0';
		}
	}
	return (0);
}
#endif				/* 0 */

FILE *OpenDefaultFile(char *mode)
{
/* Open the settings file for reading or writing.

   Try first ${HOME}/.AUMIXRC, then AUMIXRC_PATH/AUMIXRC;
   become an error generator if neither can be opened.

   Input:

   mode should be either 'r' (read) or 'w' (write)

   Return:

   Success:   pointer to the default settings file structure
   Failure:   NULL
 */
	FILE *setfile;
	char *home;
	char filename[80];
	home = getenv("HOME");
	sprintf(filename, "%s/.%s", home, AUMIXRC);
	setfile = fopen(filename, mode);
	if (setfile == NULL) {
		sprintf(filename, "%s/%s", AUMIXRC_PATH, AUMIXRC);
		setfile = fopen(filename, mode);
	}
	if (setfile == NULL) {
		close(mixer_fd);
		return (NULL);
	}
	return (setfile);
}

int SetShowNoninter(int dev)
{
/* Change or display settings from the command line. */
	char *devstr;
	int tmp;
	if ((*optarg == 'R') || (*optarg == 'P')) {
		if ((1 << dev) & recmask) {
			recsrc = (*optarg == 'R') ? recsrc | (1 << dev) : recsrc & ~(1 << dev);
			if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc) == -1)
				return (7);
		}
		return (0);
	}
	if ((*optarg == 'q') || (*optarg == ' ')) {
		devstr = dev_name[dev];
		if (ioctl(mixer_fd, MIXER_READ(dev), &tmp) == -1)
			return (8);
		printf("%s %i, %i", dev_name[dev], (tmp & 0xFF), ((tmp >> 8) & 0xFF));
		if ((1 << (dev)) & recmask) {
			if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1)
				return (5);
			printf(", %c", ((1 << dev) & recsrc ? 'R' : 'P'));
		}
		printf("\n");
		return (0);
	} else {
		tmp = atoi(optarg);
		tmp = (tmp > 100) ? 100 : tmp;
		tmp = (tmp < 0) ? 0 : 257 * tmp;
		if (ioctl(mixer_fd, MIXER_WRITE(dev), &tmp) == -1)
			return (9);
		return (0);
	}
}

void Usage(void)
{
	char version[] = "@(#) aumix.c " AUMIX_VERSION "\n";
	char copyright[] =
	"@(#) copyright (c) 1993, 1996-1998 the authors--see AUTHORS file\n";
#ifdef HAVE_LIBNCURSES
	fprintf(stdout, "%s", LOCAL_TEXT("aumix %s usage: aumix [[ -<channel option>[ ]<level>|P[lay]|R[ecord]|q[uery],\n[-d] [-h] [-I{d|c}] [-L] [-q] [-S]\n\n"),
		AUMIX_VERSION);
#else
	fprintf(stdout, "%s", LOCAL_TEXT("aumix %s usage: aumix [[ -<channel option>[ ]<level>|P[lay]|R[ecord]|q[uery],\n[-d] [-h] [-L] [-q] [-S]\n\n"),
		AUMIX_VERSION);
#endif
	fprintf(stdout, "%s", LOCAL_TEXT("\
channel options:\n\n\
  v:  main volume           x:  mix monitor\n\
  b:  bass                  W:  PCM 2\n\
  t:  treble                r:  record\n\
  s:  synthesizer           i:  input gain\n\
  w:  PCM                   o:  output gain\n\
  p:  PC speaker            1:  line 1\n\
  l:  line                  2:  line 2\n\
  m:  microphone            3:  line 3\n\
  c:  CD\n\n\
other options:\n\
  d:  adjust a device besides /dev/mixer\n\
  h:  this helpful message\n"));
#ifdef HAVE_LIBNCURSES
	fprintf(stdout, "%s", LOCAL_TEXT("\
  I:  start in interactive mode after doing non-interactive functions\n"));
#endif
	fprintf(stdout, "%s", LOCAL_TEXT("\
  L:  load settings\n\
  q:  query all channels and print their settings\n\
  S:  save settings to ~/.aumixrc\n"));
	exit(1);
}

/* Initialize I18N.  The initialization amounts to invoking
   setlocale(), bindtextdomain() and textdomain().
   Does nothing if NLS is disabled or missing.  */
void i18n_initialize(void)
{
	/* If HAVE_NLS is defined, assume the existence of the three
	   functions invoked here.  */
#ifdef HAVE_NLS
	/* Set the current locale.  */
	/* Here we use LC_MESSAGES instead of LC_ALL, for two reasons.
	   First, message catalogs are all of I18N aumix uses anyway.
	   Second, setting LC_ALL has a dangerous potential of messing
	   things up. */
	setlocale(LC_MESSAGES, "");
	/* Set the text message domain.  */
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif				/* HAVE_NLS */
}

#ifdef HAVE_ALSA
/* Unmuting "hardware mute" in ALSA driver.  Currently unmuting is done only
 * for the first sound system.  This code requires alsa-lib but can be done
 * without the library XXX-huh?.
 */
void Alsa_Anty_Mute(void)
{
	int channel;
	snd_mixer_channel_t cdata =
	{0};
	static snd_mixer_info_t mixer_info =
	{0};
	static int card_id = 0;
	static int mixer_id = 0;
	static void *mixer_handle;
	static struct snd_ctl_hw_info hw_info;
	void *ctl_handle;

	if (!snd_mixer_open(&mixer_handle, card_id, mixer_id)) {
		snd_mixer_info(mixer_handle, &mixer_info);
		channel = snd_mixer_channels(mixer_handle);
		while (channel >= 0) {
			snd_mixer_channel_read(mixer_handle, channel, &cdata);
/* Turn off hardware mute for all channels. */
			cdata.flags &= ~SND_MIXER_FLG_MUTE_LEFT;
			cdata.flags &= ~SND_MIXER_FLG_MUTE_RIGHT;
			snd_mixer_channel_write(mixer_handle, channel, &cdata);
			--channel;
		}
		snd_mixer_close(mixer_handle);
	}
}
#endif
