// textfile.c -- file class for dealing with text files

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#include "localdefs.h"
#ifdef __GNUG__
#pragma implementation
#endif
#include "textfile.h"

TextFile::TextFile(const char* filename, io_mode m, access_mode a) { 
	open(filename, m, a); 
}

TextFile::TextFile(const char* filename, const char* m) { 
	open(filename, m); 
}

//------------------------------------------------------------------


TextFile& TextFile::put(const char* s)
{ 
  return failif(!writable() || fputs(s, fp) == EOF);
}

TextFile& TextFile::get(char* s, int n, char terminator)
{
  if (!readable())
  {
    set(_fail);
    return *this;
  }

  char ch;
  stat = --n;

  if (n > 0 && (get(ch)))
  {
    if (ch == terminator) {
      unget(ch);
      stat= 0;	// This is not an error condition !
    }
    else
    {
      *s++ = ch; --n;
      while (n > 0 && (get(ch)))
      {
        if (ch == terminator)
        {
          unget(ch);
          break;
        }
        else
        {
          *s++ = ch; --n;
        }
      }
    }
  }

  *s = 0;
  return failif((stat != 0) && ((stat -= n) == 0));
}

TextFile& TextFile::getline(char* s, int n, char terminator)
{
  if (!readable())
  {
    set(_fail);
    return *this;
  }

  char ch;
  stat = --n;

  while (n > 0 && (get(ch)))
  {
    --n;
    if ((*s++ = ch) == terminator)
      break;
  }

  *s = 0;
  return failif((stat != 0) && ((stat -= n) == 0));
}

// from Doug Schmidt

// This should probably be a page size....
#define CHUNK_SIZE 512

/* Reads an arbitrarily long input line terminated by a user-specified
   TERMINATOR.  Super-nifty trick using recursion avoids unnecessary calls
   to NEW! */

char *TextFile::readline (int chunk_number, char terminator) 
{
  char buf[CHUNK_SIZE];
  register char *bufptr = buf;
  register char *ptr;
  char ch;
  int continu;

  while ((continu = !!get(ch)) && ch != terminator) /* fill the current buffer */
    {
      *bufptr++ = ch;
      if (bufptr - buf >= CHUNK_SIZE) /* prepend remainder to ptr buffer */
        {
          if (ptr = readline (chunk_number + 1, terminator))

            for (; bufptr != buf; *--ptr = *--bufptr);

          return ptr;
        }
    }
  if (!continu && bufptr == buf)
    return NULL;

  int size = (chunk_number * CHUNK_SIZE + bufptr - buf) + 1;

  if (ptr = new char[stat = size])
    {

      for (*(ptr += (size - 1)) = '\0'; bufptr != buf; *--ptr = *--bufptr)
        ;

      return ptr;
    } 
  else 
    return NULL;
}

/* Reads an arbitrarily long input line terminated by TERMINATOR.
   This routine allocates its own memory, so the user should
   only supply the address of a (char *). */

TextFile& TextFile::gets(char **s, char terminator)
{
  if (!readable())
  {
    set(_fail);
    return *this;
  }

  return failif(!(*s = readline (0, terminator)));
}
  
#ifndef VMS
TextFile& TextFile::scan(const char* fmt ...)
{
  if (readable())
  {
    va_list args;
    va_start(args, fmt);
#ifndef HAVE_VSCANF
    stat = _doscan(fp, fmt, args);
#else
    stat = vfscanf(fp, fmt, args);
#endif
    va_end(args);
    failif(stat <= 0);
  }
  return *this;
}
#endif

TextFile& TextFile::form(const char* fmt ...)
{
  va_list args;
  va_start(args, fmt);
#ifndef HAVE_VPRINTF
  stat = _doprnt(fmt, args, fp);
#ifdef HAVE_VOID_DOPRNT
  stat = ferror(fp) ? -1 : 0;
#endif
#else
  stat = vfprintf(fp, fmt, args);
#endif
  va_end(args);
  failif(stat < 0);
  return *this;
}

