#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <malloc.h>
#include <sys/stat.h>
#include "icfg.h"

#define yyerror gus_icfg_error

extern FILE *gus_icfg_in;
int gus_icfg_parse( void );

struct gus_icfg_config_stru *gus_icfg_config = NULL;
static struct gus_icfg_config_stru *gus_icfg_configs = NULL;

static int gus_icfg_parse_go( int action, char *source, char *options );
static void gus_icfg_unload1( int mask );

/* ---- */

#ifdef DEBUG

void __gus_icfg_dprintf( char *fmt, ... )
{
  va_list va;
  
  fprintf( stderr, "icfg error >>> " );
  va_start( va, fmt );
  vfprintf( stderr, fmt, va );
  va_end( va );
  fprintf( stderr, "\n" );
  fflush( stderr );
}

#endif

void gus_icfg_error( int warning, char *format, ... )
{
  va_list va;
  
  if ( !warning && gus_icfg_config -> exit_value <= 0 )
    gus_icfg_config -> exit_value = -1;
  va_start( va, format );
  gus_icfg_config -> error( gus_icfg_config -> privatedata, gus_icfg_config -> device, warning, gus_icfg_config -> filename, gus_icfg_in != NULL ? gus_icfg_config -> line_count + 1 : -1, format, va );
  va_end( va );
}
              
/* ---- */

void gus_icfg_free_gf1path( struct gus_icfg_gf1path *pgf1path )
{
  struct gus_icfg_gf1path *ngf1path;

  while ( pgf1path ) {
    ngf1path = pgf1path -> next;
    free( pgf1path -> path );
    free( pgf1path );
    pgf1path = ngf1path;
  }
}

void gus_icfg_free_iwfile( struct gus_icfg_iwfile *piwfile )
{
  struct gus_icfg_iwfile *niwfile = NULL;

  while ( piwfile ) {
    niwfile = piwfile -> next;
    if ( piwfile -> flags & GUS_ICFG_IWF_ROM ) {
      if ( piwfile -> data.rom.name )
        free( piwfile -> data.rom.name );
    } else {
      if ( piwfile -> data.file.name )
        free( piwfile -> data.file.name );
      if ( piwfile -> data.file.data )
        free( piwfile -> data.file.data );
    }   
    free( piwfile );   
    piwfile = niwfile;
  }
}

void gus_icfg_free_instrument( struct gus_icfg_instrument *pinstrument )
{
  struct gus_icfg_instrument *ninstrument = NULL;

  while ( pinstrument ) {
    ninstrument = pinstrument -> next;
    if ( pinstrument -> name )
      free( pinstrument -> name );
    free( pinstrument );
    pinstrument = ninstrument;
  }
}

void gus_icfg_free_group_range( struct gus_icfg_group_range *prange )
{
  struct gus_icfg_group_range *nrange;

  while ( prange ) {
    nrange = prange -> next;
    free( prange );
    prange = nrange;
  }
}

void gus_icfg_free_group( struct gus_icfg_group *pgroup )
{
  struct gus_icfg_group *ngroup = NULL;

  while ( pgroup ) {
    ngroup = pgroup -> next;
    gus_icfg_free_group_range( pgroup -> first );
    free( pgroup );
    pgroup = ngroup;
  }  
}

void gus_icfg_free_preload_download( struct gus_icfg_preload_download *pdownload )
{
  struct gus_icfg_preload_download *ndownload;
  
  while ( pdownload ) {
    ndownload = pdownload -> next;
    free( pdownload );
    pdownload = ndownload;
  }
}

void gus_icfg_free_preload_aliases( struct gus_icfg_preload_alias *paliases )
{
  struct gus_icfg_preload_alias *naliases;
  
  while ( paliases ) {
    naliases = paliases -> next;
    free( paliases );
    paliases = naliases;
  }
}

void gus_icfg_free_preload_format( struct gus_icfg_preload_format *pformat )
{
  struct gus_icfg_preload_format *nformat;

  while ( pformat ) {
    nformat = pformat -> next;  
    gus_icfg_free_preload_download( pformat -> downloads );
    gus_icfg_free_preload_aliases( pformat -> aliases );
    free( pformat );
    pformat = nformat;
  }
}

void gus_icfg_free_preload( struct gus_icfg_preload *ppreload )
{
  struct gus_icfg_preload *npreload;
  
  while ( ppreload ) {
    npreload = ppreload -> next;
    gus_icfg_free_preload_format( ppreload -> formats );
    free( ppreload );
    ppreload = npreload;
  }
}

void gus_icfg_free_info( struct gus_icfg_info *pinfo )
{
  struct gus_icfg_info *ninfo = NULL;

  while ( pinfo ) {
    ninfo = pinfo -> next;
    if ( pinfo -> name )
      free( pinfo -> name );
    if ( pinfo -> description )
      free( pinfo -> description );
    if ( pinfo -> options ) {
      int idx;
      for ( idx = 0; pinfo -> options[ idx ]; idx++ )
        free( pinfo -> options[ idx ] );
      free( pinfo -> options );
    }
    free( pinfo );
    pinfo = ninfo;
  }
}

/* ---- */

char *gus_icfg_look_for_patch_file( char *filename )
{
  char result[ 256 + 1 ];
  struct stat st;
  struct gus_icfg_gf1path *path;
  int size;

  size = strlen( filename );        
  for ( path = gus_icfg_config -> gf1paths; path; path = path -> next )
    {
      if ( size + strlen( path -> path ) + 1 > sizeof( result ) )
        continue;
      strcpy( result, path -> path );
      strcat( result, filename );
#if 0
      printf( "look_for_file '%s'..\n", result );
#endif
      if ( !stat( result, &st ) )
        return strdup( result );
      if ( size + strlen( path -> path ) + 2 > sizeof( result ) )
        continue;
      strcpy( result, path -> path );
      strcpy( result, "/" );
      strcat( result, filename );
      if ( !stat( result, &st ) ) {
        return strdup( result );
      }
    }
  return NULL;
}

static struct gus_icfg_instrument *look_for_instrument( unsigned int instrument )
{
  struct gus_icfg_instrument *instr;

  for ( instr = gus_icfg_config -> instruments; instr; instr = instr -> next )
    if ( instr -> number == instrument ) break;
  return instr;
}

struct gus_icfg_iwfile *get_ffff_file( int file )
{
  struct gus_icfg_iwfile *ifile;

  for ( ifile = gus_icfg_config -> iwfiles; ifile; ifile = ifile -> next )
    if ( ifile -> number == file ) return ifile;
  return NULL;
}

int download_ffff_blocks( void )
{
  struct gus_icfg_iwfile *iwf;

  for ( iwf = gus_icfg_config -> iwfiles; iwf; iwf = iwf -> next )
    if ( iwf -> flags & GUS_ICFG_IWF_ROM )
      {
        gus_rom_interwave_header_t header;
      
        if ( gus_instr_ffff_get_rom_header( gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, iwf -> data.rom.bank, &header ) < 0 )
          {
            gus_dprintf( "can't get header of internal ROM '%s'.. aborting..", iwf -> data.rom.name );
            return -1;
          }
        if ( strncmp( header.series_name, iwf -> data.rom.name, 16 ) )
          {
            gus_dprintf( "wrong header name = '%s' - current = '%s'.. aborting..", iwf -> data.rom.name, header.series_name );
            return -1;
          }
        if ( ( iwf -> handle = gus_instr_ffff_open_rom( gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, iwf -> data.rom.bank, iwf -> data.rom.file ) ) < 0 )
          {
            gus_dprintf( "can't open internal ROM '%s'.. aborting..", iwf -> data.rom.name );
            return -1;
          }
      }
     else
      {
        if ( ( iwf -> handle = gus_instr_ffff_open( iwf -> data.file.name, iwf -> data.file.data ) ) < 0 )
          {
            gus_dprintf( "can't open/read file '%s' - '%s'.. aborting..", iwf -> data.file.name, iwf -> data.file.data );
            return -1;
          }
        if ( (iwf -> flags & GUS_ICFG_IWF_DOWNLOAD) && gus_instr_ffff_download( iwf -> handle, gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, 0 ) < 0 )
          {
            gus_dprintf( "can't download whole file '%s' to onboard RAM.. aborting..", iwf -> data.file.data );
            return -1;
          }
      }
  return 0;
}

int download_ffff_blocks_free( void )
{
  struct gus_icfg_iwfile *iwf;

  for ( iwf = gus_icfg_config -> iwfiles; iwf; iwf = iwf -> next )
    {
      if ( !(iwf -> flags & GUS_ICFG_IWF_ROM) )
        {
          if ( iwf -> flags & GUS_ICFG_IWF_DOWNLOAD )
            gus_instr_ffff_download_free( iwf -> handle, gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, 0 );
           else
          if ( iwf -> flags & GUS_ICFG_IWF_TEMP )
            gus_instr_ffff_download_free( iwf -> handle, gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, 1 );
        }
      gus_instr_ffff_close( iwf -> handle );
    }
  return 0;
}

/* --- */

int gus_icfg_open( void **handle, struct gus_icfg_open *icfg )
{
  struct gus_icfg_config_stru *config, *config1;
  
  *handle = NULL;
  if ( icfg -> version != 11 )
    return -1;
  gus_icfg_in = NULL;
  if ( ( config = calloc( 1, sizeof( struct gus_icfg_config_stru ) ) ) == NULL )
    return -1;
  config -> filename = icfg -> filename ? strdup( icfg -> filename ) : NULL;
  config -> gus_fd = icfg -> gus_fd;
  config -> gus_access = icfg -> gus_access;
  config -> device = icfg -> device;
  config -> midi_emul = icfg -> emulation;
  config -> error = icfg -> error;
  config -> memory_alloc = icfg -> memory_alloc;
  config -> memory_test = icfg -> memory_test;
  config -> gf1paths = NULL;
  config -> iwfiles = NULL;
  config -> instruments = NULL;
  config -> groups = NULL;
  config -> preloads = NULL;
  if ( config -> gus_fd < 0 || config -> gus_access < 0 ) {
    config -> gus_fd = config -> gus_access = -1;
    if ( icfg -> chip == GUS_ICFG_CHIP_UART ) {
      config -> uart_flag = 1;
    }
    if ( icfg -> chip == GUS_ICFG_CHIP_INTERWAVE ) {
      config -> pnp_flag = 1;
    }
  } else {
    if ( config -> gus_access == GUS_ACCESS_SYNTH ) {
      if ( ioctl( config -> gus_fd, GUS_IOCTL_INFO, &config -> info_data ) < 0 ) {
        free( config );
        return -1;
      }
    }
    if ( config -> gus_access == GUS_ACCESS_INSMAN ) {
      if ( ioctl( config -> gus_fd, GUS_IOCTL_DMN_INFO, &config -> info_data ) < 0 ) {
        free( config );
        return -1;
      }
    }
    if ( (config -> gus_access & GUS_ACCESS_MIDI) && (config -> device & 0x0f) == GUS_MIDID_SYNTH ) {
      struct GUS_STRU_MIDI_CARD_INFO info;
      info.device = config -> device & 0x0f;
      info.info = &config -> info_data;
      if ( ioctl( config -> gus_fd, GUS_MIDI_CARD_INFO, &info ) < 0 ) {
        free( config );
        return -1;
      }
    }
    if ( config -> info_data.flags & GUS_STRU_INFO_F_ENHANCED )
      config -> pnp_flag = 1;
  }
  config1 = gus_icfg_configs;
  if ( !config1 ) {
    gus_icfg_configs = config;
  } else { 
    while ( config1 -> next ) config1 = config1 -> next;
    config1 -> next = config;
  }
  *handle = (void *)config;
  return 0;
}

int gus_icfg_close( void *handle )
{
  struct gus_icfg_config_stru *config, *config1;
  
  config = NULL;
  config1 = (struct gus_icfg_config_stru *)handle;
  if ( !config1 ) return -1;
  if ( config )
    config -> next = config1 -> next;
   else
    gus_icfg_configs = config1 -> next;
  config = gus_icfg_config;
  if ( config == config1 ) config = NULL;
  gus_icfg_unload( config1 );
  free( config1 -> filename );
  free( config1 );
  gus_icfg_config = config;
  return 0;
}

void *gus_icfg_handle( int device )
{
  struct gus_icfg_config_stru *result;

  for ( result = gus_icfg_configs; result && result -> device != device; result = result -> next );
  return (void *)result;
}

void *gus_icfg_ptrget( void *handle, int ptrtype )
{
  struct gus_icfg_config_stru *config;

  config = handle;
  if ( !config ) return NULL;
  switch ( ptrtype ) {
    case GUS_ICFG_PTR_GF1PATHS:
      return config -> gf1paths;
    case GUS_ICFG_PTR_IWFILES:
      return config -> iwfiles;
    case GUS_ICFG_PTR_INSTRUMENTS:
      return config -> instruments;
    case GUS_ICFG_PTR_GROUPS:
      return config -> groups;
  }
  return NULL;
}

static int gus_icfg_parse_go( int action, char *source, char *options )
{
  int res, mask;

  if ( action == GUS_ICFG_ACTION_LOAD ) {
    mask = 1;
    if ( gus_icfg_config -> instruments )
      gus_icfg_unload1( mask );			/* already loaded */
    gus_icfg_config -> load_source = source ? strdup( source ) : GUS_ICFG_SOURCE_AUTO;
    gus_icfg_config -> load_options = options ? strdup( options ) : NULL;
  } else {
    mask = 2;
    if ( gus_icfg_config -> preloads )
      gus_icfg_unload1( mask );
    gus_icfg_config -> preload_source = source ? strdup( source ) : GUS_ICFG_SOURCE_AUTO;
    gus_icfg_config -> preload_options = options ? strdup( options ) : NULL;
  }
  gus_icfg_in = NULL;
  if ( ( gus_icfg_in = fopen( gus_icfg_config -> filename, "r" ) ) == NULL )
    return -1;
  gus_icfg_parser_init( action );
  res = gus_icfg_parse();
  if ( !res && gus_icfg_config -> exit_value != 0 )
    res = -1;
  gus_icfg_parser_done();
  fclose( gus_icfg_in );
  gus_icfg_in = NULL;  
  if ( res ) {
    gus_icfg_unload1( mask );	/* for sure */
  }
  return res ? -1 : 0;
}

int gus_icfg_load( void *handle, char *source, char *options )
{
  int res;

  if ( !source ) { source = "auto"; options = NULL; }
  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  res = gus_icfg_parse_go( GUS_ICFG_ACTION_LOAD, source, options );
  if ( !res )
    {
      if ( ( res = download_ffff_blocks() ) < 0 )
        download_ffff_blocks_free();
    }
  return res;
}

static void gus_icfg_unload1( int mask )
{
  struct gus_icfg_gf1path *pgf1path = NULL;
  struct gus_icfg_iwfile *piwfile = NULL;
  struct gus_icfg_instrument *pinstrument = NULL;
  struct gus_icfg_group *pgroup = NULL;
  struct gus_icfg_preload *ppreload = NULL;
  struct gus_icfg_info *pinfo = NULL;
  
  if ( mask & 1 ) {
    if ( gus_icfg_config -> load_source ) {
      free( gus_icfg_config -> load_source );
      gus_icfg_config -> load_source = NULL;
    }
    if ( gus_icfg_config -> load_options ) {
      free( gus_icfg_config -> load_options );
      gus_icfg_config -> load_options = NULL;
    }
    download_ffff_blocks_free();
    pgf1path = gus_icfg_config -> gf1paths;
    gus_icfg_config -> gf1paths = NULL;
    piwfile = gus_icfg_config -> iwfiles;
    gus_icfg_config -> iwfiles = NULL;  
    pinstrument = gus_icfg_config -> instruments;
    gus_icfg_config -> instruments = NULL;
    pgroup = gus_icfg_config -> groups;
    gus_icfg_config -> groups = NULL;
  }
  if ( mask & 2 ) {
    if ( gus_icfg_config -> preload_source ) {
      free( gus_icfg_config -> preload_source );
      gus_icfg_config -> preload_source = NULL;
    }
    if ( gus_icfg_config -> preload_options ) {
      free( gus_icfg_config -> preload_options );
      gus_icfg_config -> preload_options = NULL;
    }
    ppreload = gus_icfg_config -> preloads;
    gus_icfg_config -> preloads = NULL;
  }
  if ( mask & 4 ) {
    pinfo = gus_icfg_config -> infos;
    gus_icfg_config -> infos = NULL;
  }
  if ( mask & 1 ) {
    gus_icfg_free_gf1path( pgf1path );
    gus_icfg_free_iwfile( piwfile );
    gus_icfg_free_instrument( pinstrument );
    gus_icfg_free_group( pgroup );
  }
  if ( mask & 2 ) {
    gus_icfg_free_preload( ppreload );
  }
  if ( mask & 4 ) {
    gus_icfg_free_info( pinfo );
  }
}

void gus_icfg_unload( void *handle )
{
  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return;
  gus_icfg_unload1( 0xffffffff );
}

/* --- */

static int write_not_found( unsigned int instrument )
{
  gus_instrument_t *ginstr;
  struct gus_icfg_instrument *instr;

  ginstr = gus_instr_alloc();
  if ( !ginstr ) return -1;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
  ginstr -> number.instrument = instrument;
  instr = look_for_instrument( instrument );
  if ( instr )
    ginstr -> name = strdup( instr -> name );
  gus_icfg_config -> memory_alloc( gus_icfg_config -> privatedata, gus_icfg_config -> device, ginstr );
  gus_instr_free( ginstr );
  return 0;
}

static int write_instrument( unsigned int instrument, int s8bit )
{
  gus_instrument_t *ginstr;
  struct gus_icfg_instrument *instr;
  struct gus_icfg_iwfile *iwf;
  int alias;

  __again:
#if 0
  fprintf( stderr, "instrument = %i, 8bit = %i\n", instrument, s8bit ); fflush( stdout );
#endif
  gus_icfg_config -> global_write_status = 0;		/* not found */
  if ( gus_icfg_config -> memory_test( gus_icfg_config -> privatedata, gus_icfg_config -> device, instrument ) )
    {
      gus_icfg_config -> global_write_status++;
      return 0;
    }
  alias = 0;
  ginstr = gus_instr_alloc();
  if ( !ginstr ) return 0;
  instr = look_for_instrument( instrument );
  if ( !instr || ( instr && (instr -> flags & GUS_ICFG_IF_DISABLE) ) )
    {
      gus_instr_free( ginstr );
      return 0;
    } 
  ginstr -> number.instrument = instrument;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
  ginstr -> name = instr -> name ? strdup( instr -> name ) : NULL;
  if ( instr -> type == GUS_ICFG_IT_ALIAS )
    {
      ginstr -> flags = GUS_INSTR_F_ALIAS;
      ginstr -> info.alias = alias = ( instr -> data.alias.bank << 16 ) | instr -> data.alias.prog;
      alias |= 0x80000000;
      gus_icfg_config -> global_write_status++;
    }
   else
  switch ( instr -> type ) {
    case GUS_ICFG_IT_GF1_PATCH:
      {
        char *filename;
        
        if ( ( filename = gus_icfg_look_for_patch_file( instr -> data.patch.filename ) ) == NULL )
          gus_dprintf( "patch file '%s' not found", instr -> data.patch.filename );
         else
        if ( gus_instr_patch_load( ginstr, filename, s8bit ) >= 0 )
          {
            ginstr -> flags = GUS_INSTR_F_NORMAL;
            ginstr -> exclusion = instr -> data.patch.exclusion;
            ginstr -> exclusion_group = instr -> data.patch.exclusion_group;
            gus_icfg_config -> global_write_status++;
          }
        if ( filename )
          free( filename );
      }
      break;
    case GUS_ICFG_IT_IW_FILE:
      iwf = get_ffff_file( instr -> data.iwfile.file );
      if ( iwf && gus_instr_ffff_load( ginstr, iwf -> handle, instr -> data.iwfile.bank, gus_icfg_config -> global_whole_flag | instr -> data.iwfile.prog ) >= 0 )
        {
          ginstr -> flags = GUS_INSTR_F_NORMAL;
          gus_icfg_config -> global_write_status++;
        }
      break;
    default:
      if ( instrument & 0xffff0000 ) {
        instrument &= 0xffff;
        gus_instr_free( ginstr );
        goto __again;
      }
      ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
      break;
  }
  if ( gus_icfg_config -> memory_alloc( gus_icfg_config -> privatedata, gus_icfg_config -> device, ginstr ) < 0 ) {
    gus_instr_free( ginstr );
    return -1;
  }
  gus_instr_free( ginstr );
  if ( alias )
    {
      instrument = (unsigned short)alias;
      goto __again;
    }
  return 0;
}

static int write_alias( unsigned int instrument, unsigned int to_instrument, int type, int s8bit )
{
  gus_instrument_t *ginstr;
  struct gus_icfg_instrument *instr;
  
  ginstr = gus_instr_alloc();
  ginstr -> flags |= GUS_INSTR_F_ALIAS;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> number.instrument = instrument;
  ginstr -> info.alias = to_instrument;
  instr = look_for_instrument( instrument );
  if ( instr )
    ginstr -> name = instr -> name ? strdup( instr -> name ) : NULL;
  gus_icfg_config -> memory_alloc( gus_icfg_config -> privatedata, gus_icfg_config -> device, ginstr );
  gus_instr_free( ginstr );
  return 0;
}

static int write_bank( struct gus_icfg_preload_format *formats )
{
  struct gus_icfg_preload_download *download;
  struct gus_icfg_preload_alias *aliases;
  struct gus_icfg_iwfile *iwf;
  int i, handle = -1;

  while ( formats )
    {
      gus_icfg_config -> global_whole_flag = 0;
      if ( formats -> type == GUS_ICFG_IT_IW_FILE )
        {
          iwf = get_ffff_file( formats -> data.iwfile.file );
          if ( iwf ) handle = iwf -> handle;
          if ( !(iwf -> flags & GUS_ICFG_IWF_DOWNLOAD) && (formats -> flags & GUS_ICFG_PF_FFFF_WHOLE) )
            {
              if ( gus_instr_ffff_download( handle, gus_icfg_config -> gus_fd, gus_icfg_config -> gus_access, 1 ) < 0 )
                gus_dprintf( "download error of file '%s'\n", (iwf ? (char *)iwf -> data.file.data : "???") );
               else 
                {
                  if ( iwf ) iwf -> flags |= GUS_ICFG_IWF_TEMP;
                  gus_icfg_config -> global_whole_flag = 0x100;
                }
            }
        }
      for ( download = formats -> downloads; download; download = download -> next ) {
#if 0
        printf( "download: bank = %i, min = %i, max = %i\n", download -> bank, download -> min, download -> max );
#endif
        for ( i = download -> min; i <= download -> max; i++ )
          write_instrument( ( download -> bank << 16 ) | i, formats -> flags & GUS_ICFG_PF_8BIT );
      }
      for ( aliases = formats -> aliases; aliases; aliases = aliases -> next ) {
#if 0
        printf( "aliases: bank = %i, min = %i, max = %i, dst_bank = %i, dst_prog = %i\n", aliases -> bank, aliases -> min, aliases -> max, aliases -> dest_bank, aliases -> dest_prog );
#endif
        for ( i = aliases -> min; i <= aliases -> max; i++ )
          write_alias( ( aliases -> bank << 16 ) | i, ( aliases -> dest_bank << 16 ) | aliases -> dest_prog, formats -> type, formats -> flags & GUS_ICFG_PF_8BIT );
      }
      formats = formats -> next;
      gus_icfg_config -> global_whole_flag = 0;
    }
  return 0;
}

/* --- */

int gus_icfg_preload( void *handle, char *source, char *options )
{
  struct gus_icfg_preload *pre;
  int res;

  if ( !source ) { source = "all"; options = NULL; }
  if ( !strcmp( source, "ALL" ) ) source = "all";
  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  if ( gus_icfg_parse_go( GUS_ICFG_ACTION_PRELOAD, source, options ) < 0 )
    return -1;
  for ( pre = gus_icfg_config -> preloads; pre; pre = pre -> next )
    {
      if ( strcmp( source, pre -> name ) ) continue;
      if ( download_ffff_blocks() < 0 ) {
        download_ffff_blocks_free();
        return -1;
      }
      res = write_bank( pre -> formats );
      gus_icfg_unload1( 2 );
      return res;
    }
  gus_dprintf( "preload - bank '%s' not found\n", source );
  gus_icfg_unload1( 2 );
  return -1;
}

int gus_icfg_download( void *handle, unsigned int instrument )
{
  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  __again:
  gus_icfg_config -> global_whole_flag = 0;
  gus_icfg_config -> global_write_status = 0;
  if ( write_instrument( instrument, 0 ) )
    return -1;
  if ( !gus_icfg_config -> global_write_status )
    {
      if ( ( instrument >> 16 ) > 0 )
        {
          instrument &= 0xffff;
          goto __again;
        }
      return write_not_found( instrument );
    }
  return 0;
}

int gus_icfg_emulation_get( void *handle )
{
  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  return gus_icfg_config -> midi_emul;
}

int gus_icfg_emulation_set( void *handle, int emulation )
{
  int result;

  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  if ( gus_icfg_config -> midi_emul == emulation )
    return 0;
  gus_icfg_unload( handle );
  gus_icfg_config -> midi_emul = emulation;
  if ( ( result = gus_icfg_load( handle, gus_icfg_config -> load_source, gus_icfg_config -> load_options ) ) < 0 )
    return result;
  if ( gus_icfg_config -> preload_source != NULL ) {
    if ( ( result = gus_icfg_preload( handle, gus_icfg_config -> preload_source, gus_icfg_config -> preload_options ) ) < 0 )
      return result;
  }
  return 0;
}

int gus_icfg_info( void *handle, struct gus_icfg_info **info )
{
  int res;

  gus_icfg_config = (struct gus_icfg_config_stru *)handle;
  if ( !gus_icfg_config ) return -1;
  *info = NULL;
  gus_icfg_in = NULL;
  if ( ( gus_icfg_in = fopen( gus_icfg_config -> filename, "r" ) ) == NULL )
    return -1;
  if ( gus_icfg_config -> infos )
    gus_icfg_unload1( 4 );
  gus_icfg_parser_init( GUS_ICFG_ACTION_INFO );
  res = gus_icfg_parse();
  if ( !res ) {
    *info = gus_icfg_config -> infos;
    gus_icfg_config -> infos = NULL;
  }
  if ( !res && gus_icfg_config -> exit_value != 0 )
    res = -1;
  gus_icfg_parser_done();
  fclose( gus_icfg_in );
  gus_icfg_in = NULL;  
  if ( res ) {
    gus_icfg_unload1( 4 );
  }
  return res ? -1 : 0;
}
