/******************************************************************************
*   WMSound Setup       for WindowMaker Sound Server            [Version 0.7] *
*   Copyright (C) 1998  Pascal Hofstee <daeron@shadowmere.student.utwente.nl> *
*                                                                             *
*   WindowMaker Sound Server                                                  *
*   Copyright (C) 1998  Anthony Quinn <southgat@frontiernet.net>              *
*                       William Moore <billy@mud.mind.net>                    *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*******************************************************************************/

#include <WINGs.h>
#include <WUtil.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <proplist.h>

#include "wsconfig.h"
#include "wmss.h"
#include "soundpath.h"
#include "info.h"
#include "config.h"

/*
 * You need to define this function to link any program to WINGs.
 * This will be called when the application will be terminated because
 * on a fatal error.
 */
void
wAbort()
{
    exit(1);
}

static char *options[] = {
        "AppExit",
        "AppStart",
        "Hide",
        "Iconify",
        "Maximize",
        "UnMaximize",
        "Shade",
        "Startup",
        "UnHide",
        "DeIconify",
        "UnShade",
        "UserDefined",
        NULL
};

Display		*dpy;
proplist_t	prop, style, val, key, text;

void
closeAction(WMWidget *self, void *data)
{
    exit(0);
}

char* defaultsPathForDomain(char *domain)
{
    char path[1024];
    char *gspath, *tmp;

    gspath = getenv("GNUSTEP_USER_ROOT");
    if (gspath) {
        strcpy(path, gspath);
        strcat(path, "/");
    } else {
        char *home;

        home = getenv("HOME");
        if (!home) {
            wwarning("could not get HOME environment variable!\n");
            exit(0);
        }
        strcpy(path, home);
        strcat(path, "/GNUstep/");
    }
    strcat(path, DEFAULTS_DIR);
    strcat(path, "/");
    strcat(path, domain);

    tmp = malloc(strlen(path)+2);
    strcpy(tmp, path);
    return tmp;
}


static WMSSSoundPathPanel*
wmssSoundPath(WMButton *bPtr, WMSoundSetupPanel *parentpanel)
{
    WMSSSoundPathPanel	*panel;
    WMListItem	*pathitem;
    proplist_t	path_key, path_val, storage;
    char	*tmp_val;
    char	**soundpath;
    int		elem, nr_elem, result;


    WMSetButtonEnabled(parentpanel->soundpathButton, NO);

    path_key = PLMakeString("SoundPath");
    path_val = PLGetDictionaryEntry(prop,path_key);

    if (path_val) {
      nr_elem = PLGetNumberOfElements(path_val);
      soundpath = malloc(sizeof(char *)*(nr_elem+1));
      if(!soundpath) {
        wwarning("Out of memory when reading SoundPath\n");
      }

      for(elem=0; elem < nr_elem; elem++)
      {
        tmp_val = PLGetString(PLGetArrayElement(path_val, elem));
        if(!tmp_val) {
          wwarning("Out of memory when processing SoundPath\n");
        }
        soundpath[elem] = tmp_val;
      }
      soundpath[elem]=NULL;
    }
    else {
      soundpath = NULL;
    }
    panel = wmssCreateSoundPathPanel(WMWidgetScreen(parentpanel->win), soundpath);

    result = wmssRunSoundPathPanel(panel);
    if (result==1) {
      PLRelease (path_val);
      path_val = PLMakeArrayFromElements(NULL);
      for (elem = 0; (pathitem=WMGetListItem(panel->soundpathList, elem)); elem++) {
        storage = PLMakeString(pathitem->text);
        if (!storage) {
          wwarning("Out of memory when processing SoundPath Changes\n"); 
        }
        path_val = PLAppendArrayElement(path_val, storage);
      }
      PLRemoveDictionaryEntry(style, path_key);
      PLInsertDictionaryEntry(style, path_key, path_val);
      PLMergeDictionaries(prop, style);
      WMSetLabelText(parentpanel->statusLabel, "Activated New SoundPath Settings");
    }    
    else {
      WMSetLabelText(parentpanel->statusLabel, "SoundPath Changes Ignored");
    }
    wmssDestroySoundPathPanel(panel);


    PLRelease (path_key);
    free(soundpath);
    WMSetButtonEnabled(parentpanel->soundpathButton, YES);

    return panel;
}


static void
wmssInfo(WMButton *bPtr, WMSoundSetupPanel *parentpanel)
{
    WMSSInfoPanel *panel;

    WMSetButtonEnabled(parentpanel->infoButton, NO);
    panel = wmssCreateInfoPanel(WMWidgetScreen(parentpanel->win));
    wmssRunInfoPanel(panel);
    wmssDestroyInfoPanel(panel);
    WMSetButtonEnabled(parentpanel->infoButton, YES);
}

static void
wmssBrowseSound(WMButton *bPtr, WMSoundSetupPanel *parentpanel)
{
    char        path[1024];
    WMOpenPanel *panel;
    char        *spath;

    WMSetButtonEnabled(parentpanel->browseButton, NO);

    spath = getenv("HOME");
    strcpy(path, spath);
    strcat(path, "/GNUstep/Library/WindowMaker/Sounds");
    spath = malloc(strlen(path)+2);
    strcpy(spath, path);

    wwarning("Path: %s", spath);

    panel = WMGetOpenPanel(WMWidgetScreen(bPtr));
    if (WMRunModalOpenPanelForDirectory(panel, spath, NULL, NULL)) {
      char *result;
      result = WMGetFilePanelFile(panel);
      WMSetTextFieldText(parentpanel->soundField, result);
    }
    WMFreeFilePanel(panel);
    WMSetButtonEnabled(parentpanel->browseButton, YES);
    free(spath);
}

static void
wmssInitSoundStatus(WMSoundSetupPanel *panel)
{
    proplist_t  WindowMaker, sound_key, sound_val, tmp_val;
    char        *path;
    WMPixmap	*pixmap;

    path = defaultsPathForDomain("WindowMaker");
    WindowMaker = PLGetProplistWithPath(path);
    if(!WindowMaker) {
        wwarning("could not load WindowMaker configuration file '%s'\n",path);
        exit(1);
    }

    sound_key = PLMakeString("DisableSound");
    sound_val = PLGetDictionaryEntry(WindowMaker,sound_key);
      
    if (!sound_val) {
      sound_key = PLMakeString("NoSound");
      sound_val = PLGetDictionaryEntry(WindowMaker,sound_key);
      if (!sound_val) {
        wwarning("could not find SoundStatus configuration in '%s' activated default value\n",path);
        sound_key = PLMakeString ("DisableSound");
        sound_val = PLMakeString ("No");
        PLInsertDictionaryEntry(WindowMaker, sound_key, sound_val);
      }
      else {
        tmp_val = PLDeepCopy(sound_val);
        PLRemoveDictionaryEntry(WindowMaker, sound_key);
        sound_key = PLMakeString ("DisableSound");
        PLInsertDictionaryEntry(WindowMaker, sound_key, tmp_val);
        PLRelease (tmp_val);
      }
    }
   

    if (strcmp(PLGetString(sound_val), "No") == 0) {
      /* Sound is currently enabled */
      pixmap = WMCreatePixmapFromFile(WMWidgetScreen(panel->win), soundon_tiff);
      WMSetLabelImage(panel->soundStatusLabel, pixmap);
      WMReleasePixmap(pixmap);
    }
    else {
      /* Sound is currently disabled */
      pixmap = WMCreatePixmapFromFile(WMWidgetScreen(panel->win), soundoff_tiff);
      WMSetLabelImage(panel->soundStatusLabel, pixmap);
      WMReleasePixmap(pixmap);
    }

    PLSave(WindowMaker, YES);
    PLRelease (WindowMaker);
    PLRelease (sound_key);
    PLRelease (sound_val);
}



static void
wmssSoundOn(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    proplist_t	WindowMaker, cur_conf, sound_key, sound_val;
    char	*path;
    WMPixmap	*pixmap;

    path = defaultsPathForDomain("WindowMaker");
    WindowMaker = PLGetProplistWithPath(path);
    if(!WindowMaker) {    
        wwarning("could not load WindowMaker configuration file '%s'\n",path);
        exit(1);
    }

    cur_conf = PLMakeDictionaryFromEntries(NULL,NULL,NULL);
    sound_key = PLMakeString("DisableSound");

    sound_val = PLMakeString("No");
    pixmap = WMCreatePixmapFromFile(WMWidgetScreen(panel->win), soundon_tiff);
    WMSetLabelImage(panel->soundStatusLabel, pixmap);
    WMReleasePixmap(pixmap);
    WMSetLabelText(panel->statusLabel, "Sound Enabled");

    PLRemoveDictionaryEntry(cur_conf, sound_key);
    PLInsertDictionaryEntry(cur_conf, sound_key, sound_val);

    PLMergeDictionaries(WindowMaker, cur_conf);
    PLSave(WindowMaker,YES);

    PLRelease (WindowMaker);
    PLRelease (cur_conf);
    PLRelease (sound_key);
    PLRelease (sound_val);
}

static void
wmssSoundOff(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    proplist_t  WindowMaker, cur_conf, sound_key, sound_val;
    char        *path;
    WMPixmap	*pixmap;

    path = defaultsPathForDomain("WindowMaker");
    WindowMaker = PLGetProplistWithPath(path);
    if(!WindowMaker) {
        wwarning("could not load WindowMaker configuration file '%s'\n",path);
        exit(1);
    }

    cur_conf = PLMakeDictionaryFromEntries(NULL,NULL,NULL);
    sound_key = PLMakeString("DisableSound");

    sound_val = PLMakeString("Yes");
    pixmap = WMCreatePixmapFromFile(WMWidgetScreen(panel->win), soundoff_tiff);
    WMSetLabelImage(panel->soundStatusLabel, pixmap);
    WMReleasePixmap(pixmap);
    WMSetLabelText(panel->statusLabel, "Sound Disabled");

    PLRemoveDictionaryEntry(cur_conf, sound_key);
    PLInsertDictionaryEntry(cur_conf, sound_key, sound_val);

    PLMergeDictionaries(WindowMaker, cur_conf);
    PLSave(WindowMaker,YES);

    PLRelease (WindowMaker);
    PLRelease (cur_conf);
    PLRelease (sound_key);
    PLRelease (sound_val);
}

static void
wmssLoadScheme(WMButton *bPtr, WMSoundSetupPanel *origpanel)
{
    char	path[1024];
    WMOpenPanel	*panel;
    proplist_t  scheme;
    char	*spath;

    WMSetButtonEnabled(origpanel->loadSchemeButton, NO);

    spath = getenv("HOME");
    strcpy(path, spath);
    strcat(path, "/GNUstep/Library/WindowMaker/SoundSets");
    spath = malloc(strlen(path)+2);
    strcpy(spath, path);

    if (chdir(spath) == -1) {
      wwarning("User's SoundSets directory not found, switching to homedir.");
      spath = getenv("HOME");
      strcpy(path, spath);
      spath = malloc(strlen(path)+2);
      strcpy(spath, path);
    }

    WMSetWindowTitle(origpanel->win, "Loading SoundScheme");
    panel = WMGetOpenPanel(WMWidgetScreen(bPtr));
    if (WMRunModalOpenPanelForDirectory(panel, spath, NULL, NULL)) {
      char *result;
      result = WMGetFilePanelFile(panel);

      scheme = PLGetProplistWithPath(result);
      if(!scheme) {
        wwarning("could not load soundscheme file '%s'\n",result);
        WMSetLabelText(origpanel->statusLabel, "Error: Could not Load Sound Scheme");
        free(result);
      }
      else {
        PLMergeDictionaries(style,scheme);
	val = PLGetDictionaryEntry(style,key);
        text = PLGetString(val);
        WMSetTextFieldText(origpanel->soundField, text);
        WMSetLabelText(origpanel->statusLabel, "Sound Scheme Loaded");
      }
    }
    WMFreeFilePanel(panel);
    WMSetWindowTitle(origpanel->win, "WMSound Setup");
    WMSetButtonEnabled(origpanel->loadSchemeButton, YES);
    free(spath);
}

static void
wmssSaveScheme(WMButton *bPtr, WMSoundSetupPanel *origpanel)
{
    char	path[1024];
    WMSavePanel *panel;
    char        *spath;
    FILE	*file;

    WMSetButtonEnabled(origpanel->saveSchemeButton, NO);

    spath = getenv("HOME");
    strcpy(path, spath);
    strcat(path, "/GNUstep/Library/WindowMaker/SoundSets");
    spath = malloc(strlen(path)+2);
    strcpy(spath, path);

    if (chdir(spath) == -1) {
      wwarning("User's SoundSets directory not found, switching to homedir.");
      spath = getenv("HOME");
      strcpy(path, spath);
      spath = malloc(strlen(path)+2);
      strcpy(spath, path);
    }

    WMSetWindowTitle(origpanel->win, "Saving SoundScheme");
    panel = WMGetOpenPanel(WMWidgetScreen(bPtr));
    if (WMRunModalOpenPanelForDirectory(panel, spath, NULL, NULL)) {
      char *result;
      result = WMGetFilePanelFile(panel);
      file = fopen (result, "w+");
      if (file) {
        fprintf (file, PLGetDescriptionIndent(style,0));
        fclose (file);
        WMSetLabelText(origpanel->statusLabel, "Sound Scheme Saved");
      }
      else {
        wwarning("could open '%s' for write, please check file-permissions.\n", result);
        WMSetLabelText(origpanel->statusLabel, "Error: Could not Write Sound Scheme");
      }
    }
    WMFreeFilePanel(panel);
    WMSetWindowTitle(origpanel->win, "WMSound Setup");
    WMSetButtonEnabled(origpanel->saveSchemeButton, YES);
    free(spath);
}


static void
wmssClearSound(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    WMSetTextFieldText(panel->soundField, "None");
}

static void
wmssPlaySound(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    int pid; /* Process id */

    text = WMGetTextFieldText(panel->soundField);
    WMSetLabelText(panel->statusLabel, "Activating NoiseMaker(tm)");

    if ((pid=vfork()) == 0) /* Parent gets pid of child process or -1 on error */
    { /* Child gets pid = 0 to indicate it is a child */
         if (execlp("nmaker", "nmaker", text, (char *)NULL) != 0)
/* 
   This only returns something when there is an error! 
   Otherwise the exec(3)'ed program takes over
*/
         {
            perror("execlp");
            _exit(1); /* IMPORTANT: use _exit(2) instead of exit(3) in a vfork(2) */
            /* NOTREACHED */
         }
    }
   /* If needed, you can do a wait(2) on the child process here... */
   /* NOTREACHED */
}

static void
wmssChangeEvent(WMPopUpButton *bPtr, WMSoundSetupPanel *panel)
{
    int event;

    text = WMGetTextFieldText(panel->soundField);
    
    val=PLMakeString(text);
    PLRemoveDictionaryEntry(style, key);
    PLInsertDictionaryEntry(style, key, val);

    event = WMGetPopUpButtonSelectedItem(bPtr);
    switch (event) {
      case 0:
	key = PLMakeString(options[7]);
        break;
      case 1:
        key = PLMakeString(options[1]);
        break;
      case 2:
        key = PLMakeString(options[0]);
        break;
      case 3:
        key = PLMakeString(options[4]);
        break;
      case 4:
        key = PLMakeString(options[5]);
        break;
      case 5:
        key = PLMakeString(options[3]);
        break;
      case 6:
        key = PLMakeString(options[9]);
        break;
      case 7:
        key = PLMakeString(options[2]);
        break;
      case 8:
        key = PLMakeString(options[8]);
        break;
      case 9:
        key = PLMakeString(options[6]);
        break;
      case 10:
        key = PLMakeString(options[10]);    
        break;    
    }
    val = PLGetDictionaryEntry(style,key);
    text = PLGetString(val);
    WMSetTextFieldText(panel->soundField, text);
}


static void
wmssSaveChanges(WMPopUpButton *bPtr, WMSoundSetupPanel *panel)
{

    text = WMGetTextFieldText(panel->soundField);

    val=PLMakeString(text);
    PLRemoveDictionaryEntry(style, key);
    PLInsertDictionaryEntry(style, key, val);


    PLMergeDictionaries(prop, style);
    PLSave(prop,YES);

    PLRelease(prop);
    PLRelease(style);
    PLRelease(key);
    PLRelease(val);
    PLRelease(text);
    
    free(panel);
    exit(0);
}

static void
wmssApplyScheme(WMButton *bPtr, WMSoundSetupPanel *panel)
{

    text = WMGetTextFieldText(panel->soundField);

    val=PLMakeString(text);
    PLRemoveDictionaryEntry(style, key);
    PLInsertDictionaryEntry(style, key, val);


    PLMergeDictionaries(prop, style);
    PLSave(prop,YES);
    WMSetLabelText(panel->statusLabel, "Applied Current Settings");
}

static void
wmssRevertChanges(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    int i;
    proplist_t temp_key;

    temp_key = PLDeepCopy(key);

    PLRelease(style);
    style = PLMakeDictionaryFromEntries(NULL,NULL,NULL);
    for(i=0;options[i]!=NULL;i++) {
        key = PLMakeString(options[i]);
        val = PLGetDictionaryEntry(prop,key);
        if(val) PLInsertDictionaryEntry(style,key,val);
    }
    key = PLMakeString("SoundPath");
    val = PLGetDictionaryEntry(prop,key);
    if(val) PLInsertDictionaryEntry(style,key,val);

    PLRelease(key);

    key = PLDeepCopy(temp_key);
    PLRelease (temp_key);
    val = PLGetDictionaryEntry(style,key);
    text = PLGetString(val);
    WMSetTextFieldText(panel->soundField, text);
    WMSetLabelText(panel->statusLabel, "Reverted Changes");
}

static void
wmssCancel(WMButton *bPtr, WMSoundSetupPanel *panel)
{
    PLRelease(prop);
    PLRelease(style);
    PLRelease(key);
    PLRelease(val);
    PLRelease(text);

    free(panel);
    exit(0);
}


static WMSoundSetupPanel*
wmssMain(WMScreen *scr)
{
    WMSoundSetupPanel	*panel;
    WMPixmap		*pixmap;

    /*
    ------------------------------
    The Actual WMSound Setup Panel
    ------------------------------
    */
    panel = malloc(sizeof(WMSoundSetupPanel));
    if (!panel) {
        wwarning("not enough memory to open soundsetup panel");
        return NULL;
    }


    panel->win = WMCreateWindow(scr, "wmsoundsetup");

    WMSetWindowTitle(panel->win, "WMSound Setup");
    
    WMSetWindowCloseAction(panel->win, closeAction, NULL);
    
    WMResizeWidget(panel->win, 352, 236);




    /*
    -------------------
    Enable Sound Button
    -------------------
    */
    panel->soundonButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->soundonButton, 36, 36);
    WMMoveWidget(panel->soundonButton, 8, 8);
    WMSetButtonAction(panel->soundonButton, (WMAction *)wmssSoundOn, panel);
    pixmap = WMCreatePixmapFromFile(scr, on_tiff);
    WMSetButtonImagePosition(panel->soundonButton, WIPImageOnly);
    WMSetButtonImage(panel->soundonButton, pixmap);
    WMReleasePixmap(pixmap);
    /*
    --------------------
    Disable Sound Button
    --------------------
    */
    panel->soundoffButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->soundoffButton, 36, 36);
    WMMoveWidget(panel->soundoffButton, 44, 8);
    WMSetButtonAction(panel->soundoffButton, (WMAction *)wmssSoundOff, panel);
    pixmap = WMCreatePixmapFromFile(scr, off_tiff);
    WMSetButtonImagePosition(panel->soundoffButton, WIPImageOnly);
    WMSetButtonImage(panel->soundoffButton, pixmap);
    WMReleasePixmap(pixmap);
    /*
    --------------------
    Apply Scheme Button
    --------------------
    */
    panel->applySchemeButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->applySchemeButton, 36, 36);
    WMMoveWidget(panel->applySchemeButton, 80, 8);
    WMSetButtonAction(panel->applySchemeButton, (WMAction *)wmssApplyScheme, panel);
    pixmap = WMCreatePixmapFromFile(scr, apply_tiff);
    WMSetButtonImagePosition(panel->applySchemeButton, WIPImageOnly);
    WMSetButtonImage(panel->applySchemeButton, pixmap);
    WMReleasePixmap(pixmap);
    /*
    --------------------
    Save Scheme Button
    --------------------
    */
    panel->saveSchemeButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->saveSchemeButton, 36, 36);
    WMMoveWidget(panel->saveSchemeButton, 116, 8);
    WMSetButtonAction(panel->saveSchemeButton, (WMAction *)wmssSaveScheme, panel);
    pixmap = WMCreatePixmapFromFile(scr, save_tiff);
    WMSetButtonImagePosition(panel->saveSchemeButton, WIPImageOnly);
    WMSetButtonImage(panel->saveSchemeButton, pixmap);
    WMReleasePixmap(pixmap);
    /*
    --------------------
    Load Scheme Button
    --------------------
    */
    panel->loadSchemeButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->loadSchemeButton, 36, 36);
    WMMoveWidget(panel->loadSchemeButton, 152, 8);
    WMSetButtonAction(panel->loadSchemeButton, (WMAction *)wmssLoadScheme, panel);
    pixmap = WMCreatePixmapFromFile(scr, open_tiff);
    WMSetButtonImagePosition(panel->loadSchemeButton, WIPImageOnly);
    WMSetButtonImage(panel->loadSchemeButton, pixmap);
    WMReleasePixmap(pixmap);
    /*
    --------------------
    Info Button
    --------------------
    */
    panel->infoButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->infoButton, 36, 36);
    WMMoveWidget(panel->infoButton, 308, 8);
    WMSetButtonAction(panel->infoButton, (WMAction *)wmssInfo, panel);
    pixmap = WMCreatePixmapFromFile(scr, info_tiff);
    WMSetButtonImagePosition(panel->infoButton, WIPImageOnly);
    WMSetButtonImage(panel->infoButton, pixmap);
    WMReleasePixmap(pixmap);




    /*
    ------------------
    Sound Status Label
    ------------------
    */
    panel->soundStatusLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->soundStatusLabel, 65, 20);
    WMMoveWidget(panel->soundStatusLabel, 8, 68);
    WMSetLabelText(panel->soundStatusLabel, "Status");
    WMSetLabelImagePosition(panel->soundStatusLabel, WIPRight);
    wmssInitSoundStatus(panel);
    /*
    ---------------------------
    Event Selection PopUpButton
    ---------------------------
    */
    panel->eventLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->eventLabel, 75, 20);
    WMMoveWidget(panel->eventLabel, 117, 68);
    WMSetLabelText(panel->eventLabel, "Sound Event");

    panel->eventPopUp = WMCreatePopUpButton(panel->win);
    WMResizeWidget (panel->eventPopUp, 144, 20);
    WMMoveWidget (panel->eventPopUp, 200, 68);
    WMSetPopUpButtonAction(panel->eventPopUp, (WMAction *)wmssChangeEvent, panel);

    WMAddPopUpButtonItem(panel->eventPopUp, "WMSound StartUp");
    WMAddPopUpButtonItem(panel->eventPopUp, "Application StartUp");
    WMAddPopUpButtonItem(panel->eventPopUp, "Application Exit");
    WMAddPopUpButtonItem(panel->eventPopUp, "Maximize");
    WMAddPopUpButtonItem(panel->eventPopUp, "UnMaximize");
    WMAddPopUpButtonItem(panel->eventPopUp, "Iconify");
    WMAddPopUpButtonItem(panel->eventPopUp, "DeIconify");
    WMAddPopUpButtonItem(panel->eventPopUp, "Hide");
    WMAddPopUpButtonItem(panel->eventPopUp, "UnHide");
    WMAddPopUpButtonItem(panel->eventPopUp, "Shade");
    WMAddPopUpButtonItem(panel->eventPopUp, "UnShade");

    WMSetPopUpButtonSelectedItem(panel->eventPopUp, 0);



    /*
    ---------------------
    Sound Selection Frame
    ---------------------
    */
    panel->soundFrame = WMCreateFrame(panel->win);
    WMResizeWidget(panel->soundFrame, 336, 72);
    WMMoveWidget(panel->soundFrame, 8, 96);
    WMSetFrameTitle(panel->soundFrame, "Sound File");
    /*
    -----------------
    Sound Input Field
    -----------------
    */
    panel->soundField = WMCreateTextField(panel->win);
    WMResizeWidget(panel->soundField, 324, 20);
    WMMoveWidget(panel->soundField, 14, 110);
    key = PLMakeString(options[7]);
    val = PLGetDictionaryEntry(prop, key);
    text = PLGetString(val);
    WMSetTextFieldText(panel->soundField, text);
    /*
    ------------------
    Clear Sound Button
    ------------------
    */
    panel->nosoundButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->nosoundButton, 75, 24);
    WMMoveWidget(panel->nosoundButton, 180, 138);
    WMSetButtonText(panel->nosoundButton, "Clear");
    WMSetButtonAction(panel->nosoundButton, (WMAction *)wmssClearSound, panel);
    /*
    -----------------
    Play Sound Button
    -----------------
    */
    panel->playButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->playButton, 75, 24);
    WMMoveWidget(panel->playButton, 263, 138);
    WMSetButtonAction(panel->playButton, (WMAction *)wmssPlaySound, panel);
    WMSetButtonText(panel->playButton, "Play");
    /*
    -------------------
    Browse Sound Button
    -------------------
    */
    panel->browseButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->browseButton, 75, 24);
    WMMoveWidget(panel->browseButton, 97, 138);
    WMSetButtonAction(panel->browseButton, (WMAction *)wmssBrowseSound, panel);
    WMSetButtonText(panel->browseButton, "Browse");
    /*
    ---------------------
    Edit SoundPath Button
    ---------------------
    */
    panel->soundpathButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->soundpathButton, 75, 24);
    WMMoveWidget(panel->soundpathButton, 14, 138);
    WMSetButtonAction(panel->soundpathButton, (WMAction *)wmssSoundPath, panel);
    WMSetButtonText(panel->soundpathButton, "Sound Path");




    /*
    ----------
    Status Bar
    ----------
    */
    panel->statusLabel = WMCreateLabel(panel->win);
    WMResizeWidget(panel->statusLabel, 336, 20);
    WMMoveWidget(panel->statusLabel, 8, 176);
    WMSetLabelText(panel->statusLabel, "Ready ...");
    WMSetLabelRelief(panel->statusLabel, WRSunken);
    WMSetLabelTextAlignment(panel->statusLabel, WALeft);




    /*
    -------------
    Cancel Button
    -------------
    */
    panel->cancelButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->cancelButton, 75, 24);
    WMMoveWidget(panel->cancelButton, 103, 204);
    WMSetButtonText(panel->cancelButton, "Cancel");
    WMSetButtonAction(panel->cancelButton, (WMAction *)wmssCancel, panel);
    /*
    -------------
    Revert Button
    -------------
    */
    panel->revertButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->revertButton, 75, 24);
    WMMoveWidget(panel->revertButton, 186, 204);
    WMSetButtonText(panel->revertButton, "Revert");
    WMSetButtonAction(panel->revertButton, (WMAction *)wmssRevertChanges, panel);
    /*
    -----------
    Save Button
    -----------
    */
    panel->saveButton = WMCreateCommandButton(panel->win);
    WMResizeWidget(panel->saveButton, 75, 24);
    WMMoveWidget(panel->saveButton, 269, 204);
    WMSetButtonText(panel->saveButton, "Save");
    WMSetButtonAction(panel->saveButton, (WMAction *)wmssSaveChanges, panel);




    WMRealizeWidget(panel->win);
    WMMapSubwidgets(panel->win);
    WMMapWidget(panel->win);
    return panel;
}


int main(int argc, char **argv)
{
    WMScreen 	*scr;
    WMPixmap	*pixmap;
    proplist_t 	key = NULL, val = NULL;
    char 	*path;
    int 	i;

    WMInitializeApplication("WMSoundSetup", &argc, argv);

    dpy = XOpenDisplay("");
    if (!dpy) {
	puts("could not open display");
	exit(1);
    }
    
    XSynchronize(dpy, True);

    path = defaultsPathForDomain(CFG_DOMAIN);
    prop = PLGetProplistWithPath(path);
    if(!prop) {
	wwarning("could not load WindowMaker configuration file '%s'\n",path);
	exit(1);
    }

    style = PLMakeDictionaryFromEntries(NULL,NULL,NULL);
    for(i=0;options[i]!=NULL;i++) {
        key = PLMakeString(options[i]);
        val = PLGetDictionaryEntry(prop,key);
        if(val) PLInsertDictionaryEntry(style,key,val);
    }
    scr = WMCreateScreen(dpy, DefaultScreen(dpy));

    pixmap = WMCreatePixmapFromFile(scr, wmss_tiff);
    WMSetApplicationIconImage(scr, pixmap);
    WMReleasePixmap(pixmap);

    wmssMain(scr);
    WMScreenMainLoop(scr);
    PLRelease(prop);
    PLRelease(style);
    PLRelease(key);
    PLRelease(val);
    PLRelease(text);

    return 0;
}
