/*
 * nvram.c -- read and set values in Atari NV-RAM
 *
 * Copyright (C) 1997 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <linux/nvram.h>

#define VERSION		"0.1"


struct nvram {
	unsigned char reserved0;
	unsigned char bootpref;
	unsigned char reserved1[4];
	unsigned char language;
	unsigned char keyboard;
	struct {
		unsigned unused1 : 3;
		unsigned time : 1;
		unsigned unused2 : 1;
		unsigned date : 3;
	} datetime;
	unsigned char separator;
	unsigned char bootdelay;
	unsigned char reserved2[3];
	struct {
		unsigned unused : 7;
		unsigned dbl : 1;
		unsigned compat : 1;
		unsigned overscan : 1;
		unsigned pal : 1;
		unsigned vga : 1;
		unsigned columns : 1;
		unsigned colors : 3;
	} vmode;
	unsigned char scsiid;
	unsigned char reserved_for_us[1];
};

static struct option long_options[] = {
	{ "boot-preference", required_argument, NULL, 'b' },
	{ "recalculate-checksum", no_argument, NULL, 'c' },
	{ "boot-delay", required_argument, NULL, 'd' },
	{ "date-format", required_argument, NULL, 'D' },
	{ "init-nvram", no_argument, NULL, 'i' },
	{ "keyboard-language", required_argument, NULL, 'k' },
	{ "language", required_argument, NULL, 'l' },
	{ "scsi-id", required_argument, NULL, 's' },
	{ "time-format", required_argument, NULL, 't' },
	{ "video-mode", required_argument, NULL, 'v' },
	{ "version", required_argument, NULL, 'V' }
};

int boot_pref = -1;
int recalc_checksum = 0;
int delay = -1;
int date_format = -1;
int separator = -1;
int init_nvram = 0;
int kbd_lang = -1;
int os_lang = -1;
int scsiid = -1;
int time_format = -1;
int video_colors = -1;
int video_columns = -1;
int video_VGA = -1;
int video_PAL = -1;
int video_overscan = -1;
int video_compat = -1;
int video_double = -1;

typedef struct {
	char *str;
	int value;
} STRTAB;

static STRTAB boot_prefs[] = {
	{ "unspecified", 0 },
	{ "TOS", 0x80 },
	{ "GEMDOS", 0x80 },
	{ "AtariSystemV", 0x40 },
	{ "SystemV", 0x40 },
	{ "SysV", 0x40 },
	{ "ASV", 0x40 },
	{ "NetBSD", 0x20 },
	{ "BSD", 0x20 },
	{ "Linux", 0x10 },
	{ NULL, 0 }
};

static STRTAB languages[] = {
	{ "English (US)", 0 },
	{ "English(us)", 0 },
	{ "English", 0 },
	{ "US-English", 0 },
	{ "English-US", 0 },
	{ "German", 1, },
	{ "French", 2 },
	{ "English (UK)", 3 },
	{ "English (GB)", 3 },
	{ "English(uk)", 3 },
	{ "English(gb)", 3 },
	{ "UK-English", 3 },
	{ "GB-English", 3 },
	{ "English-UK", 3 },
	{ "English-GB", 3 },
	{ "Spanish", 4 },
	{ "Italian", 5 },
	{ "Swiss(French)", 6 },
	{ "Swiss-fr", 6 },
	{ "Swiss(German)", 7 },
	{ "Swiss-de", 7 },
	{ NULL, 0 }
};

static struct {
	unsigned char val;
	char *name;
} r_boot_prefs[] = {
	{ 0x80, "TOS" },
	{ 0x40, "ASV" },
	{ 0x20, "NetBSD (?)" },
	{ 0x10, "Linux" },
	{ 0x00, "unspecified" }
};

static char *r_languages[] = {
	"English (US)",
	"German",
	"French",
	"English (UK)",
	"Spanish",
	"Italian",
	"6 (undefined)",
	"Swiss (French)",
	"Swiss (German)"
};

static char *r_dateformat[] = {
	"MM%cDD%cYY",
	"DD%cMM%cYY",
	"YY%cMM%cDD",
	"YY%cDD%cMM",
	"4 (undefined)",
	"5 (undefined)",
	"6 (undefined)",
	"7 (undefined)"
};

static char *r_colors[] = {
	"2", "4", "16", "256", "65536", "??", "??", "??"
};

#define fieldsize(a)	(sizeof(a)/sizeof(*a))



static void usage( void )
{
	fprintf( stderr, "\
Usage: nvram [-bcdDiklstv] [--boot-preference=pref|unspecified]\n\
             [--recalculate-checksum] [--boot-delay=delay|default]\n\
             [--date-format=format] [--init-nvram]\n\
             [--keyboard-language=language] [--language=language]\n\
             [scsi-id=id|off] [--time-format=12|24] [video-mode=modespec]\n" );
	exit( 1 );
}

static int getstringval( const char *str, STRTAB *table )
{
	STRTAB *p, *exact = NULL, *partial = NULL;
	int len = strlen(str);
	
	for( p = table; p->str; ++p ) {
		if (strncasecmp( p->str, str, len ) == 0) {
			if (strlen(p->str) == len) {
				exact = p;
				break;
			}
			else {
				if (partial) {
					fprintf( stderr, "Ambigous abbreviation: %s\n", str );
					return( -1 );
				}
				partial = p;
			}
		}
	}
	if (exact)
		return( exact->value );
	else if (partial)
		return( partial->value );
	else
		return( -1 );
}

static int parse_dateformat( const char *str, int *datef, int *sep )
{
	const char *p;
	char format[4], seps[2];
	int i;

	for( p = str, i = 0; *p && i < 3; ++i ) {
		if (strchr( "DMY", toupper(*p) )) {
			format[i] = toupper(*p++);
		}
		else {
			fprintf( stderr, "Bad character '%c' in date format\n", *p );
			return( -1 );
		}
		/* char doubled (usual, but no must) */
		if (toupper(*p) == format[i])
			++p;
		if (i == 2)
			break;
		if (!*p) {
			fprintf( stderr, "Short date string!\n" );
			return( -1 );
		}
		seps[i] = *p++;
	}
	if (i != 2) {
		fprintf( stderr, "Short date string!\n" );
		return( -1 );
	}
	if (*p) {
		fprintf( stderr, "Junk at end of date string\n" );
		return( -1 );
	}
	
	if (seps[0] != seps[1]) {
		fprintf( stderr, "Need equal separator characters\n" );
		return( -1 );
	}
	*sep = seps[0];

	format[3] = 0;
	if (strcmp( format, "MDY" ) == 0) {
		*datef = 0;
		return( 1 );
	}
	else if (strcmp( format, "DMY" ) == 0) {
		*datef = 1;
		return( 1 );
	}
	else if (strcmp( format, "YMD" ) == 0) {
		*datef = 2;
		return( 1 );
	}
	else if (strcmp( format, "YDM" ) == 0) {
		*datef = 3;
		return( 1 );
	}
	else {
		fprintf( stderr, "Bad date format\n" );
		return( 0 );
	}
}

static int parse_video_mode( char *str )
{
	char *p, *value, *end;
	unsigned val;
	int double_used = 0, half_used = 0;
	
	for( p = strtok( str, "," ); p; p = strtok( NULL, "," ) ) {
		if ((value = strchr( p, '=' ))) {
			*value++ = 0;
			val = strtoul( value, &end, 0 );
			if (*end) {
				fprintf( stderr, "Bad number in video argument: %s\n", value );
				return( 0 );
			}
		}

		if (strcasecmp( p, "colors" ) == 0) {
			if (!value) {
				fprintf( stderr, "colors missing argument\n" );
				return( 0 );
			}
			if (val != 2 && val != 4 && val != 16 && val != 256 &&
				val != 65536) {
				fprintf( stderr, "Bad number of colors: %d\n", val );
				return( 0 );
			}
			video_colors = val;
		}
		else if (strcasecmp( p, "columns" ) == 0) {
			if (!value) {
				fprintf( stderr, "columns missing argument\n" );
				return( 0 );
			}
			if (val != 40 && val != 80) {
				fprintf( stderr, "Bad number for columns: %d (40 or 80)\n",
						 val );
				return( 0 );
			}
			video_columns = val;
		}
		else if (strcasecmp( p, "col40" ) == 0) {
			video_columns = 40;
		}
		else if (strcasecmp( p, "col80" ) == 0) {
			video_columns = 80;
		}
		else if (strcasecmp( p, "tv" ) == 0) {
			video_VGA = 0;
		}
		else if (strcasecmp( p, "vga" ) == 0) {
			video_VGA = 1;
		}
		else if (strcasecmp( p, "ntsc" ) == 0) {
			video_PAL = 0;
		}
		else if (strcasecmp( p, "pal" ) == 0) {
			video_PAL = 1;
		}
		else if (strcasecmp( p, "overscan" ) == 0) {
			if (!value)
				video_overscan = 1;
			else
				video_overscan = !!val;
		}
		else if (strcasecmp( p, "no-overscan" ) == 0) {
			video_overscan = 0;
		}
		else if (strcasecmp( p, "compat" ) == 0) {
			if (!value)
				video_compat = 1;
			else
				video_compat = !!val;
		}
		else if (strcasecmp( p, "no-compat" ) == 0) {
			video_compat = 0;
		}
		else if (strcasecmp( p, "double" ) == 0) {
			if (!value)
				video_double = 1;
			else
				video_double = !!val;
			double_used = 1;
		}
		else if (strcasecmp( p, "no-double" ) == 0) {
			video_double = 0;
			double_used = 1;
		}
		else if (strcasecmp( p, "half" ) == 0) {
			if (!value)
				video_double = 1;
			else
				video_double = !!val;
			half_used = 1;
		}
		else if (strcasecmp( p, "no-half" ) == 0) {
			video_double = 0;
			half_used = 1;
		}
	}

	if (double_used && half_used) {
		fprintf( stderr, "double and half are mutually exclusive\n" );
		return( 0 );
	}
	if (double_used && video_VGA == 0) {
		fprintf( stderr, "double can be used only with VGA monitor\n" );
		return( 0 );
	}
	if (half_used && video_VGA == 1) {
		fprintf( stderr, "double can be used only with TV monitor\n" );
		return( 0 );
	}
	return( 1 );
}

static void print_nvram( struct nvram *nvram )
{
	int i;
	char sep;
	
	printf( "Boot preference  : " );
	for( i = fieldsize(r_boot_prefs)-1; i >= 0; --i ) {
		if (nvram->bootpref == r_boot_prefs[i].val) {
			printf( "%s\n", r_boot_prefs[i].name );
			break;
		}
	}
	if (i < 0)
		printf( "0x%02x (undefined)\n", nvram->bootpref );

	printf( "SCSI arbitration : %s\n", (nvram->scsiid & 0x80) ? "on" : "off" );
	printf( "SCSI host ID     : " );
	if (nvram->scsiid & 0x80)
		printf( "%d\n", nvram->scsiid & 7 );
	else
		printf( "n/a\n" );

	printf( "Only valid on Falcon:\n" );
	
	printf( "OS language      : " );
	if (nvram->language < fieldsize(r_languages))
		printf( "%s\n", r_languages[nvram->language] );
	else
		printf( "%u (undefined)\n", nvram->language );

	printf( "Keyboard language: " );
	if (nvram->keyboard < fieldsize(r_languages))
		printf( "%s\n", r_languages[nvram->keyboard] );
	else
		printf( "%u (undefined)\n", nvram->keyboard );

	sep = nvram->separator ? nvram->separator : '/';
	printf( "Date format      : " );
	printf( r_dateformat[nvram->datetime.date], sep, sep );
	printf( ", %dh clock\n", nvram->datetime.time ? 24 : 12 );

	printf( "Boot delay       : " );
	if (nvram->bootdelay == 0)
		printf( "default\n" );
	else
		printf( "%ds%s\n", nvram->bootdelay,
				nvram->bootdelay < 8 ? ", no memory test" : "" );

	printf( "Video mode       : " );
	printf( "%s colors, ", r_colors[nvram->vmode.colors] );
	printf( "%d columns, ", nvram->vmode.columns ? 80 : 40 );
	printf( "%s %s monitor\n",
			nvram->vmode.vga ? "VGA" : "TV",
			nvram->vmode.pal ? "PAL" : "NTSC" );
	printf( "                   " );
	printf( "%soverscan, ", nvram->vmode.overscan ? "" : "no " );
	printf( "compat. mode %s", nvram->vmode.compat ? "on" : "off" );
	if (nvram->vmode.dbl)
		printf( ", %s", nvram->vmode.vga ? "line doubling" : "half screen" );
	printf( "\n" );
}

int main( int argc, char *argv[] )
{
	int c;
	char *end;
	int any_set = 0, write_action = 0;
	int fd, cnt;
	struct nvram nvram;
	
	while( (c = getopt_long( argc, argv, "b:cd:D:ik:l:s:t:v:",
							 long_options, (int *)0 )) != EOF) {
		switch( c ) {
			
		  case 'b':
			if (isdigit(*optarg)) {
				boot_pref = strtoul( optarg, &end, 0 );
				if (*end || (boot_pref != 0x80 && boot_pref != 0x40 &&
							 boot_pref != 0x20 && boot_pref != 0x10 &&
							 boot_pref != 0x08 && boot_pref != 0)) {
					fprintf( stderr, "Bad numeric boot preference: %s\n",
							 optarg );
					usage();
				}
			}
			else {
				if ((boot_pref = getstringval( optarg, boot_prefs )) < 0) {
					printf( "Bad boot preference: %s\n", optarg );
					usage();
				}
			}
			any_set = 1;
			break;
			
		  case 'c':
			recalc_checksum = 1;
			break;
	
		  case 'd':
			if (strcasecmp( optarg, "default" ) == 0)
				delay = 0;
			else {
				delay = strtoul( optarg, &end, 0 );
				if (*end || delay > 255) {
					fprintf( stderr, "Bad boot delay: %s\n", optarg );
					usage();
				}
			}
			any_set = 1;
			break;

		  case 'D':
			if (!parse_dateformat( optarg, &date_format, &separator )) {
				fprintf( stderr, "Bad date format: %s\n", optarg );
				usage();
			}
			any_set = 1;
			break;

		  case 'i':
			init_nvram = 1;
			break;
	
		  case 'k':
			if ((kbd_lang = getstringval( optarg, languages )) < 0) {
				printf( "Bad language: %s\n", optarg );
				usage();
			}
			any_set = 1;
			break;
			
		  case 'l':
			if ((os_lang = getstringval( optarg, languages )) < 0) {
				printf( "Bad language: %s\n", optarg );
				usage();
			}
			any_set = 1;
			break;
			
		  case 's':
			if (strcasecmp( optarg, "off" ) == 0)
				scsiid = 8;
			else {
				scsiid = strtoul( optarg, &end, 0 );
				if (*end || scsiid > 7) {
					fprintf( stderr, "Bad SCSI ID: %s\n", optarg );
					usage();
				}
			}
			any_set = 1;
			break;
			
		  case 't':
			time_format = strtoul( optarg, &end, 0 );
			if (!(*end == 0 || (tolower(*end) == 'h' && end[1] == 0)) ||
				(time_format != 12 && time_format != 24)) {
				fprintf( stderr, "Bad time format: %s\n", optarg );
				usage();
			}
			any_set = 1;
			break;
			
		  case 'v':
			if (!parse_video_mode( optarg )) {
				printf( "Bad video mode specification: %s\n", optarg );
				usage();
			}
			any_set = 1;
			break;

		  case 'V':
			printf( "Atari nvram " VERSION "\n" );
			exit( 0 );
			
		  default:
			usage();
		}
	}

	/* no non-option args */
	if (optind != argc)
		usage();
	
	write_action = any_set + recalc_checksum + init_nvram;
	if (write_action > 1) {
		fprintf( stderr, "-c, -i, and other options are mutually exclusive\n");
		exit( 1 );
	}
	
	if ((fd = open( "/dev/nvram", write_action ? O_RDWR : O_RDONLY )) < 0) {
		perror( "open(\"/dev/nvram\")" );
		exit( 1 );
	}

	if ((cnt = read( fd, &nvram, sizeof(nvram) )) != sizeof(nvram)) {
		if (cnt < 0 && errno == EIO)
			fprintf( stderr, "NVRAM checksum bad.\n" );
		else if  (cnt < 0)
			perror( "read" );
		else
			fprintf( stderr, "read error on /dev/nvram "
					 "(%d bytes instead of %d\n",
					 cnt, sizeof(nvram) );
		exit( 1 );
	}

	if (!write_action)
		print_nvram( &nvram );
	else if (recalc_checksum) {
		if (ioctl( fd, NVRAM_SETCKS, 0 )) {
			perror( "ioctl(NVRAM_SETCKS)" );
			exit( 1 );
		}
		printf( "NVRAM checksum updated.\n" );
	}
	else if (init_nvram) {
		if (ioctl( fd, NVRAM_INIT, 0 )) {
			perror( "ioctl(NVRAM_INIT)" );
			exit( 1 );
		}
		printf( "NVRAM initialized.\n" );
	}
	else {
		if (boot_pref >= 0)
			nvram.bootpref = boot_pref;
		if (os_lang >= 0)
			nvram.language = os_lang;
		if (kbd_lang >= 0)
			nvram.keyboard = kbd_lang;
		if (date_format >= 0)
			nvram.datetime.date = date_format;
		if (time_format >= 0)
			nvram.datetime.time = time_format == 24;
		if (delay >= 0)
			nvram.bootdelay = delay;
		if (separator >= 0)
			nvram.separator = separator == '/' ? 0 : separator;
		if (delay >= 0)
			nvram.bootdelay = delay;
		if (video_colors >= 0)
			nvram.vmode.colors = video_colors;
		if (video_columns >= 0)
			nvram.vmode.columns = video_columns == 80;
		if (video_VGA >= 0)
			nvram.vmode.vga = video_VGA;
		if (video_PAL >= 0)
			nvram.vmode.pal = video_PAL;
		if (video_overscan >= 0)
			nvram.vmode.overscan = video_overscan;
		if (video_compat >= 0)
			nvram.vmode.compat = video_compat;
		if (video_double >= 0)
			nvram.vmode.dbl = video_double;
		if (scsiid >= 0)
			nvram.scsiid = scsiid == 8 ? 0 : scsiid | 0x80;
	}

	if (write_action) {
		if (lseek( fd, 0, SEEK_SET ) < 0) {
			perror( "lseek" );
			exit( 1 );
		}
		if ((cnt = write( fd, &nvram, sizeof(nvram) )) != sizeof(nvram)) {
			if (cnt < 0 && errno == EIO)
				fprintf( stderr, "NVRAM checksum bad on write.\n" );
			else if  (cnt < 0)
				perror( "write" );
			else
				fprintf( stderr, "write error on /dev/nvram "
						 "(%d bytes instead of %d\n",
						 cnt, sizeof(nvram) );
			exit( 1 );
		}
	}

	close( fd );
	return( 0 );
}
