/*
 * Display the file in the window.
 *
 * The current top line is displayed at the top of the window,
 * the pixel offset is the number of pixels from the start of the document.
 */

#include <stdio.h>

#include "arena.h"
#include "bridge.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.24-01-97 */
#  include "debug.h"
#endif
#include "display_i.h"
#include "display.h"
#include "dither.h"
#include "forms.h"
#include "html.h"
#include "icon.h"
#include "image.h"
#include "main.h"
#include "parsehtml.h"
#include "scrollbar.h"
#include "status.h"
#include "style.h"
#include "toolbar.h"
#include "util.h"
#include "x11.h"


Window DocDispWin = None;
GC     DocDispGC  = NULL;

#if 0	/* QingLong.31-01-98: Does that ``optimization'' make sense? */
const float winBufferWidthN  = 0.1;
const float winBufferHeightN = 1.0;
# else
const float winBufferWidthN  = 0.0;
const float winBufferHeightN = 0.0;
#endif

char *buffer;            /* the start of the document buffer */
int  hdrlen;             /* offset to start of data from top of buffer*/
int  buf_width;
long buf_height;
long chDescent;
int  chStrike;
int  chWidth;             /* width of average char */
int  spWidth;             /* width of space char */
int  font = -1;           /* index into Fonts[] */
long lineHeight;

XFontStruct* pFontInfo;

static char* FindNextString = NULL;

char* targetptr  = NULL;   /* for toggling view between HTML/TEXT views */
char* targetId   = NULL;   /* for locating named Id during ParseHTML() */
long  IdOffset   = 0;      /* offset for targetId */
long  ViewOffset = 0;      /* for toggling between HTML/TEXT views */


static XPoint     DocDispWinOrigin = { 0, 0 };
static ViewWindow DocViewWin       = { None, { 0, 0, 0, 0 } };
static ViewWindow DocViewCache     = { None, { 0, 0, 0, 0 } };
static GC         DocViewCacheGC   = NULL;


Bool UpdateDocDispWinGeometry(void)
{
 XRectangle rect;

 if (UpdateScrollBarGeometry())
   {
    rect = GetScrollXBarGeometry();
    DocDispWinOrigin.x     = rect.x;
    DocViewWin.rect.width  = rect.width;

    rect = GetScrollYBarGeometry();
    DocDispWinOrigin.y     = rect.y;
    DocViewWin.rect.height = rect.height;

    return True;
   }
  else
   return False;
}


/*
 * LAME: get default document view window.
 */
ViewWindow GetDefaultDocViewWindow(void)
{
 return DocViewWin;
}


/*
 * Get current document display window geometry.
 */
XRectangle GetDocDispWinGeometry(void)
{
 XRectangle DocDispWinRect = { DocDispWinOrigin.x,    DocDispWinOrigin.y,
			       DocViewWin.rect.width, DocViewWin.rect.height };

 return DocDispWinRect;
}


/*
 * Get current document view area geometry.
 */
ViewRectangle GetDocViewWinGeometry(void)
{
 return DocViewWin.rect;
}


/*
 * Get current document display area width.
 */
unsigned int GetDocDispWinWidth(void)
{
 return DocViewWin.rect.width;
}


/*
 * Get current document display area height.
 */
unsigned int GetDocDispWinHeight(void)
{
 return DocViewWin.rect.height;
}


/*
 * Initialize document display subwindow stuff.
 */
Bool DocDispInit(Window theWindow)
{
 if (theWindow != None && DocDispWin == None)
   {
    UpdateDocDispWinGeometry();

    DocDispWin = DocViewWin.win = CreateSubWindow(display, theWindow,
						  DocDispWinOrigin.x,
						  DocDispWinOrigin.y,
						  DocViewWin.rect.width,
						  DocViewWin.rect.height,
						  0, windowColour);
   }

 if (DocDispWin != None && DocDispGC == NULL)
   {
    DocDispGC = XCreateGC(display, DocDispWin, 0, NULL);
   }

 if (DocDispWin != None && DocViewCacheGC == NULL)
   {
    DocViewCacheGC = XCreateGC(display, DocDispWin, 0, NULL);
    XSetGraphicsExposures(display, DocViewCacheGC, False);
   }

 return (DocDispWin != None && (DocDispGC) && (DocViewCacheGC));
}


void SetDisplayFont(XFontStruct* pf)
{
 pFontInfo = pf;
 XSetFont(display, DocDispGC, pFontInfo->fid);
 XSetForeground(display, DocDispGC, textColour);
 XSetBackground(display, DocDispGC, windowColour);
}


void SetColour(GC gc, int colour_text_ix, int colour_background_ix)
{
 XSetForeground(display, gc, ix2colour(colour_text_ix));
 /* GRR:  this works now that html.c is fixed */
 XSetBackground(display, gc, ix2colour(colour_background_ix));

#if 0

 XGCValues values;
 unsigned int valuemask = 0;

 if (colour_text_ix < 128)
   {
    if (depth == 24)
      values.foreground = magic2colour(colour_text_ix);
     else
      values.foreground = stdcmap[colour_text_ix];

    valuemask |= GCForeground; 
    }
  else
   if (colour_text_ix < 144)
     {
      if (depth == 24)
	values.foreground = magic2colour(colour_text_ix & 0xf);
       else
	values.foreground = greymap[colour_text_ix & 0xf];

      valuemask |= GCForeground; 
     }

 /* background colourmap entry has to be found based on
  * the index stored on the paint stream.
  */

 if (colour_background_ix < 128)
   {
    if (depth == 24)
      values.background = magic2colour(colour_background_ix);
     else
      values.background = stdcmap[colour_background_ix];

    valuemask |= GCBackground;
   }
  else
   if (colour_background_ix < 144)
     {
      if (depth == 24)
	values.background = magic2colour(colour_background_ix & 0xf);
       else
	values.background = greymap[colour_background_ix & 0xf];

      valuemask |= GCBackground;
     }

 XChangeGC(display, gc, valuemask, &values);
#endif
}


void SetFont(GC gc, int fontIndex)
{
 font = fontIndex;
 pFontInfo = Fonts[fontIndex];
 XSetFont(display, gc, pFontInfo->fid);
#if 0
 XSetForeground(display, gc, textColour);
 XSetBackground(display, gc, windowColour);
#endif
 lineHeight = 2 + pFontInfo->max_bounds.ascent + pFontInfo->max_bounds.descent;
 chDescent = pFontInfo->max_bounds.descent;
 chStrike = lineHeight - 2 - (pFontInfo->max_bounds.ascent + chDescent)/2;
 spWidth = XTextWidth(pFontInfo, " ", 1);
 chWidth = XTextWidth(pFontInfo, "ABCabc", 6)/6;
}


void SetEmphFont(GC gc, XFontStruct* pFont, XFontStruct* pNormal)
{
 pFontInfo = pFont;
 XSetFont(display, gc, pFont->fid);
 XSetForeground(display, gc,   textColour);
 XSetBackground(display, gc, windowColour);
 lineHeight = 2 + pNormal->max_bounds.ascent + pNormal->max_bounds.descent;
 chDescent = pNormal->max_bounds.descent;
 chStrike = lineHeight - 2 - (pFontInfo->max_bounds.ascent + chDescent)/2;
 spWidth = XTextWidth(pFontInfo, " ", 1);
 chWidth = XTextWidth(pFontInfo, "ABCabc", 6)/6;
}


/*
 * Check visibility. Find the visible subarea of the rectangle.
 * Actually just two rectangles intersection.
 * If visible, return visible subarea rectangle.
 */
Bool VisibleViewRectangle(ViewRectangle  theView,
			  ViewRectangle  theArea,
			  ViewRectangle* visibleArea)
{
 if (theView.width > 0 && theView.height > 0 &&
     theArea.width > 0 && theArea.height > 0)
   {
    int    left1 = theView.x;
    long    top1 = theView.y;
    int   right1 = left1 + theView.width;
    long bottom1 =  top1 + theView.height;
    int    left2 = theArea.x;
    long    top2 = theArea.y;
    int   right2 = left2 + theArea.width;
    long bottom2 =  top2 + theArea.height;


    if (left1 <  right2 && left2 <  right1 &&
	 top1 < bottom2 &&  top2 < bottom1)
      {
       if (visibleArea)
	 {
	  if (  left2 >   left1)   left1 =   left2;
	  if ( right2 <  right1)  right1 =  right2;
	  if (   top2 >    top1)    top1 =    top2;
	  if (bottom2 < bottom1) bottom1 = bottom2;

	  visibleArea->x = left1;
	  visibleArea->y =  top1;
	  visibleArea->width  =  right1 - left1;
	  visibleArea->height = bottom1 -  top1;
	 }

       return True;
      }
   }

 return False;
}


/*
 * Create document predisplay cache-pixmap.
 */
ViewWindow MakeDocViewCacheWin(int x, long y)
{
 extern  int buf_width;
 extern long buf_height;
 extern ViewWindow DocViewCache;


 if (DocViewCache.win == None)
   {
    DocViewCache.rect.width   = (winBufferWidthN  ?
     min(buf_width,
	 DocViewWin.rect.width * (winBufferWidthN * 2 + 1)) : buf_width);

    DocViewCache.rect.height  = (winBufferHeightN ?
     min(buf_height,
       DocViewWin.rect.height * (winBufferHeightN * 2 + 1)) : buf_height);

    DocViewCache.rect.x = 0;
    if (DocViewCache.rect.width < buf_width)
      {
       if ((DocViewWin.rect.width * winBufferWidthN) < x)
	 {
	  int rightmost = buf_width -
                          DocViewWin.rect.width * (winBufferWidthN + 1);

	  DocViewCache.rect.x = min(x, rightmost) - DocViewWin.rect.width;
	 }
      }

    DocViewCache.rect.y = 0;
    if (DocViewCache.rect.height < buf_height)
      {
       if ((DocViewWin.rect.height * winBufferHeightN) < y)
	 {
	  long topmost = buf_height -
                         DocViewWin.rect.height * (winBufferHeightN + 1);

	  DocViewCache.rect.y = min(y, topmost) - DocViewWin.rect.height;
	 }
      }

    DocViewCache.win = XCreatePixmap(display, DocDispWin,
				     DocViewCache.rect.width,
				     DocViewCache.rect.height,
				     depth);
   }

 return DocViewCache;
}


/*
 * Free document display cache-pixmap.
 */
Bool FreeDocDispCacheWin(void)
{
 extern ViewWindow DocViewCache;

 if (DocViewCache.win != None)
   {
    XFreePixmap(display, DocViewCache.win);
    DocViewCache.win = None;
    DocViewCache.rect.x     = DocViewCache.rect.y      = 0;
    DocViewCache.rect.width = DocViewCache.rect.height = 0;
    return True;
   }
  else
   return False;
}


/*
 * When a new file buffer is created, the first thing to do is to measure
 * the length of the buffer in pixels and set up the scrollbar appropriately.
 * The reference point should be set to the beginning of the buffer.
 *
 * Assumes that pFontInfo has been set up in advance,
 *              and that buffer and hdrlen are ok.
 */
void NewBuffer(Doc* doc)
{
 char *buf;
#ifdef ARENA_DEBUG
 char Iam[] = "NewBuffer";
#endif


 if (doc == NULL)
   {
#ifdef ARENA_DEBUG
    if (ANY_TRACE) Arena_TracePrint(Iam, " the given doc is NULL!\n");
#endif
    return;
   }

 buf = doc->content_buffer;

 if ((buf == NULL) && (doc->image == NULL))
   {
    char* url = doc->url ? doc->url : "???";
#ifdef ARENA_DEBUG	/* QingLong.16-12-96 */
    if (DISPLAY_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " doc given to parse has no contents:\n"
		       "\tdoc = "POINTER_FORMAT",\n"
		       "\tdoc->anchor = "POINTER_FORMAT",\n"
		       "\tdoc->anchor->document = "POINTER_FORMAT",\n"
		       "\tdoc->url =\"%s\",\n"
		       "\tdoc->content_buffer = NULL,\n"
		       "\tdoc->image = "POINTER_FORMAT",\n"
		       "\tdoc->state = 0x%x.\n",
		       doc, doc->anchor, HTAnchor_document(doc->anchor), url,
		       doc->image, doc->state);
#endif
    Announce("Is the \"%s\" empty?", url);
    return;
   }
 /* End ``if (buf == NULL)'' */

#ifdef ARENA_DEBUG
 if (doc->anchor)
   {
    HTAtom* a = HTAnchor_format(HTAnchor_parent((HTAnchor*)(doc->anchor)));

    if (a)
      {
       if (DISPLAY_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam,
			  " doc "POINTER_FORMAT" has type \"%s\".\n",
			  doc, a->name);
      }
     else
      {
       if (ANY_TRACE)
	 Arena_TracePrint(Iam,
			  " doc "POINTER_FORMAT" MIME type UNdefined!\n", doc);
      }
   }
#endif


/* should freeing be taken care of by the library? */
/*
 * if (buffer && buffer != buf) Free(buffer);
 */

 if (doc->state != DOC_PARSING_LOCK) StyleClearDoc(doc);

 buffer = buf;
#if 0
 hdrlen   = doc.hdrlen;
 document = doc.content_type;
#endif
 StartOfLine = buffer + hdrlen;
 DocViewWin.rect.y = 0;
 DocViewWin.rect.x = 0;

 FreeDocDispCacheWin();

 background.child = NULL;
 DocIsIndex = False;  /* clear the HTML searchable flag */
 IdOffset = 0; /* howcome 10/1/95 */

 if (doc->state != DOC_PARSING_LOCK)
   {
#ifdef ARENA_DEBUG	/* QingLong.15-12-96 */
    if (DISPLAY_TRACE && VERBOSE_TRACE)
      Arena_TracePrint(Iam,
		       " doc "POINTER_FORMAT" state is %i, parsing...\n",
		       doc, doc->state);
#endif
    DocDimension(doc, &buf_width, &buf_height);
   }

 /* DisplayDoc() will now take care of correct display...
  * including any necessary offset to show the correct fragment tag.
  */
 if (doc->state == DOC_PROCESSED) DisplayDoc(doc, False);
 return;
}


void DisplaySizeChanged(Doc* theDoc, Bool all)
{
 long h, target;


 FreeDocDispCacheWin();

 if (DocHTML(theDoc))
   {
    ResizeForm();

    DocViewWin.rect.y = 0;
    buf_height = ParseHTML(&buf_width, False);

    if (ViewOffset > 0)
      {
       target = ViewOffset;
       h = buf_height - DocViewWin.rect.height;

       if (h <= 0)
	 target = 0;
        else
	 if (target > h) target = h;

       /* FIXME for fragments */
       DeltaHTMLPosition(theDoc, &(DocViewWin.rect), target);
      }
   }
  else
   if (all || buf_height == 0)
     {
      DocViewWin.rect.y = DocHeight(theDoc, buffer);

      if (IdOffset > 0)
	{
	 target = IdOffset;

	 if (target > buf_height - DocViewWin.rect.height)
	   {
	    target = buf_height - DocViewWin.rect.height;

	    if (target < 0) target = 0;
	   }

	 /* FIXME for fragments */
	 DeltaHTMLPosition(theDoc, &(DocViewWin.rect), target);
        }

      DocDimension(theDoc, &buf_width, &buf_height);
     }

 {
  int max_indent;

  max_indent = buf_width - DocViewWin.rect.width;
  if (max_indent < 0) max_indent = 0;
  if (DocViewWin.rect.x > max_indent) DocViewWin.rect.x = max_indent;
 }

 {
  long max_offset;

  max_offset = buf_height - DocViewWin.rect.height;
  if (max_offset < 0) max_offset = 0;
  if (DocViewWin.rect.y > max_offset) DocViewWin.rect.y = max_offset;
 }

 SetScrollXBarPosition(DocViewWin.rect.x, buf_width);
 SetScrollYBarPosition(DocViewWin.rect.y, buf_height);
}


int LineLength(char *buf)
{
 char *s;
 int len, c;

 s = buf;
 len = 0;

 while ((c = *s++) && c != '\n') ++len;
 if (buf[len-1] == '\r') --len;

 return len;
}


/*
 * DEBUG: return pointer to null terminated line of text from string s
 */
char *TextLine(char *s)
{
 static char buf[128];
 int i, c;

 for (i = 0; i < 127; ++i)
   {
    c = *s;

    if (c == '\0' || c == '\r' || c == '\n') break;

    buf[i] = c;
    ++s;
   }

 buf[i] = '\0';
 return buf;
}


/*
 * Work out how far window has moved relative to document.
 */
long DeltaTextPosition(Doc* theDoc,
		       ViewRectangle* theViewWindowRect_p, long theOffset)
{
                    /* the number of pixels hidden for this line */
 int nClipped = theViewWindowRect_p->y % lineHeight;
 long d1;           /* pixel offset to new top line */
 char *p1 = NULL;   /* points to its text */
#ifdef ARENA_DEBUG
 char Iam[] = "DeltaTextPosition";
#endif


#ifdef ARENA_DEBUG
 if (PAINTSTREAM_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " %d ---> %d.\n", theViewWindowRect_p->y, theOffset);
#endif

 /*
  * Find the text line which intersects/starts from top of window.
  * d1 is pixel offset to new top line, p1 points to its text.
  * theViewWindowRect.y is the pixel offset to the previous top line.
  */
 if (theOffset > theViewWindowRect_p->y)
   {     /* search forward */
    d1 = theViewWindowRect_p->y - nClipped;
    p1 = StartOfLine;

    while (d1 + lineHeight <= theOffset)
      {
       while (*p1)
	 {
	  if (*p1++ != '\n') continue;

	  d1 += lineHeight;
	  break;
	 }

       if (*p1 == '\0') break;         /* this should be unnecessary */
      }
   }
  else  /* search backward */
   {
    d1 = theViewWindowRect_p->y - nClipped;
    p1 = StartOfLine;

    while (d1 > theOffset)
      {
       if (p1 == buffer + hdrlen) break; /* this should be unnecessary */

       /* now move back to start of previous line*/

       --p1;  /* to first point to \n at end of previous line */

       for (;;)   /* and find start of that line */
	 {
	  if (p1 == buffer + hdrlen)
	    break;
	   else
	    if (*--p1 == '\n')
	      {
	       ++p1;
	       break;
	      }
	 }

       /* finally adjust pixel offset to start of that line */
       d1 -= lineHeight;
      }
   }

 {
 /* delta is required movement of window in pixels */
  long delta = theOffset - theViewWindowRect_p->y;

  theViewWindowRect_p->y = theOffset;
  StartOfLine = p1;
  return delta;
 }
}


/*
 * Scroll the view window horizontally. Returns the gap rectangle.
 */
ViewRectangle ScrollViewWinX(GC theGC, ViewWindow theViewWindow,
			     ViewRectangle theScrollArea, int delta)
{
 /* See if change in pixel indent from left hand side of document
  * is small enough to justify a scroll of window contents.
  */
 if (abs(delta) < (2 * theViewWindow.rect.width)/3)
   {
   /* document moves left/right by delta pixels thru window */
    XRectangle scroll_area = { theScrollArea.x - theViewWindow.rect.x,
			       theScrollArea.y - theViewWindow.rect.y,
			       theScrollArea.width, theScrollArea.height };
    ViewRectangle gap = { theScrollArea.x, theScrollArea.y,
			       abs(delta), theScrollArea.height };
     int scroll_src_x  = scroll_area.x,
         scroll_dest_x = scroll_area.x;
    long scroll_y      = scroll_area.y;
    unsigned int scroll_width  = scroll_area.width - gap.width,
                 scroll_height = scroll_area.height;


    if (delta > 0)
      {
      /* move RIGHT */
       scroll_dest_x += delta;
      }
     else
      {
      /* move LEFT */
       scroll_src_x  -= delta;
       gap.x += scroll_width;
      }

    XCopyArea(display, theViewWindow.win, theViewWindow.win, theGC,
	      scroll_src_x,  scroll_y,
	      scroll_width,  scroll_height,
	      scroll_dest_x, scroll_y);

    /* We must note that a copy request has been issued,
     * and avoid further such requests until all resulting
     * GraphicsExpose events are handled as these will repair
     * any holes caused by windows above this one.
     */
    ExposeCount = 1;

    return gap;
   }
  else
   return theViewWindow.rect; /* I.e. request total redisplay */
}


/*
 * Scroll the subarea of the view window vertically. Return the gap rectangle.
 */
ViewRectangle ScrollViewWinY(GC theGC, ViewWindow theViewWindow,
			     ViewRectangle theScrollArea, int delta)
{
 /* See if change in pixel offset from start of document
  * is small enough to justify a scroll of window contents.
  */
 if (abs(delta) < (2 * theViewWindow.rect.height)/3)
   {
   /* document moves up/down by delta pixels thru window */
    XRectangle scroll_area = { theScrollArea.x - theViewWindow.rect.x,
			       theScrollArea.y - theViewWindow.rect.y,
			       theScrollArea.width, theScrollArea.height };
    ViewRectangle gap = { theScrollArea.x,     theScrollArea.y,
			  theScrollArea.width, abs(delta) };
     int scroll_x      = scroll_area.x;
    long scroll_src_y  = scroll_area.y,
         scroll_dest_y = scroll_area.y;
    unsigned int scroll_width  = scroll_area.width,
                 scroll_height = scroll_area.height - gap.height;


    if (delta > 0)
      {
      /* DOWN */
       scroll_dest_y += delta;
      }
     else
      {
      /* UP */
       scroll_src_y  -= delta;
       gap.y += scroll_height;
      }

    XCopyArea(display, theViewWindow.win, theViewWindow.win, theGC,
	      scroll_x,     scroll_src_y,
	      scroll_width, scroll_height,
	      scroll_x,     scroll_dest_y);

    /* We must note that a copy request has been issued,
     * and avoid further such requests until all resulting
     * GraphicsExpose events are handled as these will repair
     * any holes caused by windows above this one.
     */
    ExposeCount = 1;

    return gap;
   }
  else
   return theViewWindow.rect; /* I.e. request total redisplay */
}


/*
 * Display rendered document in(to) the given window.
 */
Bool _displayRenderedDoc(Doc* theDoc,
			 GC theGC, ViewWindow theViewWindow,
			 ViewRectangle theArea)
{
 return (theDoc ?
	 theDoc->already_displayed = DisplayHTML(theDoc,
						 theGC, theViewWindow,
						 theArea)
	 :
	 False);
}


/*
 * Display the text in the buffer appearing in the view defined
 * by the rectangle with upper left origin (x, y) and extent (w, h).
 *
 * The first line of text is pointed to by (char*)StartOfLine
 * and is (theViewWindow.rect.y - nClipped) pixels from
 * the start of the document.
 * There are nClipped pixels of this line hidden above the top
 * of the window, so that the window starts at theViewWindow.rect.y pixels
 * from the start of the document itself.
 */
Bool _displayRawDoc(Doc* theDoc,
		    GC theGC, ViewWindow theViewWindow, ViewRectangle theArea)
{
#ifdef ARENA_DEBUG
 char Iam[] ="_displayRawDoc";
#endif


 if (theDoc)
   {
    int line_number, len, y1;
    char c;
    char *p, *r, lbuf[LineBuffSize];
    int nClipped;            /* the number of pixels hidden for this line */
    Bool TotalSuccess;


    if (theDoc->show_raw ||
	(!(DocHTML(theDoc) || DocImage(theDoc))))
      {
       XRectangle theRect = { theArea.x - theViewWindow.rect.x,
			      theArea.y - theViewWindow.rect.y,
			      theArea.width,
			      theArea.height };

       theDoc->already_displayed = True;

       ParseHTMLerror = 0;
       PaintVersion();

       nClipped = theViewWindow.rect.y % lineHeight;

       XSetClipRectangles(display, theGC,   0, 0, &theRect, 1, Unsorted);
       XSetClipRectangles(display, gc_fill, 0, 0, &theRect, 1, Unsorted);

       y1 = -nClipped;
       line_number = 0;
       p = StartOfLine;

       if (UsePaper)
	 {
	  XSetTSOrigin(display, gc_fill,
		       (-(long)theViewWindow.rect.x) % tileWidth,
		       (-(long)theViewWindow.rect.y) % tileHeight);
	  XFillRectangles(display, theViewWindow.win, gc_fill, &theRect, 1);
	 }

#ifdef ARENA_DEBUG	/* QingLong.22-01-97 */
       if (DISPLAY_TRACE && VERBOSE_TRACE)
	 {
	  char* x = p;
	  int StringLength = 0;
	  while (*x++) StringLength++;
	  Arena_TracePrint(Iam, " string length = %i+1\n", StringLength);
	 }
#endif

       while ((y1 < (int)theArea.height) && (*p != 0))
	 {
	  r = lbuf;
	  len = 0;
	  y1 += lineHeight;

	  while (((c = *p) != 0) &&      /* QingLong.28-10-96 */
		 (len < LineBuffSize))   /* QingLong.29-03-97 */
	    {
	     if (c == '\n') break;
	     p++;

	     switch (c)
	       {
	        case '\t' :
		  if (len < LineBuffSize-1)
		    {
		     while ((len < LineBuffSize) && ((len%TABSIZE) != 0))
		       {
			*(r++) = ' ';
			len++;
		       }
		    }
		  /* End if */
		  continue;
		  break;

	        case '\r' :
		  continue;
		  break;

	        case '\b' :
		  if (len > 0) { len--; r--; };              
		  continue;
		  break;

	        default :
		  *(r++) = c;
		  len++;
		  continue;
		  break;
	       }
	     /* End switch */
	    }
	  /* End while */

	  if (len == LineBuffSize) len--;

	  lbuf[len] = 0;	/* Ensure zero ending */

	  /* howcome 1/7/95 */
	  if (y1 > theArea.y - theViewWindow.rect.y)
	    XDrawString(display, theViewWindow.win, theGC,
			4, y1 - chDescent,
			lbuf, len);

	  if (*p == '\n') p++;
	 }

       TotalSuccess = True;
      }
     else
      TotalSuccess = False;

    XSetClipMask(display, theGC, None);
    XSetClipMask(display, gc_fill, None);

    return TotalSuccess;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to display is NULL.\n");
#endif
    return False;
   }
}


/*
 * Display doc.
 */
Bool _display_doc(Doc* theDoc,
		  GC theGC, ViewWindow theViewWindow, ViewRectangle theArea)
{
 ViewRectangle rect;
#ifdef ARENA_DEBUG
 char Iam[] = "_display_doc";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " Window %d:\n"
		    "\tthe win area: %dx%d%+d%+d,\n"
		    "\tdisplay area: %dx%d%+d%+d.\n",
		    theViewWindow.win,
		    theViewWindow.rect.width, theViewWindow.rect.height,
		    theViewWindow.rect.x, theViewWindow.rect.y,
		    theArea.width, theArea.height, theArea.x, theArea.y);
#endif

 if (VisibleViewRectangle(theViewWindow.rect, theArea, &rect))
   {
    if ((DocHTML(theDoc) || DocImage(theDoc)) && (!theDoc->show_raw))
      return _displayRenderedDoc(theDoc, theGC, theViewWindow, rect);
     else
      return _displayRawDoc(theDoc, theGC, theViewWindow, rect);
   }
  else
   return False;
}


/*
 * Update document displaiing cache window X location, if necessary.
 */
Bool MoveDocViewCacheWinX(Doc* theDoc, int indent)
{
 int x = indent - DocViewWin.rect.width * winBufferWidthN;
 Bool DoMoveIt = False;


 if (indent < DocViewCache.rect.x)
   {
    DoMoveIt = True;
    if (x < 0) x = 0;
   }
  else
   if (indent + DocViewWin.rect.width > DocViewCache.rect.x + DocViewCache.rect.width)
     {
      int max_x = buf_width - DocViewCache.rect.width;

      DoMoveIt = True;
      if (x > max_x) x = max_x;
     }

 if (DoMoveIt)
   {
    int delta = x - DocViewCache.rect.x;

    if ((DoMoveIt = (delta != 0)))
      {
       ViewRectangle gap;

       DocViewCache.rect.x = x;
       gap = ScrollViewWinX(DocViewCacheGC, DocViewCache, DocViewCache.rect,
			    -delta);
       return _display_doc(theDoc, DocDispGC, DocViewCache, gap);
      }
   }

 return DoMoveIt;
}


/*
 * Update document displaiing cache window Y location, if necessary.
 */
Bool MoveDocViewCacheWinY(Doc* theDoc, long offset)
{
 long y = offset - DocViewWin.rect.height * winBufferHeightN;
 Bool DoMoveIt = False;


 if (offset < DocViewCache.rect.y)
   {
    DoMoveIt = True;
    if (y < 0) y = 0;
   }
  else
   if (offset + DocViewWin.rect.height > DocViewCache.rect.y + DocViewCache.rect.height)
     {
      long max_y = buf_height - DocViewCache.rect.height;

      DoMoveIt = True;
      if (y > max_y) y = max_y;
     }

 if (DoMoveIt)
   {
    long delta;


    if (DocImage(theDoc) || (DocHTML(theDoc) && !theDoc->show_raw))
      delta = DeltaHTMLPosition(theDoc, &(DocViewCache.rect), y);
     else
      delta = DeltaTextPosition(theDoc, &(DocViewCache.rect), y);

    if ((DoMoveIt = (delta != 0)))
      {
       ViewRectangle gap;

       gap = ScrollViewWinY(DocViewCacheGC, DocViewCache, DocViewCache.rect,
			    -delta);
       return _display_doc(theDoc, DocDispGC, DocViewCache, gap);
      }
   }

 return DoMoveIt;
}


/*
 * Copy appropriate area from the window cache to the window.
 */
Bool ViewDocArea_from_Cache(Doc* theDoc,
			    GC theGC,
			    ViewWindow theViewWindow, ViewRectangle theArea)
{
 Bool DocViewCache_is_fresh, TotalSuccess = False;
 ViewRectangle rect;
#ifdef ARENA_DEBUG
 char Iam[] = "ViewDocArea_from_Cache";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " View window %d:\n"
		    "\tthe win area: %dx%d%+d%+d,\n"
		    "\tdisplay area: %dx%d%+d%+d.\n",
		    theViewWindow.win,
		    theViewWindow.rect.width, theViewWindow.rect.height,
		    theViewWindow.rect.x,     theViewWindow.rect.y,
		    theArea.width, theArea.height, theArea.x, theArea.y);
#endif


 if ((DocViewCache_is_fresh = (DocViewCache.win == None)))
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " NO view cache window! Creating.\n");
#endif
    MakeDocViewCacheWin(theArea.x, theArea.y);
   }

#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " Doc view cache window %d: %dx%d%+d%+d.\n",
		    DocViewCache.win,
		    DocViewCache.rect.width, DocViewCache.rect.height,
		    DocViewCache.rect.x, DocViewCache.rect.y);
#endif

 if (VisibleViewRectangle(DocViewCache.rect, theArea, &rect))
   {
    TotalSuccess = (DocViewCache_is_fresh ?
		    _display_doc(theDoc,
				 theGC, DocViewCache, DocViewCache.rect)
		    :
		    True);

    if (TotalSuccess)
      XCopyArea(display, DocViewCache.win, theViewWindow.win, theGC,
		theArea.x - DocViewCache.rect.x,
		theArea.y - DocViewCache.rect.y,
		theArea.width, theArea.height,
		theArea.x - theViewWindow.rect.x,
		theArea.y - theViewWindow.rect.y);
   }

 return TotalSuccess;
}


/*
 * Display doc area in the given view window. Do buffering.
 */
Bool ViewDocArea(Doc* theDoc,
		 GC theGC, ViewWindow theViewWindow, ViewRectangle theArea)
{
 ViewRectangle visibleArea;
#ifdef ARENA_DEBUG
 char Iam[] = "ViewDocArea";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " Window %d:\n"
		    "\tthe win area: %dx%d%+d%+d,\n"
		    "\tdisplay area: %dx%d%+d%+d.\n",
		    theViewWindow.win,
		    theViewWindow.rect.width, theViewWindow.rect.height,
		    theViewWindow.rect.x, theViewWindow.rect.y,
		    theArea.width, theArea.height, theArea.x, theArea.y);
#endif

 if (VisibleViewRectangle(theViewWindow.rect, theArea, &visibleArea))
   return ViewDocArea_from_Cache(theDoc, theGC, theViewWindow, visibleArea);
  else
   return False;
}


/*
 * Display doc in the given view window. Do buffering.
 */
Bool ViewDoc(Doc* theDoc,
	     GC theGC, ViewWindow theViewWindow)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ViewDoc";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " Window %d: %dx%d%+d%+d.\n",
		    theViewWindow.win,
		    theViewWindow.rect.width, theViewWindow.rect.height,
		    theViewWindow.rect.x,     theViewWindow.rect.y);
#endif


 if (ViewDocArea_from_Cache(theDoc, theGC, theViewWindow, theViewWindow.rect))
   {
    PaintVersion();
    return True;
   }
  else
   return False;
}


/*
 * Display doc area in default view window. Do buffering.
 */
Bool DisplayDocArea(Doc* theDoc, XRectangle theArea)
{
 ViewRectangle theViewArea = { theArea.x + DocViewWin.rect.x,
			       theArea.y + DocViewWin.rect.y,
			       theArea.width, theArea.height };
#ifdef ARENA_DEBUG
 char Iam[] = "DisplayDocArea";
#endif


#ifdef ARENA_DEBUG
 if (DISPLAY_TRACE)
   Arena_TracePrint(Iam,
		    " %dx%d%+d%+d.\n",
		    theArea.width, theArea.height, theArea.x, theArea.y);
#endif

 return ViewDocArea(theDoc, DocDispGC, DocViewWin, theViewArea);
}


/*
 * Display document in DocViewWin.
 */
Bool DisplayDoc(Doc* theDoc, Bool reparsing)
{
#ifdef ARENA_DEBUG
 char Iam[] = "DisplayDoc";
#endif


 if (theDoc)
   {
    Bool update_position = !reparsing;

    /* Always see if we have a fragment tag specified...
     * If so, get it's position so we can display the proper part of the doc.
     */
    if (theDoc->Fragment)
      {
       update_position = True;
       context->current_history->y = Y_of_Fragment(theDoc,
						   theDoc->Fragment);     
      }

    if (update_position)
      DeltaHTMLPosition(theDoc,
			&(DocViewWin.rect), context->current_history->y);

    if (ViewDoc(theDoc, DocDispGC, DocViewWin))
      {
       if (update_position)
	 {
	  SetScrollXBarPosition(DocViewWin.rect.x, buf_width);
	  SetScrollYBarPosition(DocViewWin.rect.y, buf_height);
	 }

       DisplayScrollBar();
       DisplayUrl(theDoc);

       return True;
      }
    else
      return False;
   }
  else
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to display is NULL.\n");
#endif
    return False;
   }
}


void DisplayAll(Doc* theDoc, Bool reparsing)
{
 DisplayToolBar();
 DisplayStatusLine();
 DisplayIcon();
 DisplayDoc(theDoc, reparsing);
 XFlush(display);
}


/* Move display to place the left of the window
 * at indent pixels from left hand edge.
 */
void MoveXDisplay(Doc* theDoc, int theIndent)
{
 int delta;


 delta = theIndent - DocViewWin.rect.x;
 DocViewWin.rect.x = theIndent;

 /* See if change in pixel offset from start of document
  * is small enough to justify a scroll of window contents.
  */
 if (delta)
   {
    if (DocViewCache.win != None)
      {
       ViewRectangle gap;

       MoveDocViewCacheWinX(theDoc, theIndent);
       gap = ScrollViewWinX(DocDispGC, DocViewWin, DocViewWin.rect, -delta);
       ViewDocArea_from_Cache(theDoc, DocDispGC, DocViewWin, gap);
      }
     else
      {
       MakeDocViewCacheWin(DocViewWin.rect.x, DocViewWin.rect.y);
       _display_doc(theDoc, DocViewCacheGC, DocViewCache, DocViewCache.rect);
       ViewDocArea_from_Cache(theDoc, DocDispGC, DocViewWin, DocViewWin.rect);
      }
   }
}


/* Move display to make h the top of the window
 * in pixels from start of document.
 */
void MoveYDisplay(Doc* theDoc, long theOffset)
{
#ifdef ARENA_DEBUG
 char Iam[] = "MoveYDisplay";
#endif


 if (theDoc)
   {
    long delta;

    /*
     * Remember where we are.
     */
    /* Requires testing of context availability.
     * I.e. if context is allocated.
     */
    /* The doc->request appears is able to be used as such flag.
     * If the displayed document hasn't been requested,
     * then it has no context.
     * The default (compiled-in) document seems to be the only such one.
     */
    if (theDoc && context)
      if (theDoc->request && context->current_history)
	context->current_history->y = theOffset ;
    /* End if */
    /* End if */

    theDoc->already_displayed = False;

    if ((delta = theOffset - DocViewWin.rect.y))
      {
       DocViewWin.rect.y = theOffset;
       if (DocViewCache.win != None)
	 {
	  ViewRectangle gap;

	  MoveDocViewCacheWinY(theDoc, theOffset);
	  gap = ScrollViewWinY(DocDispGC, DocViewWin, DocViewWin.rect, -delta);
	  ViewDocArea_from_Cache(theDoc,
				 DocDispGC, DocViewWin, gap);
	 }
        else
	 {
	  MakeDocViewCacheWin(DocViewWin.rect.x, DocViewWin.rect.y);
	  _display_doc(theDoc,
		       DocViewCacheGC, DocViewCache, DocViewCache.rect);
	  ViewDocArea_from_Cache(theDoc,
				 DocDispGC, DocViewWin, DocViewWin.rect);
	 }
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to display is NULL.\n");
#endif
   }
}


/*
 * Move (horizontally) to x indent.
 */
void MoveToX(Doc* theDoc, int x)
{
 extern ViewWindow DocViewWin;
 extern int buf_width;
 int theW = buf_width - DocViewWin.rect.width;


 if (theW <= 0) return;	/* The doc fits in one page, so it cannot be moved. */

 if (x < 0)
   x = 0;
  else
   if (x > theW) x = theW;

 if (x == DocViewWin.rect.x) return;	/* No move is actually needed. */

 MoveXDisplay(theDoc, x);
 MoveXSlider(x, buf_width);
 return;
}


/*
 * Move (vertically) to y offset.
 */
void MoveToY(Doc* theDoc, long y)
{
 extern ViewWindow DocViewWin;
 extern long buf_height;
 long theH = buf_height - DocViewWin.rect.height;


 if (theH <= 0) return;	/* The doc fits in one page, so it cannot be moved. */

 if (y < 0)
   y = 0;
  else
   if (y > theH) y = theH;

 if (y == DocViewWin.rect.y) return;	/* No move is actually needed. */

 MoveYDisplay(theDoc, y);
 MoveYSlider(y, buf_height);
 return;
}


/*
 * Move (horizontally) to the x indent vicinity (page-width).
 */
void MoveToXpage(Doc* theDoc, int x)
{
 extern ViewWindow DocViewWin;
 extern int buf_width;
 int theW = buf_width - DocViewWin.rect.width;


 if (theW > 0)	/* If the doc fits in one page, it cannot be moved along X. */
   {
    int EffectiveWinWidth = DocViewWin.rect.width - lineHeight;

    if (x < 0) x = 0;
    if (x > theW) x = theW;

    if (EffectiveWinWidth >= 0)
      {
       int dx = x - DocViewWin.rect.x;
                                                         /* the same page */
       if (dx >= 0 && dx <= EffectiveWinWidth) return;
      }

    MoveToX(theDoc, x);
   }

 return;	
}


/*
 * Move (vertically) to the y offset vicinity (page-height).
 */
void MoveToYpage(Doc* theDoc, long y)
{
 extern ViewWindow DocViewWin;
 extern long buf_height;
 long theH = buf_height - DocViewWin.rect.height;


 if (theH > 0)	/* If the doc fits in one page, it cannot be moved along Y. */
   {
    long EffectiveWinHeiht = DocViewWin.rect.height - lineHeight;

    if (y < 0) y = 0;
    if (y > theH) y = theH;

    if (EffectiveWinHeiht >= 0)
      {
       long dy = y - DocViewWin.rect.y;
                                                         /* the same page */
       if (dy >= 0 && dy <= EffectiveWinHeiht) return;
      }

    MoveToY(theDoc, y);
   }

 return;	
}


/* Should these adjust window offset to ensure that nClipped == 0 ?
 * i.e. so that top line is never clipped after MoveUpLine.
 */
void MoveLeftLine(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToX(theDoc, DocViewWin.rect.x - lineHeight);
 /* What the ^%&# lineHeight does here? */
}


void MoveLeftPage(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToX(theDoc,
	 DocViewWin.rect.x - (DocViewWin.rect.width - lineHeight * 2));
 /* What the ^%&# lineHeight does here? */
}


void MoveToLeft(Doc* theDoc)
{
 MoveToX(theDoc, 0);
}


void MoveRightLine(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToX(theDoc, DocViewWin.rect.x + lineHeight);
 /* What the ^%&# lineHeight does here? */
}


void MoveRightPage(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToX(theDoc, DocViewWin.rect.x + (DocViewWin.rect.width - lineHeight * 2));
 /* What the ^%&# lineHeight does here? */
}


void MoveToRight(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern int buf_width;

 MoveToX(theDoc, buf_width - DocViewWin.rect.width);
}


void MoveUpLine(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToY(theDoc, DocViewWin.rect.y - lineHeight);
}


void MoveUpPage(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToY(theDoc,
	 DocViewWin.rect.y - (DocViewWin.rect.height - lineHeight * 2));
}


void MoveToStart(Doc* theDoc)
{
 MoveToY(theDoc, 0);
}


void MoveDownLine(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToY(theDoc, DocViewWin.rect.y + lineHeight);
}


void MoveDownPage(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long lineHeight;

 MoveToY(theDoc,
	 DocViewWin.rect.y + (DocViewWin.rect.height - lineHeight * 2));
}


void MoveToEnd(Doc* theDoc)
{
 extern ViewWindow DocViewWin;
 extern long buf_height;

 MoveToY(theDoc, buf_height - DocViewWin.rect.height);
}



void SlideXDisplay(Doc* theDoc, int slider, int scrollExtent)
{
 double dh;

 /* compute the new pixel offset to top of window */

 dh = ((double)slider * buf_width) / scrollExtent;
 MoveXDisplay(theDoc, (long)(dh + 0.5));
}


void SlideYDisplay(Doc* theDoc, int slider, int scrollExtent)
{
 double dh;

 /* compute the new pixel offset to top of window */

 dh = ((double)slider * buf_height) / scrollExtent;
 MoveYDisplay(theDoc, (long)(dh + 0.5));
}


/*
 * what is the offset from the start of the file to the current line?
 */
long DocHeight(Doc* theDoc, char *buf)
{
#ifdef ARENA_DEBUG
 char Iam[] = "DocHeight";
#endif

 if (theDoc)
   {
    long height;

    if (buf)
      {
       if (DocHTML(theDoc) && !theDoc->show_raw)
	 {
	  height = 0;
	  /* ParseSGML(HEIGHT, &height, &buf, 0, 0, StartOfLine); */
	 }
        else
	 height = ((*buf) ? DocViewWin.rect.y : 0);

       return height;
      }
     else
      {
#ifdef ARENA_DEBUG
       if (DISPLAY_TRACE)
	 Arena_TracePrint(Iam, " ERROR! The buffer is NULL.\n");
#endif
       return 0;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to display is NULL.\n");
#endif
    return 0;
   }
}


/*
 * how big (in pixels) is the file ?
 */
void DocDimension(Doc* doc, int *width, long *height)
{
#ifdef ARENA_DEBUG
 char Iam[] = "DocDimension";
#endif

 if (doc)
   {
    DocStatus doc_state_previuos = doc->state;

    if (doc_state_previuos != DOC_PARSING_LOCK)
      {
       int w;
       char *p, *buf;
       HTAtom* a = NULL;


#ifdef ARENA_DEBUG
       if (STATUS_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam,
			  " doc "POINTER_FORMAT" state %d.\n",
			  doc, doc_state_previuos);
#endif

       doc->state = DOC_PARSING_LOCK;

       if (doc->anchor) a = HTAnchor_format(doc->anchor);

       *width = DocViewWin.rect.width;
       buf = doc->content_buffer;

       if (DocHTML(doc) || (DocRawText(doc)))
	 {
	 /* document is HTML, but we're in source mode */
	  if (doc->show_raw || (a == text_atom))
	    {
	     *height = (*buf ? lineHeight : 0);
	     p = buf;

	     while (*buf)
	       {
		if (*buf++ == '\n')
		  {
		   *height += lineHeight;
		   w = chWidth * (buf - p);
		   p = buf;

		   if (w > *width) *width = w;
		  }
	       }

	     w = chWidth * (buf - p - 1);

	     if (w > *width)  *width = w;

	     /* should we call StyleParse manually here?? */
	    }
	   else
	    {
	    /* we're about to parse an HTML document
	     * (and thereby generate a paint stream)
	     */
	     *height = ParseHTML(width, True);
	    }

	  doc->state = DOC_PROCESSED;
	 }
        else
	 {
	  if (DocImage(doc))
	    {
	    /* the document is of image type we can display internally */
	     if (doc->image == NULL) ProcessLoadedImage(doc);

#ifdef ARENA_DEBUG
	     if (STATUS_TRACE && IMAGE_TRACE && VERBOSE_TRACE)
	       Arena_TracePrint(Iam,
				" image doc "POINTER_FORMAT" state is 0x%x"
				" (previous state was 0x%x).\n",
				doc, doc->state, doc_state_previuos);
#endif
	     if ((doc_state_previuos == DOC_PROCESSED) ||
		 (doc_state_previuos == DOC_LOADED))
	       *height = ParseImage(doc, width);
	     /* End if */

	     if (doc->state == DOC_PARSING_LOCK)
	       doc->state = doc_state_previuos;
	    }
	   else
	    {
	    /* unknown format */
	     if (a && a->name)
	       Announce("Please register application"
			" to handle %s in ~/.mailcap",
			(char *)a->name);
	     /* End ``if (a && a->name)'' */

	     doc->state = doc_state_previuos; 
	    }
	  /*  End ``if (DocImage(doc))''*/
	 }
       /* End ``if (DocHTML(doc) || (DocRawText(doc)))'' */
      }
#ifdef ARENA_DEBUG
     else
      {
       if (STATUS_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam,
			  " doc "POINTER_FORMAT" is locked for parsing.\n",
			  doc);
      }
#endif
   }
#ifdef ARENA_DEBUG
  else
   {
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The given doc is NULL.\n");
   }
#endif
}


/*
 * setup skip table for searching str forwards thru document
 */
void ForwardSkipTable(unsigned char *skip, int len, char *str)
{
 int i;
 unsigned char c;

 for ((i = 0); i < 256; ++i) skip[i] = len;

 for (i = 1; (c = *(unsigned char *)str++); ++i) skip[c] = len - i;
}


/*
 * setup skip table for searching str backwards thru document
 */
void BackwardSkipTable(unsigned char *skip, int len, char *str)
{
 int i;

 for ((i = 0); i < 256; ++i) skip[i] = len;

 str += len;

 for (i = len - 1; i >= 0; --i) skip[(unsigned char) *--str] = i;
}



void ResetFindStringPos(void)
{
 FindNextString = NULL;
}


void FindString(Doc* theDoc, char* theString)
{
#ifdef ARENA_DEBUG
 char Iam[] = "FindString";
#endif


 if (theDoc)
   {
    static char* FindStrVal = NULL;
    char* aString;
    char* p;
    int i, j, c1, c2, patlen, patlen1;
    long len;
    unsigned char skip[256];


    if (Arena_StrLen(theString))
      aString = theString;
     else
      if ((aString = ArenaPromptUser("Find?", FindStrVal, True)))
	{
	 Free(FindStrVal);
	 FindStrVal = aString;
	 FindNextString = NULL;
	}
       else
	return;

    patlen  = Arena_StrLen(aString);
    patlen1 = patlen - 1;
    ForwardSkipTable(skip, patlen, aString);
    len = theDoc->loaded_length;

    p = FindNextString;
    if ((p == NULL) || (p <  buffer) || (p > buffer + len - 2))
      p = StartOfLine;

    i = p - buffer + patlen1;
    j = patlen1;

    while (j >= 0 && i < len && i >= 0)
      {
       c1 = buffer[i];
       c1 = TOLOWER(c1);

       c2 = aString[j];
       c2 = TOLOWER(c2);

       if (isspace(c1)) c1 = ' ';

       if (c1 == c2)
	 {
	  --i;
	  --j;
	  continue;
	 }

       i += patlen - j;    /* to next New char */
       j = patlen1;

       if (i >= len) break;

       c1 = buffer[i];
       c1 = TOLOWER(c1);

       if (isspace(c1)) c1 = ' ';

       i += skip[c1];
      }

    if (++j > 0)
      {
       p = NULL;
       Warn(_("Can't find \"%s\""), aString);
      }
     else
      {
      /* move to start of current line */

       p = buffer + i + patlen1;

       while (i > 0)
	 {
	  if (buffer[i] != '\n')
	    {
	     --i;
	     continue;
	    }

	  ++i;
	  break;
	 }

       /* and display accordingly */

       len = 0;

#if 0
       if (document == HTML_DOCUMENT)
	 {
	  offset = 0;
	  p = StartOfLine;
	  ParseSGML(HEIGHT, &offset, &p, 0, 0, buffer + i);
	  DeltaHTMLPosition(&(theViewWindow.rect), DocViewWin.rect.y + offset);

	  if (AtLastPage())
	    {
	     offset = buf_height - DocViewWin.rect.height + lineHeight;

	     if (offset > buf_height) offset = buf_height - lineHeight;;

	     if (offset < 0) offset = 0;

	     DeltaHTMLPosition(&(theViewWindow.rect), offset);
	    }
	 }
        else
#endif	/* if 0 */
	 {
	  p = buffer;
	  StartOfLine = buffer + i;
	  DocViewWin.rect.y = (*p ? lineHeight : 0);

	  while (*p && p < StartOfLine)
	    {
	     if (*p++ == '\n') DocViewWin.rect.y += lineHeight;
	    }

	  if (DocViewWin.rect.y + DocViewWin.rect.height > buf_height)
	    {
	     len = buf_height - DocViewWin.rect.height;

	     if (len < 0) len = 0;

	     MoveYDisplay(theDoc, len);
	    }
	 }

       FindNextString = p;
       Announce("Found \"%s\", press F3 for next match", aString);
       DisplayDoc(theDoc, False);
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (DISPLAY_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc to dig through is NULL.\n");
#endif
   }
}


/*
 * toggle view between HTML and PLAIN
 */
void ToggleView(Doc* theDoc)
{
#ifdef ARENA_DEBUG
 char Iam[] = "ToggleView";
#endif


 if (theDoc)
   {
    char *p, *start;
    long offset, maxOffset, target;


    if (DocHTML(theDoc))
      {
       if (theDoc->show_raw)
	 {
	  theDoc->show_raw = False;

#if 0
	  SetFont(DocDispGC, IDX_NORMALFONT);
#endif
#if 0
	  SetFont(DocDispGC,
		  GetFont(0, (char*)"courier",
			  F_ENCODE(F_WEIGHT_BOLD, F_SLANT_ROMAN,
				   F_SERIF_DEFAULT),
			  pt2px(18.0 * lens_size_factor)));
#endif
	  targetptr = StartOfLine;
	  DocViewWin.rect.y = 0;
	  buf_height = ParseHTML(&buf_width, False);

	  if (ViewOffset > 0)
	    {
	     target = ViewOffset;

	     if (target > buf_height - DocViewWin.rect.height)
	       {
		target = buf_height - DocViewWin.rect.height;

		if (target < 0) target = 0;
	       }

	     maxOffset = buf_height - DocViewWin.rect.height;

	     if (target > maxOffset && maxOffset >= 0) target = maxOffset;

	     DeltaHTMLPosition(theDoc, &(DocViewWin.rect), target);
	    }
	 }
        else
	 {
	 /* howcome 12/10/94: added support for external editing */
	  theDoc->show_raw = True;

#ifdef STYLE_COLOUR
	  {
#if 0
	   Byte colour_text_ix = rgb2ix(0, 0, 0, 0, False);  /* set colour to black */

	   colour_text_ix = rgb2colour(0, 0, 0);
	   Byte colour_text_ix = rgb2magic(0, 0, 0);  /* set colour to black */
	   if (colour_text_ix < 128)
	     textColour = stdcmap[colour_text_ix];
	    else
	     if (colour_text_ix < 144)
	       textColour = stdcmap[colour_text_ix & 0xF]; /* set colour to black */
#endif
	  }
#endif
#if 0
	  SetFont(DocDispGC, IDX_FIXEDFONT);
#endif
#if 0
	  SetFont(DocDispGC,
		  GetFont(0, str2list("lucidasanstypewriter", "courier"),
			  F_ENCODE(F_WEIGHT_BOLD, F_SLANT_ROMAN,
				   F_SERIF_IGNORE), 14));
#endif

	  /* at this point, we need to set the font for html source.
	   * It will be used to calculate the display height.
	   */
	  FormatElementStart(theDoc, TAG_HTML, NULL, 0);
	  FormatElementStart(theDoc, TAG_HTML_SOURCE, NULL, 0);
	  SetFont(DocDispGC, (int)StyleGet(theDoc, ArenaSI_FONT));

	  p = buffer + hdrlen;
	  start = TopStr(DocViewWin, &background);
	  offset = 0;   /* (*p ? lineHeight : 0); */

	  while (*p && p < start)
	    {
	     if (*p++ == '\n') offset += lineHeight;
	    }

	  DocDimension(theDoc, &buf_width, &buf_height);
	  maxOffset = buf_height - DocViewWin.rect.width;

	  if (offset > maxOffset) offset = maxOffset;

	  DocViewWin.rect.y = 0;
	  StartOfLine = buffer+hdrlen;
	  FormatElementEnd(theDoc);
	  FormatElementEnd(theDoc);
	 }

       DisplayDoc(theDoc, False);
      }
     else
      Warn(_("You can only see the source of HTML documents"));
   }
#ifdef ARENA_DEBUG
 else
  if (DISPLAY_TRACE) Arena_TracePrint(Iam, " ERROR! The doc is NULL!\n");
#endif
}
