/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#include "icewm.h"

YColor::YColor(unsigned short red, unsigned short green, unsigned short blue) {
    fDarker = fBrighter = 0;

    color.red   = red;
    color.green = green;
    color.blue  = blue;
    color.flags = DoRed | DoGreen | DoBlue;

    alloc();
}

YColor::YColor(const char *clr) {
    fDarker = fBrighter = 0;

    XParseColor(app->display(),
                defaultColormap,
                clr,
                &color);
    alloc();
}

void YColor::alloc() {
    if (XAllocColor(app->display(),
                    defaultColormap,
                    &color) == 0)
    {
        int j, ncells;
        long d = 0x7FFFFFFF, d1;
        XColor clr;
        unsigned long pix;
        long d_red, d_green, d_blue;
        unsigned long u_red, u_green, u_blue;

        pix = 0xFFFFFFFF;
        ncells = DisplayCells(app->display(), DefaultScreen(app->display()));
        for (j = 0; j < ncells; j++) {
            clr.pixel = j;
            XQueryColor(app->display(), defaultColormap, &clr);
            
            d_red   = color.red   - clr.red;   d_red   /= 4;
            d_green = color.green - clr.green; d_green /= 4;
            d_blue  = color.blue  - clr.blue;  d_blue  /= 4;

            u_red   = 3UL * (d_red   * d_red);
            u_green = 4UL * (d_green * d_green);
            u_blue  = 2UL * (d_blue  * d_blue);
            
            d1 = u_red + u_blue + u_green;
            
            if (pix == 0xFFFFFFFF || d1 < d) {
                pix = j;
                d = d1;
            }
        }
        if (pix != 0xFFFFFFFF) {
            clr.pixel = pix;
            XQueryColor(app->display(), defaultColormap, &clr);
            /*DBG(("color=%04X:%04X:%04X, match=%04X:%04X:%04X\n",
                   color.red, color.blue, color.green,
                   clr.red, clr.blue, clr.green));*/
            color = clr;
        } 
        if (XAllocColor(app->display(), defaultColormap, &color) == 0)
            if (color.red + color.green + color.blue >= 32768)
                color.pixel = WhitePixel(app->display(),
                                         DefaultScreen(app->display()));
            else
                color.pixel = BlackPixel(app->display(),
                                         DefaultScreen(app->display()));
    }
}

YColor *YColor::darker() { // !!! fix
    if (fDarker == 0) {
        unsigned short red, blue, green;
        
        red = ((unsigned int)color.red) * 2 / 3;
        blue = ((unsigned int)color.blue) * 2 / 3;
        green = ((unsigned int)color.green) * 2 / 3;
        fDarker = new YColor(red, green, blue);
    }
    return fDarker;
}

YColor *YColor::brighter() { // !!! fix
    if (fBrighter == 0) {
        unsigned short red, blue, green;

#define maxx(x) (((x) <= 0xFFFF) ? (x) : 0xFFFF)

        red = maxx(((unsigned int)color.red) * 4 / 3);
        blue = maxx(((unsigned int)color.blue) * 4 / 3);
        green = maxx(((unsigned int)color.green) * 4 / 3);

        fBrighter = new YColor(red, green, blue);
    }
    return fBrighter;
}

YFont::YFont(const char *name) {
#ifndef I18N
    font = XLoadQueryFont(app->display(), name);
    if (font == 0)  {
        fprintf(stderr, "Could not load font '%s'.", name);
        font = XLoadQueryFont(app->display(), "fixed");
    }
#else
    char **missing, *def_str;
    int missing_num;
    font = XCreateFontSet(app->display(), name, &missing, &missing_num, &def_str);
    if (missing_num) {
	int i;
	fprintf(stderr, "missing fontset in loading %s\n", name);
	for (i = 0; i < missing_num; i++) fprintf(stderr, "%s\n", missing[i]);
	XFreeStringList(missing);
    }
    XFontSetExtents *extents = XExtentsOfFontSet(font);
    fontascent = -extents->max_logical_extent.y;
    fontdescent = extents->max_logical_extent.height - fontascent;
#endif
}

YFont::~YFont() {
#ifndef I18N
    XFreeFont(app->display(), font);
#else
    XFreeFontSet(app->display(), font);
#endif
}

Graphics::Graphics(YWindow *window) {
    XGCValues gcv;

    display = app->display();
    drawable = window->handle();

    gc = XCreateGC(display, drawable, 0, &gcv);
}

Graphics::Graphics(YPixmap *pixmap) {
    XGCValues gcv;

    display = app->display();
    drawable = pixmap->pixmap();

    gc = XCreateGC(display, drawable, 0, &gcv);
}

Graphics::~Graphics() {
    XFreeGC(display, gc);
}

void Graphics::drawPixmap(YPixmap *pix, int x, int y) {
    if (pix->mask())
        drawClippedPixmap(pix->pixmap(),
                          pix->mask(),
                          0, 0, pix->width(), pix->height(), x, y);
    else
        XCopyArea(display, pix->pixmap(), drawable, gc,
                  0, 0, pix->width(), pix->height(), x, y);
}

void Graphics::drawClippedPixmap(Pixmap pix, Pixmap clip,
                                 int x, int y, int w, int h, int toX, int toY)
{
    XGCValues gcv;

    gcv.clip_mask = clip;
    gcv.clip_x_origin = toX;
    gcv.clip_y_origin = toY;
    XChangeGC(display, clipPixmapGC,
              GCClipMask|GCClipXOrigin|GCClipYOrigin, &gcv);
    XCopyArea(display, pix, drawable, clipPixmapGC,
              x, y, w, h, toX, toY);
    gcv.clip_mask = None;
    XChangeGC(display, clipPixmapGC, GCClipMask, &gcv);
}

void Graphics::draw3DRect(int x, int y, int w, int h, bool raised) {
    YColor *back = getColor();
    YColor *bright = back->brighter();
    YColor *dark = back->darker();
    YColor *t = raised ? bright : dark;
    YColor *b = raised ? dark : bright;
    
    setColor(t);
    drawLine(x, y, x + w, y);
    drawLine(x, y, x, y + h);
    setColor(b);
    drawLine(x, y + h, x + w, y + h);
    drawLine(x + w, y, x + w, y + h);
    setColor(back);
    drawPoint(x + w, y);
    drawPoint(x, y + h);
}

void Graphics::drawBorderW(int x, int y, int w, int h, bool raised) {
    YColor *back = getColor();
    YColor *bright = back->brighter();
    YColor *dark = back->darker();

    if (raised) {
        setColor(bright);
        drawLine(x, y, x + w - 1, y);
        drawLine(x, y, x, y + h - 1);
        setColor(black);
        drawLine(x, y + h, x + w, y + h);
        drawLine(x + w, y, x + w, y + h);
        setColor(dark);
        drawLine(x + 1, y + h - 1, x + w - 1, y + h - 1);
        drawLine(x + w - 1, y + 1, x + w - 1, y + h - 1);
    } else {
        setColor(bright);
        drawLine(x + 1, y + h, x + w, y + h);
        drawLine(x + w, y + 1, x + w, y + h);
        setColor(black);
        drawLine(x, y, x + w, y);
        drawLine(x, y, x, y + h);
        setColor(dark);
        drawLine(x + 1, y + 1, x + w - 1, y + 1);
        drawLine(x + 1, y + 1, x + 1, y + h - 1);
    }
    setColor(back);
}

// doesn't move... needs two pixels on all sides for up and down
// position.
void Graphics::drawBorderM(int x, int y, int w, int h, bool raised) {
    YColor *back = getColor();
    YColor *bright = back->brighter();
    YColor *dark = back->darker();

    if (raised) {
        setColor(bright);
        drawLine(x + 1, y + 1, x + w, y + 1);
        drawLine(x + 1, y + 1, x + 1, y + h);
        drawLine(x + 1, y + h, x + w + 1, y + h);
        drawLine(x + w, y + 1, x + w, y + h);
        
        setColor(dark);
        drawLine(x, y, x + w - 1, y);
        drawLine(x, y, x, y + h);
        drawLine(x, y + h - 1, x + w - 1, y + h - 1);
        drawLine(x + w - 1, y, x + w - 1, y + h - 1);
        
        ///  how to get the color of the taskbar??
        setColor(back);
        
        drawPoint(x, y + h);
        drawPoint(x + w, y);
    } else {
        setColor(dark);
        drawLine(x, y, x + w - 1, y);
        drawLine(x, y, x, y + h - 1);

        setColor(bright);
        drawLine(x + 1, y + h, x + w, y + h);
        drawLine(x + w, y + 1, x + w, y + h);
        
        setColor(back);
        drawRect(x + 1, y + 1, x + w - 2, y + h - 2);
        //drawLine(x + 1, y + 1, x + w - 1, y + 1);
        //drawLine(x + 1, y + 1, x + 1, y + h - 1);
        //drawLine(x, y + h - 1, x + w - 1, y + h - 1);
        //drawLine(x + w - 1, y, x + w - 1, y + h - 1);
        
        ///  how to get the color of the taskbar??
        drawPoint(x, y + h);
        drawPoint(x + w, y);
    }
}

void Graphics::drawBorderG(int x, int y, int w, int h, bool raised) {
    YColor *back = getColor();
    YColor *bright = back->brighter();
    YColor *dark = back->darker();

    if (raised) {
        setColor(bright);
        drawLine(x, y, x + w - 1, y);
        drawLine(x, y, x, y + h - 1);
        setColor(dark);
        drawLine(x, y + h, x + w, y + h);
        drawLine(x + w, y, x + w, y + h);
        setColor(black);
        drawLine(x + 1, y + h - 1, x + w - 1, y + h - 1);
        drawLine(x + w - 1, y + 1, x + w - 1, y + h - 1);
    } else {
        setColor(bright);
        drawLine(x + 1, y + h, x + w, y + h);
        drawLine(x + w, y + 1, x + w, y + h);
        setColor(black);
        drawLine(x, y, x + w, y);
        drawLine(x, y, x, y + h);
        setColor(dark);
        drawLine(x + 1, y + 1, x + w - 1, y + 1);
        drawLine(x + 1, y + 1, x + 1, y + h - 1);
    }
    setColor(back);
}

void Graphics::drawCenteredPixmap(int x, int y, int w, int h, YPixmap *pixmap) {
    int r = x + w;
    int b = y + h;
    int pw = pixmap->width();
    int ph = pixmap->height();

    drawOutline(x, y, r, b, pw, ph);
    drawPixmap(pixmap, x + w / 2 - pw / 2, y + h / 2 - ph / 2);
}

void Graphics::drawOutline(int l, int t, int r, int b, int iw, int ih) {
    if (l + iw >= r && t + ih >= b)
        return ;

    int li = (l + r) / 2 - iw / 2;
    int ri = (l + r) / 2 + iw / 2;
    int ti = (t + b) / 2 - ih / 2;
    int bi = (t + b) / 2 + ih / 2;

    if (l < li)
        fillRect(l, t, li - l, b - t);

    if (ri < r)
        fillRect(ri, t, r - ri, b - t);

    if (t < ti)
        if (li < ri)
            fillRect(li, t, ri - li, ti - t);

    if (bi < b)
        if (li < ri)
            fillRect(li, bi, ri - li, b - bi);
}

void Graphics::repHorz(YPixmap *pixmap, int x, int y, int w) {
    int pw = pixmap->width();
    int ph = pixmap->height();

    while (w > 0) {
        XCopyArea(display, pixmap->pixmap(), drawable, gc,
                  0, 0, (w > pw ? pw : w), ph, x, y);
        x += pw;
        w -= pw;
    }
}

void Graphics::repVert(YPixmap *pixmap, int x, int y, int h) {
    int pw = pixmap->width();
    int ph = pixmap->height();

    while (h > 0) {
        XCopyArea(display, pixmap->pixmap(), drawable, gc,
                  0, 0, pw, (h > ph ? ph : h), x, y);
        y += ph;
        h -= ph;
    }
}


void Graphics::drawArrow(int direction, int style, int x, int y, int size) {
    XPoint points[3];

    switch (direction) {
    case 0:
        points[0].x = x;
        points[0].y = y;
        points[1].x = x + ((style == 0) ? size / 2 : size);
        points[1].y = y + size / 2;
        points[2].x = x;
        points[2].y = y + size;
        break;
    default:
        return ;
    }
    switch (style) {
    case 0:
        fillPolygon(points, 3, Convex, CoordModeOrigin);
        break;
    case 1:
    case -1:
        {
            YColor *back = getColor();
            YColor *c1 = (style == 1) ? black /*back->darker()*/ : back->brighter();
            YColor *c2 = (style == 1) ? back->brighter() : black; // back->darker();

            setColor(c1);
            drawLine(points[0].x, points[0].y, points[1].x, points[1].y);
            drawLine(points[0].x, points[0].y, points[2].x, points[2].y);
            setColor(c2);
            drawLine(points[2].x, points[2].y, points[1].x, points[1].y);
        }
        break;
    }
}

