// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: dpointer.h,v 1.3 1997/08/18 20:49:34 tom Exp $
/* ######################################################################
   
   Double Smart Pointer Class - Reference counted pointer to reference
                                counted pointer implementation

   This source is placed in the Public Domain, do with it what you will
   It was originally written by Brian C. White.
   
   ##################################################################### */
									/*}}}*/
// Header section: pkglib
#ifndef PKGLIB_DPOINTER_H
#define PKGLIB_DPOINTER_H

#include <pkglib/pointer.h>

/* Double Smart-Pointer template definition
  
   This template allows for making smart double-pointers to an arbitrary type.
   It also overloads the standard pointer operators to allow them to be used  
   normally. */
class DPointer 
{
   protected:
   IPtr <Pointer> dPtr; // double pointer to data
   
   public:

   // Assignment 
   DPointer& operator= (const DPointer& rhs);
   DPointer& operator= (const Pointer& rhs);
   
   // Comparision
   bool operator ==(const DPointer& rhs) const;
   bool operator !=(const DPointer& rhs) const;
   bool operator ==(const Pointer& rhs) const;
   bool operator !=(const Pointer& rhs) const;
   bool operator !(void) const;
   bool isValid(void) const;
   bool isNull(void)  const;
   
   // Casting
   operator const void*() const {return isValid()?(const void *)*dPtr : NULL;};
   operator const Pointer&() const {return *dPtr;};
   operator void*() {return(isValid() ? (void*)*dPtr : NULL);};
   operator Pointer&() {return *dPtr;};

   // Allocation
   void malloc(size_t size) {(*dPtr).malloc(size);};
   void free(void) {(*dPtr).free();};
   void resize(size_t o, size_t n) {(*dPtr).resize(o,n);};

   // Construct/destruct
   DPointer(void);
   DPointer(const DPointer& rhs);
   ~DPointer(void);
};

/* Double Smart-Pointer template definition
  
   This template allows for making smart double-pointers to an arbitrary type.
   It also overloads the standard pointer operators to allow them to be used  
   normally. */
template <class Type> class SDPtr : public DPointer 
{
   void malloc(size_t size) {DPointer::malloc(size);};
   void free(void) {DPointer::free();};
   void resize(size_t o,size_t n) {DPointer::resize(o,n);};
   
   public:

   // Dereferencing
   const Type& operator *(void) const {return *(const Type*)(const void*)(*dPtr);}
   const Type* operator->(void) const {return  (const Type*)(const void*)(*dPtr);}
   Type& operator* (void) {return *(Type*)(void*)(*dPtr);}
   Type* operator->(void) {return (Type*)(void*)(*dPtr);}
   
   // Casting
   operator const Type*() const {return (const Type*)(const void*)(*dPtr);};
   operator const SPtr<Type>&() const {return *(const SPtr<Type>*)(&(*dPtr));};
   operator const Pointer&() const {return *(const Pointer*)(&(*dPtr));};
   operator Type*() {return (Type*)(void*)(*dPtr);};
   operator SPtr<Type>&() {return *(SPtr<Type>*)(&(*dPtr));};
   operator Pointer&() {return *(Pointer*)(&(*dPtr));}

   // Assignment
   SDPtr<Type> &operator =(const SDPtr<Type>& rhs);
   
   // Construction
   void New(void);
   void New(const Type& rhs);
   void Delete(void);

   SDPtr(void) : DPointer() {};
   SDPtr(const SDPtr& rhs) : DPointer(rhs) {};
   ~SDPtr(void) {Delete();}   
   
   // These variable-size structure routines simply call the appropriate
   // routine of the base smart-pointer.
   /* Removed because of a casting bug
   void newSlots(unsigned int Number) {((SPtr<Type>&)(*this)).newSlots(Number);}
   unsigned int slotCount(void) {return ((SPtr<Type>&)(*this)).slotCount();}
   void resizeSlots(unsigned int Number){((SPtr<Type>&)(*this)).resizeSlots(Number);}
   unsigned int compactSlots(void)   {return ((SPtr<Type>&)(*this)).compactSlots();}*/
};

/* Constant Double Smart-Pointer template definition
  
   This template allows for making smart double-pointers to an arbitrary type.
   It also overloads the standard pointer operators to allow them to be used  
   normally. This is similar to the CSPtr. */
template <class Type> class CSDPtr : public SDPtr<Type> 
{
   public:

   // Casting
   const Type &operator *(void) const {return *(const Type*)(const void*)(*dPtr);};
   const Type *operator->(void) const {return  (const Type*)(const void*)(*dPtr);};
   operator const Type*()  const {return (const Type*)(const void*)(*dPtr);};
   
   // Change operator
   Type *change(void)
   { 
      dPtr.setPFlag(PTRF_CHANGED);
      return((Type*)(void*)(*dPtr));
   }

   CSDPtr(void) : SDPtr<Type>() {};
   CSDPtr(const SDPtr<Type>& rhs) : SDPtr<Type>(rhs) {};
};

#include <pkglib/dpointer.icc>

#endif
