{-
ToDo:
* Almost every use of "B.mov" in the examples uses an argument of the form
  "repeat (x,y)".  We should either introduce a new ADT for sequences of
  arbitrary objects (not just sequences of pictures and functions) or we
  should add a special case for when the argument of B.mov happens to
  be "repeat (x,y)".
* We treat Poly, Pic and Movie as ADTs - but haven't gotten round to doing
  this for Behaviours yet.
-}


{-**********************************************************************
  MODULE BEHAVIOUR

    This module defines the basic Behaviours available to manipulate
  Movies. These functions can either be used directly, or used to
  easily create personalized Behaviours (see Defaults).
    There are the Behaviours that affect one Movie, which are mov,movto
  circ_mov,scale,scale_rel,rot,flip and set_color. These change some 
  aspect of the movie over time.
    There are functions that combine several movies into one, namely
  bseq,bSeq,bpar and bPar.
    Some functions modify the Behaviours. These are do, rpt and forever.
  They put limits on how long the Behaviour is. 
    Finally, there are the functions that apply the Behaviours to a Movie.
  These are apply and while. Apply applies a Behaviour to a Movie until
  it runs out of Movie or Behaviour. While takes a conditional and
  applies the Behaviour to it until that condition is fullfilled.

***********************************************************************-}

module Behaviour(
	Behaviour,
	mov, movto, circ_mov, scale, scale_rel, rot, flip,
	set_Color, 
	bseq, bSeq, bpar, bPar, 
	takeB, rpt, forever, 
	apply, while 
        )  where

import Prelude hiding (flip)
import Color(
	Color
	)
import Vector(
	Vec, VecFloat,
	vftov,
	vmin
	)
import Picture(Pic)
import qualified Picture as Pic
{-
	(
	flip, mov, movto,
	scale, scale_rel,
	twist, rot,
	set_Color
	)
-}
import Movie(
	Movie, 
	picsOf, mkMovie
	)
import Utility(
	power
	)

{-
  ADRNote: These are often infinite lists - 
  but some operations (eg bseq) are for finite lists.
-}
type Behaviour = [Pic -> Pic]

  -- takes a movie and flips it around the x-origin using flip
flip :: Behaviour
flip = repeat Pic.flip

  -- mov takes a list of Vec's and applies each Pic-to-Pic in the Behaviour
  -- list to its corresponding Vec, and gives back a new Behaviour
mov :: [Vec] -> Behaviour
mov = map Pic.mov

  -- movto creates a list of Pic-to-Pic Behaviours that move each Pic to its 
  -- corresponding Vec
movto :: [Vec] -> Behaviour
movto = map Pic.movto

  -- produces a Behaviour that produces movement in a circle, taking
  -- the radius and the increment as arguments.
circ_mov :: Float -> Float -> Behaviour
circ_mov r inc = 
  mov [ vftov (r * cos theta, r * sin theta) | theta <- [ 0.0, inc .. ] ]


  -- scale outputs a list of Pic-to-Pic's that scale according to its 
  -- corresponding Int in the input list
scale :: [Int] -> Behaviour
scale = map Pic.scale

  -- scale_rel does the same thing, but centers on the lower-left corner of
  -- the image
scale_rel :: Vec -> [Int] -> Behaviour
scale_rel v = map (Pic.scale_rel v)

  -- twist outputs a list of Behaviours that rotate each pick by its 
  -- corresponding Float in the list
twist :: [Float] -> Behaviour
twist = map Pic.twist

  -- set_color takes a list of Colors, and returns a list of Pic-to-Pic's
  -- that change to the corresponding color in the list
set_Color :: [Color] -> Behaviour
set_Color = map Pic.set_Color

-- ADRNote: I suspect both lists are intended to be infinite!
  -- rot produces behaviours rotating by the Float, around the point
  -- of the Vec, both provided by lists.
rot :: [Vec] -> [Float] -> Behaviour
rot = zipWith Pic.rot

  -- bseq takes two Behaviours and combines them into one, in sequence. 
  -- It first applies all of the first Behaviour, then all of the second
bseq :: Behaviour -> Behaviour -> Behaviour
bseq ps qs = ps ++ qs

  -- bSeq takes a list of Behaviour and makes them into one Behaviour, in
  -- sequence.
bSeq :: [Behaviour] -> Behaviour
bSeq = foldr bseq []

  -- bpar takes two behaviours and applies them both simultaneously,
  -- producing a list of Pic-to-Pic's, each one made up of a function
  -- from the first list combined with a function from the second list
bpar :: Behaviour -> Behaviour -> Behaviour
bpar [] qs = qs
bpar ps [] = ps
bpar (p:ps) (q:qs) = (p.q):(bpar ps qs)

  -- bPar takes a list of Behaviours and makes them all into one Behaviour,
  -- in paralell
bPar :: [Behaviour] -> Behaviour
bPar = foldr bpar []

  -- takes the first n Pic-to-Pics in a Behaviour and returns that Behaviour 
takeB :: Int -> Behaviour -> Behaviour
takeB n f = take n f

  -- applies bseq to the list of behaviours, so that the nth element of
  -- the returned list has n-1 behaviours in it, applied in sequence
rpt :: Int -> Behaviour -> Behaviour
rpt n f = power n (bseq f) []

  -- takes the behaviour and applies all the behaviours up the nth element
  -- to the nth element, in an infinite list
forever :: Behaviour -> Behaviour
forever f = f `bseq` forever f

  -- takes a behaviour, applies each function to a Pic in a Movie and returns
  -- the new Movie
apply :: Behaviour -> Movie -> Movie
apply fs m = mkMovie $ zipWith ($) fs (picsOf m)

  -- applies the Behaviour to the Movie until the condition is fullfilled,
  -- then returns the movie to that point
while :: (Pic -> Bool) -> Behaviour -> Movie -> Movie
while p fs m = mkMovie $ zipWith ($) fs (takeWhile p (picsOf m))

