module UrlEncoded where

{-
Decoding application/x-www-form-urlencoded data (last modified: Friday, March 07, 1997)

Erik Meijer (erik@cs.ruu.nl)
-}

import Common
import Mime
import Parsing
import Pretty

{-
The MIME type application/x-www-form-urlencoded is used to encode
tables of (name, value)-pairs that are transmitted from the client to
the server.
-}

newtype ApplicationX_Www_Form_UrlEncoded
 = URLEncoded [(String,String)]

instance Read ApplicationX_Www_Form_UrlEncoded where
   readsPrec _ e = papply (do{e <- env; return (URLEncoded e)}) e

instance Show ApplicationX_Www_Form_UrlEncoded where
   showsPrec r env = showsPrec r (ppEnv env)

instance Mime ApplicationX_Www_Form_UrlEncoded where
   mimeType _ = "application/x-www-form-urlencoded"

{-
An URL encoded value consist of a sequence of
zero or more name "=" value pairs separated by "&"

Env ::= [Name "=" Value {"&" Name "=" Value}]

Names and values are URL-encoded,
according to the following table

   character | encoding
   ----------|---------
    ' '      | '+'
    '<'      | "%XX"
     c       | "%"hexval(ord c)

-}

urlDecode :: String -> [(Name,Value)]
urlDecode s
 = case readsPrec 0 s of
     [] -> []
     ((URLEncoded e,_):_) -> e

env
 = do{ n <- urlEncoded
     ; string "="
     ; v <- urlEncoded
     ; return (n,v)
     } `sepby` (string "&")

urlEncoded
 = many ( alphanum ++ extra ++ safe
         ++ do{ char '+' ; return ' '}
         ++ do{ char '%'
              ; d <- hexadecimal
              ; return $ chr (hex2int d)
              }
         )

extra
 = sat (`elem` "!*'(),")

safe
 = sat (`elem` "$-_.")

hexadecimal :: Parser HexString
hexadecimal
 = do{ d1 <- hexdigit
     ; d2 <- hexdigit
     ; return [d1,d2]
     }

type HexString = String

hex2int :: HexString -> Int
hex2int ds
 = foldl (\n d -> n*16+d) 0 (map (toInt.toUpper) ds)
   where
      toInt d | isDigit d = ord d - ord '0'
      toInt d | isHex d   = (ord d - ord 'A') + 10
--
-- Pretty printing URL encoded name=value pairs
--

ppEnv (URLEncoded [])
 = text ""
ppEnv (URLEncoded (e:es))
 = sep (ppAssoc e : [ text "&" <+> ppAssoc e | e <- es ])
   where
     ppAssoc (n , v) = text n <+> text "=" <+> text v

{-
A function to do URL encoding and proving its correctness might be a
nice exercise for the book.

We don't usually need it for CGI scripts though. The browser does the
encoding and the CGI script does the decoding.
-}
