\subsection{Midi-File Datatypes and I/O Functions}

\begin{verbatim}

> module MidiFile(
>	 MidiFile(..), Division(..), Track, MFType, MEvent(..), DeltaTime,
>	 MPitch, Velocity, ControlNum, PBRange, ProgNum, Pressure,
>	 MidiChannel, ControlVal, 
>	 MidiEvent(..),
>	 MTempo, SMPTEHours, SMPTEMins, SMPTESecs, SMPTEFrames, SMPTEBits,
>	 MetaEvent(..),
>	 KeyName(..),
>	 midiFileToString,
>	 outputMidiFile
>	 ) where
> 
> import Monads (Output, runO, outO)
> import HaskoreUtils (unlinesS, rightS, concatS)
> import Maybe (fromJust)
> import IOExtensions (readBinaryFile, writeBinaryFile)

\end{verbatim} 

{\tt OutputMidiFile} is the main function for writing {\tt MidiFile}
values to an actual file; its first argument is the filename:
\begin{verbatim} 

> outputMidiFile :: String -> MidiFile -> IO ()
> outputMidiFile fn mf = writeBinaryFile fn (midiFileToString mf)

\end{verbatim} 

\begin{exercise} Take as many examples as you like from the previous
sections, create one or more {\tt UserPatchMaps}, write the examples
to a file, and play them using a conventional Midi player.
\end{exercise}
Appendix \ref{test-functions} defines some functions which should make
the above exercise easier.  Appendices \ref{examples}, \ref{chick},
and \ref{self-similar} contain more extensive examples.

Midi files are first converted to a monadic string computation using
the function {\tt outMF}, and then "executed" using
{\tt runM :: MidiWriter a -> String}.

\begin{verbatim} 

> midiFileToString :: MidiFile -> String
> midiFileToString = runM . outMF
>
> outMF :: MidiFile -> MidiWriter ()
> outMF (MidiFile mft divisn trks) = 
>   outChunk "MThd" (
>     out2 mft                  >> -- format (type 0, 1 or 2)
>     out2 (length trks)        >> 
>     outputDivision divisn        -- time unit
>   ) >>
>   outputTracks trks

\end{verbatim} 

The remaining code in this section is not very well documented.  Sorry
about that!

\begin{verbatim}\tiny

> -- The datatypes for Midi Files and Midi Events
> ------------------------------------------------------------------------
> data MidiFile = MidiFile MFType Division [Track] deriving Show
> 
> data Division = Ticks Int | SMPTE Int Int
>      deriving Show
> 
> type Track  = [MEvent]
> type MFType = Int
> 
> data MEvent = MidiEvent DeltaTime MidiEvent
>             | MetaEvent DeltaTime MetaEvent
>      deriving Show
> 
> type DeltaTime  = Int
> 
> -- Midi Events
> 
> type MPitch     = Int
> type Velocity   = Int
> type ControlNum = Int
> type PBRange    = Int
> type ProgNum    = Int
> type Pressure   = Int
> type MidiChannel= Int
> type ControlVal = Int
> 
> data MidiEvent = NoteOff    MidiChannel MPitch Velocity
>                | NoteOn     MidiChannel MPitch Velocity
>                | PolyAfter  MidiChannel MPitch Pressure
>                | ProgChange MidiChannel ProgNum
>                | Control    MidiChannel ControlNum ControlVal
>                | PitchBend  MidiChannel PBRange
>                | MonoAfter  MidiChannel Pressure
>      deriving Show
> 
> -- Meta Events
> 
> type MTempo      = Int
> type SMPTEHours  = Int
> type SMPTEMins   = Int
> type SMPTESecs   = Int
> type SMPTEFrames = Int
> type SMPTEBits   = Int
> 
> data MetaEvent = SequenceNum Int
>                | TextEvent String
>                | Copyright String
>                | TrackName String
>                | InstrName String
>                | Lyric String
>                | Marker String
>                | CuePoint String
>                | MIDIPrefix MidiChannel
>                | EndOfTrack
>                | SetTempo MTempo
>                | SMPTEOffset SMPTEHours SMPTEMins SMPTESecs SMPTEFrames SMPTEBits
>                | TimeSig Int Int Int Int
>                | KeySig KeyName Bool
>                | SequencerSpecific [Int]
>                | UserEvent
>      deriving Show
> 
> -- Enumerated type useful for Key Signatures
> 
> data KeyName = KeyCf | KeyGf | KeyDf | KeyAf | KeyEf | KeyBf | KeyF
>              | KeyC | KeyG | KeyD | KeyA | KeyE | KeyB | KeyFs | KeyCs
>              deriving (Eq, Ord, Ix, Enum, Show)
> 
> -- A parser monad
> ------------------------------------------------------------------------
> data Parser s a = P (s -> Maybe (a,s))
>
> unP :: Parser s a -> (s -> Maybe (a,s))
> unP (P a) = a
> 
> -- Access to state
> tokenP  :: (s -> Maybe (a,s)) -> Parser s a
> runP    :: Parser s a -> s -> Maybe (a,s)
> 
> tokenP get   = P $ get
> runP m s     = (unP m) s
> 
> instance Monad (Parser s) where
>   m >>= k  = P $ \ s -> do { (a,s') <- unP m s; unP (k a) s' }
>   m >>  k  = P $ \ s -> do { (_,s') <- unP m s; unP k     s' }
>   return a = P $ \ s -> return (a,s)
> 
> instance MonadZero (Parser s) where
>   zero = P $ \ s -> zero
> 
> instance MonadPlus (Parser s) where
>   p ++ q = P $ \ s -> unP p s ++ unP q s
> 
> -- Wadler's force function
> force             :: Parser s a -> Parser s a
> force (P p)        = P $ \ s -> let x = p s in
> 				  Just (fst (fromJust x), snd (fromJust x))
> 
>
> zeroOrMore        :: Parser s a -> Parser s [a]
> zeroOrMore p       = force (oneOrMore p ++ return [])
> 
> oneOrMore         :: Parser s a -> Parser s [a]
> oneOrMore p        = do {x <- p; xs <- zeroOrMore p; return (x:xs)}
> 
> -- The MidiReader monad
> ------------------------------------------------------------------------
> type MidiReader a = Parser String a
> 
> getCh :: MidiReader Char
> getCh = tokenP myHead
>   where  myHead []     = Nothing
>          myHead (c:cs) = Just (c,cs)
>  
> getN :: Int -> MidiReader String
> getN n = tokenP (Just . splitAt n)
> 
> getAll :: MidiReader String
> getAll = tokenP (\s -> return (s,""))
> 
> get1 :: MidiReader Int
> get1 = getCh >>= \ c ->
>        return (fromEnum c)
> 
> get2 :: MidiReader Int
> get2 = get1 >>= \ x1 ->
>        get1 >>= \ x2 ->
>        return (polynomial 256 [x1,x2])
> 
> get3 :: MidiReader Int
> get3 = get1 >>= \ x1 ->
>        get1 >>= \ x2 ->
>        get1 >>= \ x3 ->
>        return (polynomial 256 [x1,x2,x3])
> 
> get4 :: MidiReader Int
> get4 = get1 >>= \ x1 ->
>        get1 >>= \ x2 ->
>        get1 >>= \ x3 ->
>        get1 >>= \ x4 ->
>        return (polynomial 256 [x1,x2,x3,x4])
> 
> get7 :: MidiReader Char
> get7 = getCh               >>= \ c ->
>        guard (c < '\128')  >>
>        return c
> 
> getVar :: MidiReader Int
> getVar = getVarAux 0
>   where getVarAux a = get1 >>= \ n ->
>                       let a' = a*128 + n
>                       in if n < 128
>                          then return a'
>                          else getVarAux a'
> 
> -- polynomial b [x,y,z] = x*b^2 + y*b^1 + z*b^0 
> polynomial base xs = foldl (\a x -> base*a+x) 0 xs
> 
> -- Note: the following parsers do not combine directly into a single
> -- master parser.  Rather, they should be used to chop parts of a midi
> -- file up into chunks of bytes which can be taken apart using 
> -- parsers for those parts.
> 
> -- Chop a Midi file into chunks returning:
> -- * list of "chunk-type"-contents pairs; and
> -- * leftover slop (should be empty in correctly formatted file)
> chunks :: MidiReader [(String,String)]
> chunks = zeroOrMore chunk
> 
> chunk :: MidiReader (String, String)
> chunk = getN 4    >>= \ ty       ->
>         get4      >>= \ size     ->
>         getN size >>= \ contents ->
>         return (ty, contents)
> 
> -- Parse a Midi Header Chunk
> header :: MidiReader (Int, Int, Division)
> header = get2       >>= \ format  -> -- should be 0, 1 or 2
>          get2       >>= \ nTracks -> -- number of tracks
>          getDivision   >>= \ div     ->
>          return (format, nTracks, div)
> 
> getDivision :: MidiReader Division
> getDivision = get1       >>= \ x ->
>               get1       >>= \ y ->
>               if x < 128
>               then return (Ticks (x*256+y))
>               else return (SMPTE (256-x) y)
> 
> data MidiFileEvent = SysexStart String         -- F0
>                    | SysexCont  String         -- F7
>                    | Midi       MidiEvent
>                    | Meta       Int String
>                    deriving Show
> 
> -- Note: Use of running status shows up in the result as a Midi event
> -- with a tag < 128.  To construct the complete event, cons the "tag"
> -- onto the contents of the event and use the tag of the previous
> -- event.  Report an error if the previous event wasn't a channel event.
> 
> track :: MidiReader [(DeltaTime,MidiFileEvent)]
> track = oneOrMore trackEvent
> 
> -- Note: we don't (can't!!!) do anything about running status here
> trackEvent :: MidiReader (DeltaTime, MidiFileEvent)
> trackEvent = getVar     >>= \ time ->
>              event      >>= \ e ->
>              return (time,e)
>    
> event :: MidiReader MidiFileEvent
> event =
>   get1          >>= \ tag ->
>   if tag == 240 then
>     get2        >>= \ size ->
>     getN size   >>= \ contents ->
>     return (SysexStart contents)
>   else if tag == 247 then
>     get2        >>= \ size ->
>     getN size   >>= \ contents ->
>     return (SysexCont contents)
>   else if tag == 255 then
>     get1        >>= \ ty ->   -- should be < 128
>     getVar      >>= \ size ->
>     getN size   >>= \ contents ->
>     return (Meta ty contents)
>   else 
>     midiEvent tag >>= \ e ->
>     return (Midi e)
> 
> midiEvent :: Int -> MidiReader MidiEvent
> midiEvent tag
>   | top4 == 8
>   = get1 >>= \ k ->
>     get1 >>= \ v ->
>     return (NoteOff channel k v)
> 
>   | top4 == 9
>   = get1 >>= \ k ->
>     get1 >>= \ v ->
>     return (NoteOn channel k v)
> 
>   | top4 == 10
>   = get1 >>= \ k ->
>     get1 >>= \ v ->
>     return (PolyAfter channel k v)
> 
>   | top4 == 11
>   = get1 >>= \ c ->
>     get1 >>= \ v ->
>     return (Control channel c v)
> 
>   | top4 == 12
>   = get1 >>= \ p ->
>     return (ProgChange channel p)
> 
>   | top4 == 13
>   = get1 >>= \ v ->
>     return (MonoAfter channel v)
> 
>   | top4 == 14
>   = get1 >>= \ lsb ->
>     get1 >>= \ msb ->
>     return (PitchBend channel (lsb+256*msb))
> 
>   | otherwise
>   = zero
> 
>  where (top4,channel) = tag `divMod` 16 -- oh for some bitops!
> 
> metaEvent :: Int -> MidiReader MetaEvent
> metaEvent  0 = get2     >>= \ x -> return (SequenceNum x)
> metaEvent  1 = getAll   >>= \ s -> return (TextEvent   s)
> metaEvent  2 = getAll   >>= \ s -> return (Copyright   s)
> metaEvent  3 = getAll   >>= \ s -> return (TrackName   s)
> metaEvent  4 = getAll   >>= \ s -> return (InstrName   s)
> metaEvent  5 = getAll   >>= \ s -> return (Lyric       s)
> metaEvent  6 = getAll   >>= \ s -> return (Marker      s)
> metaEvent  7 = getAll   >>= \ s -> return (CuePoint    s)
>                                                        
> metaEvent 32 = get1     >>= \ c -> return (MIDIPrefix  c)
> metaEvent 47 =                     return EndOfTrack   
> metaEvent 81 = get3     >>= \ t -> return (SetTempo    t)
> 
> metaEvent 84 =
>   get1 >>= \ hrs    ->
>   get1 >>= \ mins   ->
>   get1 >>= \ secs   ->
>   get1 >>= \ frames ->
>   get1 >>= \ bits   ->
>   return (SMPTEOffset hrs mins secs frames bits)
> 
> metaEvent 88 =
>   get1 >>= \ n ->
>   get1 >>= \ d ->
>   get1 >>= \ c ->
>   get1 >>= \ b ->
>   return (TimeSig n d c b)
> 
> metaEvent 89 =
>   get1 >>= \ sf ->
>   get1 >>= \ mi ->
>   return (KeySig (toKeyName sf) (mi /= 0))
> 
> metaEvent 127 =
>   getAll >>= \ contents -> return (SequencerSpecific (map fromEnum contents))
> 
> metaEvent _ =
>   zero
> 
> toKeyName sf = [ KeyCf .. ] !! (sf+7 `mod` 15)
>
>
> -- Auxiliary IO functions
> --------------------------------------------------------------------------
> outputDivision :: Division -> MidiWriter ()
> outputDivision (Ticks nticks)      = out2 nticks
> outputDivision (SMPTE mode nticks) = out1 (256-mode) >> 
>                                      out1 nticks
> 
> outputTracks :: [Track] -> MidiWriter ()
> outputTracks trks = mapM_ outputTrack trks
> 
> outputTrack :: Track -> MidiWriter ()
> outputTrack trk = 
>   outChunk "MTrk" (mapM_ (\e -> outputEvent e 0) (delta trk))
> 
> -- convert a track using absolute time to one using delta time
> delta :: Track -> Track
> delta trk = trk' ++ [MetaEvent t EndOfTrack]
>  where
>   (t,trk') = mscanl delta' 0 trk
> 
>   delta' :: Int ->       -- current time
>            MEvent ->    -- event
>            (Int,        -- new time
>             MEvent)     -- event
>   delta' t (MidiEvent dt e) = (dt, MidiEvent (dt-t) e)
>   delta' t (MetaEvent dt e) = (dt, MetaEvent (dt-t) e)
> 
> 
> -- t is start time, it returns end time
> outputEvent :: MEvent -> Int -> MidiWriter Int
> outputEvent (MidiEvent dt mevent) t = 
>   outVar (dt-t)    >> 
>   outputMidiEvent mevent  >>
>   return dt
> outputEvent (MetaEvent dt mevent) t = 
>   outVar (dt-t)    >> 
>   outputMetaEvent mevent  >>
>   return dt
> 
> outputMidiEvent :: MidiEvent -> MidiWriter ()
> outputMidiEvent (NoteOff    c p v)   = outChan 128 c [p,v]
> outputMidiEvent (NoteOn     c p v)   = outChan 144 c [p,v]
> outputMidiEvent (PolyAfter  c p pr)  = outChan 160 c [p,pr]
> outputMidiEvent (Control    c cn cv) = outChan 176 c [cn,cv]
> outputMidiEvent (ProgChange c pn)    = outChan 192 c [pn]
> outputMidiEvent (MonoAfter  c pr)    = outChan 208 c [pr]
> outputMidiEvent (PitchBend  c pb)    = outChan 224 c [lo,hi] -- small-endian!!
>  where (lo,hi) = pb `divMod` 128 
> 
> -- output a channel event
> outChan :: Int -> MidiChannel -> [Int] -> MidiWriter ()
> outChan code chan bytes = 
>   out1 (code+chan-1) >>
>   mapM_ out1 bytes
> 
> 
> outMeta    :: Int -> [Int] -> MidiWriter ()
> outMeta code bytes =
>   out1 255              >>
>   out1 code             >>
>   outVar (length bytes) >>
>   outList bytes
> 
> outMetaStr :: Int -> String -> MidiWriter ()
> outMetaStr code bytes =
>   out1 255              >>
>   out1 code             >>
>   outVar (length bytes) >>
>   outStr bytes
> 
> -- As with outChunk, there are other ways to do this - but
> -- it's not obvious which is best or if performance is a big issue.
> outMetaMW :: Int -> MidiWriter a -> MidiWriter a
> outMetaMW code m =
>   out1 255              >>
>   out1 code             >>
>   outVar (mLength m)    >>
>   m
> 
> outputMetaEvent :: MetaEvent -> MidiWriter ()
> outputMetaEvent (SequenceNum num) = outMetaMW   0 (out2 num)
> outputMetaEvent (TextEvent s)     = outMetaStr  1 s
> outputMetaEvent (Copyright s)     = outMetaStr  2 s
> outputMetaEvent (TrackName s)     = outMetaStr  3 s
> outputMetaEvent (InstrName s)     = outMetaStr  4 s
> outputMetaEvent (Lyric s)         = outMetaStr  5 s
> outputMetaEvent (Marker s)        = outMetaStr  6 s
> outputMetaEvent (CuePoint s)      = outMetaStr  7 s
> outputMetaEvent (MIDIPrefix c)    = outMeta    32 [c]
> outputMetaEvent EndOfTrack        = outMeta    47 []
> 
> outputMetaEvent (SetTempo tp)     = outMetaMW  81 (out3 tp)
> outputMetaEvent (SMPTEOffset hr mn se fr ff) 
>                                   = outMeta    84 [hr,mn,se,fr,ff]
> outputMetaEvent (TimeSig n d c b) = outMeta    88 [n,d,c,b]
> outputMetaEvent (KeySig sf mi)    = outMeta    89 [255 - getKeyNum sf, bool mi]
> outputMetaEvent (SequencerSpecific codes) 
>                                   = outMeta    127 codes
> outputMetaEvent UserEvent         = return ()
> 
> getKeyNum :: KeyName -> Int
> getKeyNum key = index (KeyCf,KeyCs) key - 7
> 
> 
> -- MidiWriters
> ------------------------------------------------------------------------
> -- The midiwriter accumulates a String.
> -- For all the usual reasons, the String is represented by ShowS.
> -- (It might also be worth storing the string length?)
> 
> type MidiWriter a = Output Char a
> 
> out1 :: Int -> MidiWriter ()
> out2 :: Int -> MidiWriter ()
> out3 :: Int -> MidiWriter ()
> out4 :: Int -> MidiWriter ()
> outVar :: Int -> MidiWriter ()
> outList :: [Int] -> MidiWriter ()
> outStr  :: String -> MidiWriter ()
> 
> 
> runM :: MidiWriter a -> String
> runM m = snd (runO m)
> 
> mLength  :: MidiWriter a -> Int
> mLength m = length (runM m)
> 
> -- Note: divMod gives correct rounding for negative numbers
> 
> out1 x    = outO [toEnum x]
> outStr cs = outO cs
> 
> out2 x = out1 (hi `mod` 256) >> out1 lo
>  where (hi,lo) = x `divMod` 256
> 
> out3 x = out2 hi >> out1 lo
>  where (hi,lo) = x `divMod` 256
> 
> out4 x = out1 (x3 `mod` 256) >> out1 x2 >> out1 x1 >> out1 x0
>  where
>   (x0',x0) = x `divMod` 256
>   (x1',x1) = x0' `divMod` 256
>   (x3,x2)  = x1' `divMod` 256
> 
> outList xs = outStr (map toEnum xs)
> 
> -- Numbers of variable size are represented by sequences of 7 bit blocks
> -- tagged (in the top bit) with a bit indicating:
> -- (1) that more data follows; or
> -- (0) that this is the last block.
> 
> outVar n | n >= 0 = let (hi,lo) = n `divMod` 128 in out hi >> out1 lo
>  where
>   out 0 = return ()
>   out n = let (hi,lo) = n `divMod` 128 in  out hi >> out1 (128+lo)
> 
> bool :: Bool -> Int
> bool False = 0
> bool True  = 1
> 
> 
> 
> -- Note: here I've chosen to compute the track twice 
> -- rather than store it.  Other options are worth exploring.
> outChunk :: String -> MidiWriter a -> MidiWriter a
> outChunk tag m | length tag == 4 =
>   outStr tag          >>
>   out4 (mLength m)    >>
>   m
> 
> -- Mapping scan:
> --
> --          x                 xs
> --          |                 |
> --          V                 V
> --        +---+         +----------+
> --  l ->  | f | -> m -> | mscanl f | -> r
> --        +---+         +----------+
> --          |                 |
> --          V                 V
> --          y                 ys
> --
> 
> mscanl :: (a -> b -> (a,c)) -> a -> [b] -> (a,[c])
> mscanl f l [] = (l,[])
> mscanl f l (x:xs) = let (m, y ) = f l x
>                         (r, ys) = mscanl f m xs
>                     in (r, y:ys)
> 
>                      
> test1 mf = writeBinaryFile "foo.mid" (midiFileToString mf)
> test2 mf = map fromEnum $ midiFileToString mf
> test3 mf = showChunks   $ midiFileToString mf
> test4 file = 
>   readBinaryFile file    >>= \ s ->
>   putStr (showChunks s)
> test5 file = 
>   readBinaryFile file    >>= \ s ->
>   putStr (show $ map fromEnum s) 
> 
> -- ToDo: Detect and handle running status
> showChunks mf = showMR chunks (unlinesS . map pp) mf ""
>  where
>   pp :: (String, String) -> ShowS
>   pp ("MThd",contents) = 
>     showString "Header: " .
>     showMR header shows contents
>   pp ("MTrk",contents) =
>     showString "Track:\n" . 
>     showMR track (unlinesS . map showTrackEvent) contents
>   pp (ty,contents) = 
>     showString "Chunk: " . 
>     showString ty . 
>     showString " " . 
>     shows (map fromEnum contents) .
>     showString "\n"
> 
> showTrackEvent :: (DeltaTime, MidiFileEvent) -> ShowS
> showTrackEvent (t,e) = 
>   rightS 10 (shows t) . showString " : " . showEvent e
> 
> showEvent :: MidiFileEvent -> ShowS
> showEvent (Midi e) = 
>   showString "MidiEvent " . 
>   shows e
> showEvent (Meta tag contents) = 
>   showString "MetaEvent(" . shows tag . showString ") " .
>   showMR (metaEvent tag) shows contents
> showEvent (SysexStart s) = 
>   showString "SysexStart " . concatS (map (shows.fromEnum) s)
> showEvent (SysexCont s) =
>   showString "SysexCont "  . concatS (map (shows.fromEnum) s)
> 
> showMR :: MidiReader a -> (a->ShowS) -> String -> ShowS
> showMR m pp s = 
>   case runP m s of 
>   Nothing       -> showString "Parse failed: " . shows (map fromEnum s)
>   Just (a,[])   -> pp a 
>   Just (a,junk) -> pp a . showString "Junk: " . shows (map fromEnum junk)

\end{verbatim} 
