/****************************************************************************
** $Id: qbitmap.cpp,v 2.3 1998/07/03 00:09:30 hanord Exp $
**
** Implementation of QBitmap class
**
** Created : 941020
**
** Copyright (C) 1992-1998 Troll Tech AS.  All rights reserved.
**
** This file is part of Qt Free Edition, version 1.40.
**
** See the file LICENSE included in the distribution for the usage
** and distribution terms, or http://www.troll.no/free-license.html.
**
** IMPORTANT NOTE: You may NOT copy this file or any part of it into
** your own programs or libraries.
**
** Please see http://www.troll.no/pricing.html for information about 
** Qt Professional Edition, which is this same library but with a
** license which allows creation of commercial/proprietary software.
**
*****************************************************************************/

#include "qbitmap.h"
#include "qimage.h"

/*!
  \class QBitmap qbitmap.h
  \brief The QBitmap class provides monochrome (1 bit depth) pixmaps.

  \ingroup drawing
  \ingroup shared

  The QBitmap class is a monochrome off-screen paint device, used mainly
  for creating custom \link QCursor mouse cursors \endlink or \link QBrush
  brushes\endlink.

  A QBitmap is a QPixmap with the \link QPixmap::depth() depth\endlink 1.
  If a pixmap with a depth greater than 1 is assigned to a bitmap, the
  bitmap will be automatically dithered.  A QBitmap is guaranteed to always
  have the depth 1, unless it is a \link QPixmap::isNull() null\endlink
  bitmap (has depth 0).

  When drawing in a QBitmap (or QPixmap with depth 1), we recommend using
  the global \c color0 and \c color1 QColor objects.  Painting with \c
  color0 sets the bitmap bits to 0, and painting with \c color1 sets the
  bits to 1.  For a bitmap, 0-bits indicate background (or white) and
  1-bits indicate foreground (or black).  Using the \c black and \c white
  QColor objects make no sense, because the \link QColor::pixel() pixel
  value\endlink is not necessarily 0 for black and 1 for white.

  Just like the QPixmap class, QBitmap is optimized by the use of \link
  shclass.html implicit sharing\endlink, so it is very efficient to pass
  QBitmap objects as arguments.

  \sa QPixmap, QPainter::drawPixmap(), bitBlt(), \link shclass.html Shared
  Classes\endlink
*/

/*!
  Constructs a null bitmap.
  \sa QPixmap::isNull()
*/

QBitmap::QBitmap()
{
    data->bitmap = TRUE;
}

/*!
  Constructs a bitmap with \e w width and \e h height.

  The contents of the bitmap is uninitialized if \e clear is FALSE, otherwise
  it is filled with pixel value 0 (the QColor \c color0).
*/

QBitmap::QBitmap( int w, int h, bool clear )
    : QPixmap( w, h, 1 )
{
    data->bitmap = TRUE;
    if ( clear )
	fill( color0 );
}

/*!
  Overloaded constructor; takes a QSize parameter instead of \e (w,h).
*/

QBitmap::QBitmap( const QSize &size, bool clear )
    : QPixmap( size, 1 )
{
    data->bitmap = TRUE;
    if ( clear )
	fill( color0 );
}

/*!
  Constructs a bitmap with \e w width and \e h height and sets the contents
  to \e bits.

  The \e isXbitmap should be TRUE if \e bits was generated by the
  X11 bitmap program.  The X bitmap bit order is little endian.
  The QImage documentation discusses bit order of monochrome images.

  Example (creates an arrow bitmap):
  \code
    uchar arrow_bits[] = { 0x3f, 0x1f, 0x0f, 0x1f, 0x3b, 0x71, 0xe0, 0xc0 };
    QBitmap bm( 8, 8, arrow_bits, TRUE );
  \endcode
*/

QBitmap::QBitmap( int w, int h, const uchar *bits, bool isXbitmap )
    : QPixmap( w, h, bits, isXbitmap )
{
    data->bitmap = TRUE;
}

/*!
  Overloaded constructor; takes a QSize parameter instead of \e (w,h).
*/

QBitmap::QBitmap( const QSize &size, const uchar *bits, bool isXbitmap )
    : QPixmap( size.width(), size.height(), bits, isXbitmap )
{
    data->bitmap = TRUE;
}

/*!
  Constructs a bitmap which is a copy of \e bitmap.
*/

QBitmap::QBitmap( const QBitmap &bitmap )
    : QPixmap( bitmap )
{
}

/*!  
  Constructs a pixmap from the file \e fileName. If the file does not
  exist, or is of an unknown format, the pixmap becomes a null pixmap.

  The parameters are passed on to QPixmap::load().

  \sa QPixmap::isNull(), QPixmap::load(), QPixmap::loadFromData(),
  QPixmap::save(), QPixmap::imageFormat()
*/

QBitmap::QBitmap( const char *fileName, const char *format )
    : QPixmap() // Will set bitmap to null bitmap, explicit call for clarity
{
    data->bitmap = TRUE;
    load( fileName, format, Mono );
}


/*!
  Assigns the bitmap \e bitmap to this bitmap and returns a reference to this
  bitmap.
*/

QBitmap &QBitmap::operator=( const QBitmap &bitmap )
{
    QPixmap::operator=(bitmap);
#if defined(CHECK_STATE)
    ASSERT( data->bitmap );
#endif
    return *this;
}


/*!
  Assigns the pixmap \e pixmap to this bitmap and returns a reference to this
  bitmap.

  Dithering will be performed if the pixmap has a
  \link QPixmap::depth() depth\endlink greater than 1.
*/

QBitmap &QBitmap::operator=( const QPixmap &pixmap )
{
    if ( pixmap.isNull() ) {			// a null pixmap
	QBitmap bm;
	bm.data->optim = data->optim;
	QBitmap::operator=(bm);
    } else if ( pixmap.depth() == 1 ) {		// 1-bit pixmap
	if ( pixmap.isQBitmap() ) {		// another QBitmap
	    QPixmap::operator=(pixmap);		// shallow assignment
	} else {				// not a QBitmap, but 1-bit
	    QBitmap bm( pixmap.size() );
	    bitBlt( &bm, 0,0, &pixmap, 0,0,pixmap.width(),pixmap.height() );
	    bm.data->optim = data->optim;
	    QBitmap::operator=(bm);
	}
    } else {					// n-bit depth pixmap
	QImage image;
	image = pixmap;				// convert pixmap to image
	*this = image;				// will dither image
    }
    return *this;
}

/*!
  Converts the image \e image to a bitmap that is assigned to this bitmap.
  Returns a reference to the bitmap.

  Dithering will be performed if the image has a
  \link QImage::depth() depth\endlink greater than 1.
*/

QBitmap &QBitmap::operator=( const QImage &image )
{
    convertFromImage( image );
    return *this;
}


/*!
  Transforms the bitmap using \e matrix, and returns the transformed
  bitmap.

  This function does exactly the same as QPixmap::xForm(), except that
  it returns a QBitmap instead of a QPixmap.

  \sa QPixmap::xForm()
*/

QBitmap QBitmap::xForm( const QWMatrix &matrix ) const
{
    QPixmap pm = QPixmap::xForm( matrix );
    QBitmap bm;
    bm = pm;
    return bm;
}
